#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <signal.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#ifdef HAVE_EDV2
# include <endeavour2.h>
#endif

#include "../include/disk.h"

#include "guiutils.h"
#include "animicon.h"
#include "cdialog.h"
#include "pdialog.h"
#include "fb.h"
#include "hview.h"
#include "avscanop.h"
#include "obj.h"
#include "win.h"    
#include "wincb.h"
#include "winlist.h"
#include "winresultsfio.h"
#include "windb.h"
#include "winopcb.h"
#include "core.h"
#include "help.h"
#include "cfglist.h"
#include "config.h"

#include "images/icon_trash_32x32.xpm"
#include "images/icon_avscan_32x32.xpm"
#include "images/icon_biohazard_32x32.xpm"


static const gchar *GET_NAME_FROM_PATH(const gchar *path);
static gchar *FORMATED_NUMBER_STRING(const gulong i);
static gchar *TIME_LAPSED_STRING(const gulong dt);
static gboolean WinQueryOverwrite(
	core_struct *core, const gchar *path, GtkWidget *toplevel
);
static void WinCListSelectAll(GtkCList *clist);
static void WinCListUnselectAll(GtkCList *clist);
static void WinCListInvertSelection(GtkCList *clist);

void WinCloseCB(GtkWidget *widget, gpointer data);
void WinExitCB(GtkWidget *widget, gpointer data);

void WinAddCB(GtkWidget *widget, gpointer data);
static gchar *WinPDialogBrowseLocationCB(
	gpointer p, gpointer data, gint prompt_num
);
static gint WinCheckObjectValidValues(
	win_struct *win, obj_struct *obj,
	const gboolean verbose
);
static void WinEdit(win_struct *win);
void WinEditCB(GtkWidget *widget, gpointer data);
void WinRemoveCB(GtkWidget *widget, gpointer data);
void WinCutCB(GtkWidget *widget, gpointer data);
void WinCopyCB(GtkWidget *widget, gpointer data);
void WinPasteCB(GtkWidget *widget, gpointer data);
void WinSelectAllCB(GtkWidget *widget, gpointer data);
void WinUnselectAllCB(GtkWidget *widget, gpointer data);
void WinInvertSelectionCB(GtkWidget *widget, gpointer data);

void WinStartCB(GtkWidget *widget, gpointer data);
void WinStopCB(GtkWidget *widget, gpointer data);
void WinPauseResumeCB(GtkWidget *widget, gpointer data);
void WinLocationCB(GtkWidget *widget, gpointer data);
void WinRecursiveCB(GtkWidget *widget, gpointer data);
void WinExecutablesOnlyCB(GtkWidget *widget, gpointer data);
void WinIgnoreLinksCB(GtkWidget *widget, gpointer data);
void WinScannerSettingsCB(GtkWidget *widget, gpointer data);

#ifdef HAVE_EDV2
static gint WinEDVRecycleProgressCB(
	gpointer data, const gulong i, const gulong m
);
#endif
void WinResultsRecycleCB(GtkWidget *widget, gpointer data);
void WinResultsMoveCB(GtkWidget *widget, gpointer data);
void WinResultsSelectAllCB(GtkWidget *widget, gpointer data);
void WinResultsUnselectAllCB(GtkWidget *widget, gpointer data);
void WinResultsInvertSelectionCB(GtkWidget *widget, gpointer data);
void WinResultsReportCB(GtkWidget *widget, gpointer data);
void WinResultsOpenCB(GtkWidget *widget, gpointer data);
void WinResultsSaveAsCB(GtkWidget *widget, gpointer data);
void WinResultsClearCB(GtkWidget *widget, gpointer data);

void WinDBRefreshCB(GtkWidget *widget, gpointer data);
void WinDBPatternDetailsCB(GtkWidget *widget, gpointer data);
void WinDBLocationCB(GtkWidget *widget, gpointer data);
void WinDBUpdateNetCB(GtkWidget *widget, gpointer data);

void WinRefreshCB(GtkWidget *widget, gpointer data);
void WinViewScanPageCB(GtkWidget *widget, gpointer data);
void WinViewResultsPageCB(GtkWidget *widget, gpointer data);
void WinViewDBPageCB(GtkWidget *widget, gpointer data);
void WinViewToolBarCB(GtkWidget *widget, gpointer data);
void WinViewStatusBarCB(GtkWidget *widget, gpointer data);

void WinHelpContentsCB(GtkWidget *widget, gpointer data);
void WinHelpScanningCB(GtkWidget *widget, gpointer data);
void WinHelpUpdatingCB(GtkWidget *widget, gpointer data);
void WinHelpAboutCB(GtkWidget *widget, gpointer data);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Returns the name portion of the path (if any).
 */
static const gchar *GET_NAME_FROM_PATH(const gchar *path)
{
	const gchar *s;

	if(STRISEMPTY(path))
	    return(path);

	s = (const gchar *)strrchr((const char *)path, G_DIR_SEPARATOR);
	return((s != NULL) ? (s + 1) : path);
}


/*
 *      Returns a dynamically allocated string describing the
 *      specified number.
 */
static gchar *FORMATED_NUMBER_STRING(const gulong i)
{
        gint comma_cnt, slen;
        gchar *ss, *ss_ptr, *ts, *ts_ptr;

        ss = g_strdup_printf("%ld", i);
        slen = STRLEN(ss);

        /* 3 digits or less? (no commas needed) */
        if((slen <= 3) || (ss == NULL))
            return(ss);

        ts_ptr = ts = (gchar *)g_malloc(80);
        if(ts == NULL)                    
            return(ss);

        ss_ptr = ss;

        /* Initialize comma counter */
        comma_cnt = slen % 3;
        if(comma_cnt <= 0)
            comma_cnt = 3;

        /* Iterate through size string until end is reached */
        while(*ss_ptr != '\0')
        {
            /* Reset comma counter and put in a comma? */
            if(comma_cnt <= 0)
            {
                *ts_ptr = ',';
                ts_ptr++;
                comma_cnt = 3;
            }

            *ts_ptr = *ss_ptr;
            ts_ptr++;
            ss_ptr++;
            comma_cnt--;
        }

        /* Null terminate return string */
        *ts_ptr = '\0';

        g_free(ss);

        return(ts);
}

/*
 *      Returns a dynamically allocated string describing the time
 *      lapsed specified by dt.
 */
static gchar *TIME_LAPSED_STRING(const gulong dt)
{
        gchar *s;

        /* Less than one second? */
        if(dt < 1)
        {
            s = g_strdup("less than one second");
        }
        /* Less than one minute? */
        else if(dt < (1 * 60))
        {
            const gulong ct = MAX(dt / 1, 1);
            s = g_strdup_printf(
                "%ld %s",
                ct, ((ct == 1l) ? "second" : "seconds")
            );
        }
        /* Less than one hour? */
        else if(dt < (60 * 60))
        {
            const gulong        ct = MAX(dt / 60, 1),
                                ct2 = MAX(dt / 1, 1) % 60;
            s = g_strdup_printf(
                "%ld %s %ld %s",
                ct, ((ct == 1l) ? "minute" : "minutes"),
                ct2, ((ct2 == 1l) ? "second" : "seconds")
            );
        }     
        /* Less than one day? */
        else if(dt < (24 * 60 * 60))
        {
            const gulong        ct = MAX(dt / 60 / 60, 1),
                                ct2 = MAX(dt / 60, 1) % 60;
            s = g_strdup_printf(
                "%ld %s %ld %s",
                ct, ((ct == 1l) ? "hour" : "hours"),
                ct2, ((ct2 == 1l) ? "minute" : "minutes")
            );
        }     
        /* Less than one week? */
        else if(dt < (7 * 24 * 60 * 60))
        {
            const gulong        ct = MAX(dt / 60 / 60 / 24, 1),
                                ct2 = MAX(dt / 60 / 60, 1) % 24;
            s = g_strdup_printf(
                "%ld %s %ld %s",
                ct, ((ct == 1l) ? "day" : "days"),
                ct2, ((ct2 == 1l) ? "hour" : "hours")
            );
        }
        /* Less than one month (30 days)? */
        else if(dt < (30 * 24 * 60 * 60))
        {                       
            const gulong        ct = MAX(dt / 60 / 60 / 24 / 7, 1),
                                ct2 = MAX(dt / 60 / 60 / 24, 1) % 7;    
            s = g_strdup_printf(
                "%ld %s %ld %s",
                ct, ((ct == 1l) ? "week" : "weeks"),
                ct2, ((ct2 == 1l) ? "day" : "days")
            );
        }     
        /* Less than 6 months ago? */
#if 0
        else if(dt < (6 * 30 * 24 * 60 * 60))
#else
        else
#endif
        {
            const gulong        ct = MAX(dt / 60 / 60 / 24 / 30, 1),
                                ct2 = MAX(dt / 60 / 60 / 24, 1) % 30;
            s = g_strdup_printf(
                "%ld %s %ld %s",
                ct, ((ct == 1l) ? "month" : "months"),
                ct2, ((ct2 == 1l) ? "day" : "days")
            );
        }

        return(s);
}


/*
 *	Checks if path exists and if it does then it queries
 *	the user for overwrite.
 *
 *	Returns TRUE to indicate continue or FALSE to indicate do not
 * 	continue.
 */
static gboolean WinQueryOverwrite(
	core_struct *core, const gchar *path, GtkWidget *toplevel  
)
{
	gint response;
	gchar *msg;
	struct stat stat_buf;

	/* Path not specified or object does not exist? */
	if(!STRISEMPTY(path) ? stat(path, &stat_buf) : TRUE)
	    return(TRUE);

	/* Confirm overwrite */
	msg = g_strdup_printf(
	    "Overwrite existing file:\n\n    %s\n",
	    path
	);
#ifdef HAVE_EDV2
	EDVPlaySoundWarning(core->edv_ctx);
#endif
	CDialogSetTransientFor(toplevel);
	response = CDialogGetResponse(
	    "Confirm Overwrite",
	    msg,
	    NULL,
	    CDIALOG_ICON_WARNING,
	    CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
	    CDIALOG_BTNFLAG_NO
	);
	CDialogSetTransientFor(NULL);
	g_free(msg);
	switch(response)
	{
	  case CDIALOG_RESPONSE_YES:
	  case CDIALOG_RESPONSE_YES_TO_ALL:
	  case CDIALOG_RESPONSE_OK:
	    return(TRUE);
	    break;
	  default:
	    return(FALSE);
	    break;
	}
}

/*
 *	Selects all rows on the specified GtkCList.
 */
static void WinCListSelectAll(GtkCList *clist)
{
	if(clist == NULL)
	    return;

	gtk_clist_freeze(clist);
	gtk_clist_select_all(clist);
	gtk_clist_thaw(clist);
}

/*
 *	Unselects all rows on the specified GtkCList.
 */
static void WinCListUnselectAll(GtkCList *clist)
{
	if(clist == NULL)
	    return;

	gtk_clist_freeze(clist);
	gtk_clist_unselect_all(clist);
	gtk_clist_thaw(clist);
}

/*
 *	Inverts selection on the specified GtkCList.
 */
static void WinCListInvertSelection(GtkCList *clist)
{
	gint i;
	GList *glist;

	if(clist == NULL)
	    return;

	glist = (clist->selection != NULL) ?
	    g_list_copy(clist->selection) : NULL;

	gtk_clist_freeze(clist);
	for(i = 0; i < clist->rows; i++)
	{
	    if(g_list_find(glist, (gpointer)i) != NULL)
		gtk_clist_unselect_row(clist, i, 0);
	    else
		gtk_clist_select_row(clist, i, 0);
	}
	gtk_clist_thaw(clist);

	g_list_free(glist);
}


/*
 *	Close.
 */
void WinCloseCB(GtkWidget *widget, gpointer data)
{
	gchar *path;
	GtkWidget *toplevel;
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	toplevel = win->toplevel;
	core = win->core;

	WinSetBusy(win, TRUE);
	win->freeze_count++;

	/* Check if there are any active processes in which case we
	 * should not close
	 */
	if(win->scan_context->toid != 0)
	{
#ifdef HAVE_EDV2
	    EDVPlaySoundQuestion(core->edv_ctx);
#endif
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Scan Process Active",
"This window may not be close because a scan process\n\
is currently active.\n\
\n\
You need to stop the process first, by pressing on the\n\
\"Stop\" button, before closing.",
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);

	    win->freeze_count--;
	    WinSetBusy(win, FALSE);
	    return;
	}

	if(win->db_update_net_toid != 0)
	{
#ifdef HAVE_EDV2
	    EDVPlaySoundQuestion(core->edv_ctx);
#endif
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Virus Update Process Active",
"This window may not be close because a virus database\n\
update process is currently active.\n\
\n\
You need to stop the process first, by pressing on the\n\
\"Stop\" button, before closing.",
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);

	    win->freeze_count--;
	    WinSetBusy(win, FALSE);
	    return;
	}

	/* Save the Win position */
	WinSavePositions(win);

	/* Unmap the Win */
	WinUnmap(win);

	/* Save the scan results log to the default last log */
	path = g_strconcat(                 
            core->prog_data_dir,
            G_DIR_SEPARATOR_S,
            AVSCAN_LAST_SCAN_RESULTS_LOG_FILE,
            NULL
        );
	WinResultsSave(win, path, FALSE);
	g_free(path);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}

/*
 *	Exit.
 */
void WinExitCB(GtkWidget *widget, gpointer data)
{
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	core = win->core;
	if(core == NULL)
	    return;

	WinSetBusy(win, TRUE);

	/* Mark the core's close_all_windows to close all windows during
	 * the core's main management check
	 */
	core->close_all_windows = TRUE;

	WinSavePositions(win);
	WinUpdate(win);

	WinSetBusy(win, FALSE);
}


/*
 *	Add.
 *
 *	Adds a new scan item.
 */
void WinAddCB(GtkWidget *widget, gpointer data)
{
	gint row;
	GList *glist;
	GtkCList *clist;
	obj_struct *obj;
	win_struct *win = WIN(data);
	if((win == NULL) || PDialogIsQuery())
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	clist = GTK_CLIST(win->scan_clist);

	/* Get last selected row (if any) */
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;

	WinSetBusy(win, TRUE);

	/* Create a new object */
	obj = ObjNew();
	if(obj != NULL)
	{
	    gint new_row;

	    /* Set the default values for the new object */
	    obj->name = STRDUP("New Scan");
	    obj->path = STRDUP(g_getenv("HOME"));
	    obj->options = 0;
	    obj->last_runned = 0l;

	    gtk_clist_freeze(clist);

	    /* Insert or append a new row on the scan list and
	     * transfer the new object to it
	     */
	    if(row > -1)
		new_row = WinScanListInsert(win, row, obj);
	    else
		new_row = WinScanListAppend(win, obj);

	    gtk_clist_thaw(clist);

	    if(new_row > -1)
	    {
		/* Select the new row */
		gtk_clist_freeze(clist);
		gtk_clist_unselect_all(clist);
		gtk_clist_select_row(clist, new_row, 0);
		gtk_clist_thaw(clist);

		/* Edit the new object */
		WinEditCB(NULL, win);
	    }
	}

	WinSetBusy(win, FALSE);
}


/*
 *	Prompt Dialog browse location callback.
 */
static gchar *WinPDialogBrowseLocationCB(
	gpointer p, gpointer data, gint prompt_num
)
{
	static gchar path[PATH_MAX];
	gboolean status;
	GtkWidget *toplevel;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	gint nftypes = 0;
	gchar **paths_list = NULL;
	gint npaths = 0;
	win_struct *win = WIN(data);
	if((win == NULL) || FileBrowserIsQuery())
	    return(NULL);

	*path = '\0';

	toplevel = PDialogGetToplevel();

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);                    

	/* Query the user for the new location */
	FileBrowserSetTransientFor(toplevel);
	status = FileBrowserGetResponse(
	    "Select Location",
	    "Select", "Cancel",
	    PDialogGetPromptValue(prompt_num),
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(status)
	{
	    const gchar *s = (npaths > 0) ?
		paths_list[0] : NULL;
	    if(!STRISEMPTY(s))
	    {
		strncpy(path, s, sizeof(path));
		path[sizeof(path) - 1] = '\0';
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	return(STRISEMPTY(path) ? NULL : path);
}

/*
 *	Checks if the object has valid values.
 */
static gint WinCheckObjectValidValues(
	win_struct *win, obj_struct *obj,
	const gboolean verbose
)
{
	gint status;
	GtkWidget *toplevel;
	core_struct *core;

	if((win == NULL) || (obj == NULL))
	    return(-2);

	status = 0;
	toplevel = win->toplevel;
	core = win->core;

	if(STRISEMPTY(obj->name))
	{

	    status = -2;
	}
	else
	{


	}

	if(STRISEMPTY(obj->path))
	{

	    status = -2;
	}
	else
	{
	    /* Check if each path is valid */
	    gchar **strv = g_strsplit(obj->path, ",", -1);
	    if(strv != NULL)
	    {
		struct stat stat_buf;
		gint i;
		const gchar *path;

		for(i = 0; strv[i] != NULL; i++)
		{
		    path = strv[i];

		    /* Does this path have a problem? */
		    if(stat((const char *)path, &stat_buf))
		    {
			const gint error_code = (gint)errno;
			if(verbose)
			{
			    gchar *msg = g_strdup_printf(
				"%s:\n\n    %s",
				g_strerror(error_code),
				path
			    );
#ifdef HAVE_EDV2   
			    EDVPlaySoundWarning(core->edv_ctx);
#endif
                            CDialogSetTransientFor(toplevel);
                            CDialogGetResponse(
				"Scan Item Warning",
				msg,
				NULL,
                                CDIALOG_ICON_WARNING,
                                CDIALOG_BTNFLAG_OK,
                                CDIALOG_BTNFLAG_OK
                            );
			    g_free(msg);
			    CDialogSetTransientFor(NULL);
			}
			if(status == 0)
			    status = -1;
			break;
		    }
		}

		g_strfreev(strv);
	    }
	}

	return(status);
}

/*
 *	Edit.
 *
 *	Edits the selected scan item.
 */
static void WinEdit(win_struct *win)
{
	gchar **strv;
	gint strc, row;
	GList *glist;
	GtkWidget *toplevel;
	GtkCList *clist;
	obj_struct *obj;
	core_struct *core;

	if((win == NULL) || PDialogIsQuery())
	    return;

	toplevel = win->toplevel;
	clist = GTK_CLIST(win->scan_clist);
	core = CORE(win->core);

	/* Get last selected row */
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;
	if((row < 0) || (row >= clist->rows))
	    return;

	/* Get object */
	obj = OBJ(gtk_clist_get_row_data(clist, row));
	if(obj == NULL)
	    return;

	WinSetBusy(win, TRUE);

	/* Set up the Prompt Dialog */
	PDialogDeleteAllPrompts();
	PDialogSetSize(350, -1);
	PDialogAddPrompt(NULL, "Name:", obj->name);
	PDialogAddPromptWithBrowse(
	    NULL, "Location:", obj->path,
	    win, WinPDialogBrowseLocationCB
	);
	PDialogSetPromptCompletePath(-1);
	PDialogSetPromptTip(
	    -1,
"Enter a comma-separated list of full paths to the file(s)\
 and/or directory(ies) to scan"
	);
	PDialogAddPromptToggle(
	    NULL, "Recursive", obj->options & AVSCAN_OPT_RECURSIVE
	);
	PDialogSetPromptTip(
	    -1,
"Check this to scan recursively into each sub directory (if the\
 specified location is a directory)"
	);
	PDialogAddPromptToggle(
	    NULL, "Executables Only", obj->options & AVSCAN_OPT_EXECUTABLES_ONLY
	);
	PDialogSetPromptTip(
	    -1,
"Check this to only scan objects who's permissions are set executable"
	);
	PDialogAddPromptToggle(
	    NULL, "Ignore Links", obj->options & AVSCAN_OPT_IGNORE_LINKS
	);
	PDialogSetPromptTip(
	    -1,
"Check this to exclude links from the scan"
	);
	if(TRUE)
	{
#ifdef HAVE_EDV2
	    const gulong t = obj->last_runned;
	    const gchar *s = (t > 0l) ?
		EDVDateString(core->edv_ctx, t) : "Unknown";
#else
	    time_t t = (time_t)((obj != NULL) ? obj->last_runned : 0l);
	    const gchar *s = (t > 0l) ? ctime(&t) : "Unknown";
#endif
	    gchar *buf = g_strdup_printf(
		"Last Scanned: %s", s
	    );
	    PDialogAddPromptLabel(buf);
	    g_free(buf);
	}
	/* Query user */
	gtk_window_set_transient_for(
	    GTK_WINDOW(PDialogGetToplevel()), GTK_WINDOW(toplevel)
	);
/*	PDialogSetTransientFor(toplevel); */
	strv = PDialogGetResponse(
	    "Scan Item Properties",
	    NULL, NULL,
	    PDIALOG_ICON_FILE_PROPERTIES,
	    "Set", "Cancel",
	    PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
	    PDIALOG_BTNFLAG_SUBMIT,
	    &strc
	);
	gtk_window_set_transient_for(
	    GTK_WINDOW(PDialogGetToplevel()), NULL
	);
/*	PDialogSetTransientFor(NULL); */
	if((strv != NULL) && (strc >= 5))
	{
	    g_free(obj->name);
	    obj->name = STRDUP(strv[0]);

	    g_free(obj->path);
	    obj->path = STRDUP(strv[1]);

	    if(ATOI(strv[2]))
		obj->options |= AVSCAN_OPT_RECURSIVE;
	    else
		obj->options &= ~AVSCAN_OPT_RECURSIVE;

	    if(ATOI(strv[3]))
		obj->options |= AVSCAN_OPT_EXECUTABLES_ONLY;
	    else
		obj->options &= ~AVSCAN_OPT_EXECUTABLES_ONLY;

	    if(ATOI(strv[4]))
		obj->options |= AVSCAN_OPT_IGNORE_LINKS;
	    else
		obj->options &= ~AVSCAN_OPT_IGNORE_LINKS;

	    WinScanListUpdateRow(win, row, obj);
	    WinScanSettingsUpdate(win, obj);
	    WinUpdate(win);

	    /* Check for and report any problems with this
	     * scan item
	     */
	    WinCheckObjectValidValues(
		win, obj,
		TRUE				/* Verbose */
	    );
	}

	PDialogDeleteAllPrompts();

	WinSetBusy(win, FALSE);
}

/*
 *	Edit.
 *
 *	Edits the selected scan item.
 */
void WinEditCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if((win == NULL) || PDialogIsQuery())
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	if(win->freeze_count > 0)
	    return;

	win->freeze_count++;

	WinEdit(win);

	win->freeze_count--;
}

/*
 *	Remove.
 *
 *	Removes the selected scan item(s).
 */
void WinRemoveCB(GtkWidget *widget, gpointer data)
{
	gint row;
	GList *glist, *obj_list;
	GtkWidget *toplevel;
	GtkCList *clist;
	obj_struct *obj;
	core_struct *core;
	win_struct *win = WIN(data);
	if((win == NULL) || CDialogIsQuery())
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinSetBusy(win, TRUE);

	toplevel = win->toplevel;
	clist = GTK_CLIST(win->scan_clist);
	core = CORE(win->core);

	/* Iterate through each selected row and generate objects list */
	obj_list = NULL;                                      
	for(glist = clist->selection; glist != NULL; glist = g_list_next(glist))
	{
	    row = (gint)glist->data;
	    obj = gtk_clist_get_row_data(clist, row);
	    if(obj == NULL)
		continue;

	    obj_list = g_list_append(obj_list, obj);
	}     

	/* Confirm delete */
	if(obj_list != NULL)
	{
	    gint response;
	    const gint nobjs = g_list_length(obj_list);
	    obj_struct *obj = (nobjs == 1) ? OBJ(obj_list->data) : NULL;
	    gchar *msg;

	    /* Format confirm delete message */
	    if((obj != NULL) ? !STRISEMPTY(obj->name) : FALSE)
		msg = g_strdup_printf(
		    "Delete scan item \"%s\"?",
		    obj->name
		);
	    else
		msg = g_strdup_printf(
		    "Delete %i selected scan items?",
		    nobjs
		);

	    /* Query user to confirm delete */
#ifdef HAVE_EDV2
	    EDVPlaySoundQuestion(core->edv_ctx);
#endif
	    CDialogSetTransientFor(toplevel);
	    response = CDialogGetResponse(
		"Confirm Delete",
		msg,
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
		CDIALOG_BTNFLAG_NO
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(msg);
	    if(response != CDIALOG_RESPONSE_YES)
	    {
		g_list_free(obj_list);
		WinSetBusy(win, FALSE);
		return;
	    }
	}

	/* Delete the rows who's data is specified in the obj_list */
	gtk_clist_freeze(clist);
	for(glist = obj_list; glist != NULL; glist = g_list_next(glist))
	{
	    row = gtk_clist_find_row_from_data(clist, glist->data);
	    if(row > -1)
		gtk_clist_remove(clist, row);
	}
	gtk_clist_thaw(clist);

	/* Delete the objects list */
	g_list_free(obj_list);

	WinUpdate(win);

	WinSetBusy(win, FALSE);
}

/*
 *	Cut.
 *
 *	Cuts the selected scan item(s) to the clipboard.
 */
void WinCutCB(GtkWidget *widget, gpointer data)
{
	guint8 *buf;
	gint row, buf_len;
	GList *glist, *obj_list;
	GtkCList *clist;
	obj_struct *obj;
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	clist = GTK_CLIST(win->scan_clist);
	core = win->core;

	/* Begin copying */

	/* Iterate through each selected row and generate the buffer
	 * and objects list
	 */
	obj_list = NULL;
	buf = NULL;
	buf_len = 0;
	for(glist = clist->selection; glist != NULL; glist = g_list_next(glist))
	{
	    row = (gint)glist->data;
	    obj = OBJ(gtk_clist_get_row_data(clist, row));
	    if(obj == NULL)
		continue;

	    obj_list = g_list_append(obj_list, obj);
	    buf = ObjDDEBufferAppend(buf, &buf_len, obj);
	}

	/* Transfer the data to the clipboard */
	if(buf != NULL)
	    GUIDDESetDirect(
		GTK_WIDGET(clist),
		GDK_SELECTION_PRIMARY,
		GDK_CURRENT_TIME,
		core->target_scan_item_atom,
		buf, buf_len
	    );


	/* Begin deleting */

	/* Delete the rows who's data is specified in the obj_list */
	gtk_clist_freeze(clist);
	for(glist = obj_list; glist != NULL; glist = g_list_next(glist))
	{
	    row = gtk_clist_find_row_from_data(clist, glist->data);
	    if(row > -1)
		gtk_clist_remove(clist, row);
	}
	gtk_clist_thaw(clist);

	/* Delete the objects list */
	g_list_free(obj_list);

	WinUpdate(win);
	 
	win->freeze_count--;
	WinSetBusy(win, FALSE);
}

/*
 *	Copy.
 *
 *	Coppies the selected scan item(s) to the clipboard.
 */
void WinCopyCB(GtkWidget *widget, gpointer data)
{
	guint8 *buf;
	gint row, buf_len;
	GList *glist;
	GtkCList *clist;
	const obj_struct *obj;
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	clist = GTK_CLIST(win->scan_clist);
	core = win->core;

	/* Iterate through each selected row and generate the buffer */
	buf = NULL;
	buf_len = 0;
	for(glist = clist->selection; glist != NULL; glist = g_list_next(glist))
	{
	    row = (gint)glist->data;
	    obj = OBJ(gtk_clist_get_row_data(clist, row));
	    if(obj == NULL)
		continue;

	    buf = ObjDDEBufferAppend(buf, &buf_len, obj);
	}

	/* Transfer the data to the clipboard */
	if(buf != NULL)
	    GUIDDESetDirect(
		GTK_WIDGET(clist),
		GDK_SELECTION_PRIMARY,
		GDK_CURRENT_TIME,
		core->target_scan_item_atom,
		buf, buf_len
	    );

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}

/*
 *	Paste.
 *
 *	Paste the scan items from the clipboard.
 */
void WinPasteCB(GtkWidget *widget, gpointer data)
{
	gboolean got_data;
	GList *obj_list;
	GtkWidget *toplevel;
	GtkCList *clist;
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	toplevel = win->toplevel;
	clist = GTK_CLIST(win->scan_clist);
	core = win->core;

	/* Get the data from the clipboard and convert it into a
	 * list of objects
	 *
	 * First attempt to get the data as AVSCAN_DND_TARGET_SCAN_ITEM,
	 * then as GUI_TARGET_NAME_STRING.
	 */
	got_data = FALSE;
	obj_list = NULL;

	if(got_data == FALSE)
        {
	    gint buf_len;
	    guint8 *buf = GUIDDEGet(
		GTK_WIDGET(clist),
		GDK_SELECTION_PRIMARY,
		GDK_CURRENT_TIME,
		core->target_scan_item_atom,
		&buf_len
	    );
	    if(buf != NULL)
	    {
		got_data = TRUE;
		obj_list = ObjDDEBufferParse(buf, buf_len);
		g_free(buf);
	    }
	}

	if(got_data == FALSE)
	{
	    gint buf_len;
	    guint8 *buf = GUIDDEGetBinary(
		GTK_WIDGET(clist),
		GDK_SELECTION_PRIMARY,
		GDK_CURRENT_TIME,
		&buf_len
	    );
	    if(buf != NULL)
            {
                got_data = TRUE;
                obj_list = ObjDDEBufferParsePath(buf, buf_len);
                g_free(buf);
            }
	}

	if(got_data == FALSE)
	{
	    gchar *s = GUIDDEGetString(
		GTK_WIDGET(clist),
		GDK_SELECTION_PRIMARY,
		GDK_CURRENT_TIME
	    );
	    if(s != NULL)
            {
                got_data = TRUE;
                obj_list = ObjDDEBufferParsePath((guint8 *)s, STRLEN(s));
                g_free(s);
            }
	}

	if(got_data)
	{
	    /* Paste the objects into the list */
	    gint last_new_row = -1;
	    GList *glist = clist->selection_end;
	    const gint row = (glist != NULL) ? (gint)glist->data : -1;
	    obj_struct *obj;

	    gtk_clist_freeze(clist);

	    /* Insert or append? */
	    if(row > -1)
	    {
		gint ins_row = row;

		for(glist = obj_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
 		{
		    obj = OBJ(glist->data);
		    ins_row = WinScanListInsert(
			win, ins_row, ObjCopy(obj)
		    );
		    if(ins_row < 0)
			break;

		    last_new_row = ins_row;

		    ins_row++;
		}
 	    }
	    else
	    {
		gint new_row = -1;

		for(glist = obj_list; glist != NULL; glist = g_list_next(glist))
		{
		    obj = OBJ(glist->data);
		    new_row = WinScanListAppend(win, ObjCopy(obj));
		    if(new_row < 0)
			break;

		    last_new_row = new_row;
		}
	    }

	    gtk_clist_thaw(clist);

	    /* If the row is not visible then scroll to make it visible */
	    if(last_new_row > -1)
	    {
		if(gtk_clist_row_is_visible(clist, last_new_row) !=
		    GTK_VISIBILITY_FULL
		)
                    gtk_clist_moveto(
			clist,
			last_new_row, -1,	/* Row, column */
			0.5f, 0.0f		/* Row, column */
		    );
	    }
	}
	else
	{
#ifdef HAVE_EDV2
            EDVPlaySoundWarning(core->edv_ctx);
#endif
            CDialogSetTransientFor(toplevel);
            CDialogGetResponse(  
                "Paste Failed",
"The clipboard is either empty or contains data\n\
that is not supported by this application.",
                NULL,
                CDIALOG_ICON_WARNING,
                CDIALOG_BTNFLAG_OK,
                CDIALOG_BTNFLAG_OK
            );
            CDialogSetTransientFor(NULL);
	}

	/* Delete the object list */
	if(obj_list != NULL)
	{
	    g_list_foreach(obj_list, (GFunc)ObjDelete, NULL);
	    g_list_free(obj_list);
	}

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}

/*
 *	Scan List Select All.
 */
void WinSelectAllCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinCListSelectAll(GTK_CLIST(win->scan_clist));
}

/*
 *	Scan List Unselect All.
 */
void WinUnselectAllCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinCListUnselectAll(GTK_CLIST(win->scan_clist));
}

/*
 *	Scan List Invert Selection.
 */
void WinInvertSelectionCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinCListInvertSelection(GTK_CLIST(win->scan_clist));
}


/*
 *	Start.
 *
 *	Start scanning.
 */
void WinStartCB(GtkWidget *widget, gpointer data)
{
	GList *paths_list;
	GtkWidget *toplevel;
	avscan_options options;
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	toplevel = win->toplevel;
	core = CORE(win->core);

	/* Is there a scan or update already in progress? */
	if((win->scan_context->toid != 0) ||
	   (win->db_update_net_toid != 0)
	)
	{
#ifdef HAVE_EDV2
	    EDVPlaySoundWarning(core->edv_ctx);
#endif
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Scan Failed",
"A scan or virus database update is already in progress.\n\
\n\
You may either wait for the current scan or update to\n\
complete itself or click on the stop button to stop the\n\
current scan or update before starting a new scan.",
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    return;
	}

	WinSetBusy(win, TRUE);

	/* Get the paths list and scan options */
	paths_list = WinGetCurrentPathsList(win);
	options = WinGetCurrentOptions(win);

	/* Scan location not specified? */
	if(paths_list == NULL)
	{
#ifdef HAVE_EDV2  
	    EDVPlaySoundWarning(core->edv_ctx);
#endif
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Scan Failed",
"The scan location was not been specified.",
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    ); 
	    CDialogSetTransientFor(NULL);

	    WinSetBusy(win, FALSE);
	    return;
	}


	/* Find the scan item who's location matches the specified
	 * location and updates its last runned time
	 */
	WinScanListTouchByLocation(
	    win,
	    WinGetCurrentLocation(win),
	    (gulong)time(NULL)
	);

	/* Start the scan */
	WinScanProcessStart(
	    win,
	    NULL,		/* Use the database path specified in
				 * the configuration */
	    paths_list,		/* List of objects to scan */
	    options
	);

	/* Delete the paths list */
	g_list_foreach(paths_list, (GFunc)g_free, NULL);
	g_list_free(paths_list);

	/* Switch to the results page */
	WinSwitchPage(win, WIN_PAGE_NUM_RESULTS);

	WinSetBusy(win, FALSE);
}

/*
 *	Stop.
 *
 *	Stop scanning by incrementing the Win's stop_count.
 */
void WinStopCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->stop_count++;
}

/*
 *	Pause/Resume.
 */
void WinPauseResumeCB(GtkWidget *widget, gpointer data)
{
	win_scan_context_struct *scan_ctx;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	scan_ctx = win->scan_context;
	if(scan_ctx->pid > 0)
	{
	    const int p = (int)scan_ctx->pid;
	    if(win->paused)
	    {
#ifdef SIGCONT
		kill(p, SIGCONT);
#else
#warning SIGCONT not defined, pausing will not function.
#endif
		win->paused = FALSE;
		AnimIconPlay(win->anim_icon);
		WinResultsStatusUpdate(win, "Scan Resumed...");
		WinStatusMessage(
		    win,
		    "Scan process resumed",
		    FALSE
		);
	    }
	    else
	    {
#ifdef SIGSTOP
		kill(p, SIGSTOP);
#else
#warning SIGSTOP not defined, pausing will not function.
#endif
		win->paused = TRUE;
		AnimIconPause(win->anim_icon);
		WinResultsStatusUpdate(win, "Scan Paused");
		WinStatusMessage(
		    win,
		    "Scan process paused",
		    FALSE
		);
	    }
	}
	else if(win->db_update_net_pid > 0)
	{
	    const int p = (int)win->db_update_net_pid;
	    if(win->paused)
	    {
#ifdef SIGCONT
		kill(p, SIGCONT);
#endif
		win->paused = FALSE;
		WinStatusMessage(
		    win,
		    "Virus database update process resumed",
		    FALSE
		);
	    }
	    else
	    {
#ifdef SIGSTOP
		kill(p, SIGSTOP);
#endif
		win->paused = TRUE;
		WinStatusMessage(
		    win,
		    "Virus database update process paused",
		    FALSE
		);
	    }
	}

	WinUpdateTitle(win);
	WinUpdate(win);
}

/*
 *	Location.
 *
 *	Sets the scan location.
 */
void WinLocationCB(GtkWidget *widget, gpointer data)
{
	gboolean status;
	const gchar *prev_location, *s;
	GtkWidget *toplevel;
	GtkEntry *entry;
	fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
	gint nftypes = 0;
	gchar **paths_list = NULL;
	gint npaths = 0;
	win_struct *win = WIN(data);
	if((win == NULL) || FileBrowserIsQuery())
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	toplevel = win->toplevel;
	entry = GTK_ENTRY(win->scan_location_entry);

	prev_location = gtk_entry_get_text(entry);
	s = strrchr(prev_location, ',');
	if(s != NULL)
	    prev_location = s + 1;

	WinSetBusy(win, TRUE);

	/* Create file types list */
	FileBrowserTypeListNew(
	    &ftype, &nftypes,
	    "*.*", "All Files"
	);

	/* Query user for path */
	FileBrowserSetTransientFor(toplevel);
	status = FileBrowserGetResponse(
	    "Select Location",
	    "Select", "Cancel",
	    prev_location,
	    ftype, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);
					 
	/* Got user response? */
	if(status)
	{         
	    gint i;
	    gchar *s, *location = STRDUP("");
	    const gchar *path;

	    for(i = 0; i < npaths; i++)
	    {
		path = paths_list[i];
		if(STRISEMPTY(path))
		    continue;

		s = g_strconcat(
		    location,
		    (i > 0) ? "," : "",
		    path,
		    NULL
		);
		g_free(location);
		location = s;
	    }

	    gtk_entry_set_text(entry, location);
	    g_free(location);
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftype, nftypes);

	WinSetBusy(win, FALSE);
}

/*
 *	Recursive.
 *
 *	Toggles recursive.
 */
void WinRecursiveCB(GtkWidget *widget, gpointer data)
{
	gboolean v;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;
		   
	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	v = GTK_TOGGLE_BUTTON_GET_ACTIVE(win->scan_recursive_check) ? FALSE : TRUE;
	GUIMenuItemSetCheck(
	    win->recursive_micheck, v, TRUE
	);
	gtk_toggle_button_set_active(
	    GTK_TOGGLE_BUTTON(win->scan_recursive_check), v
	);  

	WinUpdate(win);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}

/*
 *	Executables Only.
 *
 *	Toggles executables only.
 */
void WinExecutablesOnlyCB(GtkWidget *widget, gpointer data)
{
	gboolean v;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	v = GTK_TOGGLE_BUTTON_GET_ACTIVE(win->scan_executables_only_check) ? FALSE : TRUE;
	GUIMenuItemSetCheck(
	    win->executables_only_micheck, v, TRUE
	);
	gtk_toggle_button_set_active(
	    GTK_TOGGLE_BUTTON(win->scan_executables_only_check), v
	);  

	WinUpdate(win);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}

/*
 *	Ignore Links.
 *
 *	Toggles ignore links.
 */
void WinIgnoreLinksCB(GtkWidget *widget, gpointer data)
{
	gboolean v;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	v = GTK_TOGGLE_BUTTON_GET_ACTIVE(win->scan_ignore_links_check) ? FALSE : TRUE;
	GUIMenuItemSetCheck(
	    win->ignore_links_micheck, v, TRUE
	);
	gtk_toggle_button_set_active(
	    GTK_TOGGLE_BUTTON(win->scan_ignore_links_check), v
	);

	WinUpdate(win);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}

/*
 *	Scanner options.
 */
void WinScannerSettingsCB(GtkWidget *widget, gpointer data)
{
	gint strc;
	gchar **strv;
	GList *glist;
	cfg_item_struct *cfg_list;
	GtkWidget *toplevel;
	core_struct *core;
	win_struct *win = WIN(data);
	if((win == NULL) || PDialogIsQuery())
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	toplevel = win->toplevel;
	core = CORE(win->core);
	cfg_list = core->cfg_list;

	WinSetBusy(win, TRUE);

	/* Set up the Prompt Dialog */
	PDialogDeleteAllPrompts();
	PDialogSetSize(350, -1);
	glist = NULL;
	glist = g_list_append(glist, "Highest");
	glist = g_list_append(glist, "High");
	glist = g_list_append(glist, "Medium");
	glist = g_list_append(glist, "Low");
	glist = g_list_append(glist, "Lowest");
	PDialogAddPromptPopupList(
	    NULL, "CPU Priority:",
	    glist,
	    CFGItemListGetValueI(
		cfg_list, CFG_PARM_SCAN_CPU_PRIORITY
	    ),
	    g_list_length(glist)
	);
	g_list_free(glist);
	PDialogAddPromptToggle(
	    NULL, "Report All Objects",
	    (gboolean)CFGItemListGetValueI(
                cfg_list, CFG_PARM_SCAN_REPORT_ALL_OBJECTS
            )
	);
	PDialogSetPromptTip(
	    -1,
"Check this to include all objects scanned in the Results\
 List (even if they are clean)"
	);
	PDialogAddPromptSpin(
	    NULL, "Max Files:",
	    (gfloat)CFGItemListGetValueUL(
		cfg_list, CFG_PARM_SCAN_MAX_FILES
	    ),
	    0.0f, (gfloat)((guint32)-1),
	    1.0f, 5.0f,
	    1.0f, 0
	);
	PDialogSetPromptTip(
	    -1,
"Set the maximum number of files to scan within an archive\
 or set this value to 0 for no maximum (default value is 500)"
	);
	PDialogAddPromptSpin(
	    NULL, "Max File Size (Bytes):",
	    (gfloat)CFGItemListGetValueUL(
		cfg_list, CFG_PARM_SCAN_MAX_FILE_SIZE
	    ),
	    0.0f, (gfloat)((gulong)-1),
	    10.0f, 50.0f,
	    1.0f, 0
	);
	PDialogSetPromptTip(
	    -1,
"Set the maximum size of a file to scan within an archive in bytes\
 or set this value to 0 for no maximum (default value is 10000000)"
	);
	PDialogAddPromptSpin(
	    NULL, "Max Recursions:",
	    (gfloat)CFGItemListGetValueUL(
		cfg_list, CFG_PARM_SCAN_MAX_RECURSIONS
	    ),
	    0.0f, (gfloat)((guint32)-1),
	    1.0f, 5.0f,
	    1.0f, 0
	);
	PDialogSetPromptTip(
	    -1,
"Set the maximum number of recursions or 0 for no maximum\
 (default value is 5)"
	);
	PDialogAddPromptSpin(
	    NULL, "Max Mail Recursions:",
	    (gfloat)CFGItemListGetValueUL(
		cfg_list, CFG_PARM_SCAN_MAX_MAIL_RECURSIONS
	    ),
	    0.0f, (gfloat)((guint32)-1),
	    1.0f, 5.0f,
	    1.0f, 0
	);
	PDialogSetPromptTip(
	    -1,
"Set the maximum number of mail recursions or 0 for no maximum\
 (default value is 64)"
	);
	PDialogAddPromptSpin(
	    NULL, "Max Compression Ratio:",
	    (gfloat)CFGItemListGetValueUL(
		cfg_list, CFG_PARM_SCAN_MAX_COMPRESSION_RATIO
	    ),
	    0.0f, (gfloat)((guint32)-1),
	    1.0f, 5.0f,
	    1.0f, 0
	);
	PDialogSetPromptTip(
	    -1,
"Set the maximum archive compression ratio limit (default value is 200)"
	);
	/* Query user */
	gtk_window_set_transient_for(
	    GTK_WINDOW(PDialogGetToplevel()), GTK_WINDOW(toplevel)
	);
/*	PDialogSetTransientFor(toplevel); */
	strv = PDialogGetResponse(
	    "Scanner Settings",
	    NULL, NULL,
	    PDIALOG_ICON_SETTINGS,
	    "Set", "Cancel",
	    PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
	    PDIALOG_BTNFLAG_SUBMIT,
	    &strc
	);
	gtk_window_set_transient_for(
	    GTK_WINDOW(PDialogGetToplevel()), NULL
	);
/*	PDialogSetTransientFor(NULL); */
	if(strv != NULL)
	{
	    if(strc > 0)
	    {
		scanner_cpu_priority p = SCANNER_CPU_PRIORITY_HIGHEST;
		const gchar *arg = strv[0];
		if(!g_strcasecmp(arg, "highest"))
		    p = SCANNER_CPU_PRIORITY_HIGHEST;
		else if(!g_strcasecmp(arg, "high"))
		    p = SCANNER_CPU_PRIORITY_HIGH;
		else if(!g_strcasecmp(arg, "medium"))
		    p = SCANNER_CPU_PRIORITY_MEDIUM;
		else if(!g_strcasecmp(arg, "low"))
		    p = SCANNER_CPU_PRIORITY_LOW;
		else if(!g_strcasecmp(arg, "lowest"))
		    p = SCANNER_CPU_PRIORITY_LOWEST;

		CFGItemListSetValueI(
		    cfg_list, CFG_PARM_SCAN_CPU_PRIORITY,
		    (gint)p, FALSE
		);
	    }
	    if(strc > 1)
		CFGItemListSetValueI(
		    cfg_list, CFG_PARM_SCAN_REPORT_ALL_OBJECTS,
		    (gboolean)ATOI(strv[1]), FALSE
		);
	    if(strc > 2)
		CFGItemListSetValueL(
		    cfg_list, CFG_PARM_SCAN_MAX_FILES,
		    ATOL(strv[2]), FALSE
		);
	    if(strc > 3)
		CFGItemListSetValueL(
		    cfg_list, CFG_PARM_SCAN_MAX_FILE_SIZE,
		    ATOL(strv[3]), FALSE
		);
	    if(strc > 4)
		CFGItemListSetValueL(
		    cfg_list, CFG_PARM_SCAN_MAX_RECURSIONS,
		    ATOL(strv[4]), FALSE
		);
	    if(strc > 5)
		CFGItemListSetValueL(
		    cfg_list, CFG_PARM_SCAN_MAX_MAIL_RECURSIONS,
		    ATOL(strv[5]), FALSE
		);
	    if(strc > 6)
		CFGItemListSetValueL(
		    cfg_list, CFG_PARM_SCAN_MAX_COMPRESSION_RATIO,
		    ATOL(strv[6]), FALSE
		);

	    WinUpdate(win);
	}

	PDialogDeleteAllPrompts();

	WinSetBusy(win, FALSE);
}


#ifdef HAVE_EDV2
/*
 *	Endeavour Recycle progress callback.
 */
static gint WinEDVRecycleProgressCB(
	gpointer data, const gulong i, const gulong m
)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return(FALSE);

	WinStatusProgress(
	    win,
	    (m > 0l) ? (gfloat)i / (gfloat)m : 0.0f,
	    TRUE
	);

	return(FALSE);
}
#endif

/*
 *	Results Recycle.
 *
 *	Recycles the selected objects in the Results List.
 */
void WinResultsRecycleCB(GtkWidget *widget, gpointer data)
{
	gint row;
	gchar *name = NULL;
	GList *glist, *obj_list;
	GtkWidget *toplevel;
	GtkCList *clist;
	obj_struct *obj;
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)             
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinSetBusy(win, TRUE);

	toplevel = win->toplevel;
	clist = GTK_CLIST(win->results_clist);
	core = CORE(win->core);

	/* Iterate through each selected row and generate objects list */
	obj_list = NULL;
	for(glist = clist->selection; glist != NULL; glist = g_list_next(glist))
	{
	    row = (gint)glist->data;
	    obj = gtk_clist_get_row_data(clist, row);
	    if(obj == NULL)
		continue;

	    obj_list = g_list_append(obj_list, obj);
	}

	/* Confirm recycle */
	if(obj_list != NULL)
	{
	    gint response;
	    gchar *msg;
	    const gint nobjs = g_list_length(obj_list);

	    /* Format confirm recycle message */
	    obj = (nobjs == 1) ? OBJ(obj_list->data) : NULL;
	    if((obj != NULL) ? !STRISEMPTY(obj->path) : FALSE)
	    {
		name = STRDUP(GET_NAME_FROM_PATH(obj->path));
		msg = g_strdup_printf(
		    "Recycle object \"%s\"?",
		    name
		);
	    }
	    else
		msg = g_strdup_printf(
		    "Recycle %i selected objects?",
		    nobjs
		);

	    /* Query user to confirm recycle */
#ifdef HAVE_EDV2
	    EDVPlaySoundQuestion(core->edv_ctx);
#endif
	    CDialogSetTransientFor(toplevel);
	    response = CDialogGetResponseIconData(
		"Confirm Recycle",
		msg,
		NULL,
		(guint8 **)icon_trash_32x32_xpm,
		CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
		CDIALOG_BTNFLAG_NO
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(msg);
	    if(response != CDIALOG_RESPONSE_YES)
	    {
		g_list_free(obj_list);
		g_free(name);
		WinSetBusy(win, FALSE);
		return;
	    }
	}    

	/* Delete the objects and rows who's row data is specified in
	 * the obj_list
	 */
	win->freeze_count++;
	gtk_clist_freeze(clist);
	for(glist = obj_list; glist != NULL; glist = g_list_next(glist))
	{
	    row = gtk_clist_find_row_from_data(clist, glist->data);
	    if(row > -1)
	    {
		gchar *msg, *path;

		obj = OBJ(glist->data);

		/* Make copy of the object's path since the object's
		 * path may become invalid
		 */
		path = STRDUP(obj->path);

		msg = g_strdup_printf(
		    "Recycling object \"%s\"",
		    GET_NAME_FROM_PATH(path)
		);
		WinStatusMessage(win, msg, TRUE);
		g_free(msg);

#ifdef HAVE_EDV2
		if(path != NULL)
		{
		    /* Recycle this object */
		    const guint index = EDVRecycle(
			core->edv_ctx, path,
			TRUE,
			WinEDVRecycleProgressCB, win
		    );
		    if(index != 0)
		    {
			EDVPlaySoundCompleted(core->edv_ctx);
			EDVContextSync(core->edv_ctx);
			gtk_clist_remove(clist, row);
		    }
		}
#else
/* TODO */
#endif
		WinStatusProgress(win, 0.0f, TRUE);
		g_free(path);
	    }
	}
	gtk_clist_columns_autosize(clist);
	gtk_clist_thaw(clist);
	win->freeze_count--;

	/* Print final status */
	if(name != NULL)
	{
	    gchar *msg = g_strdup_printf(
		"Recycled object \"%s\"",
		name
	    );
	    WinStatusMessage(win, msg, FALSE);
	    g_free(msg);
	}
	else
	{
	    const gint nobjs = g_list_length(obj_list);
	    gchar *msg = g_strdup_printf(
		"Recycled %i object%s",
		nobjs,
		(nobjs == 1) ? "" : "s"
	    );
	    WinStatusMessage(win, msg, FALSE);
	    g_free(msg);
	}

	/* Delete objects list */
	g_list_free(obj_list);
	g_free(name);

	WinUpdate(win);

	WinSetBusy(win, FALSE);
}

/*
 *	Results Move.
 *
 *	Moves the selected objects in the Results List.
 */
void WinResultsMoveCB(GtkWidget *widget, gpointer data)
{
	gint row;
	gchar *name = NULL, *dest_path = NULL;
	GList *glist, *obj_list;
	GtkWidget *toplevel;
	GtkCList *clist;
	obj_struct *obj;
	core_struct *core;
	struct stat stat_buf;
	win_struct *win = WIN(data);
	if((win == NULL) || PDialogIsQuery())
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinSetBusy(win, TRUE);

	toplevel = win->toplevel;
	clist = GTK_CLIST(win->results_clist);
	core = CORE(win->core);

	/* Iterate through each selected row and generate objects list */
	obj_list = NULL;
	for(glist = clist->selection; glist != NULL; glist = g_list_next(glist))
	{
	    row = (gint)glist->data;
	    obj = gtk_clist_get_row_data(clist, row);
	    if(obj == NULL)
		continue;

	    obj_list = g_list_append(obj_list, obj);
	}

	/* Query Move */
	if(obj_list != NULL)
	{
	    gint strc;
	    gchar *mesg, **strv;
	    const gint nobjs = g_list_length(obj_list);

	    obj = (nobjs == 1) ? OBJ(obj_list->data) : NULL;
	    if((obj != NULL) ? !STRISEMPTY(obj->path) : FALSE)
	    {
		name = STRDUP(GET_NAME_FROM_PATH(obj->path));
		mesg = g_strdup_printf(
		    "Move object \"%s\"",
		    name
		);
	    }
	    else
		mesg = g_strdup_printf(
		    "Move %i selected objects", 
		    nobjs
		);

	    /* Set up the Prompt Dialog */
	    PDialogDeleteAllPrompts();
	    PDialogAddPromptWithBrowse(
		NULL, "To:", NULL,
		win, WinPDialogBrowseLocationCB
	    );
	    PDialogSetPromptCompletePath(-1);
	    PDialogSetSize(350, -1);
	    gtk_window_set_transient_for(
		GTK_WINDOW(PDialogGetToplevel()), GTK_WINDOW(toplevel)
	    );
	    strv = PDialogGetResponse(
		"Move", mesg, NULL,
		PDIALOG_ICON_FILE_MOVE,
		"Move", "Cancel",
		PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
		PDIALOG_BTNFLAG_SUBMIT,
		&strc
	    );
	    g_free(mesg);
	    gtk_window_set_transient_for(
		GTK_WINDOW(PDialogGetToplevel()), NULL
	    );
	    if((strv != NULL) && (strc > 0))
	    {
		dest_path = STRDUP((strc > 0) ? strv[0] : NULL);
	    }

	    PDialogDeleteAllPrompts();
	}

	/* Did not get destination path or user aborted? */
	if(dest_path == NULL)
	{
	    g_list_free(obj_list);
	    g_free(name);
	    WinSetBusy(win, FALSE);
	    return;
	}
	/* Destination does not exist? */
	if(stat((const char *)dest_path, &stat_buf))
	{
	    gchar *msg = g_strdup_printf(
"Destination directory does not exist:\n\
\n\
    %s\n",
		dest_path
	    );
#ifdef HAVE_EDV2
	    EDVPlaySoundError(core->edv_ctx);
#endif
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Move Failed", msg, NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(msg);

	    g_list_free(obj_list);
	    g_free(name);
	    g_free(dest_path);
	    WinSetBusy(win, FALSE);
	    return;
	}
	/* Destination is not a directory? */
#ifdef S_ISDIR
	else if(!S_ISDIR(stat_buf.st_mode))
#else
	else if(TRUE)
#endif
	{
#ifdef HAVE_EDV2
	    EDVPlaySoundError(core->edv_ctx);
#endif
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Move Failed",
		"Destination is not a directory.",
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_list_free(obj_list);
	    g_free(name);
	    g_free(dest_path);
	    WinSetBusy(win, FALSE);
	    return;
	}

	/* Move the objects who's row data is specified in the
	 * obj_list
	 */
	win->freeze_count++;
	gtk_clist_freeze(clist);
	for(glist = obj_list; glist != NULL; glist = g_list_next(glist))
	{
	    row = gtk_clist_find_row_from_data(clist, glist->data);
	    if(row > -1)
	    {
		gchar *old_path, *new_path;

		obj = OBJ(glist->data);

		/* Get old path */
		old_path = STRDUP(obj->path);

		/* Get new path */
		new_path = STRDUP(PrefixPaths(
		    dest_path,
		    GET_NAME_FROM_PATH(old_path)
		));

		/* Got old & new paths? */
		if(!STRISEMPTY(old_path) && !STRISEMPTY(new_path))
		{
		    /* Move */
		    if(!rename(old_path, new_path))
		    {
#ifdef HAVE_EDV2
			EDVNotifyQueueObjectAdded(
			    core->edv_ctx, new_path
			);
			EDVNotifyQueueObjectRemoved(
			    core->edv_ctx, old_path
			);
			EDVContextSync(core->edv_ctx);
#endif
			gtk_clist_remove(clist, row);
		    }
		}
		g_free(old_path);
		g_free(new_path);
	    }
	} 
	gtk_clist_columns_autosize(clist);
	gtk_clist_thaw(clist);
	win->freeze_count--;  

	/* Print final status */
	if(name != NULL)
	{
	    gchar *buf = g_strdup_printf(
		"Moved object \"%s\"",
		name
	    );
	    WinStatusMessage(win, buf, FALSE);
	    g_free(buf);
	}
	else
	{
	    const gint nobjs = g_list_length(obj_list);
	    gchar *buf = g_strdup_printf(
		"Moved %i object%s",
		nobjs,
		(nobjs == 1) ? "" : "s"
	    );
	    WinStatusMessage(win, buf, FALSE);
	    g_free(buf);
	}

	/* Delete objects list */
	g_list_free(obj_list);
	g_free(name);
	g_free(dest_path);

	WinUpdate(win);

	WinSetBusy(win, FALSE);
}

/*
 *	Results List Select All.
 */
void WinResultsSelectAllCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinCListSelectAll(GTK_CLIST(win->results_clist));
}

/*
 *	Results List Unselect All.
 */
void WinResultsUnselectAllCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;
 
	if(win->freeze_count > 0)
	    return;
 
	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinCListUnselectAll(GTK_CLIST(win->results_clist));
}

/*
 *	Results List Invert Selection.
 */
void WinResultsInvertSelectionCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinCListInvertSelection(GTK_CLIST(win->results_clist));
}

/*
 *	Results Report.
 */
void WinResultsReportCB(GtkWidget *widget, gpointer data)
{
	gchar	*time_start_s,
		*duration_s,
		*nscanned_s,
		*ninfected_s,
		*nproblems_s,
		*data_scanned_s,
		*data_scanned_units_s;
	GtkWidget *toplevel;
#ifdef HAVE_EDV2
	edv_context_struct *edv_ctx;
#endif
	win_scan_context_struct *scan_ctx;
	core_struct *core;
	win_struct *win = WIN(data);
	if((win == NULL) || CDialogIsQuery())
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	toplevel = win->toplevel;
	core = win->core;
	scan_ctx = win->scan_context;
#ifdef HAVE_EDV2
	edv_ctx = core->edv_ctx;
#endif

	if(scan_ctx->start_time == 0l)
	{
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
                "Scan Report",
		"No scan results available.",
		NULL,
                CDIALOG_ICON_WARNING,
                CDIALOG_BTNFLAG_OK,
                CDIALOG_BTNFLAG_OK
            );
	    CDialogSetTransientFor(NULL);
	    return;
	}

	/* Start time */
#ifdef HAVE_EDV2
	time_start_s = STRDUP(EDVDateString(
	    edv_ctx, scan_ctx->start_time
	));
#else
	{
	    gchar *s;
	    time_t t = scan_ctx->start_time;
	    time_start_s = STRDUP(ctime(&t));
	    s = (gchar *)strchr((char *)time_start_s, '\n');
	    if(s != NULL)
		*s = '\0';
	}
#endif

	/* Duration */
	duration_s = TIME_LAPSED_STRING(scan_ctx->duration);

	/* Objects Scanned */
	nscanned_s = FORMATED_NUMBER_STRING(scan_ctx->nscanned);

	/* Objects Infected */
	ninfected_s = FORMATED_NUMBER_STRING(scan_ctx->ninfected);

	/* Problems */
	nproblems_s = FORMATED_NUMBER_STRING(scan_ctx->nproblems);

	/* Data Scanned & Units */
        if(scan_ctx->blocks_scanned >= (1024 * 1024))
        {
            data_scanned_s = FORMATED_NUMBER_STRING(
                scan_ctx->blocks_scanned / 1024l / 1024l
            );
            data_scanned_units_s = STRDUP("gb");
        }
        else if(scan_ctx->blocks_scanned >= 1024)
        {
            data_scanned_s = FORMATED_NUMBER_STRING(
                scan_ctx->blocks_scanned / 1024l
            );
            data_scanned_units_s = STRDUP("mb");
        }
        else
        {
            data_scanned_s = FORMATED_NUMBER_STRING(
                scan_ctx->blocks_scanned
            );
            data_scanned_units_s = STRDUP("kb");
        }


	/* Display report */
	CDialogSetTransientFor(toplevel);
	if(scan_ctx->ninfected > 0l)
	{
	    gchar *msg = g_strdup_printf(
"Started On: %s\n\
Duration: %s\n\
\n\
Objects Infected: %s\n\
Objects Scanned: %s\n\
Problems: %s\n\
Data Scanned: %s %s\n\
\n\
DO NOT VIEW OR RUN THE INFECTED FILES!",
		time_start_s,
		duration_s,
		ninfected_s,
		nscanned_s,
		nproblems_s,
		data_scanned_s, data_scanned_units_s
	    );
	    CDialogGetResponseIconData(
                "Scan Report", msg, NULL,
                (guint8 **)icon_biohazard_32x32_xpm,
                CDIALOG_BTNFLAG_OK,
                CDIALOG_BTNFLAG_OK
            );
	    g_free(msg);
	}
	else
	{
	    gchar *msg = g_strdup_printf(
"Started On: %s\n\
Duration: %s\n\
\n\
No Infected Files Found\n\
Objects Scanned: %s\n\
Problems: %s\n\
Data Scanned: %s %s\n",
		time_start_s,
		duration_s,
		nscanned_s,
		nproblems_s,
		data_scanned_s, data_scanned_units_s
	    );
            CDialogGetResponseIconData(
                "Scan Report", msg, NULL,
                (guint8 **)icon_avscan_32x32_xpm,
                CDIALOG_BTNFLAG_OK,
                CDIALOG_BTNFLAG_OK
            );
	    g_free(msg);
	}
        CDialogSetTransientFor(NULL);

	g_free(time_start_s);
	g_free(duration_s);
	g_free(nscanned_s);
	g_free(ninfected_s);
	g_free(nproblems_s);
	g_free(data_scanned_s);
	g_free(data_scanned_units_s);
}

/*
 *	Results Open.
 *
 *	Opens a results log file.
 */
void WinResultsOpenCB(GtkWidget *widget, gpointer data)
{
	gboolean status;
	GtkWidget *toplevel;
	fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
	gint nftypes = 0;
	gchar **paths_list = NULL;
	gint npaths = 0;
	win_struct *win = WIN(data);
	if((win == NULL) || FileBrowserIsQuery())
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinSetBusy(win, TRUE);

	toplevel = win->toplevel;

	/* Create file types list */
	FileBrowserTypeListNew(     
	    &ftype, &nftypes,
	    ".log", "Scan Results Log"
	);
	FileBrowserTypeListNew(
	    &ftype, &nftypes,
	    "*.*", "All Files"
	);

	/* Query user */
	FileBrowserSetTransientFor(toplevel);
	status = FileBrowserGetResponse(
	    "Open Scan Results Log",
	    "Open", "Cancel",
	    NULL,
	    ftype, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(status)
	{
	    const gchar *new_path = (npaths > 0) ?
		paths_list[0] : NULL;
	    if(!STRISEMPTY(new_path))
	    {
		/* Open the results log */
		WinResultsOpen(win, new_path, TRUE);
	    }
	}
	 
	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftype, nftypes);

	WinSetBusy(win, FALSE);
}

/*
 *	Results Save As. 
 *
 *	Save as a results log file.
 */
void WinResultsSaveAsCB(GtkWidget *widget, gpointer data)
{
	gboolean status;
	GtkWidget *toplevel;
	fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
	gint nftypes = 0;
	gchar **paths_list = NULL;
	gint npaths = 0;
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)             
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinSetBusy(win, TRUE);
      
	toplevel = win->toplevel; 
	core = CORE(win->core);

	/* Create file types list */
	FileBrowserTypeListNew(     
	    &ftype, &nftypes,
	    ".log", "Scan Results Log"
	);
	FileBrowserTypeListNew(     
	    &ftype, &nftypes,
	    "*.*", "All Files"
	);

	/* Query user */
	FileBrowserSetTransientFor(toplevel);
	status = FileBrowserGetResponse(
	    "Save Scan Results Log As",
	    "Save", "Cancel",
	    NULL,                       /* Initial path */
	    ftype, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(status)
	{
	    const gchar *new_path = (npaths > 0) ?
		paths_list[0] : NULL;
	    if(!STRISEMPTY(new_path))
	    {
		/* Do overwrite check */
		if(WinQueryOverwrite(core, new_path, toplevel))
		{
		    /* Save the results log */
		    WinResultsSave(win, new_path, TRUE);
		}
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftype, nftypes);

	/* Reset due to possible file related change */
	FileBrowserReset();

	WinSetBusy(win, FALSE);
}

/*
 *	Results Clear.
 */
void WinResultsClearCB(GtkWidget *widget, gpointer data)
{
	win_scan_context_struct *scan_ctx;
	win_struct *win = WIN(data);
	if(win == NULL)             
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	scan_ctx = win->scan_context;
	scan_ctx->start_time = 0l;
	scan_ctx->duration = 0l;
	scan_ctx->nscanned = 0l;
	scan_ctx->ninfected = 0l;
	scan_ctx->nproblems = 0l;
	scan_ctx->total = 0l;
	scan_ctx->blocks_scanned = 0l;

	WinResultsLocationUpdate(win, NULL);
	WinResultsStatusUpdate(win, NULL);
	WinResultsListClear(win);

	WinStatusMessage(
	    win,
"Scan results cleared",
	    FALSE
	);

	WinUpdateTitle(win);
	WinUpdate(win);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}


/*
 *	Virus Database Refresh.
 *
 *	Reloads the virus database to the Virus Database Tree.
 */
void WinDBRefreshCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;             

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinDBClear(win);
	WinDBQueueUpdate(win);
}

/*
 *	Virus Database Pattern Details.
 */
void WinDBPatternDetailsCB(GtkWidget *widget, gpointer data)
{
	gchar *s;
	guint8 *pattern_data;
	gint pattern_data_length;
	gulong	last_updated_time,
		discovery_time;
	GList *glist;
	GtkCList *clist;
	GtkCTreeNode *node;
	hview_struct *hview;
	win_db_item_struct *pattern;
	win_pattern_dlg_struct *d;
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;             

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	core = win->core;

	clist = GTK_CLIST(win->db_ctree);
	glist = clist->selection_end;
	node = (glist != NULL) ?
	    (GtkCTreeNode *)glist->data : NULL;
	if(node == NULL)
	    return;

	pattern = WIN_DB_ITEM(gtk_ctree_node_get_row_data(
	    GTK_CTREE(clist), node
	));
	if(pattern == NULL)
	    return;

	/* Get the pattern dialog (create it as needed) */
	d = win->pattern_dlg;
	if(d == NULL)
	    win->pattern_dlg = d = WinPatternDlgNew();
	if(d == NULL)
	    return;

	gtk_window_set_transient_for(
	    GTK_WINDOW(d->toplevel), GTK_WINDOW(win->toplevel)
	);

	switch(pattern->type)
	{
	  case WIN_DB_ITEM_TYPE_PATTERN:
	    s = g_strdup_printf(
		"Pattern: \"%s\"",
		pattern->name
	    );
	    break;
	  case WIN_DB_ITEM_TYPE_FOLDER:
	    s = g_strdup_printf(
		"Folder: \"%s\"",
		pattern->name
	    );
	    break;
	  default:
	    s = g_strdup_printf(
		"\"%s\"",
		pattern->name
	    );
	    break;
	}
	gtk_window_set_title(GTK_WINDOW(d->toplevel), s);
	g_free(s);

	hview = d->hview;
	HViewClear(hview);

	if(!WinDBGetPatternStats(
	    win,
	    pattern->name,
	    &pattern_data,
	    &pattern_data_length,
	    &last_updated_time,
	    &discovery_time
	))
	{
	    if(pattern_data != NULL)
	    {
		HViewSetPosition(hview, 0);
		HViewOpenData(
		    hview,
		    pattern_data, pattern_data_length
		);
		g_free(pattern_data);
	    }

#ifdef HAVE_EDV2
	    if(d->stats_label != NULL)
	    {
		edv_context_struct *ctx = core->edv_ctx;
		gchar	*ds1 = (last_updated_time > 0l) ?
		STRDUP(EDVDateString(ctx, last_updated_time)) : STRDUP(""),
			*ds2 = (discovery_time > 0l) ?
		STRDUP(EDVDateString(ctx, discovery_time)) : STRDUP("");

		s = g_strconcat(
		    ds1, "\n",
		    ds2, NULL
		);
		gtk_label_set_text(GTK_LABEL(d->stats_label), s);
		g_free(s);

		g_free(ds1);
		g_free(ds2);
	    }
#else
	    if(d->stats_label != NULL)
	    {
		time_t t;
		gchar	ds1[80],
			ds2[80];

		t = last_updated_time;
		strncpy(
		    (char *)ds1,
		    (t > 0l) ? ctime(&t) : "",
		    sizeof(ds1)
		);
		s = (gchar *)strchr((char *)ds1, '\n');
		if(s != NULL)
		    *s = '\0';

		t = discovery_time;
		strncpy(
		    (char *)ds2,
		    (t > 0l) ? ctime(&t) : "",
		    sizeof(ds2)
		);
		s = (gchar *)strchr((char *)ds2, '\n');
		if(s != NULL)
		    *s = '\0';

		s = g_strconcat(
		    ds1, "\n",
		    ds2, NULL
		);
		gtk_label_set_text(GTK_LABEL(d->stats_label), s);
		g_free(s);
	    }
#endif
	}


	gtk_widget_show_raise(d->toplevel);
	gtk_widget_grab_focus(d->close_btn);
	gtk_widget_grab_default(d->close_btn);
}

/*
 *	Virus Database Location
 *
 *	Sets the Virus Scanner's Database Location.   
 */
void WinDBLocationCB(GtkWidget *widget, gpointer data)
{
	gboolean status;
	GtkWidget *toplevel;
	fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
	gint nftypes = 0;
	gchar **paths_list = NULL;
	gint npaths = 0;
	cfg_item_struct *cfg_list;
	core_struct *core;
	win_struct *win = WIN(data);
	if((win == NULL) || FileBrowserIsQuery())
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	toplevel = win->toplevel;
	core = CORE(win->core);
	cfg_list = core->cfg_list;

	WinSetBusy(win, TRUE);

	/* Create file types list */
#ifdef HAVE_CLAMAV
	FileBrowserTypeListNew(
	    &ftype, &nftypes,
	    ".cvd", "Clam Antivirus Database"
	);
#endif
	FileBrowserTypeListNew(
	    &ftype, &nftypes,
	    "*.*", "All Files"
	);                    

	/* Query user for path */
	FileBrowserSetTransientFor(toplevel);
	status = FileBrowserGetResponse(     
	    "Set Virus Scanner Database Location",
	    "Set", "Cancel",
	    CFGItemListGetValueS(cfg_list, CFG_PARM_DB_LOCATION),
	    ftype, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(status) 
	{
	    const gchar *path = (npaths > 0) ?
		paths_list[0] : NULL;
	    if(!STRISEMPTY(path))
	    {
		/* Set new virus database location */
		CFGItemListSetValueS(
		    cfg_list, CFG_PARM_DB_LOCATION,
		    path, FALSE
		);
		  
		/* Update Virus Database Tree due to virus database
		 * location change
		 */
		WinDBClear(win);
		if(win->page_num == WIN_PAGE_NUM_DB)
		    WinDBQueueUpdate(win);

		WinUpdate(win);
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftype, nftypes);

	WinSetBusy(win, FALSE); 
}

/*
 *	Virus Database Update From Network.
 *
 *	Updates the virus database from the network.
 */
void WinDBUpdateNetCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)                         
	    return;             

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinSetBusy(win, TRUE);

	/* Start the virus database update */
	WinDBUpdateNetProcessStart(win);

	WinSetBusy(win, FALSE);
}


/*
 *	Refresh.
 */
void WinRefreshCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	WinUpdate(win);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}

/*
 *	Go to Scan Page.
 */
void WinViewScanPageCB(GtkWidget *widget, gpointer data)
{
	const win_page_num page_num = WIN_PAGE_NUM_SCAN;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;
		   
	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	if(win->page_num != page_num)
	    gtk_notebook_set_page(
		GTK_NOTEBOOK(win->notebook), (gint)page_num
	    );
}

/*
 *	Go to Results Page.
 */
void WinViewResultsPageCB(GtkWidget *widget, gpointer data)
{
	const win_page_num page_num = WIN_PAGE_NUM_RESULTS;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;
		   
	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	if(win->page_num != page_num)
	    gtk_notebook_set_page(
		GTK_NOTEBOOK(win->notebook), (gint)page_num
	    );
}

/*
 *	Go to Virus Database Page.
 */
void WinViewDBPageCB(GtkWidget *widget, gpointer data)  
{
	const win_page_num page_num = WIN_PAGE_NUM_DB;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	if(win->page_num != page_num)
	    gtk_notebook_set_page(
		GTK_NOTEBOOK(win->notebook), (gint)page_num
	    );
}

/*
 *	Show/Hide Tool Bar.
 */
void WinViewToolBarCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	if(WinToolBarIsShown(win))
	    WinToolBarSetShow(win, FALSE);
	else
	    WinToolBarSetShow(win, TRUE);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}

/*
 *	Show/Hide Status Bar.
 */
void WinViewStatusBarCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	if(WinStatusBarIsShown(win))
	    WinStatusBarSetShow(win, FALSE);
	else
	    WinStatusBarSetShow(win, TRUE);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}


/*
 *	Help Contents.
 */
void WinHelpContentsCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	Help(CORE(win->core), "Contents", win->toplevel);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}

/*
 *	Help Scanning.
 */
void WinHelpScanningCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	Help(CORE(win->core), "Scanning", win->toplevel);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}

/*
 *	Help Updating.
 */
void WinHelpUpdatingCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	Help(CORE(win->core), "Updating", win->toplevel);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}

/*
 *	About.
 */
void WinHelpAboutCB(GtkWidget *widget, gpointer data)
{
	gchar *msg, *av_engine_name_version;
	GtkWidget *toplevel;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinSetBusy(win, TRUE);

	toplevel = win->toplevel;

	av_engine_name_version = AVScanGetAVEngineNameVersionString();
	msg = g_strdup_printf(
PROG_NAME_FULL "\n\
Version " PROG_VERSION "\n\
\n\
" PROG_URL "\n\
\n\
" PROG_COPYRIGHT "\
\n\
AntiVirus Engine: %s",
	    (av_engine_name_version != NULL) ?
		av_engine_name_version : "None"
	);
	g_free(av_engine_name_version);

	CDialogSetTransientFor(toplevel);
	CDialogGetResponseIconData(
	    "About " PROG_NAME_FULL,
	    msg,
	    NULL,
	    (guint8 **)icon_avscan_32x32_xpm,
	    CDIALOG_BTNFLAG_OK,
	    CDIALOG_BTNFLAG_OK
	);
	g_free(msg);
	CDialogSetTransientFor(NULL);


	WinSetBusy(win, FALSE);
}
