/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <string.h>

#include <kozo.h>

#include "bkl-db.h"
#include "bkl-entry.h"
#include "bkl-item-audio.h"
#include "bkl-item-broken.h"
#include "bkl-item-image.h"
#include "bkl-item-video.h"

static const char *fieldnames[N_BKL_FIELD_TYPE] = {
    "type", "file", "audio", "image", "video", "broken", "playlist", "extended"
};
int fieldids[N_BKL_FIELD_TYPE] = {0, };

static gboolean
register_fields (KozoDB  *db,
                 GError **error)
{
    int i, id;

    for (i = 0; i < N_BKL_FIELD_TYPE; i++) {
        id = kozo_db_register_field (db, fieldnames[i], error);
        if (id < 0) {
            return FALSE;
        }

        fieldids[i] = id;
    }

    return TRUE;
}

static gboolean
setup_db (BklDB   *db,
          GError **error)
{
    if (register_fields (db->db, error) == FALSE) {
        return FALSE;
    }

    kozo_db_flush (db->db);

    return TRUE;
}

BklDB *
bkl_db_get (const char *db_name,
            GError    **error)
{
    BklDB *db;

    db = g_slice_new (BklDB);
    db->db = kozo_db_get (db_name, BICKLEY_DB_VERSION, error);
    if (db->db == NULL) {
        g_slice_free (BklDB, db);
        return NULL;
    }

    if (setup_db (db, error) == FALSE) {
        kozo_db_unref (db->db);
        if (db->dbus) {
            g_object_unref (db->dbus);
        }

        g_slice_free (BklDB, db);
        return NULL;
    }

    return db;
}

BklDB *
bkl_db_get_for_path (const char *path,
                     const char *db_name,
                     GError    **error)
{
    BklDB *db;

    db = g_slice_new (BklDB);
    db->db = kozo_db_get_for_path (path, db_name,
                                   BICKLEY_DB_VERSION, error);
    if (db->db == NULL) {
        g_slice_free (BklDB, db);
        return NULL;
    }

    if (setup_db (db, error) == FALSE) {
        kozo_db_unref (db->db);
        if (db->dbus) {
            g_object_unref (db->dbus);
        }

        g_slice_free (BklDB, db);
        return NULL;
    }

    return db;
}

BklDB *
bkl_db_get_for_name (const char *filename,
                     const char *db_name,
                     GError    **error)
{
    BklDB *db;

    db = g_slice_new (BklDB);
    db->db = kozo_db_get_for_name (filename, db_name,
                                   BICKLEY_DB_VERSION, error);
    if (db->db == NULL) {
        g_slice_free (BklDB, db);
        return NULL;
    }

    if (setup_db (db, error) == FALSE) {
        kozo_db_unref (db->db);
        if (db->dbus) {
            g_object_unref (db->dbus);
        }

        g_slice_free (BklDB, db);
        return NULL;
    }

    return db;
}

void
bkl_db_free (BklDB *db)
{
    kozo_db_unref (db->db);
    g_slice_free (BklDB, db);
}

gboolean
bkl_db_add_item (BklDB      *db,
                 const char *key,
                 BklItem    *item,
                 GError    **error)
{
    GSList *fields;
    gboolean ret;

    fields = bkl_item_get_fields (item);
    ret = kozo_db_add (db->db, key, fields, error);
    kozo_field_list_free (fields);

    return ret;
}

gboolean
bkl_db_replace_item (BklDB      *db,
                     const char *key,
                     BklItem    *item,
                     GError    **error)
{
    GSList *fields;
    gboolean ret;

    fields = bkl_item_get_fields (item);

    ret = kozo_db_set (db->db, key, fields, error);
    kozo_field_list_free (fields);

    return ret;
}

gboolean
bkl_db_remove (BklDB      *db,
               const char *key,
               GError    **error)
{
    return kozo_db_remove (db->db, key, error);
}

static BklItem *
item_from_entry (KozoEntry *entry,
                 gboolean   want_broken)
{
    BklItem *item;
    BklItemType type;
    KozoField *field;

    field = kozo_entry_get_field (entry, BKL_FIELD_TYPE);
    type = kozo_field_get_value_int (field, 0);
    kozo_field_free (field);

    switch (type) {
    case BKL_ITEM_TYPE_AUDIO:
        field = kozo_entry_get_field (entry, BKL_FIELD_AUDIO);
        item = (BklItem *) bkl_item_audio_new_from_field (field);
        kozo_field_free (field);
        break;

    case BKL_ITEM_TYPE_IMAGE:
        field = kozo_entry_get_field (entry, BKL_FIELD_IMAGE);
        item = (BklItem *) bkl_item_image_new_from_field (field);
        kozo_field_free (field);
        break;

    case BKL_ITEM_TYPE_VIDEO:
        field = kozo_entry_get_field (entry, BKL_FIELD_VIDEO);
        item = (BklItem *) bkl_item_video_new_from_field (field);
        kozo_field_free (field);
        break;

    case BKL_ITEM_TYPE_BROKEN:
        if (want_broken == FALSE) {
            return NULL;
        }

        field = kozo_entry_get_field (entry, BKL_FIELD_BROKEN);
        item = (BklItem *) bkl_item_broken_new_from_field (field);
        kozo_field_free (field);
        break;

    case BKL_ITEM_TYPE_PLAYLIST:
    default:
        return NULL;
    }

    field = kozo_entry_get_field (entry, BKL_FIELD_FILE);
    bkl_item_set_from_field (item, field);
    kozo_field_free (field);

    field = kozo_entry_get_field (entry, BKL_FIELD_EXTENDED);
    bkl_item_extended_set_from_field ((BklItemExtended *) item, field);
    kozo_field_free (field);

    return item;
}

BklItem *
bkl_db_get_item (BklDB      *db,
                 const char *key,
                 GError    **error)
{
    KozoEntry *entry;

    entry = kozo_db_lookup (db->db, key, NULL, error);
    if (entry == NULL) {
        return NULL;
    }

    return item_from_entry (entry, TRUE);
}

struct _build_list_struct {
    GPtrArray *array;
    gboolean include_broken;
};

static gboolean
build_item_list (KozoDB     *db,
                 const char *key,
                 KozoEntry  *entry,
                 gpointer    userdata)
{
    struct _build_list_struct *d = userdata;
    BklItem *item;

    item = item_from_entry (entry, d->include_broken);
    if (item != NULL) {
        g_ptr_array_add (d->array, item);
    }

    return TRUE;
}

GPtrArray *
bkl_db_get_items (BklDB   *db,
                  gboolean include_broken,
                  GError **error)
{
    struct _build_list_struct *d = g_newa (struct _build_list_struct, 1);

    d->array = g_ptr_array_new ();
    d->include_broken = include_broken;
    kozo_db_foreach (db->db, build_item_list, d);

    return d->array;
}

GSequence *
bkl_db_get_index_words (BklDB *db)
{
    GSequence *words;

    words = kozo_db_get_index_words (db->db);
    return words;
}

struct _DBForeachData {
    BklDB *db;
    BklDBForeachFunc func;
    gpointer data;
};

static gboolean
db_foreach_proxy (KozoDB     *db,
                  const char *key,
                  KozoEntry  *entry,
                  gpointer    data)
{
    struct _DBForeachData *closure = (struct _DBForeachData *) data;
    BklItem *item;
    gboolean ret;

    item = item_from_entry (entry, FALSE);
    if (item == NULL) {
        return TRUE;
    }

    ret = closure->func (closure->db, key, item, closure->data);

    g_object_unref (item);
    return ret;
}

void
bkl_db_foreach (BklDB           *db,
                BklDBForeachFunc func,
                gpointer         data)
{
    struct _DBForeachData closure;

    closure.db = db;
    closure.func = func;
    closure.data = data;
    kozo_db_foreach (db->db, db_foreach_proxy, &closure);
}
