/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2008  Marcel Holtmann <marcel@holtmann.org>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib/gi18n.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include "upa.h"

#include "dialog.h"
#include "tracer.h"
#include "import.h"
#include "export.h"

static GtkWidget *notebook = NULL;
static GtkWidget *menuitem_debug;
static GtkWidget *menuitem_close;
static GtkWidget *menuitem_print;
static GtkWidget *menuitem_saveas;
static GtkWidget *menuitem_save;
static GtkWidget *toolitem_save;
static GtkWidget *toolitem_stop;
static GtkWidget *toolitem_hci;
static GtkWidget *toolitem_l2cap;
static GtkWidget *toolbar_tracer;

static void draw_trace(GtkTreeModel *model, cairo_t *cr, PangoLayout *layout,
					gdouble width, gdouble height)
{
	GtkTreeIter iter;
	gboolean valid;
	gchar *packet;
	gdouble pos = 10;

#if 0
	cairo_set_line_width(cr, 2);

	cairo_move_to(cr, 10, 10);
	cairo_line_to(cr, width - 10, height - 10);
	cairo_stroke(cr);

	cairo_move_to(cr, width - 10, 10);
	cairo_line_to(cr, 10, height - 10);
	cairo_stroke(cr);
#endif

	valid = gtk_tree_model_get_iter_first(model, &iter);

	while (valid == TRUE) {
		int layout_width, layout_height;
		gdouble offset;

		gtk_tree_model_get(model, &iter, COLUMN_PACKET, &packet, -1);

		pango_layout_set_markup(layout, packet, -1);
		pango_layout_get_size(layout, &layout_width, &layout_height);

		offset = (layout_height / PANGO_SCALE) + 10;
		if (pos + offset > height - 5) {
			pango_layout_set_markup(layout, "...", -1);

			cairo_move_to(cr, 10, pos);
			pango_cairo_show_layout(cr, layout);
			break;
		}

		cairo_move_to(cr, 10, pos);
		pango_cairo_show_layout(cr, layout);

		pos += offset;

		valid = gtk_tree_model_iter_next(model, &iter);
	}
}

static void draw_page(GtkPrintOperation *operation,
		GtkPrintContext *context, gint page_nr, gpointer user_data)
{
	GtkTreeModel *model = user_data;
	cairo_t *cr;
	gdouble width, height;
	PangoLayout *layout;

	cr = gtk_print_context_get_cairo_context(context);
	width = gtk_print_context_get_width(context);
	height = gtk_print_context_get_height(context);
	layout = gtk_print_context_create_pango_layout(context);

	draw_trace(model, cr, layout, width, height);

	g_object_unref(layout);
}

static void print_callback(GtkWidget *item, gpointer user_data)
{
	GtkWidget *child;
	GtkTreeModel *model;
	GtkPrintOperation *print;
	GtkPrintOperationResult result;
	gint page;

	page = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
	if (page < 0)
		return;

	child = gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), page);

	model = g_object_get_data(G_OBJECT(child), "filter");
	if (model == NULL)
		model = g_object_get_data(G_OBJECT(child), "model");

	print = gtk_print_operation_new();

	gtk_print_operation_set_n_pages(print, 1);

	g_signal_connect(print, "draw_page",
				G_CALLBACK(draw_page), model);

	result = gtk_print_operation_run(print,
			GTK_PRINT_OPERATION_ACTION_PRINT_DIALOG, NULL, NULL);

	g_object_unref(print);
}

static void update_notebook_label(GtkNotebook *notebook, GtkWidget *child);

static gboolean save_command(GtkWidget *child, const gchar *currentname)
{
	GtkRecentManager *recent;
	GtkTreeModel *model;
	gchar *filename, *uri;
	gboolean saved = TRUE;

	filename = show_save_dialog(currentname);
	if (filename == NULL)
		return FALSE;

	model = g_object_get_data(G_OBJECT(child), "model");

	if (export_model(model, filename) < 0) {
		show_error_dialog(_("Saving of protocol trace failed"));
		g_free(filename);
		return FALSE;
	}

	g_object_set_data(G_OBJECT(child), "filename", filename);

	g_object_set_data(G_OBJECT(child), "saved",
					GUINT_TO_POINTER(saved));

	update_notebook_label(GTK_NOTEBOOK(notebook), child);

	gtk_widget_set_sensitive(toolitem_save, !saved);
	gtk_widget_set_sensitive(menuitem_save, !saved);
	gtk_widget_set_sensitive(menuitem_saveas, saved);

	recent = gtk_recent_manager_get_default();

	uri = g_strconcat("file://", filename, NULL);
	gtk_recent_manager_add_item(recent, uri);
	g_free(uri);

	return TRUE;
}

static void save_callback(GtkWidget *item, gpointer user_data)
{
	GtkWidget *child;
	gint page;

	page = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
	if (page < 0)
		return;

	child = gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), page);

	save_command(child, NULL);
}

static void saveas_callback(GtkWidget *item, gpointer user_data)
{
	GtkWidget *child;
	gchar *filename;
	gint page;

	page = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
	if (page < 0)
		return;

	child = gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), page);

	filename = g_object_get_data(G_OBJECT(child), "filename");

	save_command(child, filename);
}

static void stop_callback(GtkWidget *item, gpointer user_data)
{
	GtkWidget *child = user_data;
	GtkTreeModel *model;
	gint page;

	if (child == NULL) {
		page = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
		child = gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), page);
		if (child == NULL)
			return;
	}

	model = g_object_get_data(G_OBJECT(child), "model");
	if (model == NULL)
		return;

	finish_import(model);
}

static void close_callback(GtkWidget *item, gpointer user_data)
{
	GtkWidget *child = user_data;
	GtkTreeModel *model;
	gboolean saved;
	gint page;

	if (child == NULL) {
		page = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
		child = gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), page);
	} else
		page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook), child);

	if (page >= 0) {
		model = g_object_get_data(G_OBJECT(child), "model");
		if (model)
			finish_import(model);

		saved = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(child),
								"saved"));
		if (saved == FALSE) {
			gchar *result = show_close_dialog();
			if (result == NULL)
				return;

			if (g_ascii_strcasecmp(result, "save") == 0)
				saved = save_command(child, NULL);
			else
				saved = TRUE;

			g_free(result);

			if (saved == FALSE)
				return;
		}

		if (model)
			g_object_unref(model);

		gtk_notebook_remove_page(GTK_NOTEBOOK(notebook), page);
	}

	if (gtk_notebook_get_n_pages(GTK_NOTEBOOK(notebook)) == 0) {
		gtk_widget_set_sensitive(menuitem_debug, FALSE);
		gtk_widget_set_sensitive(menuitem_close, FALSE);
		gtk_widget_set_sensitive(menuitem_print, FALSE);
		gtk_widget_set_sensitive(menuitem_saveas, FALSE);
		gtk_widget_set_sensitive(menuitem_save, FALSE);
		gtk_widget_set_sensitive(toolitem_save, FALSE);
		gtk_widget_hide(toolbar_tracer);
	}
}

static void update_notebook_label(GtkNotebook *notebook, GtkWidget *child)
{
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *image;
	GtkWidget *button;
	gchar *string, *title, *name, *basename = NULL;
	gboolean saved, running;

	running = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(child),
								"running"));

	saved = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(child), "saved"));

	hbox = gtk_hbox_new(FALSE, 2);

	if (running == TRUE) {
		if (saved == TRUE) {
			title = _("Loading Trace");
			name = g_object_get_data(G_OBJECT(child), "filename");
			basename = g_filename_display_basename(name);
		} else {
			title = _("Live Import");
			name = g_object_get_data(G_OBJECT(child), "hostname");
		}
	} else {
		title = _("Protocol Trace");
		if (saved) {
			name = g_object_get_data(G_OBJECT(child), "filename");
			basename = g_filename_display_basename(name);
		} else
			name = _("Unsaved");
	}

	label = gtk_label_new(NULL);
	string = g_strdup_printf("<b>%s</b>\n<small>%s</small>", title,
						basename ? basename : name);
	g_free(basename);
	gtk_label_set_markup(GTK_LABEL(label), string);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
	g_free(string);
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 2);

	button = gtk_button_new();
	gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
	gtk_button_set_focus_on_click(GTK_BUTTON(button), FALSE);
	image = gtk_image_new_from_stock(GTK_STOCK_CLOSE, GTK_ICON_SIZE_MENU);
	gtk_button_set_image(GTK_BUTTON(button), image);
	gtk_widget_show(image);
	gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(button), "clicked",
					G_CALLBACK(close_callback), child);

	g_object_set_data(G_OBJECT(child), "button", button);

	gtk_widget_show_all(hbox);

	gtk_notebook_set_tab_label(GTK_NOTEBOOK(notebook), child, hbox);
}

static void tracer_callback(GObject *object,
				GParamSpec *spec, gpointer user_data)
{
	GtkWidget *child = user_data;
	GtkTreeModel *model;
	gboolean active = FALSE;
	gint page;

	if (child == NULL)
		return;

	g_object_set_data(G_OBJECT(child), "running", GUINT_TO_POINTER(active));

	model = g_object_get_data(G_OBJECT(child), "model");
	if (model == NULL)
		return;

	analyze_import(model);

	update_notebook_label(GTK_NOTEBOOK(notebook), child);

	page = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
	if (page < 0)
		return;

	if (child != gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), page))
		return;

	gtk_widget_set_sensitive(toolitem_stop, FALSE);
	gtk_widget_set_sensitive(toolitem_hci, TRUE);
	gtk_widget_set_sensitive(toolitem_l2cap, TRUE);

	active = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(child), "saved"));

	gtk_widget_set_sensitive(toolitem_save, !active);
	gtk_widget_set_sensitive(menuitem_save, !active);
	gtk_widget_set_sensitive(menuitem_saveas, active);
	gtk_widget_set_sensitive(menuitem_print, TRUE);

	g_object_unref(child);
}

static gint append_notebook_page(GtkNotebook *notebook, GtkWidget *child)
{
	gint page;

	g_object_set_data(G_OBJECT(child), "hci", GUINT_TO_POINTER(TRUE));
	g_object_set_data(G_OBJECT(child), "l2cap", GUINT_TO_POINTER(FALSE));

	page = gtk_notebook_append_page(GTK_NOTEBOOK(notebook), child, NULL);
	if (page < 0)
		return -1;

	gtk_notebook_set_tab_label_packing(GTK_NOTEBOOK(notebook), child,
						FALSE, TRUE, GTK_PACK_START);

	update_notebook_label(notebook, child);

	gtk_notebook_set_tab_reorderable(GTK_NOTEBOOK(notebook), child, TRUE);

	gtk_widget_show(child);

	return page;
}

static void protocol_callback(GtkWidget *item, gpointer user_data)
{
	gchar *key = user_data;
	GtkWidget *child;
	GtkWidget *tree;
	GtkTreeModel *model;
	guint flags, layer;
	gboolean active;
	gint page;

	if (notebook == NULL)
		return;

	page = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
	if (page < 0)
		return;

	child = gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), page);

	active = gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(item));

	g_object_set_data(G_OBJECT(child), key, GUINT_TO_POINTER(active));

	flags = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(child), "flags"));

	if (g_strcasecmp(key, "hci") == 0)
		layer = LAYER_HCI;

	if (g_strcasecmp(key, "l2cap") == 0)
		layer = LAYER_L2CAP;

	if (active)
		flags |= (1 << layer);
	else
		flags &= ~(1 << layer);

	g_object_set_data(G_OBJECT(child), "flags", GUINT_TO_POINTER(flags));

	tree = g_object_get_data(G_OBJECT(child), "tree");

	model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree));

	gtk_tree_model_filter_refilter(GTK_TREE_MODEL_FILTER(model));
}

static void notebook_callback(GtkNotebook *notebook, GtkNotebookPage *page,
					guint page_num, gpointer user_data)
{
	GtkWidget *child = gtk_notebook_get_nth_page(notebook, page_num);
	gint i, n_pages = gtk_notebook_get_n_pages(notebook);
	gboolean active;

	active = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(child), "running"));
	gtk_widget_set_sensitive(toolitem_stop, active);
	gtk_widget_set_sensitive(toolitem_hci, !active);
	gtk_widget_set_sensitive(toolitem_l2cap, !active);

	if (active == FALSE) {
		active = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(child),
								"saved"));
		gtk_widget_set_sensitive(menuitem_print, TRUE);
		gtk_widget_set_sensitive(menuitem_saveas, active);
		gtk_widget_set_sensitive(menuitem_save, !active);
		gtk_widget_set_sensitive(toolitem_save, !active);
	} else {
		gtk_widget_set_sensitive(menuitem_print, FALSE);
		gtk_widget_set_sensitive(menuitem_saveas, FALSE);
		gtk_widget_set_sensitive(menuitem_save, FALSE);
		gtk_widget_set_sensitive(toolitem_save, FALSE);
	}

	g_signal_handlers_block_by_func(G_OBJECT(toolitem_hci),
					G_CALLBACK(protocol_callback), "hci");
	g_signal_handlers_block_by_func(G_OBJECT(toolitem_l2cap),
					G_CALLBACK(protocol_callback), "l2cap");

	active = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(child), "hci"));
	gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(toolitem_hci),
									active);

	active = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(child), "l2cap"));
	gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(toolitem_l2cap),
									active);

	g_signal_handlers_unblock_by_func(G_OBJECT(toolitem_l2cap),
					G_CALLBACK(protocol_callback), "l2cap");
	g_signal_handlers_unblock_by_func(G_OBJECT(toolitem_hci),
					G_CALLBACK(protocol_callback), "hci");

	for (i = 0; i < n_pages; i++) {
		GtkWidget *button;

		child = gtk_notebook_get_nth_page(notebook, i);

		button = g_object_get_data(G_OBJECT(child), "button");
		if (button)
			gtk_widget_set_sensitive(button, i == page_num);
	}
}

static gint open_file(const gchar *filename)
{
	GtkRecentManager *recent;
	GtkWidget *child;
	gchar *uri;
	gint page;

	child = create_tracer_from_file(filename, G_CALLBACK(tracer_callback));
	if (child == NULL)
		return -1;

	recent = gtk_recent_manager_get_default();

	uri = g_strconcat("file://", filename, NULL);
	gtk_recent_manager_add_item(recent, uri);
	g_free(uri);

	page = append_notebook_page(GTK_NOTEBOOK(notebook), child);
	if (page < 0) {
		gtk_widget_destroy(child);
		return -1;
	}

	gtk_widget_set_sensitive(menuitem_debug, TRUE);
	gtk_widget_set_sensitive(menuitem_close, TRUE);
	gtk_widget_show(toolbar_tracer);

	return page;
}

static void open_callback(GtkWidget *item, gpointer user_data)
{
	gchar *filename;
	gint page;

	filename = show_open_dialog();
	if (filename == NULL)
		return;

	page = open_file(filename);

	g_free(filename);

	if (page < 0)
		return;

	gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), page);
}

static void recent_callback(GtkRecentChooser *chooser, gpointer user_data)
{
	GtkWidget *child;
	gchar *uri, *filename;
	gint page;

	uri = gtk_recent_chooser_get_current_uri(chooser);
	filename = g_filename_from_uri(uri, NULL, NULL);
	g_free(uri);

	child = create_tracer_from_file(filename, G_CALLBACK(tracer_callback));

	g_free(filename);

	if (child == NULL)
		return;

	page = append_notebook_page(GTK_NOTEBOOK(notebook), child);

	gtk_widget_set_sensitive(menuitem_debug, TRUE);
	gtk_widget_set_sensitive(menuitem_close, TRUE);
	gtk_widget_show(toolbar_tracer);

	gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), page);
}

static void new_callback(GtkWidget *item, gpointer user_data)
{
	GtkWidget *child;
	gchar *hostname;
	gint page;

	hostname = show_host_dialog();
	if (hostname == NULL)
		return;

	child = create_live_import(hostname, G_CALLBACK(tracer_callback));
	if (child == NULL) {
		gchar *message = g_strdup_printf("%s <i>%s</i>",
					_("Can't live import from"), hostname);
		show_error_dialog(message);
		g_free(message);
		g_free(hostname);
		return;
	}

	g_free(hostname);

	page = append_notebook_page(GTK_NOTEBOOK(notebook), child);

	gtk_widget_set_sensitive(menuitem_debug, TRUE);
	gtk_widget_set_sensitive(menuitem_close, TRUE);
	gtk_widget_show(toolbar_tracer);

	gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), page);
}

static gboolean check_unsaved(void)
{
	gint i, n_pages = gtk_notebook_get_n_pages(GTK_NOTEBOOK(notebook));
	guint unsaved = 0;

	for (i = 0; i < n_pages; i++) {
		GtkWidget *child;
		gpointer saved;

		child = gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), i);
		if (child == NULL)
			continue;

		saved = g_object_get_data(G_OBJECT(child), "saved");
		if (GPOINTER_TO_UINT(saved) == FALSE)
			unsaved++;
	}

	if (unsaved > 0)
		return show_quit_dialog(unsaved);

	return TRUE;
}

static void quit_callback(GtkWidget *item, gpointer user_data)
{
	if (check_unsaved() == FALSE)
		return;

	gtk_main_quit();
}

static void fullscreen_callback(GtkWidget *item, gpointer user_data)
{
	GtkWidget *window = user_data;
	gboolean active;

	active = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(item));

	if (active)
		gtk_window_fullscreen(GTK_WINDOW(window));
	else
		gtk_window_unfullscreen(GTK_WINDOW(window));
}

static gboolean print_packet(GtkTreeModel *model, GtkTreePath *path,
					GtkTreeIter *iter, gpointer user_data)
{
	UpaPacket *packet;

	gtk_tree_model_get(model, iter, COLUMN_UPA, &packet, -1);

	if (packet != NULL) {
		upa_packet_print(packet);
		g_object_unref(packet);
	}

	return FALSE;
}

static void debug_callback(GtkWidget *item, gpointer user_data)
{
	GtkWidget *child;
	GtkTreeModel *model;
	gint page;

	page = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
	if (page < 0)
		return;

	child = gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), page);

	model = g_object_get_data(G_OBJECT(child), "model");

	gtk_tree_model_foreach(model, print_packet, NULL);
}

static void about_callback(GtkWidget *item, gpointer user_data)
{
	show_about_dialog();
}

static gboolean delete_callback(GtkWidget *window, GdkEvent *event,
						gpointer user_data)
{
	if (check_unsaved() == FALSE)
		return TRUE;

	gtk_widget_destroy(window);

	gtk_main_quit();

	return FALSE;
}

static GtkWidget *create_tracer_toolbar(void)
{
	GtkWidget *toolbar;
	GtkWidget *label;
	GtkToolItem *item;

	toolbar = gtk_toolbar_new();
	gtk_toolbar_set_show_arrow(GTK_TOOLBAR(toolbar), FALSE);

	item = gtk_separator_tool_item_new();
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), item, -1);

	item = gtk_tool_button_new_from_stock(GTK_STOCK_STOP);
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), item, -1);

	toolitem_stop = GTK_WIDGET(item);
	g_signal_connect(item, "clicked",
				G_CALLBACK(stop_callback), NULL);

	item = gtk_tool_item_new();
	label = gtk_label_new(NULL);
	gtk_label_set_markup(GTK_LABEL(label), "<small>Packet\nFilters</small>");
	gtk_misc_set_padding(GTK_MISC(label), 6, 0);
	gtk_container_add(GTK_CONTAINER(item), label);
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), item, -1);

#if 0
	item = gtk_toggle_tool_button_new();
	gtk_tool_button_set_label(GTK_TOOL_BUTTON(item), "BB");
	gtk_tool_item_set_is_important(item, TRUE);
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), item, -1);
	gtk_widget_set_sensitive(GTK_WIDGET(item), FALSE);
#endif

	item = gtk_toggle_tool_button_new();
	gtk_tool_button_set_label(GTK_TOOL_BUTTON(item), "HCI");
	gtk_tool_item_set_is_important(item, TRUE);
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), item, -1);

	toolitem_hci = GTK_WIDGET(item);
	g_signal_connect(item, "toggled",
				G_CALLBACK(protocol_callback), "hci");

	item = gtk_separator_tool_item_new();
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), item, -1);

	item = gtk_tool_item_new();
	label = gtk_label_new(NULL);
	gtk_label_set_markup(GTK_LABEL(label), "<small>Protocol\nFilters</small>");
	gtk_misc_set_padding(GTK_MISC(label), 6, 0);
	gtk_container_add(GTK_CONTAINER(item), label);
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), item, -1);

#if 0
	item = gtk_toggle_tool_button_new();
	gtk_tool_button_set_label(GTK_TOOL_BUTTON(item), "LMP");
	gtk_tool_item_set_is_important(item, TRUE);
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), item, -1);
	gtk_widget_set_sensitive(GTK_WIDGET(item), FALSE);
#endif

	item = gtk_toggle_tool_button_new();
	gtk_tool_button_set_label(GTK_TOOL_BUTTON(item), "L2CAP");
	gtk_tool_item_set_is_important(item, TRUE);
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), item, -1);

	toolitem_l2cap = GTK_WIDGET(item);
	g_signal_connect(item, "toggled",
				G_CALLBACK(protocol_callback), "l2cap");

	item = gtk_separator_tool_item_new();
	gtk_separator_tool_item_set_draw(GTK_SEPARATOR_TOOL_ITEM(item), FALSE);
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), item, -1);

	return toolbar;
}

static GtkWidget *create_main_toolbar(GtkWidget *window)
{
	GtkWidget *toolbar;
	GtkToolItem *item;

	toolbar = gtk_toolbar_new();
	gtk_toolbar_set_show_arrow(GTK_TOOLBAR(toolbar), FALSE);

	item = gtk_tool_button_new_from_stock(GTK_STOCK_NEW);
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), item, -1);
	g_signal_connect(item, "clicked", G_CALLBACK(new_callback), NULL);

	item = gtk_tool_button_new_from_stock(GTK_STOCK_OPEN);
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), item, -1);
	g_signal_connect(item, "clicked", G_CALLBACK(open_callback), NULL);

	item = gtk_tool_button_new_from_stock(GTK_STOCK_SAVE);
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), item, -1);
	g_signal_connect(item, "clicked", G_CALLBACK(save_callback), NULL);

	toolitem_save = GTK_WIDGET(item);
	gtk_widget_set_sensitive(GTK_WIDGET(item), FALSE);

	return toolbar;
}

static void create_window(void)
{
	GtkRecentFilter *filter;
	GtkAccelGroup *accel;
	GtkWidget *window;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *menubar;
	GtkWidget *item;
	GtkWidget *menu;
	GtkWidget *recent;
	GtkWidget *toolbar;
	GtkWidget *statusbar;

	accel = gtk_accel_group_new();

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(window), _("Bluetooth Analyzer"));
	gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER);
	gtk_window_set_default_size(GTK_WINDOW(window), 850, 650);
	g_signal_connect(G_OBJECT(window), "delete-event",
					G_CALLBACK(delete_callback), NULL);

	gtk_window_add_accel_group(GTK_WINDOW(window), accel);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(window), vbox);

	menubar = gtk_menu_bar_new();
	gtk_box_pack_start(GTK_BOX(vbox), menubar, FALSE, FALSE, 0);

	item = gtk_menu_item_new_with_mnemonic(_("_File"));
	gtk_container_add(GTK_CONTAINER(menubar), item);
	menu = gtk_menu_new();
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), menu);

	item = gtk_image_menu_item_new_from_stock(GTK_STOCK_NEW, accel);
	gtk_container_add(GTK_CONTAINER(menu), item);
	g_signal_connect(item, "activate", G_CALLBACK(new_callback), NULL);
	gtk_widget_add_accelerator(item, "activate", accel,
					GDK_F4, 0, GTK_ACCEL_VISIBLE);

	item = gtk_image_menu_item_new_from_stock(GTK_STOCK_OPEN, accel);
	gtk_container_add(GTK_CONTAINER(menu), item);
	g_signal_connect(item, "activate", G_CALLBACK(open_callback), NULL);
	gtk_widget_add_accelerator(item, "activate", accel,
					GDK_F5, 0, GTK_ACCEL_VISIBLE);

	filter = gtk_recent_filter_new();
	gtk_recent_filter_add_pattern(filter, "*.hcidump");
	gtk_recent_filter_add_pattern(filter, "*.btsnoop");
	gtk_recent_filter_add_pattern(filter, "*.pklg");

	recent = gtk_recent_chooser_menu_new();
	gtk_recent_chooser_add_filter(GTK_RECENT_CHOOSER(recent), filter);
	gtk_recent_chooser_set_local_only(GTK_RECENT_CHOOSER(recent), TRUE);
	gtk_recent_chooser_set_limit(GTK_RECENT_CHOOSER(recent), 10);

	g_signal_connect(G_OBJECT(recent), "item-activated",
					G_CALLBACK(recent_callback), NULL);

	item = gtk_menu_item_new_with_mnemonic(_("Open _Recent"));
	gtk_container_add(GTK_CONTAINER(menu), item);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), recent);

	item = gtk_separator_menu_item_new();
	gtk_container_add(GTK_CONTAINER(menu), item);

	item = gtk_image_menu_item_new_from_stock(GTK_STOCK_SAVE, accel);
	gtk_container_add(GTK_CONTAINER(menu), item);
	g_signal_connect(item, "activate", G_CALLBACK(save_callback), NULL);

	menuitem_save = item;
	gtk_widget_set_sensitive(menuitem_save, FALSE);

	item = gtk_image_menu_item_new_from_stock(GTK_STOCK_SAVE_AS, accel);
	gtk_container_add(GTK_CONTAINER(menu), item);
	g_signal_connect(item, "activate", G_CALLBACK(saveas_callback), NULL);

	menuitem_saveas = item;
	gtk_widget_set_sensitive(menuitem_saveas, FALSE);

	item = gtk_separator_menu_item_new();
	gtk_container_add(GTK_CONTAINER(menu), item);

	item = gtk_image_menu_item_new_from_stock(GTK_STOCK_PRINT, accel);
	gtk_container_add(GTK_CONTAINER(menu), item);
	g_signal_connect(item, "activate", G_CALLBACK(print_callback), NULL);

	menuitem_print = item;
	gtk_widget_set_sensitive(menuitem_print, FALSE);

	item = gtk_separator_menu_item_new();
	gtk_container_add(GTK_CONTAINER(menu), item);

	item = gtk_image_menu_item_new_from_stock(GTK_STOCK_CLOSE, accel);
	gtk_container_add(GTK_CONTAINER(menu), item);
	g_signal_connect(item, "activate", G_CALLBACK(close_callback), NULL);
	gtk_widget_add_accelerator(item, "activate", accel,
					GDK_F6, 0, GTK_ACCEL_VISIBLE);

	menuitem_close = item;
	gtk_widget_set_sensitive(menuitem_close, FALSE);

	item = gtk_image_menu_item_new_from_stock(GTK_STOCK_QUIT, accel);
	gtk_container_add(GTK_CONTAINER(menu), item);
	g_signal_connect(item, "activate", G_CALLBACK(quit_callback), NULL);

	item = gtk_menu_item_new_with_mnemonic(_("_Edit"));
	gtk_container_add(GTK_CONTAINER(menubar), item);
	menu = gtk_menu_new();
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), menu);

	item = gtk_image_menu_item_new_from_stock(GTK_STOCK_PREFERENCES, accel);
	gtk_widget_set_sensitive(item, FALSE);
	gtk_container_add(GTK_CONTAINER(menu), item);

	item = gtk_menu_item_new_with_mnemonic(_("_View"));
	gtk_container_add(GTK_CONTAINER(menubar), item);
	menu = gtk_menu_new();
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), menu);

	item = gtk_check_menu_item_new_with_mnemonic(_("_Full Screen"));
	gtk_container_add(GTK_CONTAINER(menu), item);
	g_signal_connect(item, "toggled",
				G_CALLBACK(fullscreen_callback), window);
	gtk_widget_add_accelerator(item, "activate", accel,
					GDK_F11, 0, GTK_ACCEL_VISIBLE);

	item = gtk_menu_item_new_with_mnemonic(_("_Tools"));
	gtk_container_add(GTK_CONTAINER(menubar), item);
	menu = gtk_menu_new();
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), menu);

	item = gtk_image_menu_item_new_with_mnemonic(_("_Debug Packet List"));
	gtk_container_add(GTK_CONTAINER(menu), item);
	g_signal_connect(item, "activate", G_CALLBACK(debug_callback), NULL);
	gtk_widget_add_accelerator(item, "activate", accel,
					GDK_F12, 0, GTK_ACCEL_VISIBLE);

	menuitem_debug = item;
	gtk_widget_set_sensitive(menuitem_debug, FALSE);

	item = gtk_menu_item_new_with_mnemonic(_("_Help"));
	gtk_container_add(GTK_CONTAINER(menubar), item);
	menu = gtk_menu_new();
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), menu);

	item = gtk_image_menu_item_new_from_stock(GTK_STOCK_ABOUT, NULL);
	gtk_container_add(GTK_CONTAINER(menu), item);
	g_signal_connect(item, "activate", G_CALLBACK(about_callback), NULL);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	toolbar = create_main_toolbar(window);
	gtk_box_pack_start(GTK_BOX(hbox), toolbar, FALSE, TRUE, 0);

	toolbar = create_tracer_toolbar();
	gtk_box_pack_start(GTK_BOX(hbox), toolbar, FALSE, TRUE, 0);
	toolbar_tracer = toolbar;

	toolbar = gtk_toolbar_new();
	gtk_toolbar_set_show_arrow(GTK_TOOLBAR(toolbar), FALSE);
	gtk_box_pack_start(GTK_BOX(hbox), toolbar, TRUE, TRUE, 0);

	notebook = gtk_notebook_new();
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(notebook), TRUE);
	gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), FALSE);
	gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 4);
	g_signal_connect(notebook, "switch-page",
				G_CALLBACK(notebook_callback), NULL);

	statusbar = gtk_statusbar_new();
	gtk_box_pack_end(GTK_BOX(vbox), statusbar, FALSE, FALSE, 0);

	gtk_widget_grab_focus(GTK_WIDGET(notebook));

	gtk_widget_show_all(window);

	gtk_widget_hide(toolbar_tracer);
}

static GOptionEntry options[] = {
	{ NULL },
};

int main(int argc, char *argv[])
{
	GError *error = NULL;
	int i;

	bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
	textdomain(GETTEXT_PACKAGE);

	if (gtk_init_with_args(&argc, &argv, "[FILE...]",
				options, GETTEXT_PACKAGE, &error) == FALSE) {
		if (error) {
			g_print("%s\n", error->message);
			g_error_free(error);
		} else
			g_print("An unknown error occurred\n");

		gtk_exit(1);
	}

	g_set_application_name(_("Bluetooth Analyzer"));

	gtk_window_set_default_icon_name("bluetooth");

	create_window();

	for (i = 1; i < argc; i++)
		open_file(argv[i]);

	gtk_main();

	return 0;
}
