#include <stdio.h>

#include "bognor-marshal.h"
#include "bognor-queue.h"

enum {
    PROP_0,
};

enum {
    ADDED,
    REMOVED,
    NOW_PLAYING,
    POSITION_CHANGED,
    PLAYING_CHANGED,
    LAST_SIGNAL
};

struct _BognorQueuePrivate {
    GQueue *play_queue;

    char *name;
    gboolean playing;

    BognorQueueItem *current_item;

    gboolean can_play_visual;
};

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), BOGNOR_TYPE_QUEUE, BognorQueuePrivate))
G_DEFINE_TYPE (BognorQueue, bognor_queue, G_TYPE_OBJECT);
static guint32 signals[LAST_SIGNAL] = {0, };

static gboolean bognor_queue_stop (BognorQueue *queue,
                                   GError     **error);
static gboolean bognor_queue_next (BognorQueue *queue,
                                   GError     **error);
static gboolean bognor_queue_play_uri (BognorQueue *queue,
                                       const char  *uri,
                                       const char  *mimetype,
                                       GError     **error);
static gboolean bognor_queue_get_playing (BognorQueue *queue,
                                          gboolean    *playing,
                                          GError     **error);

static gboolean bognor_queue_get_now_playing (BognorQueue *queue,
                                              char       **audio_uri,
                                              char       **visual_uri,
                                              GError     **error);
static gboolean bognor_queue_list_uris (BognorQueue *queue,
                                        char      ***uris,
                                        GError     **error);
static gboolean bognor_queue_add_uri (BognorQueue *queue,
                                      const char  *uri,
                                      const char  *mimetype,
                                      GError     **error);
static gboolean bognor_queue_clear (BognorQueue *queue,
                                    GError     **error);
static gboolean bognor_queue_remove (BognorQueue *queue,
                                     int          index,
                                     GError     **error);
static gboolean bognor_queue_insert_uri (BognorQueue *queue,
                                         const char  *uri,
                                         const char  *mimetype,
                                         int          position,
                                         GError     **error);
static gboolean bognor_queue_get_name (BognorQueue *queue,
                                       char       **name,
                                       GError     **error);
static gboolean bognor_queue_set_position (BognorQueue *queue,
                                           int          type,
                                           double       position,
                                           GError     **error);
static gboolean bognor_queue_get_position (BognorQueue *queue,
                                           int          type,
                                           double      *position,
                                           GError     **error);

#include "bognor-queue-glue.h"

static void
free_queue_item (BognorQueueItem *item)
{
    g_free (item->uri);
    g_free (item->mimetype);
    g_slice_free (BognorQueueItem, item);
}

gboolean
bognor_queue_play (BognorQueue *queue,
                   GError     **error)
{
    BognorQueueClass *klass = BOGNOR_QUEUE_GET_CLASS (queue);
    BognorQueuePrivate *priv = queue->priv;

    if (priv->current_item == NULL) {
        bognor_queue_get_next_item (queue);
    }

    if (priv->current_item) {
        if (priv->current_item->type == AUDIO_TYPE) {
            klass->set_audio_playing (queue, TRUE);
        } else {
            klass->set_visual_playing (queue, TRUE);
        }
    }

    priv->playing = TRUE;
    g_signal_emit (queue, signals[PLAYING_CHANGED], 0, TRUE);
    return TRUE;
}

static gboolean
bognor_queue_stop (BognorQueue *queue,
                   GError     **error)
{
    BognorQueueClass *klass = BOGNOR_QUEUE_GET_CLASS (queue);
    BognorQueuePrivate *priv = queue->priv;

    klass->set_audio_playing (queue, FALSE);
    klass->set_visual_playing (queue, FALSE);

#if 0
    g_signal_emit (queue, signals[NOW_PLAYING], 0,
                   NULL, AUDIO_TYPE);
    g_signal_emit (queue, signals[NOW_PLAYING], 0,
                   NULL, VISUAL_TYPE);
#endif

    priv->playing = FALSE;
    g_signal_emit (queue, signals[PLAYING_CHANGED], 0, FALSE);
    return TRUE;
}

static gboolean
bognor_queue_next (BognorQueue *queue,
                   GError     **error)
{
    bognor_queue_play_next (queue);
    return TRUE;
}

static gboolean
bognor_queue_play_uri (BognorQueue *queue,
                       const char  *uri,
                       const char  *mimetype,
                       GError     **error)
{
    BognorQueueClass *klass = BOGNOR_QUEUE_GET_CLASS (queue);
    BognorQueuePrivate *priv = queue->priv;
    BognorQueueItem *item;

    item = g_slice_new (BognorQueueItem);
    item->uri = g_strdup (uri);
    item->mimetype = g_strdup (mimetype);
    if (g_str_has_prefix (item->mimetype, "audio/")) {
        item->type = AUDIO_TYPE;
    } else {
        item->type = VISUAL_TYPE;
    }

    if (priv->current_item) {
        free_queue_item (priv->current_item);
    }
    priv->current_item = item;

    klass->set_uri (queue, priv->current_item);

    if (item->type == VISUAL_TYPE &&
        priv->can_play_visual == FALSE) {
        klass->force_visual (queue);
    }

    if (item->type == VISUAL_TYPE) {
        klass->set_visual_playing (queue, TRUE);
        klass->set_audio_playing (queue, FALSE);
        g_signal_emit (queue, signals[NOW_PLAYING], 0, uri, VISUAL_TYPE);
        g_signal_emit (queue, signals[NOW_PLAYING], 0, "", AUDIO_TYPE);
    } else if (item->type == AUDIO_TYPE) {
        klass->set_visual_playing (queue, FALSE);
        klass->set_audio_playing (queue, TRUE);
        g_signal_emit (queue, signals[NOW_PLAYING], 0, uri, AUDIO_TYPE);
        g_signal_emit (queue, signals[NOW_PLAYING], 0, "", VISUAL_TYPE);
    }

    priv->playing = TRUE;
    klass->add_item_to_recent (queue, item);
    g_signal_emit (queue, signals[PLAYING_CHANGED], 0, TRUE);
    return TRUE;
}

static gboolean
bognor_queue_get_playing (BognorQueue *queue,
                          gboolean    *playing,
                          GError     **error)
{
    BognorQueuePrivate *priv = queue->priv;

    *playing = priv->playing;
    return TRUE;
}

static gboolean
bognor_queue_list_uris (BognorQueue *queue,
                        char      ***uris,
                        GError     **error)
{
    BognorQueuePrivate *priv = queue->priv;
    char **uri_array;
    GList *u = NULL;
    int n_uris, i;

    n_uris = g_queue_get_length (priv->play_queue);
    uri_array = g_new (char *, n_uris + 1);

    /* Wonder if you're allowed to do this */
    for (u = priv->play_queue->head, i = 0; u; u = u->next, i++) {
        BognorQueueItem *item = u->data;

        uri_array[i] = g_strdup (item->uri);
    }

    /* NULL terminate the array */
    uri_array[n_uris] = NULL;
    *uris = uri_array;

    return TRUE;
}

static gboolean
bognor_queue_get_now_playing (BognorQueue *queue,
                              char       **audio_uri,
                              char       **visual_uri,
                              GError     **error)
{
    BognorQueuePrivate *priv = queue->priv;

    if (priv->current_item == NULL) {
        *audio_uri = NULL;
        *visual_uri = NULL;
    } else {
        if (priv->current_item->type == AUDIO_TYPE) {
            *audio_uri = g_strdup (priv->current_item->uri);
        } else {
            *visual_uri = g_strdup (priv->current_item->uri);
        }
    }

    return TRUE;
}

static gboolean
bognor_queue_add_uri (BognorQueue *queue,
                      const char  *uri,
                      const char  *mimetype,
                      GError     **error)
{
    BognorQueuePrivate *priv = queue->priv;
    BognorQueueItem *item;
    gboolean empty;

    item = g_slice_new (BognorQueueItem);
    item->uri = g_strdup (uri);
    item->mimetype = g_strdup (mimetype);
    if (g_str_has_prefix (item->mimetype, "audio/")) {
        item->type = AUDIO_TYPE;
    } else {
        item->type = VISUAL_TYPE;
    }

    g_queue_push_tail (priv->play_queue, item);

    g_signal_emit (queue, signals[ADDED], 0, uri,
                   g_queue_get_length (priv->play_queue) - 1);
    return TRUE;
}

static gboolean
bognor_queue_clear (BognorQueue *queue,
                    GError     **error)
{
    BognorQueuePrivate *priv = queue->priv;
    BognorQueueItem *item;

    while ((item = g_queue_pop_tail (priv->play_queue))) {
        g_signal_emit (queue, signals[REMOVED], 0, item->uri,
                       g_queue_get_length (priv->play_queue));

        free_queue_item (item);
    }

    return TRUE;
}

static gboolean
bognor_queue_remove_uri (BognorQueue *queue,
                         const char  *uri,
                         GError     **error)
{
    BognorQueuePrivate *priv = queue->priv;
    GList *l;
    int position = 0;

    for (l = priv->play_queue->head; l; l = l->next) {
        BognorQueueItem *item = l->data;

        if (g_str_equal (item->uri, uri)) {
            g_queue_delete_link (priv->play_queue, l);

            g_signal_emit (queue, signals[REMOVED], 0, item->uri, position);
            g_free (item->uri);
            g_free (item->mimetype);
            g_slice_free (BognorQueueItem, item);

            break;
        }

        position++;
    }

    return TRUE;
}

static gboolean
bognor_queue_remove (BognorQueue *queue,
                     int          index,
                     GError     **error)
{
    BognorQueuePrivate *priv = queue->priv;
    BognorQueueItem *item;

    item = g_queue_pop_nth (priv->play_queue, index);

    if (item) {
        g_print ("Removing %s from %d\n", item->uri, index);
        g_signal_emit (queue, signals[REMOVED], 0, item->uri, index);
        g_free (item->uri);
        g_free (item->mimetype);
        g_slice_free (BognorQueueItem, item);
    }

    return TRUE;
}

static gboolean
bognor_queue_insert_uri (BognorQueue *queue,
                         const char  *uri,
                         const char  *mimetype,
                         int          position,
                         GError     **error)
{
    BognorQueuePrivate *priv = queue->priv;
    BognorQueueItem *item;
    gboolean empty;

    item = g_slice_new (BognorQueueItem);
    item->uri = g_strdup (uri);
    item->mimetype = g_strdup (mimetype);
    if (g_str_has_prefix (mimetype, "audio/")) {
        item->type = AUDIO_TYPE;
    } else {
        item->type = VISUAL_TYPE;
    }

    g_print ("Pushing %s to %d\n", uri, position);
    g_queue_push_nth (priv->play_queue, item, position);

    g_signal_emit (queue, signals[ADDED], 0, uri, position);

    return TRUE;
}

static gboolean
bognor_queue_get_name (BognorQueue *queue,
                       char       **name,
                       GError     **error)
{
    BognorQueuePrivate *priv = queue->priv;

    *name = g_strdup (priv->name);
    return TRUE;
}

static gboolean
bognor_queue_set_position (BognorQueue *queue,
                           int          type,
                           double       position,
                           GError     **error)
{
    BognorQueueClass *klass = BOGNOR_QUEUE_GET_CLASS (queue);

    if (type == AUDIO_TYPE) {
        klass->set_audio_position (queue, position);
    } else {
        klass->set_visual_position (queue, position);
    }

    return TRUE;
}

static gboolean
bognor_queue_get_position (BognorQueue *queue,
                           int          type,
                           double      *position,
                           GError     **error)
{
    BognorQueueClass *klass = BOGNOR_QUEUE_GET_CLASS (queue);

    if (type == AUDIO_TYPE) {
        klass->get_audio_position (queue, position);
    } else {
        klass->get_visual_position (queue, position);
    }

    return TRUE;
}

static void
bognor_queue_finalize (GObject *object)
{
    BognorQueue *self = (BognorQueue *) object;
    BognorQueuePrivate *priv = self->priv;

    if (priv->name) {
        g_free (priv->name);
        priv->name = NULL;
    }

    G_OBJECT_CLASS (bognor_queue_parent_class)->finalize (object);
}

static void
bognor_queue_dispose (GObject *object)
{
    BognorQueue *self = (BognorQueue *) object;

    G_OBJECT_CLASS (bognor_queue_parent_class)->dispose (object);
}

static void
bognor_queue_set_property (GObject      *object,
                          guint         prop_id,
                          const GValue *value,
                          GParamSpec   *pspec)
{
    BognorQueue *self = (BognorQueue *) object;

    switch (prop_id) {

    default:
        break;
    }
}

static void
bognor_queue_get_property (GObject    *object,
                          guint       prop_id,
                          GValue     *value,
                          GParamSpec *pspec)
{
    BognorQueue *self = (BognorQueue *) object;

    switch (prop_id) {

    default:
        break;
    }
}

static void
bognor_queue_class_init (BognorQueueClass *klass)
{
    GObjectClass *o_class = (GObjectClass *)klass;

    o_class->dispose = bognor_queue_dispose;
    o_class->finalize = bognor_queue_finalize;
    o_class->set_property = bognor_queue_set_property;
    o_class->get_property = bognor_queue_get_property;

    g_type_class_add_private (klass, sizeof (BognorQueuePrivate));
    dbus_g_object_type_install_info (G_TYPE_FROM_CLASS (klass),
                                     &dbus_glib_bognor_queue_object_info);

    signals[ADDED] = g_signal_new ("uri-added",
                                   G_TYPE_FROM_CLASS (klass),
                                   G_SIGNAL_RUN_FIRST |
                                   G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                   bognor_marshal_VOID__STRING_INT,
                                   G_TYPE_NONE, 2, G_TYPE_STRING,
                                   G_TYPE_INT);
    signals[REMOVED] = g_signal_new ("uri-removed",
                                     G_TYPE_FROM_CLASS (klass),
                                     G_SIGNAL_RUN_FIRST |
                                     G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                     bognor_marshal_VOID__STRING_INT,
                                     G_TYPE_NONE, 2, G_TYPE_STRING,
                                     G_TYPE_INT);
    signals[NOW_PLAYING] = g_signal_new ("now-playing-changed",
                                         G_TYPE_FROM_CLASS (klass),
                                         G_SIGNAL_RUN_FIRST |
                                         G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                         bognor_marshal_VOID__STRING_INT,
                                         G_TYPE_NONE, 2, G_TYPE_STRING,
                                         G_TYPE_INT);
    signals[POSITION_CHANGED] = g_signal_new ("position-changed",
                                              G_TYPE_FROM_CLASS (klass),
                                              G_SIGNAL_RUN_FIRST |
                                              G_SIGNAL_NO_RECURSE,
                                              0, NULL, NULL,
                                              bognor_marshal_VOID__INT_DOUBLE,
                                              G_TYPE_NONE, 2, G_TYPE_INT,
                                              G_TYPE_DOUBLE);
    signals[PLAYING_CHANGED] = g_signal_new ("playing-changed",
                                             G_TYPE_FROM_CLASS (klass),
                                             G_SIGNAL_RUN_FIRST |
                                             G_SIGNAL_NO_RECURSE,
                                             0, NULL, NULL,
                                             g_cclosure_marshal_VOID__BOOLEAN,
                                             G_TYPE_NONE, 1, G_TYPE_BOOLEAN);
}

static void
bognor_queue_init (BognorQueue *self)
{
    BognorQueuePrivate *priv;

    priv = self->priv = GET_PRIVATE (self);
    priv->play_queue = g_queue_new ();
    priv->can_play_visual = FALSE;
}

BognorQueue *
bognor_queue_new (void)
{
    BognorQueue *q;

    q = g_object_new (BOGNOR_TYPE_QUEUE, NULL);

    return q;
}

BognorQueue *
bognor_queue_new_from_playlist (const char *playlist)
{
    BognorQueue *q;
    FILE *f;

    q = g_object_new (BOGNOR_TYPE_QUEUE, NULL);

    f = fopen (playlist, "r");
    if (f == NULL) {
        g_object_unref (q);
        return NULL;
    }

    return q;
}

void
bognor_queue_set_visual_enabled (BognorQueue *queue,
                                 gboolean     enabled)
{
    BognorQueuePrivate *priv = queue->priv;
    BognorQueueClass *klass = BOGNOR_QUEUE_GET_CLASS (queue);
    BognorQueueItem *item;

    priv->can_play_visual = enabled;

    item = bognor_queue_get_current_item (queue);
    if (item == NULL) {
        return;
    }

    if (priv->can_play_visual == FALSE) {
        if (item->type == VISUAL_TYPE) {
            klass->set_visual_playing (queue, FALSE);
            priv->playing = FALSE;
            g_signal_emit (queue, signals[PLAYING_CHANGED], 0, FALSE);
        }
    } else {
        if (item->type == VISUAL_TYPE) {
            klass->set_visual_playing (queue, TRUE);
            priv->playing = TRUE;
            g_signal_emit (queue, signals[PLAYING_CHANGED], 0, TRUE);
        }
    }
}

gboolean
bognor_queue_get_visual_enabled (BognorQueue *queue)
{
    BognorQueuePrivate *priv = queue->priv;

    return priv->can_play_visual;
}

void
bognor_queue_set_name (BognorQueue *queue,
                       const char  *name)
{
    BognorQueuePrivate *priv = queue->priv;

    priv->name = g_strdup (name);
}

void
bognor_queue_emit_position_changed (BognorQueue *queue,
                                    double       position,
                                    int          type)
{
    g_signal_emit (queue, signals[POSITION_CHANGED], 0, type, position);
}

BognorQueueItem *
bognor_queue_get_current_item (BognorQueue *queue)
{
    BognorQueuePrivate *priv = queue->priv;

    return priv->current_item;
}

BognorQueueItem *
bognor_queue_get_next_item (BognorQueue *queue)
{
    BognorQueueClass *klass = BOGNOR_QUEUE_GET_CLASS (queue);
    BognorQueuePrivate *priv = queue->priv;
    BognorQueueItem *item = NULL;
    GList *l;
    int position = 0;

    for (l = priv->play_queue->head; l; l = l->next) {
        BognorQueueItem *i = l->data;

        if (i->type == AUDIO_TYPE) {
            g_queue_delete_link (priv->play_queue, l);
            item = i;
            break;
        }

        if (i->type == VISUAL_TYPE &&
            priv->can_play_visual == TRUE) {
            g_queue_delete_link (priv->play_queue, l);
            item = i;
            break;
        }

        position++;
    }

    if (priv->current_item) {
        free_queue_item (priv->current_item);
    }
    priv->current_item = item;

    klass->set_uri (queue, priv->current_item);
    if (item) {
        g_signal_emit (queue, signals[REMOVED], 0, item->uri, position);
        klass->add_item_to_recent (queue, item);
    }
    if (item == NULL) {
        g_signal_emit (queue, signals[NOW_PLAYING], 0, NULL, AUDIO_TYPE);
        g_signal_emit (queue, signals[NOW_PLAYING], 0, NULL, VISUAL_TYPE);
    } else if (item->type == AUDIO_TYPE) {
        g_signal_emit (queue, signals[NOW_PLAYING], 0, item->uri, item->type);
        g_signal_emit (queue, signals[NOW_PLAYING], 0, NULL, VISUAL_TYPE);
    } else if (item->type == VISUAL_TYPE) {
        g_signal_emit (queue, signals[NOW_PLAYING], 0, item->uri, item->type);
        g_signal_emit (queue, signals[NOW_PLAYING], 0, NULL, AUDIO_TYPE);
    }

    return item;
}

void
bognor_queue_play_next (BognorQueue *queue)
{
    BognorQueueClass *klass = BOGNOR_QUEUE_GET_CLASS (queue);
    BognorQueueItem *item = NULL;

    if (item = bognor_queue_get_next_item (queue)) {
        if (item->type == VISUAL_TYPE) {
            klass->set_audio_playing (queue, FALSE);
            klass->set_visual_playing (queue, TRUE);
        } else {
            klass->set_audio_playing (queue, TRUE);
            klass->set_visual_playing (queue, FALSE);
        }
    } else {
        klass->set_audio_playing (queue, FALSE);
        klass->set_visual_playing (queue, FALSE);
    }
}

gboolean
bognor_queue_get_is_playing (BognorQueue *queue)
{
    BognorQueuePrivate *priv = queue->priv;

    return priv->playing;
}
