/*
 * This file is part of Canola Thumbnailer
 * Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
 * Contact: Gustavo Lima Chaves <glima@profusion.mobi>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <Epsilon.h>

#include <errno.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

static const char *error_prefix = "CANOLA-THUMBNAILER:";
static const char null_str = '\0';

struct pipewrapper {
    int fd[2];
};

struct thumbnailer {
    struct pipewrapper pipe;
};

struct message {
    long id;
    int size;
    int aspect;
    int w;
    int h;
    char str[1024];
};
#define REQUEST_MSGLEN sizeof(struct message)

/**
 * read data to buffer, similar to read(2), but handles EINTR and EAGAIN.
 *
 * @param buf where to store data.
 * @param size amount of data to read, returns the number of read bytes.
 *
 * @return number of actually read bytes; in case of return -1 it
 *         will be less than the requested amount.
 */
static int
read_safe(int fd, void *buf, ssize_t *size)
{
    ssize_t done, todo;
    char *p;

    done = 0;
    todo = *size;
    p = buf;

    while (todo > 0) {
        ssize_t r;

        r = read(fd, p, todo);
        if (r > 0) {
            done += r;
            todo -= r;
            p += r;
        } else if (r == 0)
            return -2;
        else {
            if (errno == EINTR || errno == EAGAIN)
                continue;
            else {
                *size = done;
                fprintf(stderr, "%s Could not read from fd %d: %s.\n",
                        error_prefix, fd, strerror(errno));
                return -1;
            }
        }
    }

    return 0;
}

/**
 * write data from buffer, similar to write(2), but handles EINTR and EAGAIN.
 *
 * @param buf where to read data.
 * @param size amount of data to write, returns the number of read bytes.
 *
 * @return number of actually written bytes; in case of return
 *         -1 it will be less than the requested amount.
 */
static int
write_safe(int fd, const void *buf, ssize_t *size)
{
    ssize_t done, todo;
    const char *p;

    done = 0;
    todo = *size;
    p = buf;

    while (todo > 0) {
        ssize_t r;

        r = write(fd, p, todo);
        if (r > 0) {
            done += r;
            todo -= r;
            p += r;
        } else if (r == 0)
            continue;
        else {
            if (errno == EINTR || errno == EAGAIN)
                continue;
            else {
                *size = done;
                fprintf(stderr, "%s Could not write to fd %d: %s.\n",
                        error_prefix, fd, strerror(errno));
                return -1;
            }
        }
    }

    return 0;
}

static void
pipewrapper_set(struct pipewrapper *pw, int fd_r, int fd_w)
{
    pw->fd[0] = fd_r;
    pw->fd[1] = fd_w;
}

static int
pipewrapper_send(struct pipewrapper *pw, struct message *msg)
{
    ssize_t sz = sizeof(*msg);
    int r;

    r = write_safe(pw->fd[1], (const void *)msg, &sz);
    if (r == -1)
        return r;

    return (int)sz;
}

static int
pipewrapper_receive(struct pipewrapper *pw, unsigned int len, char *buf)
{
    ssize_t sz = len;
    int r;

    r = read_safe(pw->fd[0], buf, &sz);

    if (r < 0)
        return r;

    return (int)sz;
}

static int
thumbnailer_msg_send(struct thumbnailer *tn, long id, const char *path, int size, int aspect, int width, int height)
{
    struct message msg;
    int len;
    int r;

    len = strlen(path) + 1;
    if (len > sizeof(msg.str)) {
        len = sizeof(msg.str) - 1;
        msg.str[len] = '\0';
    }

    msg.id = id;
    memcpy(msg.str, path, len);
    msg.size = size;
    msg.aspect = aspect;
    msg.w = width;
    msg.h = height;

    r = pipewrapper_send(&tn->pipe, &msg);
    if (r != sizeof(msg)) {
        fprintf(stderr, "%s Error sending message, exiting.\n", error_prefix);
        abort();
    }
    return 0;
}

static int
thumbnailer_msg_receive(struct thumbnailer *tn, struct message *msg)
{
    int r;

    r = pipewrapper_receive(&tn->pipe, REQUEST_MSGLEN, (char *)msg);
    if (r == -1) {
        fprintf(stderr, "%s Error receiving message, exiting\n", error_prefix);
        abort();
    } else if (r == -2) {
        fprintf(stderr, "%s Pipe closed, exiting\n", error_prefix);
        abort();
    }

    return 0;
}

static const char *
thumbnailer_thumbnail_create(struct thumbnailer *tn, const char *path, int size, int aspect, int width, int height)
{
    Epsilon *e;
    const char *folder;

    epsilon_init();
    e = epsilon_new(path);
    if (!e)
        return NULL;

    epsilon_format_set(e, EPSILON_THUMB_JPEG);
    epsilon_aspect_set(e, aspect);
    if (aspect == EPSILON_THUMB_CROP)
        folder = "/canola-cropped";
    else
        folder = "/canola";

    if (size == EPSILON_THUMB_NORMAL) {
        width = 128;
        height = 128;
    } else if (size == EPSILON_THUMB_LARGE) {
        width = 256;
        height = 256;
    }

    epsilon_custom_thumb_size(e, width, height, folder);
    if (epsilon_exists(e) != EPSILON_OK)
        if (epsilon_generate(e) != EPSILON_OK)
            return NULL;

    return epsilon_thumb_file_get(e);
}

static int
thumbnailer_thumbnail_request(struct thumbnailer *tn)
{
    struct message msg;
    const char *path;
    int r;

    r = thumbnailer_msg_receive(tn, &msg);
    if (r != 0)
        return r;

    path = thumbnailer_thumbnail_create(tn, msg.str, msg.size, msg.aspect,
                                        msg.w, msg.h);
    if (!path) {
        fprintf(stderr, "%s Could not create thumbnail for %s.\n",
                error_prefix, msg.str);
        return thumbnailer_msg_send(tn, msg.id, &null_str, msg.size,
                                    msg.aspect, 0, 0);
    }

    return thumbnailer_msg_send(tn, msg.id, path, msg.size, msg.aspect,
                                msg.w, msg.h);
}

static void
thumbnailer_start(struct thumbnailer *tn)
{
    while (true)
        thumbnailer_thumbnail_request(tn);
}

static void
usage(char *argv[])
{
    fprintf(stdout, "usage: %s <read fd> <write fd>\n", argv[0]);
    abort();
}

int
main(int argc, char *argv[])
{
    struct thumbnailer tn;
    int fd_r, fd_w;

    if (argc != 3)
        usage(argv);

    fd_r = atoi(argv[1]);
    fd_w = atoi(argv[2]);

    pipewrapper_set(&tn.pipe, fd_r, fd_w);
    thumbnailer_start(&tn);

    return 0;
}
