# -*- coding: utf-8 -*-
# Canola2 Tuning Plugin
# Copyright (C) 2008 Instituto Nokia de Tecnologia
# Authors: Adriano Rezende <adriano.rezende@openbossa.org>
#          Renato Chencarek <renato.chencarek@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import os
import ecore
import logging

from terra.core.manager import Manager

manager = Manager()
ModalThrobberController = manager.get_class("Controller/Folder/Settings/ModalThrobber")
ModalCheckListController = manager.get_class("Controller/Folder/Settings/ModalCheckList")


log = logging.getLogger("plugins.canola-tuning.ui")


class ScanCoversController(ModalCheckListController):
    terra_type = "Controller/Settings/Folder/MediaLibrary/Tuning/ScanCovers"
    header_message = "Choose the cover art sources."

    def __init__(self, model, canvas, parent):
        ModalCheckListController.__init__(self, model, canvas, parent)
        self.ok_button_text = " Update "
        self.model.changed_callback_add(self.view.add_item)
        self.model.load()
        self.check_disable_button()

    def ok_clicked(self):
        self.colapse_wait("  Updating covers... ")
        self.model.update(self.on_finished, self.on_progress)

    def on_progress(self, fname, progress, total, status):
        if self.view and self.view.throbber and total:
            self.view.throbber.text = "  Updating covers<br>(%d of %d)" % \
                (progress, total)

    def cancel_clicked(self):
        self.model.cancel()
        self.close()
        self.refresh_view()

    def item_clicked(self, model):
        model.checked = not model.checked
        model.updated = False
        self.check_disable_button()

    def on_finished(self, error, retval):
        self.close()
        self.refresh_view()

    def refresh_view(self):
        pass

    def check_disable_button(self):
        self.ok_button_enabled = (self.model.select_count() > 0)

    def delete(self):
        self.model.changed_callback_del(self.view.add_item)
        ModalCheckListController.delete(self)
        self.model.unload()


class ScanCoversItemController(ScanCoversController):
    terra_type = "Controller/Options/Folder/Media/Audio/AlbumArt/ScanCovers"

    def __init__(self, model, canvas, parent):
        ScanCoversController.__init__(self, model, canvas, parent)
        self.parent.screen_controller.view.album_set_show()

    def ok_clicked(self):
        model = self.parent.screen_controller.model
        model = model.children[self.parent.screen_controller.current]

        self.colapse_wait("  Updating covers... ")
        self.model.update(model.artist, model.name, self.on_finished)

    def on_finished(self, error, retval):
        if retval == 0:
            self.view.colapse_message("No covers found for this album"
                                      "<br>using the selected sources")
            ecore.timer_add(3.0, ScanCoversController.on_finished,
                            self, error, retval)
        else:
            ScanCoversController.on_finished(self, error, retval)

    def refresh_view(self):
        view = self.parent.screen_controller.view
        view.current_album.cb_cover_right()


class CleanUnusedCoversController(ModalThrobberController):
    terra_type = "Controller/Settings/Folder/MediaLibrary/Tuning/CleanUnusedCovers"
    wait_message = "  Removing unused covers..."

    def do_start(self, end_callback):
        self.model.update(end_callback)

    def do_finish(self):
        pass

    def on_cancel(self):
        self.model.cancel()


class CleanUnusedCoversItemController(ModalThrobberController):
    terra_type = "Controller/Options/Folder/Media/Audio/AlbumArt/CleanUnused"
    wait_message = "  Removing unused covers..."

    def do_start(self, end_callback):
        view = self.parent.screen_controller.view
        view.album_set_show()
        self.model.update(view.current_album._cover_path, end_callback)

    def do_finish(self):
        view = self.parent.screen_controller.view
        view.current_album.cb_cover_right()

    def on_cancel(self):
        self.model.cancel()


class VideoThumbnailController(ModalThrobberController):
    terra_type = "Controller/Settings/Folder/MediaLibrary/Tuning/VideoThumbnail"
    wait_message = "  Updating thumbs..."

    def do_start(self, end_callback):
        self.model.update(end_callback, self.on_progress)

    def on_progress(self, fname, progress, total, status):
        if self.view and self.view.throbber and total:
            self.view.throbber.text = "  Updating thumbs<br>(%d of %d)" % \
                (progress, total)

    def do_finish(self):
        pass

    def on_cancel(self):
        self.model.cancel()


class VideoPlayerThumbnailController(ModalThrobberController):
    terra_type = "Controller/Options/Folder/Player/Video/Thumbnail"
    wait_message = "  Updating thumbnail..."

    def do_start(self, end_callback):
        controller = self.parent.screen_controller
        self.model.update(controller.model.uri,
                          controller.get_position(), end_callback)

    def do_finish(self):
        pass

    def on_cancel(self):
        self.model.cancel()
