#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import ecore

from terra.ui.base import EdjeWidget
from terra.core.manager import Manager

__all__ = ("ListWidget",)

mger = Manager()

KineticListWidget = mger.get_class("Widget/KineticList")
ScrollbarWidget = mger.get_class("Widget/Scrollbar")
KnobWidget = mger.get_class("Widget/Knob")

PlayerCurrentMediaURI = mger.get_class("Current/MediaURI")
def row_value_set(self, value):
    if PlayerCurrentMediaURI.uri == value.uri:
        self.signal_emit("show,marker", "")
    else:
        self.signal_emit("hide,marker", "")

    self.part_text_set("text", value.name)


ResizableRowRenderer = mger.get_class("Widget/ResizableRowRenderer")
RowRenderer = mger.get_class("Widget/RowRendererURI")

class ResizableAlbumRowRendererCurrentURIWidget(ResizableRowRenderer):
    row_group = "album_list_item_text_resizable"
    do_value_set = row_value_set


class AlbumRowRendererCurrentURIWidget(RowRenderer):
    row_group = "album_list_item_text"
    do_value_set = row_value_set
    resizable_row_renderer = ResizableAlbumRowRendererCurrentURIWidget


class ListWidget(EdjeWidget):
    row_renderer = AlbumRowRendererCurrentURIWidget

    def __init__(self, parent, elements, theme=None):
        EdjeWidget.__init__(self, parent.evas, "album_list", parent, theme)
        self.elements = elements
        self.callback_clicked = None
        self._hide_scroll_timer = None
        self._setup_gui_scrollbar()
        self._setup_gui_list()

    def _setup_gui_scrollbar(self):
        self._scroll = ScrollbarWidget(self, KnobWidget(self))
        self.part_swallow("scrollbar", self._scroll)

    def _setup_gui_list(self):
        def renderer_new(canvas):
            return self.row_renderer(self)
        self._list = KineticListWidget(self, renderer_new, self.elements)
        self._list.on_resize_add(self._cb_list_resized)
        self._list.position_changed_cb_set(self._cb_position_changed)
        self._list.clicked_cb_set(self._cb_clicked)
        self.part_swallow("contents", self._list)

    def _cb_list_resized(self, lst):
        self._scroll.scale_set(lst.visible_rows_scale_get())

    def _cb_position_changed(self, lst, percent):
        self._scroll.value_set(percent)
        if self._hide_scroll_timer is not None:
            self._hide_scroll_timer.delete()
        self._hide_scroll_timer = ecore.timer_add(0.1, self._hide_scroll_cb)
        self.signal_emit("scrollbar,transition,in", "")

    def _hide_scroll_cb(self):
        self.signal_emit("scrollbar,transition,out", "")
        self._hide_scroll_timer = None
        return False

    def _cb_clicked(self, lst, index):
        self.callback_clicked(self, index)

    def delete(self):
        self._list.delete()
        self._scroll.delete()
        if self.elements:
            del self.elements
        EdjeWidget.delete(self)

    def append(self, child):
        self._list.append(child)

    def extend(self, children):
        self._list.extend(children)

    def theme_changed(self):
        self._list.theme_changed()
        self._scroll.theme_changed()

    def select_item(self, index, end_callback=None):
        self._list.select_item(index, end_callback)
