#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import os
import dbus
from e_dbus import DBusEcoreMainLoop
DBusEcoreMainLoop(set_as_default=True)

from terra.core.terra_object import TerraObject
from terra.core.model import ModelStatus
from terra.core.controller import Controller
from terra.ui.window import StatusIcon

pjoin = os.path.join

class BatteryStatus(ModelStatus):
    terra_type = "Model/Status/Battery"

    (BATTERY_CHARGING_OFF,
     BATTERY_CHARGING_ON) = xrange(2)

    def __init__(self):
        ModelStatus.__init__(self, "Battery")
        self.status = (self.BATTERY_CHARGING_OFF, 0.0)
        self._setup()

    def _setup(self):
        bus = dbus.SessionBus()

        proxy = bus.get_object('org.freedesktop.PowerManagement',
                               '/org/freedesktop/PowerManagement')
        iface = dbus.Interface(proxy, dbus_interface='org.freedesktop.PowerManagement')

        bus.add_signal_receiver(self._on_battery, signal_name="OnBatteryChanged",
            dbus_interface='org.freedesktop.PowerManagement')

        bus.add_signal_receiver(self._battery_level_low, signal_name="LowBatteryChanged",
            dbus_interface='org.freedesktop.PowerManagement')

        self._on_battery(iface.GetOnBattery())
        if (iface.GetLowBattery()):
            self._battery_level_low()

    def _battery_level_low(self):
        self.status = (self.BATTERY_CHARGING_OFF, 0.0)

    def _on_battery(self, value):
        if bool(value):
            self.status = (self.BATTERY_CHARGING_OFF, 1.0)
        else:
            self.status = (self.BATTERY_CHARGING_OFF, 1.0)


class BatteryStatusController(Controller):
    terra_type = "Controller/Status/Battery"

    MSG_ID = 1

    def __init__(self, model, canvas, parent):
        Controller.__init__(self, model, canvas, parent)
        self.view = StatusIcon(parent.view, "battery")
        self.model.add_listener(self._update_battery)
        # force update or ui
        self.view.message_send(self.MSG_ID, model.status[1])
        self.view.callback_theme_changed = self._theme_changed

    def _theme_changed(self, *ignored):
        self._update_battery(self.model)

    def _update_battery(self, model):
        if model.status[0] == model.BATTERY_CHARGING_OFF:
            signal = "state,charging_off"
        else:
            signal = "state,charging_on"

        self.view.signal_emit(signal, "canola")
        self.view.message_send(self.MSG_ID, model.status[1])

    def delete(self):
        self.view.callback_theme_changed = None
        self.model.remove_listener(self._update_battery)
        self.view.delete()


class NetworkStatus(ModelStatus):
    terra_type = "Model/Status/Network"

    def __init__(self):
        ModelStatus.__init__(self, "Network")
        self.status = 0.0
        try:
            self._setup()
        except:
            self.status = 1.0

    def _setup(self):
        sbus = dbus.SystemBus()

        proxy = sbus.get_object('org.freedesktop.NetworkManager',
            '/org/freedesktop/NetworkManager')

        # XXX FIXME: just get the first available device
        dev_name = proxy.getDevices()[0]

        device = sbus.get_object('org.freedesktop.NetworkManager', dev_name)

        self.dev_iface = dbus.Interface(device,
            dbus_interface='org.freedesktop.NetworkManager.Devices')

        sbus.add_signal_receiver(self._on_strength_changed,
            signal_name="DeviceStrengthChanged",
            dbus_interface='org.freedesktop.NetworkManager')

        sbus.add_signal_receiver(self._on_no_longer_active,
            signal_name="DeviceNoLongerActive",
            dbus_interface='org.freedesktop.NetworkManager')

        sbus.add_signal_receiver(self._on_now_active,
            signal_name="DeviceNowActive",
            dbus_interface='org.freedesktop.NetworkManager')

        if bool(self.dev_iface.getLinkActive()):
            self._on_now_active(device)

    def _on_strength_changed(self, device, strength):
        self.status = float(strength) / 100

    def _on_no_longer_active(self, device):
        self.status = 0.0

    def _on_now_active(self, device):
        network_type = int(self.dev_iface.getType())

        # wired
        if network_type == 1 and bool(self.dev_iface.getLinkActive()):
            self.status = 1.0
        # wireless
        elif network_type == 2:
           self.status = float(self.dev_iface.getStrength()) / 100


class NetworkStatusController(Controller):
    terra_type = "Controller/Status/Network"

    MSG_ID = 1

    def __init__(self, model, canvas, parent):
        Controller.__init__(self, model, canvas, parent)
        self.view = StatusIcon(parent.view, "connection")
        self.model.add_listener(self._update_network)
        # force update or ui
        self.view.message_send(self.MSG_ID, model.status)
        self.view.callback_theme_changed = self._theme_changed

    def _theme_changed(self, *ignored):
        self._update_network(self.model)

    def _update_network(self, model):
        self.view.message_send(self.MSG_ID, model.status)

    def delete(self):
        self.view.callback_theme_changed = None
        self.model.remove_listener(self._update_network)
        self.view.delete()


class SystemProperties(TerraObject):
    terra_type = "SystemProperties"

    HOME = os.environ["HOME"]
    MEDIA_BASE = HOME
    DEFAULT_DIR_DOCS = pjoin(MEDIA_BASE, "Documents")
    DEFAULT_DIR_GAMES = None
    DEFAULT_DIR_IMAGES = pjoin(MEDIA_BASE, "Pictures")
    DEFAULT_DIR_AUDIO = pjoin(MEDIA_BASE, "Music")
    DEFAULT_DIR_VIDEO = pjoin(MEDIA_BASE, "Videos")

    DEFAULT_DIRS = {
        "photo": [DEFAULT_DIR_IMAGES],
        "audio": [DEFAULT_DIR_AUDIO],
        "video": [DEFAULT_DIR_VIDEO]}

    path_alias = {
        "/": "Root",
        HOME: "Home",
        DEFAULT_DIR_DOCS: "Documents",
        DEFAULT_DIR_IMAGES: "Images",
        DEFAULT_DIR_AUDIO: "Audio clips",
        DEFAULT_DIR_VIDEO: "Video clips",
        "/media/disk": "USB device",
        }

    storage_cards = ["/media/disk"]

    def download_dirs_get(self):
        dd = []
        for dir in [self.HOME] + self.storage_cards:
            if dir in self.path_alias:
                dd.append((os.path.join(dir, "My downloads"),
                           self.path_alias[dir]))
        return dd

    def path_alias_get(self, path):
        p = os.path.normpath(os.path.expanduser(path))
        return self.path_alias.get(p, "")

    def basemountdir(self, path):
        volatile_path = None

        for known_mount in self.storage_cards:
            if path.startswith(known_mount):
                volatile_path = known_mount
                break

        return volatile_path

    def prepare_write_path(self, path, name, min_size=-1):
        errmsg = None

        if not self.is_mounted(path):
            basedir = self.basemountdir(path)
            alias = self.path_alias_get(basedir)

            if alias != "":
                errmsg = alias + " is not available."
            else:
                errmsg = name.capitalize() + " is on a non-available device."
            raise Exception(errmsg)

        if not os.path.exists(path):
            try:
                os.makedirs(path)
            except OSError, e:
                errmsg = e.strerror + ": '" +  e.filename + "'."
                raise Exception(errmsg)

        if min_size != -1 and self.bytes_available(path) < min_size:
            errmsg = "Not enough space left in " + name.lower() + "."
            raise Exception(errmsg)

    def is_mounted(self, path):
        volatile_path = self.basemountdir(path)

        if volatile_path is not None:
            memcards_paths = self.mounted_storage_cards_get()
            if volatile_path not in memcards_paths:
                return False

        return True

    def mounted_storage_cards_get(self):
        memcards = []
        try:
            mounts = open("/proc/mounts").read()
            for memcard_dir in self.storage_cards:
                if memcard_dir in mounts:
                    memcards.insert(0, memcard_dir)
        except:
            pass

        return memcards

    def bytes_available(self, path):
        if os.path.isfile(path):
            path = os.path.dirname(path)

        return int(os.statvfs(path)[4] * os.statvfs(path)[1])
