#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import etk
import edje
import ecore
import os.path
import evas.decorators

from terra.core.controller import Controller
from terra.core.manager import Manager
from terra.ui.throbber import EtkThrobber
from terra.ui.modal import Modal, ModalSlider, SliderContentList


mger = Manager()
ModalController = mger.get_class("Controller/Modal")
SettingsFolderController = mger.get_class("Controller/Settings/Folder")


class ModalThrobber(Modal):
    def __init__(self, parent, message, theme=None, hborder=0, vborder=0):
        Modal.__init__(self, parent, "", theme, hborder, vborder)

        self._init_embed()
        self.callback_clicked = None
        self.callback_escape = None
        self.message = message

    def _init_embed(self):
        self.embed = etk.Embed(self.evas)

        self.button = etk.Button(label="   Cancel   ")
        self.button.on_clicked(self.clicked)
        self.button.show()
        self.button_align = etk.Alignment(0.5, 0.5, 0.0, 0.0, child=self.button)
        self.button_align.show()

        self.vbox = etk.VBox()
        self.vbox.border_width = 25

        self.throbber = EtkThrobber("")
        self.throbber.show()
        self.throbber_align = etk.Alignment(0.5, 0.4, 0.0, 0.0,
                                            child=self.throbber)
        self.throbber_align.show()

        self.throbber.start()

        self.vbox.append(self.throbber_align, etk.VBox.START, etk.VBox.FILL, 0)
        self.vbox.append(self.button_align, etk.VBox.END, etk.VBox.FILL, 0)
        self.vbox.show()

        self.embed.on_key_down(self._key_down_cb)
        self.embed.on_key_up(self._key_up_cb)

        self.embed.add(self.vbox)
        self.embed.show()
        self.contents_set(self.embed.object)

    def _message_set(self, value):
        self.throbber.text = value

    def _message_get(self):
        return self.throbber.text

    message = property(_message_get, _message_set)

    def clicked(self, *ignored):
        if self.callback_clicked:
            self.callback_clicked()

    def _key_down_cb(self, o, ev):
        return self.handle_key_down(ev)

    def _key_up_cb(self, o, ev):
        return self.handle_key_up(ev)

    def handle_key_down(self, ev):
        if ev.key == "Escape":
            if self.callback_escape:
                self.callback_escape()
            return False
        return True

    def handle_key_up(self, ev):
        return True

    @evas.decorators.focus_in_callback
    def _focus_in(self):
        self.embed.object.focus = True
        self.do_on_focus()

    def do_on_focus(self):
        pass

    @evas.decorators.del_callback
    def _destroy_contents(self):
        self.embed.destroy()


class CheckCollectionsController(SettingsFolderController):
    terra_type = "Controller/Settings/Folder/MediaLibrary/CheckCollections"

    def __init__(self, model, canvas, parent):
        SettingsFolderController.__init__(self, model, canvas, parent)
        self.view.header_text = \
            "These are the albums that may have multiple<br>" \
            "entries:"

        self.model.callback_on_finished = self.cb_on_finished
        self.model.callback_on_update_request = self.cb_on_update_request

        self.modal = None
        self.load_finished = False
        ecore.timer_add(0.6, self.show_modal)

    def cb_on_update_request(self, model):
        if len(model.children) <= 1:
            self.model.remove(model)

    def show_modal(self):
        if self.load_finished:
            return

        self.modal = ModalThrobber(self.parent.last_panel.view,
                                   "  Checking entries...",
                                   hborder=16, vborder=100)
        self.modal.callback_escape = self.cb_on_cancel
        self.modal.callback_clicked = self.cb_on_cancel
        self.modal.show()

    def cb_on_cancel(self):
        self.model.stop_loading()
        if self.modal:
            self.modal.delete()
            self.modal = None

    def cb_on_finished(self):
        self.load_finished = True
        if self.modal:
            self.modal.delete()
            self.modal = None


class DirItemRenderer(etk.EdjeKineticRenderer):
    states = ["unchecked", "checked"]

    def __init__(self, item_click=None, *a, **ka):
        etk.EdjeKineticRenderer.__init__(self)
        self.item_click = item_click
        self.args = a
        self.kargs = ka
        self.last_selected = None

    def create_cell(self, canvas):
        cell = edje.Edje(canvas)
        etk.theme_edje_object_set(cell, self.list.theme_file,
                                  "image_check", "canola/conf")
        cell.signal_emit("checked", "")

        return cell

    def update_cell(self, cell, row):
        img = cell.part_object_get("thumbnail")
        if os.path.exists(row.cover):
            try:
                img.file_set(row.cover)
                cell.signal_emit("thumbnail,show", "")
            except evas.EvasLoadError:
                cell.signal_emit("thumbnail,hide", "")
        else:
            cell.signal_emit("thumbnail,hide", "")

        cell.part_text_set("label", row.artist.name)
        if len(row.audios) == 1:
            cell.part_text_set("label_description", row.audios[0].title)
        else:
            cell.part_text_set("label_description", "%d files" % len(row.audios))

        if not row.updated:
            cell.signal_emit("%s" % self.states[row.state], "")
        else:
            cell.signal_emit("%s,immediate" % self.states[row.state], "")

        row.updated = True

    def show_cell(self, cell, x, y, w, h):
        cell.geometry = x, y, w, h
        cell.show()

    def click(self, cell, part, row):
        if part == "event_area":
            if self.last_selected:
                self.last_selected.signal_emit("cell,state,unselected", "")
            cell.signal_emit("cell,state,selected", "")
            if self.item_click:
                self.item_click(row, self.list, *self.args, **self.kargs)
            self.last_selected = cell


class SettingsDirContent(SliderContentList):
    def __init__(self, evas_obj, item_clicked=None):
        SliderContentList.__init__(self, evas_obj)
        self.callback_item_clicked = item_clicked
        self.list.row_height = 90 # XXX

    def columns_get(self):
        dir_item_renderer =  DirItemRenderer(item_click=self.cb_item_clicked)
        return [(100, dir_item_renderer, True)]

    def cb_item_clicked(self, row, list):
        if self.callback_item_clicked:
            self.callback_item_clicked(row)


class SettingsDirController(Controller):
    def __init__(self, model, canvas, parent):
        Controller.__init__(self, model, canvas, parent)

        self.view = SettingsDirContent(self.evas,
                                       self.cb_item_clicked)
        self.view.callback_escape = self.parent.escape
        self.callback_on_item_clicked = None

        self.last_model = None

        self.view.show()
        self.model.changed_callback_add(self._add_item)
        self.load_model()

    def load_model(self):
        ecore.idler_add(self.model.load)

    def _add_item(self, model):
        if model.children:
            self.view.item_append(model.children[-1])

    def cb_item_clicked(self, model):
        if model.state == model.STATE_UNCHECKED:
            model.state = model.STATE_CHECKED
        else:
            model.state = model.STATE_UNCHECKED
        model.updated = False

        if self.callback_on_item_clicked:
            self.callback_on_item_clicked()

    def delete(self):
        self.model.changed_callback_del(self._add_item)
        self.view.destroy()
        self.view = None
        self.model.unload()


class SettingsCompilationView(ModalSlider):
    #title = "Create compilation"

    HEADER_AREA_HEIGHT = 50

    def __init__(self, parent, album_name):
        ModalSlider.__init__(self, parent.last_panel.view, album_name,
                             parent.last_panel.view.theme, hborder=16,
                             vborder=20)

        label = etk.Label("Select the entries that are part of the album.")
        label.alignment_set(0.0, 0.5)
        label.show()

        self.embed = None
        self.throbber = None
        self.title_separator.show()

        self.caption = etk.HBox()
        self.caption.border_width_set(15)
        self.caption.append(label, etk.HBox.START, etk.HBox.EXPAND_FILL, 0)
        self.caption.show()
        self.header_vbox.append(self.caption, etk.VBox.START, etk.VBox.EXPAND_FILL, 0)

        self.header_area_show()

        self.ablum_name = album_name
        self.button_box_show()
        self.ok_button = self.button_add(" Combine ")
        self.back_button = self.button_add("   Cancel   ")

        self.callback_ok_clicked = None
        self.callback_back_clicked = None

    def _on_button_clicked(self, bt):
        if bt == self.ok_button:
            if self.callback_ok_clicked:
                self.callback_ok_clicked()
        elif bt == self.back_button:
            if self.callback_back_clicked:
                self.callback_back_clicked()

    def colapse_wait(self):
        self.title = ""
        self.embed = etk.Embed(self.evas)
        self.throbber = EtkThrobber("  Merging albums...")
        self.throbber_align = \
            etk.Alignment(0.5, 0.4, 0.0, 0.0, child=self.throbber)
        self.throbber_align.show()
        self.embed.add(self.throbber_align)
        self.contents = self.embed.object
        self.throbber.start()
        self.half_expand()
        self.embed.show()
        self.throbber.show()
        self.slider_contents.hide()

    @evas.decorators.del_callback
    def _destroy_throbber(self):
        if self.embed:
            self.embed.destroy()
            self.embed = None


class CheckCollectionsFinalController(ModalController):
    terra_type = "Controller/Settings/Folder/MediaLibrary/CheckCollectionsFinal"

    def __init__(self, model, canvas, parent):
        ModalController.__init__(self, model, canvas, parent)
        self.children = []

        controller = SettingsDirController(model, self.evas, self)
        controller.callback_on_item_clicked = self.check_disable_button
        self.children.append(controller)

        self.view = SettingsCompilationView(parent, model.title)
        self.view.center_contents_set(controller.view.object)
        self.view.callback_ok_clicked = self.cb_ok
        self.view.callback_back_clicked = self.back

        self.view.show()
        self.check_disable_button()

    def check_disable_button(self):
        if self.model.select_count() < 2:
            self.view.ok_button.disabled = 1
        else:
            self.view.ok_button.disabled = 0

    def cb_ok(self):
        self.parent.killall()
        def finished():
            self.view.throbber.stop()
            if self.model.parent.callback_on_update_request:
                self.model.parent.callback_on_update_request(self.model)
            self.close()

        self.view.colapse_wait()
        self.model.update(finished)

    def escape(self):
        self.back()

    def really_close(self):
        def cb():
            self.parent.back()
        self.view.hide(cb)

    def close(self):
        self.delete_children()
        self.really_close()

    def delete_children(self):
        while self.children:
            controller = self.children.pop()
            controller.delete()

    def delete(self):
        self.view.delete()
        self.view = None

    def back(self):
        self.close()

