#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import etk
import ecore
import evas.decorators

from terra.ui.modal import Modal
from terra.core.manager import Manager
from terra.ui.throbber import EtkThrobber, EdjeThrobber
from terra.ui.base import EdjeWidget
from terra.core.terra_object import TerraObject
from terra.ui.panel import PanelContentFrame
from terra.ui.modal import ModalSlider, SliderContentList

manager = Manager()
ModalController = manager.get_class("Controller/Modal")
CheckItemRenderer = manager.get_class("Renderer/EtkList/CheckItem")
PanelButtonWidget = manager.get_class("Widget/PanelButton")


class ModalThrobber(Modal, TerraObject):
    terra_type = "Widget/Settings/ModalThrobber"

    def __init__(self, parent, message,
                 has_cancel=True, theme=None, hborder=0, vborder=0):
        Modal.__init__(self, parent, "", theme, hborder, vborder)

        self.has_cancel = has_cancel
        self._init_embed()
        self.message = message
        self.callback_escape = None
        self.callback_clicked = None

    def _init_embed(self):
        self.embed = etk.Embed(self.evas)

        self.vbox = etk.VBox()
        self.vbox.border_width = 25

        self.throbber = EtkThrobber("")
        self.throbber.show()
        self.throbber_align = etk.Alignment(0.5, 0.4, 0.0, 0.0,
                                            child=self.throbber)
        self.throbber_align.show()
        self.throbber.start()

        self.vbox.append(self.throbber_align, etk.VBox.START, etk.VBox.FILL, 0)

        if self.has_cancel:
            self.button = etk.Button(label="   Cancel   ")
            self.button.on_clicked(self.clicked)
            self.button.show()
            self.button_align = etk.Alignment(0.5, 0.5, 0.0, 0.0, child=self.button)
            self.button_align.show()

            self.vbox.append(self.button_align, etk.VBox.END, etk.VBox.FILL, 0)

        self.vbox.show()

        self.embed.on_key_down(self._key_down_cb)
        self.embed.on_key_up(self._key_up_cb)

        self.embed.add(self.vbox)
        self.embed.show()
        self.contents_set(self.embed.object)

    def _message_set(self, value):
        self.throbber.text = value

    def _message_get(self):
        return self.throbber.text

    message = property(_message_get, _message_set)

    def clicked(self, *ignored):
        if self.callback_clicked:
            self.callback_clicked()

    def _key_down_cb(self, o, ev):
        return self.handle_key_down(ev)

    def _key_up_cb(self, o, ev):
        return self.handle_key_up(ev)

    def handle_key_down(self, ev):
        if ev.key == "Escape":
            if self.callback_escape:
                self.callback_escape()
            return False
        return True

    def handle_key_up(self, ev):
        return True

    @evas.decorators.focus_in_callback
    def _focus_in(self):
        self.embed.object.focus = True
        self.do_on_focus()

    def do_on_focus(self):
        pass

    @evas.decorators.del_callback
    def _destroy_contents(self):
        self.embed.destroy()


class PanelContentModal(Modal, TerraObject):
    terra_type = "Widget/Settings/PanelContentModal"

    def __init__(self, parent, title,
                 theme=None, hborder=16, vborder=100,
                 has_cancel=True, label_ok="OK", label_cancel="  Cancel  "):
        Modal.__init__(self, parent.view, title, theme,
                       hborder=hborder, vborder=vborder)

        self.callback_ok_clicked = None
        self.callback_cancel_clicked = None
        self.callback_escape = None
        self.message_embed = None

        self.modal_contents = PanelContentFrame(self.evas)
        self.ok_button = self.modal_contents.button_add(label_ok)
        self.ok_button.on_clicked(self._on_button_clicked)

        if has_cancel:
            self.cancel_button = self.modal_contents.button_add(label_cancel)
            self.cancel_button.on_clicked(self._on_button_clicked)

        self.modal_contents.handle_key_down = self.handle_key_down
        self.contents_set(self.modal_contents.object)

        self.init_label()
        self.init_throbber()

    def init_label(self):
        self.label_embed = etk.Embed(self.evas)
        self.label = etk.Label(text="")
        self.label.theme_group = "modal_label"
        self.label_align = etk.Alignment(0.5, 0.4, 0.0, 0.0,
                                         child=self.label)
        self.label_align.show()
        self.label_embed.add(self.label_align)
        self.label_embed.hide()
        self.label.show()

    def init_throbber(self):
        self.throbber_embed = etk.Embed(self.evas)
        self.throbber = EtkThrobber("")
        self.throbber_align = etk.Alignment(0.5, 0.4, 0.0, 0.0,
                                            child=self.throbber)
        self.throbber_align.show()
        self.throbber_embed.add(self.throbber_align)
        self.throbber.start()
        self.throbber_embed.hide()
        self.throbber.show()

    def set_content(self, content):
        self.modal_contents.frame.add(content)

    def handle_key_down(self, ev):
        if ev.key == "Escape":
            if self.callback_escape:
                self.callback_escape()
            return False
        return True

    def _on_ok_clicked(self, *ignored):
        if self.callback_ok_clicked:
            self.callback_ok_clicked()

    def _on_button_clicked(self, bt):
        if bt == self.ok_button:
            self._on_ok_clicked()
        elif bt == self.cancel_button:
            if self.callback_cancel_clicked:
                self.callback_cancel_clicked()

    def do_on_focus(self):
        self.modal_contents.object.focus = True

    @evas.decorators.del_callback
    def _destroy_contents(self):
        self.modal_contents.destroy()
        self.throbber_embed.destroy()
        self.label_embed.destroy()

    def hide_all(self):
        self.modal_contents.hide()
        self.modal_contents.frame.hide()
        self.modal_contents.button_box.hide()
        self.label_embed.hide()
        self.throbber_embed.hide()

    def message(self, msg):
        self.hide_all()

        self.title = ""
        self.label.text = msg
        self.label_embed.show()
        self.contents = self.label_embed.object

    def message_wait(self, msg):
        self.hide_all()

        self.title = ""
        self.throbber.text = msg
        self.throbber.start()
        self.half_expand()
        self.throbber_embed.show()
        self.contents = self.throbber_embed.object


class EntryModal(PanelContentModal):
    terra_type = "Widget/Settings/EntryModal"

    def __init__(self, parent, title, subtitle, value="",
                 theme=None, hborder=16, vborder=100):
        PanelContentModal.__init__(self, parent, title, theme,
                                   hborder=hborder, vborder=vborder)

        label = etk.Label(subtitle)
        label.alignment_set(0.0, 1.0)
        label.show()

        self.entry = etk.Entry(text=value)
        self.entry.on_text_activated(self._on_ok_clicked)
        self.entry.show()

        vbox = etk.VBox()
        vbox.border_width_set(25)
        vbox.append(label, etk.VBox.START, etk.VBox.FILL, 0)
        vbox.append(self.entry, etk.VBox.START, etk.VBox.FILL, 10)
        vbox.show()

        self.set_content(vbox)

    def get_text(self):
        return self.entry.text

    def set_text(self, text):
        self.entry.text = text

    text = property(get_text, set_text)

    def _on_ok_clicked(self, *ignored):
        if self.callback_ok_clicked:
            self.callback_ok_clicked(self.entry.text)


class MessageModal(PanelContentModal):
    terra_type = "Widget/Settings/MessageModal"

    def __init__(self, parent, title, subtitle, theme=None, hborder=16,
                 vborder=100, has_cancel=True, label_ok="OK",
                 label_cancel="  Cancel  "):
        PanelContentModal.__init__(self, parent, title, theme, hborder=hborder,
                                   vborder=vborder, has_cancel=has_cancel,
                                   label_ok=label_ok, label_cancel=label_cancel)

        vbox = etk.VBox()
        vbox.border_width_set(25)

        words = subtitle.split()
        line_max_len = 38
        line = ""
        line_len = 0
        chopped_text = []

        while words:
            w = words.pop(0)
            w_len = len(w)

            # Split long words
            if w_len > line_max_len:
                words.insert(0, w[line_max_len:])
                w = w[:line_max_len]
                w_len = line_max_len

            # Clear the current line if we can't fit the word and a " " in it.
            if line_len + w_len + 1 > line_max_len:
                chopped_text.append(line)
                line = ""
                line_len = 0
            elif line_len != 0:
                w = " " + w
                w_len += 1

            line += w
            line_len += w_len

        chopped_text.append(line)

        for l in chopped_text:
            label = etk.Label(l)
            label.alignment_set(0.0, 1.0)
            label.show()
            vbox.append(label, etk.VBox.START, etk.VBox.FILL, 0)

        vbox.show()

        self.set_content(vbox)


class UsernamePasswordModal(PanelContentModal):
    terra_type = "Widget/Settings/UsernamePasswordModal"

    def __init__(self, parent, title,
                 theme=None, hborder=16, vborder=50):
        PanelContentModal.__init__(self, parent, title, theme,
                                   hborder=hborder, vborder=vborder)

        label_username = etk.Label("Username:")
        label_username.alignment_set(0.0, 1.0)
        label_username.show()

        label_password = etk.Label("Password:")
        label_password.alignment_set(0.0, 1.0)
        label_password.show()

        self.entry_username = etk.Entry(text="")
        self.entry_username.on_text_activated(self._on_ok_clicked)
        self.entry_username.show()

        self.entry_password = etk.Entry(text="")
        self.entry_password.on_text_activated(self._on_ok_clicked)
        self.entry_password.show()
        self.entry_password.password_mode_set(1)

        vbox = etk.VBox()
        vbox.border_width_set(5)
        vbox.append(label_username, etk.VBox.START, etk.VBox.FILL, 5)
        vbox.append(self.entry_username, etk.VBox.START, etk.VBox.FILL, 10)

        vbox.append(label_password, etk.VBox.START, etk.VBox.FILL, 5)
        vbox.append(self.entry_password, etk.VBox.START, etk.VBox.FILL, 10)
        vbox.show()

        self.set_content(vbox)

    def get_username(self):
        return self.entry_username.text

    def set_username(self, text):
        self.entry_username.text = text

    username = property(get_username, set_username)

    def get_password(self):
        return self.entry_password.text

    def set_password(self, text):
        self.entry_password.text = text

    password = property(get_password, set_password)

    def _on_ok_clicked(self, *ignored):
        if self.callback_ok_clicked:
            self.callback_ok_clicked()


class ModalThrobberController(ModalController):
    terra_type = "Controller/Folder/Settings/ModalThrobber"
    wait_message = ""

    def __init__(self, model, canvas, parent):
        ModalController.__init__(self, model, canvas, parent)

        self.view = ModalThrobber(self.parent.last_panel.view,
                                  self.wait_message,
                                  hborder=16, vborder=100)
        self.view.callback_escape = self.cb_on_cancel
        self.view.callback_clicked = self.cb_on_cancel
        self.view.show()

        self.update()

    def update(self):
        def end(retval, error):
            self.stop_finished()
            self.do_finish()

        def cb_callback(retval, error):
            # delay for message
            ecore.timer_add(1.5, end, retval, error)

        self.do_start(cb_callback)

    def do_start(self, end_callback):
        raise NotImplementedError("must be implemented by subclasses")

    def do_finish(self):
        pass

    def on_cancel(self):
        pass

    def cb_on_cancel(self):
        self.on_cancel()
        self.stop_finished()

    def stop_finished(self):
        self.view.hide(self.parent.back)

    def delete(self):
        self.view.delete()


#############################################################################
# CheckList Modal Controller
#############################################################################

class CheckContentList(SliderContentList):
    def __init__(self, evas_obj, item_clicked=None):
        SliderContentList.__init__(self, evas_obj)
        self.callback_item_clicked = item_clicked

    def columns_get(self):
        def cb_ui_func(row):
            return (row.name, row.checked)

        def cb_item_click(row, list):
            if self.callback_item_clicked:
                self.callback_item_clicked(row)

        renderer = CheckItemRenderer(ui_func=cb_ui_func,
                                     item_click=cb_item_click)
        return [(100, renderer, True)]


class CheckListThrobberWait(EdjeWidget):

    def __init__(self, parent, theme, text="", cancel_callback=None, **kargs):
        group = "edje_throbber_wait"
        EdjeWidget.__init__(self, parent.evas, group, parent, theme)
        self.throbber = EdjeThrobber(self, self._theme, text)
        self.throbber.show()
        self.part_swallow("throbber", self.throbber)
        self.button = PanelButtonWidget(
            parent.evas, "   Cancel   ", parent=self, theme=self._theme,
            clicked_callback=self._cancel_clicked_cb)
        self.button.show()
        self.part_swallow("button", self.button)
        self.cancel_callback = cancel_callback

    def text_get(self):
        return self.throbber.text

    def text_set(self, text):
        self.throbber.text = text

    text = property(text_get, text_set)

    def start(self):
        self.throbber.start()

    def stop(self):
        self.throbber.stop()

    def _cancel_clicked_cb(self, *ignored):
        if self.cancel_callback:
            self.cancel_callback(self)

    def destroy(self):
        self.throbber.delete()
        self.button.delete()
        self.cancel_callback = None
        EdjeWidget.delete(self)


class CheckListView(ModalSlider):
    HEADER_AREA_HEIGHT = 50

    def __init__(self, parent, title, cb_item_clicked):
        ModalSlider.__init__(self, parent.last_panel.view, title,
                             parent.last_panel.view.theme, hborder=16,
                             vborder=20)

        self.lst = CheckContentList(self.evas,
                                    cb_item_clicked)
        self.lst.show()
        self.center_contents_set(self.lst.object)

        self.header_label = etk.Label("")
        self.header_label.alignment_set(0.0, 0.5)
        self.header_label.show()

        self.content = None
        self.throbber = None
        self.title_separator.show()

        self.caption = etk.HBox()
        self.caption.border_width_set(15)
        self.caption.append(self.header_label, etk.HBox.START,
                            etk.HBox.EXPAND_FILL, 0)
        self.caption.show()
        self.header_vbox.append(self.caption, etk.VBox.START,
                                etk.VBox.EXPAND_FILL, 0)

        self.header_area_show()

        self.button_box_show()
        self.ok_button = self.button_add("   OK   ")
        self.back_button = self.button_add("   Cancel   ")

        self.collapsed = False
        self.callback_ok_clicked = None
        self.callback_back_clicked = None
        self.callback_cancel_clicked = None

    def add_item(self, model):
        if model.children:
            self.lst.item_append(model.children[-1])

    def _on_button_clicked(self, bt):
        if bt == self.ok_button:
            if self.callback_ok_clicked:
                self.callback_ok_clicked()
        elif bt == self.back_button:
            if self.callback_back_clicked:
                self.callback_back_clicked()

    def _cancel_clicked(self, *ignored):
        if self.callback_cancel_clicked:
            self.callback_cancel_clicked()

    def _check_destroy_embed(self):
        if self.content is not None:
            self.content.destroy()
            self.content = None

    def _colapse_embed(self, content):
        self._check_destroy_embed()
        self.title = ""
        self.content = content
        if isinstance(content, etk.VBox):
            embed = etk.Embed(self.evas)
            embed.add(content)
            self.contents = embed.object
        else:
            self.contents = content
        if not self.collapsed:
            self.half_expand()
        self.slider_contents.hide()
        self.collapsed = True

    def colapse_wait(self, message):
        self.throbber = CheckListThrobberWait(self, self._theme, message,
                                         self._cancel_clicked)
        self.throbber.show()
        self.throbber.start()

        self._colapse_embed(self.throbber)

    def colapse_message(self, message):
        label = etk.Label(message)
        label.theme_group = "modal_label"
        label.show()
        label_align = etk.Alignment(0.5, 0.4, 0.0, 0.0, child=label)
        label_align.show()

        vbox = etk.VBox()
        vbox.border_width = 25
        vbox.append(label_align, etk.VBox.START, etk.VBox.FILL, 0)

        ok_button = etk.Button(label="      OK      ")
        ok_button.on_clicked(self._cancel_clicked)
        ok_button.show()
        button_align = etk.Alignment(0.5, 0.5, 0.0, 0.0, child=ok_button)
        button_align.show()
        vbox.append(button_align, etk.VBox.END, etk.VBox.FILL, 0)
        vbox.show()

        self._colapse_embed(vbox)

    @evas.decorators.del_callback
    def _destroy_throbber(self):
        self._check_destroy_embed()
        self.lst.destroy()
        self.lst = None


class ModalCheckList(ModalController):
    terra_type = "Controller/Folder/Settings/ModalCheckList"
    title = None
    header_message = ""

    def __init__(self, model, canvas, parent):
        ModalController.__init__(self, model, canvas, parent)

        self.view = CheckListView(self.parent, self.title or model.title,
                                  self.cb_item_clicked)
        self.view.callback_ok_clicked = self.cb_ok
        self.view.callback_back_clicked = self.back
        self.view.callback_cancel_clicked = self.cb_cancel
        self.view.header_label.text = self.header_message
        self.view.show()

    def item_clicked(self):
        pass

    def ok_clicked(self):
        pass

    def cancel_clicked(self):
        pass

    def add_item(self, model):
        self.view.add_item(model)

    def colapse_wait(self, message):
        self.view.colapse_wait(message)

    def _get_ok_button_text(self):
        return self.view.ok_button.label

    def _set_ok_button_text(self, value):
        self.view.ok_button.label = value

    ok_button_text = property(_get_ok_button_text,
                              _set_ok_button_text)

    def _get_cancel_button_text(self):
        return self.view.back_button.label

    def _set_cancel_button_text(self, value):
        self.view.back_button.label = value

    cancel_button_text = property(_get_cancel_button_text,
                                  _set_cancel_button_text)

    def _get_ok_button_enabled(self):
        return self.view.ok_button.disabled != 0

    def _set_ok_button_enabled(self, value):
        if value:
            self.view.ok_button.disabled = 0
        else:
            self.view.ok_button.disabled = 1

    ok_button_enabled = property(_get_ok_button_enabled,
                                 _set_ok_button_enabled)

    def _get_cancel_button_enabled(self):
        return self.view.back_button.disabled != 0

    def _set_cancel_button_enabled(self, value):
        if value:
            self.view.back_button.disabled = 0
        else:
            self.view.back_button.disabled = 1

    cancel_button_enabled = property(_get_cancel_button_enabled,
                                     _set_cancel_button_enabled)

    def cb_ok(self):
        self.ok_clicked()

    def cb_cancel(self):
        self.cancel_clicked()

    def cb_item_clicked(self, model):
        self.item_clicked(model)

    def close(self):
        if self.view is not None:
            self.view.hide(self.parent.back)

    def delete(self):
        self.view.delete()
        self.view = None

    def back(self):
        self.close()

