/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors: Neil J. Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>

#include <glib.h>
#include <gtk/gtk.h>

#include <libwnck/libwnck.h>

#include <panel-applet.h>
#include <panel-applet-gconf.h>

#define ICON_SIZE 26
#define LOGOUT_CMD "gnome-session-save --kill --gui"

typedef struct 
{
  GtkWidget    *image;
  GtkWidget    *applet;
  
} CloseWindowApp;

static GdkPixbuf *close_pixbuf = NULL;
static GdkPixbuf *logout_pixbuf = NULL;

static void cw_panel_background_changed (PanelApplet               *applet,
                                         PanelAppletBackgroundType  type,
				                                 GdkColor                  *colour,
				                                 GdkPixmap                 *pixmap,
                                         gpointer                   user_data);
static void display_about_dialog (BonoboUIComponent *component, 
                                  gpointer           user_data, 
                                  const gchar       *verb);

static const char Context_menu_xml [] =
   "<popup name=\"button3\">\n"
  "   <menuitem name=\"About Item\" "
   "             verb=\"MenuAbout\" "
   "           _label=\"_About\"\n"
   "          pixtype=\"stock\" "
   "          pixname=\"gnome-stock-about\"/>\n"
   "</popup>\n";

static const BonoboUIVerb _menu_verbs [] = {
        BONOBO_UI_VERB ("MenuAbout", display_about_dialog),
        BONOBO_UI_VERB_END
}; 

static const gchar *close_window_authors [] = {
	"Neil J. Patel <neil.patel@canonical.com>",
	NULL
};

static gboolean
on_button_press (GtkWidget      *eb, 
                 GdkEventButton *event,
                 CloseWindowApp *app)
{
  WnckScreen *screen;
  WnckWindow *window;
  WnckWindowType type;

  g_return_val_if_fail (app, FALSE);

  if (event->button != 1)
    return FALSE;

  screen = wnck_screen_get_default ();
  window = wnck_screen_get_active_window (screen);
  type = wnck_window_get_window_type (window);

  switch (type)
  {
    case WNCK_WINDOW_DESKTOP:
    case WNCK_WINDOW_DOCK:
      gdk_spawn_command_line_on_screen (gdk_screen_get_default (),
                                        LOGOUT_CMD, NULL);
      break;
    case WNCK_WINDOW_NORMAL:
      wnck_window_close (window, event->time);
      break;
    default:
      return FALSE;
  }

  return TRUE;
}

static void
on_active_window_changed (WnckScreen     *screen,
                          WnckWindow     *window,
                          CloseWindowApp *app)
{
  WnckWindowType type = WNCK_WINDOW_DESKTOP;
  WnckWindow *active;

  g_return_if_fail (app);

  active = wnck_screen_get_active_window (screen);
  
  if (WNCK_IS_WINDOW (active))
    type = wnck_window_get_window_type (active);

  switch (type)
  {
    case WNCK_WINDOW_DESKTOP:
    case WNCK_WINDOW_DOCK:
      gtk_image_set_from_pixbuf (GTK_IMAGE (app->image), logout_pixbuf);
      gtk_widget_queue_draw (app->applet);
      break;
    case WNCK_WINDOW_NORMAL:
      gtk_image_set_from_pixbuf (GTK_IMAGE (app->image), close_pixbuf);
      gtk_widget_queue_draw (app->applet);
      break;
    default:
      return;
  }
}

static gboolean
cw_applet_fill (PanelApplet *applet, 
                const gchar *iid, 
                gpointer     data)
{
  GtkIconTheme *theme = gtk_icon_theme_get_default ();
  WnckScreen *screen;
  CloseWindowApp *app;
  GtkWidget *eb;

  if (strcmp (iid, "OAFIID:GNOME_CloseWindow") != 0)
		return FALSE;
  	
  bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);

  if (!GDK_IS_PIXBUF (close_pixbuf))
    close_pixbuf = gtk_icon_theme_load_icon (theme, "gtk-close",
                                             ICON_SIZE, 0, NULL); 
  if (!GDK_IS_PIXBUF (logout_pixbuf))
    logout_pixbuf = gtk_icon_theme_load_icon (theme, "gnome-logout",
                                              ICON_SIZE, 0, NULL); 


  app = g_slice_new0 (CloseWindowApp);
  screen = wnck_screen_get_default ();

  app->applet = GTK_WIDGET (applet);

  eb = gtk_event_box_new ();
  g_object_set (eb, "above-child", TRUE, "visible-window", FALSE, NULL);
	gtk_container_add (GTK_CONTAINER (applet), eb);

  app->image = gtk_image_new ();
  gtk_container_add (GTK_CONTAINER (eb), app->image);

	gtk_widget_show_all (GTK_WIDGET (applet));
	
		
  /* Signals */
	g_signal_connect (applet, "change-background",
			             G_CALLBACK (cw_panel_background_changed), NULL);
  g_signal_connect (screen, "active-window-changed",
                    G_CALLBACK (on_active_window_changed), app);
  g_signal_connect (eb, "button-release-event",
                    G_CALLBACK (on_button_press), app);
	
  panel_applet_setup_menu (PANEL_APPLET (applet),
                         	Context_menu_xml,
                         	_menu_verbs,
                         	NULL);
	
	return TRUE;
}

PANEL_APPLET_BONOBO_FACTORY ("OAFIID:GNOME_CloseWindow_Factory",
                             PANEL_TYPE_APPLET,
                             "Close Active Window",
                             "0",
                             cw_applet_fill,
                             NULL);

static void 
cw_panel_background_changed (PanelApplet               *applet,
                             PanelAppletBackgroundType  type,
                             GdkColor                  *colour,
                             GdkPixmap                 *pixmap,
                             gpointer                   user_data)
{
  GtkRcStyle *rc_style;
  GtkStyle *style;

  /* reset style */
  gtk_widget_set_style (GTK_WIDGET (applet), NULL);
  rc_style = gtk_rc_style_new ();
  gtk_widget_modify_style (GTK_WIDGET (applet), rc_style);
  gtk_rc_style_unref (rc_style);

  switch (type) 
  {
    case PANEL_NO_BACKGROUND:
      break;
    case PANEL_COLOR_BACKGROUND:
      gtk_widget_modify_bg (GTK_WIDGET (applet), GTK_STATE_NORMAL, colour);
      break;
    
    case PANEL_PIXMAP_BACKGROUND:
      style = gtk_style_copy (GTK_WIDGET (applet)->style);
      if (style->bg_pixmap[GTK_STATE_NORMAL])
        g_object_unref (style->bg_pixmap[GTK_STATE_NORMAL]);
      style->bg_pixmap[GTK_STATE_NORMAL] = g_object_ref (pixmap);
      gtk_widget_set_style (GTK_WIDGET (applet), style);
      g_object_unref (style);
      break;
  }
}

static void
display_about_dialog (BonoboUIComponent *component, 
                      gpointer           user_data, 
                      const gchar       *verb)
{
  GtkWidget *panel_about_dialog;
	
  panel_about_dialog = gtk_about_dialog_new ();

  g_object_set (panel_about_dialog,
                "name", _("Close Window"),
                "comments", _("Close the active window"),
                "version", "0.1",
                "authors", close_window_authors,
                "logo-icon-name", "gtk-close",
                "copyright", "Copyright \xc2\xa9 2008 Canonical Ltd",
                NULL);

  gtk_widget_show (panel_about_dialog);

  g_signal_connect (panel_about_dialog, "response",
                    G_CALLBACK (gtk_widget_destroy), NULL);
	

  gtk_window_present (GTK_WINDOW (panel_about_dialog));
}
