/*
 * Copyright (C) 2009 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 * NB: Inspiration for column storage taken from GtkListStore
 *     API inspired by ClutterModel by Matthew Allumn <mallum@openedhand.com>
 *                                     Neil Patel <njp@o-hand.com>
 *                                     Emmanuele Bassi <ebassi@openedhand.com>
 */
/**
 * SECTION:dbus-model
 * @short_description: A generic list model that can synchronize with other
 *    #DbusModel objects across D-Bus.
 * @include: dbusmodel/dbusmodel.h
 *
 * #DbusModel is a generic list model that can hold most GLib Types as column
 * types. It is created with a path (usually unique to your program(s) and
 * something that complies with the D-Bus path spec) on which it will find
 * other #DbusModels created with the same path through D-Bus, and will keep
 * synchronized  with them.
 *
 * This allows to you build MVC programs with a sane model API, but have the
 * controller (or multiple views) in a seperate process.
 *
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <memory.h>
#include <time.h>
#include <unistd.h>
#include <gobject/gvaluecollector.h>
#include <dbus/dbus-gtype-specialized.h>

#include "dbus-model.h"
#include "dbus-model-marshal.h"

G_DEFINE_TYPE (DbusModel, dbus_model, DBUS_TYPE_PEER)

#define DBUS_MODEL_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE(obj, \
        DBUS_TYPE_MODEL, DbusModelPrivate))

#define _DbusModelIter GSequenceIter

#define DBUS_TYPE_G_VALUE_ARRAY (dbus_g_type_get_collection ("GPtrArray", G_TYPE_VALUE))

/**
 * DbusModelPrivate:
 *
 * Ignore this structure.
 **/
struct _DbusModelPrivate
{
  GSequence  *sequence;

  guint       n_columns;
  GType      *column_types;

  guint64 uid;
  guint64 row_counter;

  gboolean    setting_many;

  GHashTable *peers;

  gboolean    found_first_peer;
};

/* Globals */
static GQuark           dbus_model_error_quark       = 0;
static DBusGConnection *dbus_model_global_connection = NULL;

enum
{
  READY,
  ROW_ADDED,
  ROW_REMOVED,
  ROW_CHANGED,

  MODEL_ROW_REMOVED,
  MODEL_ROW_ADDED,
  MODEL_ROW_CHANGED,

  LAST_SIGNAL
};
static guint32 _model_signals[LAST_SIGNAL] = { 0 };

static const GType type_list[] =
{
    G_TYPE_BOOLEAN,
    G_TYPE_UCHAR,
    G_TYPE_INT,
    G_TYPE_UINT,
    G_TYPE_INT64,
    G_TYPE_UINT64,
    G_TYPE_DOUBLE,
    G_TYPE_STRING
};

static const gchar* type_names[] =
{
  "bool",
  "uchar",
  "int",
  "uint",
  "int64",
  "uint64",
  "double",
  "string"
};

/* Forwards */
static void     dbus_model_connected             (DbusPeer      *peer,
                                                  const gchar   *peer_name);
static void     dbus_model_peer_found            (DbusPeer      *peer,
                                                  const gchar   *name);
static void     dbus_model_peer_lost             (DbusPeer      *peer,
                                                  const gchar   *name);

static gboolean _dbus_model_server_get_row_types (DbusModel     *self,
                                                  gchar       ***types,
                                                  GError       **error);
static gboolean _dbus_model_server_get_rows      (DbusModel   *self,
                                                  GPtrArray  **rows,
                                                  GError      *error);

static void     disconnect_and_unref_peer        (gchar      *key,
                                                  DBusGProxy *proxy,
                                                  DbusModel  *model);

#include "dbus-model-client.h"
#include "dbus-model-server.h"

static void dbus_model_set_column_type (DbusModel *self,
                                        guint      column,
                                        GType      gtype);
static DbusModelIter * dbus_model_insert_row (DbusModel *self, gint pos);

/* GObject Init */
static void
dbus_model_finalize (GObject *object)
{
  DbusModelPrivate *priv = DBUS_MODEL (object)->priv;
  GSequenceIter    *iter;

  if (priv->sequence)
    {
      iter = g_sequence_get_begin_iter (priv->sequence);
      while (!g_sequence_iter_is_end (iter))
        {
          GValueArray *value_array = g_sequence_get (iter);

          g_value_array_free (value_array);
          iter = g_sequence_iter_next (iter);
        }
      g_sequence_free (priv->sequence);
      priv->sequence = NULL;
    }

  if (priv->column_types)
    {
      g_free (priv->column_types);
      priv->column_types = NULL;
    }

  if (priv->peers)
    {
      g_hash_table_foreach (priv->peers,
                            (GHFunc)disconnect_and_unref_peer,
                            object);
      g_hash_table_destroy (priv->peers);
      priv->peers = NULL;
    }

  dbus_g_connection_unref (dbus_model_global_connection);

  G_OBJECT_CLASS (dbus_model_parent_class)->finalize (object);
}

static void
dbus_model_set_property (GObject      *object,
                          guint         id,
                          const GValue *value,
                          GParamSpec   *pspec)
{
  switch (id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, id, pspec);
      break;
    }
}

static void
dbus_model_get_property (GObject    *object,
                          guint       id,
                          GValue     *value,
                          GParamSpec *pspec)
{
  switch (id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, id, pspec);
      break;
    }
}

static void
dbus_model_class_init (DbusModelClass *klass)
{
  GObjectClass  *obj_class = G_OBJECT_CLASS (klass);
  DbusPeerClass *per_class = DBUS_PEER_CLASS (klass);

  obj_class->finalize     = dbus_model_finalize;
  obj_class->set_property = dbus_model_set_property;
  obj_class->get_property = dbus_model_get_property;

  per_class->connected  = dbus_model_connected;
  per_class->peer_found = dbus_model_peer_found;
  per_class->peer_lost  = dbus_model_peer_lost;

  /* Add Signals */
  /**
   * DbusModel::ready:
   * @self: the #DbusModel on which the signal is emitted
   *
   * Connect to this signal to be notified when the model is ready to be
   * modified (has it's column number and column type information).
   *
   * This signal is most useful with the dbus_model_new_from_path() method,
   * where there may be a delay between that function returning and the model
   * finding out the column information from other models on D-Bus.
   **/
  _model_signals[READY] =
    g_signal_new ("ready",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (DbusModelClass, ready),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  /**
   * DbusModel::row-added:
   * @self: the #DbusModel on which the signal is emitted
   * @iter: a #DbusModelIter pointing to the newly added row
   *
   * Connect to this signal to be notified when a row is added to @self.
   **/
  _model_signals[ROW_ADDED] =
    g_signal_new ("row-added",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (DbusModelClass, row_added),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE, 1,
                  G_TYPE_POINTER);
  /**
   * DbusModel::row-removed:
   * @self: the #DbusModel on which the signal is emitted
   * @iter: a #DbusModelIter pointing to the removed row
   *
   * Connect to this signal to be notified when a row is removed from @self.
   *   The row is still valid while the signal is being emitted.
   **/
  _model_signals[ROW_REMOVED] =
    g_signal_new ("row-removed",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (DbusModelClass, row_removed),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE, 1,
                  G_TYPE_POINTER);
  /**
   * DbusModel::row-changed:
   * @self: the #DbusModel on which the signal is emitted
   * @iter: a #DbusModelIter pointing to the changed row
   *
   * Connect to this signal to be notified when a row is changed.
   **/
  _model_signals[ROW_CHANGED] =
    g_signal_new ("row-changed",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (DbusModelClass, row_changed),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE, 1,
                  G_TYPE_POINTER);

  /* Dbus signals for communication between instances */
  /*< private >*/
  /**
   * DbusModel::model-row-removed: (skip)
   *
   * Private signal leaked because of how dbus-glib work with signals
   */
  _model_signals[MODEL_ROW_REMOVED] =
    g_signal_new ("model-row-removed",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                  0,
                  NULL, NULL,
                  _dbus_model_marshal_VOID__UINT64,
                  G_TYPE_NONE, 1,
                  G_TYPE_UINT64);
  /*< private >*/
  /**
   * DbusModel::model-row-added: (skip)
   *
   * Private signal leaked because of how dbus-glib work with signals
   */
  _model_signals[MODEL_ROW_ADDED] =
    g_signal_new ("model-row-added",
                  G_OBJECT_CLASS_TYPE (klass),
                  G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                  0,
                  NULL, NULL,
                  _dbus_model_marshal_VOID__UINT64_BOXED,
                  G_TYPE_NONE, 2,
                  G_TYPE_UINT64,
                  DBUS_TYPE_G_VALUE_ARRAY);
  /*< private >*/
  /**
   * DbusModel::model-row-changed: (skip)
   *
   * Private signal leaked because of how dbus-glib work with signals
   */
  _model_signals[MODEL_ROW_CHANGED] =
    g_signal_new ("model-row-changed",
                  G_OBJECT_CLASS_TYPE (klass),
                  G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                  0,
                  NULL, NULL,
                  _dbus_model_marshal_VOID__UINT64_BOXED,
                  G_TYPE_NONE, 2,
                  G_TYPE_UINT64,
                  DBUS_TYPE_G_VALUE_ARRAY);

  /* Register as a DBus object */
  dbus_g_object_type_install_info (G_TYPE_FROM_CLASS (klass),
                                   &dbus_glib__dbus_model_server_object_info);

  /* Add private data */
  g_type_class_add_private (obj_class, sizeof (DbusModelPrivate));
}

static void
dbus_model_init (DbusModel *model)
{
  DbusModelPrivate *priv;
  static gint       n_models = 1;
  guint64           uid = 0;

  priv = model->priv = DBUS_MODEL_GET_PRIVATE (model);

  priv->sequence = g_sequence_new (NULL);
  priv->setting_many = FALSE;
  priv->peers = g_hash_table_new_full (g_str_hash, g_str_equal,
                                       g_free, NULL);

  if (!dbus_model_error_quark)
    dbus_model_error_quark = g_quark_from_string ("dbus-model-error");


  /* We need a UID to identify rows from across peers */
  uid = getpid();
  uid *= 10000000;
  uid += n_models * 1000000;
  priv->uid = uid;
  priv->row_counter = uid;

  n_models++;
}

/* Private Methods */
static gchar *
_get_string_for_type (GType type)
{
  gint i;

  for (i = 0; i < G_N_ELEMENTS (type_list); i++)
    {
      if (g_type_is_a (type, type_list[i]))
        return g_strdup (type_names[i]);
    }
  return NULL;
}

static GType
_get_type_for_string (const gchar *type_name)
{
  gint i;

  for (i = 0; i < G_N_ELEMENTS (type_names); i++)
    {
      if (g_strcmp0 (type_name, type_names[i]) == 0)
        return type_list[i];
    }

  g_warning ("Unable to recoginize type name (%s)", type_name);
  return G_TYPE_NONE;
}

/* Callbacks from peers */
static void
on_peer_row_added (DBusGProxy *proxy,
                   guint64     time_,
                   GPtrArray  *row,
                   DbusModel  *self)
{
  DbusModelPrivate *priv;
  DbusModelIter    *iter;
  gint i;

  g_return_if_fail (DBUS_IS_MODEL (self));
  priv = self->priv;

  iter = dbus_model_insert_row (self, -1);
  g_assert (iter);

  priv->setting_many = TRUE;

  for (i = 0; i < priv->n_columns+1; i++)
    {
      GValue *value = g_ptr_array_index (row, i);

      dbus_model_set_value (self, iter, i, value);
    }

  priv->setting_many = FALSE;

  g_signal_emit (self, _model_signals[ROW_ADDED], 0, iter);
}

static void
on_peer_row_removed (DBusGProxy *proxy,
                     guint64     uid,
                     DbusModel  *self)
{
  DbusModelPrivate *priv;
  GSequenceIter    *iter;

  g_return_if_fail (DBUS_IS_MODEL (self));
  priv = self->priv;

  iter = g_sequence_get_begin_iter (priv->sequence);
  while (iter != g_sequence_get_end_iter (priv->sequence))
    {
      GValue      *value;
      GValueArray *array;

      array = (GValueArray *)g_sequence_get (iter);
      value = g_value_array_get_nth (array, priv->n_columns);

      if (uid == g_value_get_uint64 (value))
        {
          g_signal_emit (self, _model_signals[ROW_REMOVED], 0, iter);

          g_value_array_free (array);
          g_sequence_remove (iter);
          break;
        }
      iter = g_sequence_iter_next (iter);
    }
}

static void
on_peer_row_changed (DBusGProxy *proxy,
                     guint64     time_,
                     GPtrArray  *row,
                     DbusModel  *self)
{
  DbusModelPrivate *priv;
  GSequenceIter    *iter;
  guint64           uid;
  GValue           *val;

  g_return_if_fail (DBUS_IS_MODEL (self));
  priv = self->priv;

  val = g_ptr_array_index (row, priv->n_columns);
  uid = g_value_get_uint64 (val);

  iter = g_sequence_get_begin_iter (priv->sequence);
  while (iter != g_sequence_get_end_iter (priv->sequence))
    {
      GValue      *value;
      GValueArray *array;

      array = (GValueArray *)g_sequence_get (iter);
      value = g_value_array_get_nth (array, priv->n_columns);

      if (uid == g_value_get_uint64 (value))
        {
          gint i;

          priv->setting_many = TRUE;

          for (i = 0; i < priv->n_columns; i++)
            {
              dbus_model_set_value (self,
                                    (DbusModelIter *)iter,
                                    i,
                                    (GValue*)g_ptr_array_index (row, i));
            }

          priv->setting_many = FALSE;

          g_signal_emit (self, _model_signals[ROW_CHANGED], 0, iter);
          break;
        }
      iter = g_sequence_iter_next (iter);
    }
}

/* Callbacks from DbusPeer */
static gboolean
emit_ready_signal (DbusModel *model)
{
  g_signal_emit (model, _model_signals[READY], 0);

  return FALSE;
}

static void
get_rows_callback_foreach (GPtrArray *ptr, DbusModel *self)
{
  DbusModelIter *iter;
  gint           i;

  self->priv->setting_many = TRUE;

  iter = dbus_model_insert_row (self, -1);

  for (i = 0; i < ptr->len; i++)
    {
      GValue *value = g_ptr_array_index (ptr, i);
      dbus_model_set_value (self, iter, i, value);

      g_value_unset (value);
    }

  self->priv->setting_many = FALSE;

  g_signal_emit (self, _model_signals[ROW_ADDED], 0, iter);

  g_ptr_array_free (ptr, TRUE);
}

static void
get_rows_callback (DBusGProxy *proxy,
                   GPtrArray  *rows,
                   GError     *error,
                   gpointer    userdata)
{
  DbusModel        *self = DBUS_MODEL (userdata);
  DbusModelPrivate *priv;

  g_return_if_fail (DBUS_IS_MODEL (self));
  priv = self->priv;

  g_ptr_array_foreach (rows, (GFunc)get_rows_callback_foreach, self);

  g_ptr_array_free (rows, TRUE);
}

static void
dbus_model_peer_found (DbusPeer      *peer,
                       const gchar   *name)
{
  DbusModel        *self = DBUS_MODEL (peer);
  DbusModelPrivate *priv = self->priv;
  DBusGProxy       *proxy;
  DBusGConnection  *conn;
  GError           *error = NULL;

  conn = dbus_peer_get_connection (peer);

  proxy = dbus_g_proxy_new_for_name_owner (conn,
                                           name,
                                           "/com/canonical/DbusModel",
                                           "com.canonical.DbusModel",
                                           &error);
  if (proxy == NULL)
    {
      g_warning ("Unable to connect to peer (%s): %s",
                 name,
                 error ? error->message: "Unknown");
      if (error)
        g_error_free (error);

      dbus_g_connection_unref (conn);
      return;
    }

  /* The proxy is valid, so let's connect to all it's signals and add it to
   * the collection
   */
  g_hash_table_insert (priv->peers, g_strdup (name), proxy);

  /* This is a good time to get the row types if we don't already have them */
  if (priv->n_columns < 1)
    {
      gchar **types = NULL;

      if (com_canonical_DbusModel_get_row_types (proxy, &types, NULL))
        {
          gint i;

          dbus_model_set_n_columns (self, g_strv_length (types));

          for (i = 0; types[i]; i++)
            {
              dbus_model_set_column_type (self,
                                          i,
                                          _get_type_for_string (types[i]));
            }

          g_idle_add ((GSourceFunc)emit_ready_signal, self);

          com_canonical_DbusModel_get_rows_async (proxy,
                                                  get_rows_callback,
                                                  self);
        }
      g_strfreev (types);
    }
  else if (!priv->found_first_peer)
    {
      com_canonical_DbusModel_get_rows_async (proxy,
                                              get_rows_callback,
                                              self);
      priv->found_first_peer = TRUE;
    }

  dbus_g_proxy_add_signal (proxy,
                           "ModelRowAdded",
                           G_TYPE_UINT64,
                           DBUS_TYPE_G_VALUE_ARRAY,
                           G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (proxy, "ModelRowAdded",
                               G_CALLBACK (on_peer_row_added), self,
                               NULL);

  dbus_g_proxy_add_signal (proxy,
                           "ModelRowRemoved",
                           G_TYPE_UINT64,
                           G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (proxy, "ModelRowRemoved",
                               G_CALLBACK (on_peer_row_removed), self,
                               NULL);

  dbus_g_proxy_add_signal (proxy,
                           "ModelRowChanged",
                           G_TYPE_UINT64,
                           DBUS_TYPE_G_VALUE_ARRAY,
                           G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (proxy, "ModelRowChanged",
                               G_CALLBACK (on_peer_row_changed), self,
                               NULL);
}

static void
disconnect_and_unref_peer (gchar      *key,
                           DBusGProxy *proxy,
                           DbusModel  *self)
{
  if (proxy)
    {
      dbus_g_proxy_disconnect_signal (proxy,
                                      "ModelRowAdded",
                                      G_CALLBACK (on_peer_row_added),
                                      self);
      dbus_g_proxy_disconnect_signal (proxy,
                                      "ModelRowRemoved",
                                      G_CALLBACK (on_peer_row_removed),
                                      self);
      dbus_g_proxy_disconnect_signal (proxy,
                                      "ModelRowChanged",
                                      G_CALLBACK (on_peer_row_changed),
                                      self);
      g_object_unref (proxy);
    }
}

static void
dbus_model_peer_lost (DbusPeer      *peer,
                      const gchar   *name)
{
  DbusModel        *self = DBUS_MODEL (peer);
  DbusModelPrivate *priv;
  DBusGProxy       *proxy;

  g_return_if_fail (DBUS_IS_MODEL (self));
  priv = DBUS_MODEL (peer)->priv;

  proxy = g_hash_table_lookup (priv->peers, name);

  disconnect_and_unref_peer (NULL, proxy, self);
  g_hash_table_remove (priv->peers, proxy);
}

static void
dbus_model_connected (DbusPeer      *peer,
                      const gchar   *peer_name)
{
  DBusGConnection *conn;

  g_return_if_fail (DBUS_IS_MODEL (peer));

  conn = dbus_peer_get_connection (peer);

  dbus_g_connection_register_g_object (conn,
                                       "/com/canonical/DbusModel",
                                       G_OBJECT (peer));

  dbus_g_connection_unref (conn);
}

/* Model stuff */
static gboolean
dbus_model_check_type (GType gtype)
{
  gint i;

  if (! G_TYPE_IS_VALUE_TYPE (gtype))
    return FALSE;

  while (type_list[i] != G_TYPE_INVALID)
    {
      if (g_type_is_a (gtype, type_list[i]))
        return TRUE;
      i++;
    }
  return FALSE;
}

static void
dbus_model_set_column_type (DbusModel *self,
                            guint      column,
                            GType      gtype)
{
  DbusModelPrivate *priv = self->priv;

  priv->column_types[column] = gtype;
}

static DbusModelIter *
dbus_model_insert_row (DbusModel *self, gint where)
{
  DbusModelPrivate *priv = self->priv;
  GValueArray      *array;
  gint              i;
  GSequenceIter    *iter = NULL;

  array = g_value_array_new (priv->n_columns+1);

  for (i = 0; i < priv->n_columns; i++)
    {
      GValue *value = NULL;

      g_value_array_append (array, NULL);

      value = g_value_array_get_nth (array, i);
      g_value_init (value, dbus_model_get_column_type (self, i));
    }
    {
      GValue *value;

      g_value_array_append (array, NULL);

      value = g_value_array_get_nth (array, i);
      g_value_init (value, G_TYPE_UINT64);
      g_value_set_uint64 (value, priv->row_counter++);
    }

  if (where < 0)
    {
      iter = g_sequence_append (priv->sequence, array);
    }
  else if (where == 0)
    {
      iter = g_sequence_prepend (priv->sequence, array);
    }
  else
    {
      iter = g_sequence_get_iter_at_pos (priv->sequence, where);
      iter = g_sequence_insert_before (iter, array);
    }

  return (DbusModelIter *)iter;
}

static void
dbus_model_set_internal_valist (DbusModel      *self,
                                DbusModelIter  *iter,
                                va_list         args)
{
  guint column = 0;

  column = va_arg (args, gint);

  while (column != -1)
    {
      GValue  value = { 0, };
      gchar  *error = NULL;

      if (column < 0 || column >= dbus_model_get_n_columns (self))
        {
          g_warning ("Column %d is out of range", column);
          break;
        }

      g_value_init (&value, dbus_model_get_column_type (self, column));

      G_VALUE_COLLECT (&value, args, 0, &error);
      if (error)
        {
          g_warning ("%s: %s", G_STRLOC, error);
          g_free (error);
          break;
        }

      dbus_model_set_value (self, iter, column, &value);

      g_value_unset (&value);

      column = va_arg (args, gint);
    }
}

static void
unset_and_free_gvalue (gpointer data)
{
  GValue *value = (GValue*)data;

  g_value_unset (value);
  g_free (value);
}

static GPtrArray *
_array_for_iter (DbusModel *self, DbusModelIter *iter)
{
  GPtrArray *array;
  gint       i;

  array = g_ptr_array_new_with_free_func (unset_and_free_gvalue);

  for (i = 0; i < self->priv->n_columns+1; i++)
    {
      GValue *value = g_new0 (GValue, 1);

      dbus_model_get_value (self, iter, i, value);

      g_ptr_array_add (array, (gpointer)value);
    }

  return array;
}

static DbusModel *
dbus_model_new_real (const gchar *path)
{
  DbusModel *self;

  /* We need to call dbus_g_bus_get at least once to initialize the the
   * specialized types
   */
  if (dbus_model_global_connection)
    {
      dbus_g_connection_ref (dbus_model_global_connection);
    }
  else
    {
      dbus_model_global_connection = dbus_g_bus_get (DBUS_BUS_SESSION, NULL);
      dbus_g_object_register_marshaller (_dbus_model_marshal_VOID__UINT64_BOXED,
                                         G_TYPE_NONE,
                                         G_TYPE_UINT64,
                                         DBUS_TYPE_G_VALUE_ARRAY,
                                         G_TYPE_INVALID);
      dbus_g_object_register_marshaller (_dbus_model_marshal_VOID__UINT64,
                                         G_TYPE_NONE,
                                         G_TYPE_UINT64,
                                         G_TYPE_INVALID);
    }

  self = g_object_new (DBUS_TYPE_MODEL,
                       "swarm-name", path,
                       NULL);

  return self;
}

static GValue *
_value_for_column (DbusModel *self, DbusModelIter *iter, guint column)
{
  GValueArray *array;
  GValue      *value;

  array = (GValueArray *)g_sequence_get ((GSequenceIter *)iter);
  value = g_value_array_get_nth (array, column);

  return value;
}

/* Public Methods */

/**
 * dbus_model_new:
 * @path: a path where models of this type with synchronize with each other
 * @n_columns: number of columns in the model
 * @VarArgs: @n_columns number of #GTypes
 *
 * Creates a new #DbusModel at the specified @path, with @n_columns of the
 * types passed in.
 *
 * Upon creation of a new #DbusModel, the model will find other models with
 * the same @path and synchronize itself with these models.
 *
 * Return value: a new #DbusModel
 **/
DbusModel *
dbus_model_new (const gchar *path, guint n_columns, ...)
{
  DbusModel *self;
  va_list    args;
  gint       i;

  g_return_val_if_fail (path != NULL, NULL);
  g_return_val_if_fail (n_columns > 0, NULL);

  self = dbus_model_new_real (path);

  dbus_model_set_n_columns (self, n_columns);

  va_start (args, n_columns);

  for (i = 0; i < n_columns; i++)
    {
      GType type = va_arg (args, GType);

      if (!dbus_model_check_type (type))
        {
          g_warning ("DbusModel: %s is an invalid type", g_type_name (type));
          g_object_unref (self);
          return NULL;
        }

      dbus_model_set_column_type (self, i, type);
    }

  dbus_model_set_column_type (self, i, G_TYPE_UINT64);

  va_end (args);

  return self;
}

DbusModel *
dbus_model_new_from_path (const gchar *path)
{
  DbusModel *self;

  g_return_val_if_fail (path != NULL, NULL);

  self = dbus_model_new_real (path);

  return self;
}

/**
 * dbus_model_set_n_columns:
 * @self: a #DbusModel instance
 * @n_columns: the number of columns
 *
 * Sets the number of columns in @self to @n_columns.
 *
 * This function should only be called once.
 *
 **/
void
dbus_model_set_n_columns (DbusModel  *self,
                          guint       n_columns)
{
  DbusModelPrivate *priv;

  g_return_if_fail (DBUS_IS_MODEL (self));
  priv = self->priv;

  if (priv->n_columns > 0)
    {
      g_warning ("%s should only be called once", G_STRFUNC);
      return;
    }

  priv->n_columns = n_columns;
  priv->column_types = g_new0 (GType, n_columns+1);
  priv->column_types[n_columns] = G_TYPE_UINT64;
}

/**
 * dbus_model_get_n_columns:
 * @self: a #DbusModel
 *
 * Gets the number of columns in @self
 *
 * Return value: the number of columns per row in @self
 **/
guint
dbus_model_get_n_columns (DbusModel *self)
{
  g_return_val_if_fail (DBUS_IS_MODEL (self), 0);

  return self->priv->n_columns;
}

/**
 * dbus_model_get_n_rows:
 * @self: a #DbusModel
 *
 * Gets the number of rows in @self
 *
 * Return value: the number of rows in @self
 **/
guint
dbus_model_get_n_rows (DbusModel *self)
{
  g_return_val_if_fail (DBUS_IS_MODEL (self), 0);

  return g_sequence_get_length (self->priv->sequence);
}

/**
 * dbus_model_get_column_type:
 * @self: a #DbusModel
 * @column: the column to get retrieve the type of
 *
 * Get the #GType of @column
 *
 * Return value: the #GType that @column holds
 **/
GType
dbus_model_get_column_type (DbusModel *self,
                            guint      column)
{
  g_return_val_if_fail (DBUS_IS_MODEL (self), G_TYPE_NONE);

  return self->priv->column_types[column];
}

/**
 * dbus_model_clear:
 * @self: a #DbusModel object to clear
 *
 * Removes all rows in the model. Signals are emitted for each row in the model
 **/
void
dbus_model_clear (DbusModel *self)
{
  DbusModelPrivate *priv;
  DbusModelIter    *iter;

  g_return_if_fail (DBUS_IS_MODEL (self));
  priv = self->priv;

  iter = dbus_model_get_first_iter (self);

  while (!dbus_model_is_last (self, iter))
    {
      dbus_model_remove (self, iter);

      iter = dbus_model_get_first_iter (self);
    }
}

/**
 * dbus_model_add:
 * @self: a #DbusModel
 * @VarArgs: pairs of column number and value, terminated with -1
 *
 * Creates and inserts a new row to the #DbusModel, setting the row values
 * upon creation. The row will be placed in the model according to the current
 * sorting.
 *
 * For example, to append a new row where column 0 is %G_TYPE_INT and column 1
 * is %G_TYPE_STRING:
 *
 * <informalexample><programlisting>
 *  DbusModel *model;
 *  model = dbus_model_new ("org.myapp.Results", 2, G_TYPE_INT, G_TYPE_STRING);
 *  dbus_model_add (model, 0, 10, 1, "Rooney", -1);
 * </programlisting></informalexample>
 **/
void
dbus_model_add (DbusModel *self,
                ...)
{
  DbusModelPrivate *priv;
  DbusModelIter    *iter;
  va_list           args;

  g_return_if_fail (DBUS_IS_MODEL (self));
  priv = self->priv;

  iter = dbus_model_insert_row (self, -1);
  g_assert (iter);

  priv->setting_many = TRUE;

  va_start (args, self);
  dbus_model_set_internal_valist (self, iter, args);
  va_end (args);

  priv->setting_many = FALSE;

  g_signal_emit (self, _model_signals[ROW_ADDED], 0, iter);
  g_signal_emit (self, _model_signals[MODEL_ROW_ADDED], 0,
                 (guint64)time(NULL),
                 _array_for_iter (self, iter));
}

/**
 * dbus_model_remove:
 * @self: a #DbusModel
 * @iter: a #DbusModelIter pointing to the row to remove
 *
 * Removes the row at the given position from the model.
 **/
void
dbus_model_remove (DbusModel     *self,
                   DbusModelIter *iter_)
{
  DbusModelPrivate *priv;
  GSequenceIter    *iter = (GSequenceIter *)iter_;

  g_return_if_fail (DBUS_IS_MODEL (self));
  priv = self->priv;

  if (iter_ == dbus_model_get_last_iter (self))
    return;

  if (iter)
    {
      GValueArray *array;
      GValue      *value;

      /* Emit the removed signal while the iter is still valid */
      g_signal_emit (self, _model_signals[ROW_REMOVED], 0, iter);

      array = g_sequence_get (iter);

      value = g_value_array_get_nth (array, priv->n_columns);

      g_signal_emit (self, _model_signals[MODEL_ROW_REMOVED], 0,
                     g_value_get_uint64 (value));

      g_value_array_free (array);

      g_sequence_remove (iter);
    }
  else
    {
      g_warning ("Unable to remove row '%p': does not exists", iter);
    }
}

/**
 * dbus_model_set:
 * @self: a #DbusModel
 * @iter: a #DbusModelIter
 * @VarArgs: a list of column/value pairs, terminated by -1
 *
 * Sets the value of one or more cells in the row references by @iter. The
 * variable argument list should contain integer column numbers, each column
 * number followed by the value to be set. The list is terminated by a -1.
 *
 * For example, to set column 0 with type %G_TYPE_STRING, use:
 * <informalexample><programlisting>
 *   dbus_model_set (model, iter, 0, "foo", -1);
 * </programlisting></informalexample>
 **/
void
dbus_model_set (DbusModel *self,
                DbusModelIter *iter,
                ...)
{
  va_list args;

  g_return_if_fail (DBUS_IS_MODEL (self));
  g_return_if_fail (iter);

  self->priv->setting_many = TRUE;

  va_start (args, iter);
  dbus_model_set_valist (self, iter, args);
  va_end (args);

  self->priv->setting_many = FALSE;

  g_signal_emit (self, _model_signals[ROW_CHANGED], 0, iter);
  g_signal_emit (self, _model_signals[MODEL_ROW_CHANGED], 0,
                 (guint64)time(NULL),
                 _array_for_iter (self, iter));
}

/**
 * dbus_model_set_valist:
 * @self: a #DbusModel
 * @iter: a #DbusModelIter
 * @args: a list of column/value location pairs, terminated by -1
 *
 * See #dbus_model_set(). This version takes a va_list for language bindings.
 **/
void
dbus_model_set_valist (DbusModel       *self,
                       DbusModelIter   *iter,
                       va_list          args)
{
  g_return_if_fail (DBUS_IS_MODEL (self));
  g_return_if_fail (iter);

  dbus_model_set_internal_valist (self, iter, args);

  if (self->priv->setting_many == FALSE)
    {
      g_signal_emit (self, _model_signals[ROW_CHANGED], 0, iter);
      g_signal_emit (self, _model_signals[MODEL_ROW_CHANGED], 0,
                     (guint64)time(NULL),
                     _array_for_iter (self, iter));
    }
}

/**
 * dbus_model_set_value:
 * @self: a #DbusModel
 * @iter: a #DbusModelIter
 * @column: column number to set the value
 * @value: new value for cell
 *
 * Sets the data in @column for the row @iter points to, to @value. The type
 * of @value must be convertable to the type of the column.
 **/
void
dbus_model_set_value (DbusModel      *self,
                      DbusModelIter  *iter,
                      guint           column,
                      const GValue   *value)
{
  GSequenceIter    *seq_iter;
  DbusModelPrivate *priv;
  GValueArray      *value_array;
  GValue           *iter_value;
  GValue            real_value = { 0, };
  gboolean          converted = FALSE;

  g_return_if_fail (DBUS_IS_MODEL (self));
  priv = self->priv;

  seq_iter = (GSequenceIter *)iter;
  g_assert (seq_iter != NULL);

  value_array = g_sequence_get (seq_iter);
  iter_value = g_value_array_get_nth (value_array, column);
  g_assert (iter_value != NULL);

  if (!g_type_is_a (G_VALUE_TYPE (value), G_VALUE_TYPE (iter_value)))
    {
      if (!g_value_type_compatible (G_VALUE_TYPE (value),
                                    G_VALUE_TYPE (iter_value)) &&
          !g_value_type_compatible (G_VALUE_TYPE (iter_value),
                                    G_VALUE_TYPE (value)))
        {
          g_warning ("Unable to convert from %s to %s\n",
                     g_type_name (G_VALUE_TYPE (value)),
                     g_type_name (G_VALUE_TYPE (iter_value)));
          return;
        }
      if (!g_value_transform (value, &real_value))
        {
          g_warning ("Unable to make conversion from %s to %s\n",
                     g_type_name (G_VALUE_TYPE (value)),
                     g_type_name (G_VALUE_TYPE (iter_value)));
          g_value_unset (&real_value);
        }

      converted = TRUE;
    }

  if (converted)
    {
      g_value_copy (&real_value, iter_value);
      g_value_unset (&real_value);
    }
  else
    {
      g_value_copy (value, iter_value);
    }

  if (self->priv->setting_many == FALSE)
    {
      g_signal_emit (self, _model_signals[ROW_CHANGED], 0, iter);
      g_signal_emit (self, _model_signals[MODEL_ROW_CHANGED], 0,
                     (guint64)time(NULL),
                     _array_for_iter (self, iter));
    }
}

/**
 * dbus_model_get:
 * @self: a #DbusModel
 * @iter: a #DbusModelIter
 * @VarArgs: a list of column/return location pairs, terminated by -1
 *
 * Gets the value of one or more cells in the row references by @iter. The
 * variable argument list should contain integer column numbers, each column
 * number followed by a place to store the value being retrieved. The list is
 * terminated by a -1.
 *
 * For example, to get a value from column 0 with type %G_TYPE_STRING use:
 * <informalexample><programlisting>
 *  dbus_model_get (model, iter, 0, &place_string_here, -1);
 * </programlisting></informalexample>
 *
 * where place_string_here is a gchar* to be filled with the string. If
 * appropriate, the returned values have to be freed or unreferenced.
 **/
void
dbus_model_get (DbusModel *self,
                DbusModelIter *iter,
                ...)
{
  va_list args;

  g_return_if_fail (DBUS_IS_MODEL (self));
  g_return_if_fail (iter);

  va_start (args, iter);
  dbus_model_get_valist (self, iter, args);
  va_end (args);
}

/**
 * dbus_model_get_valist:
 * @self: a #DbusModel
 * @iter: a #DbusModelIter
 * @args: a list of column/return location pairs, terminated by -1
 *
 * See #dbus_model_get(). This version takes a va_list for language bindings.
 **/
void
dbus_model_get_valist (DbusModel       *self,
                       DbusModelIter   *iter,
                       va_list          args)
{
  guint column;

  g_return_if_fail (DBUS_IS_MODEL (self));
  g_return_if_fail (iter);

  column = va_arg (args, gint);

  while (column != -1)
    {
      GValue  value = { 0, };
      gchar  *error = NULL;

      if (column < 0 || column >= dbus_model_get_n_columns (self) +1)
        {
          g_warning ("Column '%d' is out of range for this model", column);
          break;
        }

      dbus_model_get_value (self, iter, column, &value);

      G_VALUE_LCOPY (&value, args, 0, &error);
      if (error)
        {
          g_warning ("Unable to copy value: %s", error);
          g_free (error);

          break;
        }

      g_value_unset (&value);

      column = va_arg (args, gint);
    }
}

/**
 * dbus_model_get_value:
 * @self: a #DbusModel
 * @iter: a #DbusModelIter
 * @column: column number to retrieve the value from
 * @value: an empty #GValue to set
 *
 * Sets an intializes @value to that at @column. When done with @value
* g_value_unset() needs to be called to free any allocated memory.
 **/
void
dbus_model_get_value (DbusModel     *self,
                      DbusModelIter *iter,
                      guint          column,
                      GValue        *value)
{
  GSequenceIter *seq_iter;
  GValueArray   *value_array;
  GValue        *iter_value;
  GValue         real_value = { 0, };
  gboolean       converted = FALSE;

  g_return_if_fail (DBUS_IS_MODEL (self));
  g_return_if_fail (iter != NULL);
  g_return_if_fail (value != NULL);

  g_value_init (value, dbus_model_get_column_type (self, column));

  seq_iter = (GSequenceIter *)iter;
  g_assert (seq_iter != NULL);

  value_array = g_sequence_get (seq_iter);
  iter_value = g_value_array_get_nth (value_array, column);
  g_assert (iter_value != NULL);

  if (!g_type_is_a (G_VALUE_TYPE (value), G_VALUE_TYPE (iter_value)))
    {
      if (!g_value_type_compatible (G_VALUE_TYPE (value),
                                    G_VALUE_TYPE (iter_value)) &&
          !g_value_type_compatible (G_VALUE_TYPE (iter_value),
                                    G_VALUE_TYPE (value)))
        {
          g_warning ("Unable to convert from %s to %s",
                     g_type_name (G_VALUE_TYPE (value)),
                     g_type_name (G_VALUE_TYPE (iter_value)));
          return;
        }

      if (!g_value_transform (iter_value, &real_value))
        {
          g_warning ("Unable to make conversion from %s to %s",
                     g_type_name (G_VALUE_TYPE (value)),
                     g_type_name (G_VALUE_TYPE (iter_value)));
          g_value_unset (&real_value);
        }

      converted = TRUE;
    }

  if (converted)
    {
      g_value_copy (&real_value, value);
      g_value_unset (&real_value);
    }
  else
    g_value_copy (iter_value, value);
}

/**
 * dbus_model_get_bool:
 * @self: a #DbusModel
 * @iter: a #DbusModelIter
 * @column: the column to retrieve a boolean from
 *
 * Return value: if @iter and @column are valid, the boolean stored at @column.
 *  Otherwise %FALSE
 **/
gboolean
dbus_model_get_bool (DbusModel      *self,
                     DbusModelIter  *iter,
                     guint           column)
{
  GValue      *value;

  g_return_val_if_fail (DBUS_IS_MODEL (self), FALSE);
  g_return_val_if_fail (iter, FALSE);

  value = _value_for_column (self, iter, column);

  g_return_val_if_fail (G_VALUE_HOLDS_BOOLEAN (value), FALSE);

  return g_value_get_boolean (value);
}

/**
 * dbus_model_get_uchar:
 * @self: a #DbusModel
 * @iter: a #DbusModelIter
 * @column: the column to retrieve a uchar from
 *
 * Return value: if @iter and @column are valid, the uchar stored at @column.
 *  Otherwise 0.
 **/
guchar
dbus_model_get_uchar (DbusModel      *self,
                      DbusModelIter  *iter,
                      guint           column)
{
  GValue      *value;

  g_return_val_if_fail (DBUS_IS_MODEL (self), 0);
  g_return_val_if_fail (iter, 0);

  value = _value_for_column (self, iter, column);

  g_return_val_if_fail (G_VALUE_HOLDS_UCHAR (value), 0);

  return g_value_get_uchar (value);
}

/**
 * dbus_model_get_int:
 * @self: a #DbusModel
 * @iter: a #DbusModelIter
 * @column: the column to retrieve a int from
 *
 * Return value: if @iter and @column are valid, the int stored at @column.
 *  Otherwise 0.
 **/
gint
dbus_model_get_int (DbusModel        *self,
                    DbusModelIter    *iter,
                    guint             column)
{
  GValue      *value;

  g_return_val_if_fail (DBUS_IS_MODEL (self), 0);
  g_return_val_if_fail (iter, 0);

  value = _value_for_column (self, iter, column);

  g_return_val_if_fail (G_VALUE_HOLDS_INT (value), 0);

  return g_value_get_int (value);
}

/**
 * dbus_model_get_uint:
 * @self: a #DbusModel
 * @iter: a #DbusModelIter
 * @column: the column to retrieve a uint from
 *
 * Return value: if @iter and @column are valid, the uint stored at @column.
 *  Otherwise 0.
 **/
guint
dbus_model_get_uint (DbusModel      *self,
                     DbusModelIter  *iter,
                     guint           column)
{
  GValue      *value;

  g_return_val_if_fail (DBUS_IS_MODEL (self), 0);
  g_return_val_if_fail (iter, 0);

  value = _value_for_column (self, iter, column);

  g_return_val_if_fail (G_VALUE_HOLDS_UINT (value), 0);

  return g_value_get_uint (value);
}


/**
 * dbus_model_get_int64:
 * @self: a #DbusModel
 * @iter: a #DbusModelIter
 * @column: the column to retrieve a int64 from
 *
 * Return value: if @iter and @column are valid, the int64 stored at @column.
 *  Otherwise 0.
 **/
gint64
dbus_model_get_int64 (DbusModel      *self,
                      DbusModelIter  *iter,
                      guint           column)
{
  GValue      *value;

  g_return_val_if_fail (DBUS_IS_MODEL (self), 0);
  g_return_val_if_fail (iter, 0);

  value = _value_for_column (self, iter, column);

  g_return_val_if_fail (G_VALUE_HOLDS_INT64 (value), 0);

  return g_value_get_int64 (value);
}


/**
 * dbus_model_get_uint64:
 * @self: a #DbusModel
 * @iter: a #DbusModelIter
 * @column: the column to retrieve a uint64 from
 *
 * Return value: if @iter and @column are valid, the uint64 stored at @column.
 *  Otherwise 0.
 **/
guint64
dbus_model_get_uint64 (DbusModel      *self,
                       DbusModelIter  *iter,
                       guint           column)
{
  GValue      *value;

  g_return_val_if_fail (DBUS_IS_MODEL (self), 0);
  g_return_val_if_fail (iter, 0);

  value = _value_for_column (self, iter, column);

  g_return_val_if_fail (G_VALUE_HOLDS_UINT64 (value), 0);

  return g_value_get_uint64 (value);
}

/**
 * dbus_model_get_double:
 * @self: a #DbusModel
 * @iter: a #DbusModelIter
 * @column: the column to retrieve a double from
 *
 * Return value: if @iter and @column are valid, the double stored at @column.
 *  Otherwise 0.
 **/
gdouble
dbus_model_get_double (DbusModel      *self,
                       DbusModelIter  *iter,
                       guint           column)
{
  GValue      *value;

  g_return_val_if_fail (DBUS_IS_MODEL (self), 0);
  g_return_val_if_fail (iter, 0);

  value = _value_for_column (self, iter, column);

  g_return_val_if_fail (G_VALUE_HOLDS_DOUBLE (value), 0);

  return g_value_get_double (value);
}

/**
 * dbus_model_get_string:
 * @self: a #DbusModel
 * @iter: a #DbusModelIter
 * @column: the column to retrieve a string from
 *
 * Return value: if @iter and @column are valid, the string stored at @column.
 *  Otherwise %NULL.
 **/
const gchar *
dbus_model_get_string (DbusModel      *self,
                       DbusModelIter  *iter,
                       guint           column)
{
  GValue      *value;

  g_return_val_if_fail (DBUS_IS_MODEL (self), NULL);
  g_return_val_if_fail (iter, NULL);

  value = _value_for_column (self, iter, column);

  g_return_val_if_fail (G_VALUE_HOLDS_STRING (value), NULL);

  return g_value_get_string (value);
}

/**
 * dbus_model_get_first_iter:
 * @self: a #DbusModel
 *
 * Retrieves a #DbusModelIter representing the first row in @self.
 *
 * Return value: (transfer none): A #DbusModelIter (owned by @self, do not
 *  free it)
 */
DbusModelIter *
dbus_model_get_first_iter (DbusModel     *self)
{
  g_return_val_if_fail (DBUS_IS_MODEL (self), NULL);

  return (DbusModelIter *) g_sequence_get_begin_iter (self->priv->sequence);
}

/**
 * dbus_model_get_last_iter:
 * @self: a #DbusModel
 *
 * Retrieves a #DbusModelIter representing the last row in @self.
 *
 * Return value: (transfer none): A #DbusModelIter (owned by @self, do not
 *  free it)
 **/
DbusModelIter *
dbus_model_get_last_iter (DbusModel *self)
{
  g_return_val_if_fail (DBUS_IS_MODEL (self), NULL);

  return (DbusModelIter *) g_sequence_get_end_iter (self->priv->sequence);
}

/**
 * dbus_model_get_iter_at_row:
 * @self: a #DbusModel
 * @row: position of the row to retrieve
 *
 * Retrieves a #DbusModelIter representing the row at the given index.
 *
 * Return value: (transfer none): A new #DbusModelIter, or %NULL if @row
 *   was out of bounds. The returned iter is owned by @self, so do not free it.
 **/
DbusModelIter *
dbus_model_get_iter_at_row (DbusModel *self, guint row)
{
  g_return_val_if_fail (DBUS_IS_MODEL (self), NULL);

  return (DbusModelIter *) g_sequence_get_iter_at_pos (self->priv->sequence,
                                                       row);
}

/**
 * dbus_model_next:
 * @self: a #DbusModel
 * @iter: a #DbusModelIter
 *
 * Returns a #DbusModelIter that points to the next position in the model.
 *
 * Return value: (transfer none): A #DbusModelIter, pointing to the next row in
 *   the model. The iter is owned by @self, do not free it.
 **/
DbusModelIter *
dbus_model_next (DbusModel     *self,
                 DbusModelIter *iter)
{
  g_return_val_if_fail (DBUS_IS_MODEL (self), NULL);
  g_return_val_if_fail (iter, NULL);

  return (DbusModelIter *) g_sequence_iter_next ((GSequenceIter *)iter);
}

/**
 * dbus_model_prev:
 * @self: a #DbusModel
 * @iter: a #DbusModelIter
 *
 * Returns a #DbusModelIter that points to the previous position in the model.
 *
 * Return value: (transfer none): A #DbusModelIter, pointing to the previous
 *   row in the model. The iter is owned by @self, do not free it.
 **/
DbusModelIter *
dbus_model_prev (DbusModel     *self,
                 DbusModelIter *iter)
{
  g_return_val_if_fail (DBUS_IS_MODEL (self), NULL);
  g_return_val_if_fail (iter, NULL);

  return (DbusModelIter *) g_sequence_iter_prev ((GSequenceIter *)iter);
}

/**
 * dbus_model_is_first:
 * @self: a #DbusModel
 * @iter: a #DbusModelIter
 *
 * Where @iter is at the start of @self.
 *
 * Return value: #TRUE if @iter is the first iter in the model
 */
gboolean
dbus_model_is_first (DbusModel     *self,
                     DbusModelIter *iter)
{
  g_return_val_if_fail (DBUS_IS_MODEL (self), FALSE);
  g_return_val_if_fail (iter, FALSE);

  return g_sequence_iter_is_begin ((GSequenceIter *)iter);
}

/**
 * dbus_model_is_last:
 * @self: a #DbusModel
 * @iter: a #DbusModelIter
 *
 * Whether @iter is at the end of @self.
 *
 * Return value: #TRUE if @iter is the last iter in the model
 */
gboolean
dbus_model_is_last (DbusModel     *self,
                    DbusModelIter *iter)
{
  g_return_val_if_fail (DBUS_IS_MODEL (self), FALSE);
  g_return_val_if_fail (iter, FALSE);

  return g_sequence_iter_is_end ((GSequenceIter *)iter);
}

/*
 * Dbus Methods
 */
static gboolean
_dbus_model_server_get_row_types (DbusModel     *self,
                                  gchar       ***types,
                                  GError       **error)
{
  DbusModelPrivate *priv;
  gint              i;
  gchar           **ret;

  g_return_val_if_fail (DBUS_IS_MODEL (self), FALSE);
  priv = self->priv;

  ret = g_new0 (gchar*, priv->n_columns+1);
  for (i = 0; i < priv->n_columns; i++)
    {
      ret[i] = _get_string_for_type (priv->column_types[i]);
    }

  *types = ret;

  return TRUE;
}

static gboolean
_dbus_model_server_get_rows (DbusModel   *self,
                             GPtrArray  **rows,
                             GError      *error)
{
  DbusModelPrivate *priv;
  GPtrArray        *ret;
  GSequenceIter    *iter;

  g_return_val_if_fail (DBUS_IS_MODEL (self), FALSE);
  priv = self->priv;

  ret = g_ptr_array_new ();

  iter = g_sequence_get_begin_iter (priv->sequence);
  while (!g_sequence_iter_is_end (iter))
    {
      GValueArray *array;
      gint         i;
      GPtrArray   *row;

      array = g_sequence_get (iter);
      if (!array)
        {
          iter = g_sequence_iter_next (iter);
          continue;
        }

      row = g_ptr_array_new ();
      for (i = 0; i < priv->n_columns+1; i++)
        {
          GValue *v = g_value_array_get_nth (array, i);
          GValue *value = g_new0 (GValue, 1);

          g_value_init (value, G_VALUE_TYPE (v));
          g_value_copy (v, value);

          g_ptr_array_add (row, value);
        }

      g_ptr_array_add (ret, row);

      iter = g_sequence_iter_next (iter);
    }

  *rows = ret;

  return TRUE;
}
