/*
 * Copyright (C) 2009 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */
/**
 * SECTION:dbus-peer
 * @short_description: Finds other objects with the same swarm-name on the bus.
 * @include: dbusmodel/dbusmodel.h
 *
 * #DbusPeer allows you to build objects that can 'find eachother' on D-Bus
 * without the need for an central registration service. Think of it like
 * peer-to-peer for your application.
 *
 * Your object should sub-class #DbusPeer and connect to it's signals to be
 * notified when a matching object comes on, or falls off, the bus.
 *
 * Peers find eachother through a well-known "swarm-name", which loosely
 * translates to a DBus name, such as: org.myapp.MyPeers. Choose a name that
 * would not normally be used outside of your program.
 *
 * For example:
 * <informalexample><programlisting>
 * {
 *   DbusPeer *peer;
 *
 *   peer = g_object_new (DBUS_TYPE_PEER,
 *                        "swarm-name", "org.myapp.MyPeers",
 *                        NULL);
 *
 *   g_signal_connect (peer, "peer-found",
 *                     G_CALLBACK (on_peer_found), NULL);
 *   g_signal_connect (peer, "peer-lost",
 *                     G_CALLBACK (on_peer_lost), NULL);
 *
 *   /<!-- -->* Publish this peer and start monitoring for other peers *<!-- -->
 *   dbus_peer_connect (peer);
 * }
 * </programlisting></informalexample>
 *
 * The callbacks for these functions would use the usual DBusGlib bindings to
 * access the remote objects at the name provided.
 *
 * For example:
 * <informalexample><programlisting>
 * void
 * on_peer_found (DbusPeer *peer, const gchar *name)
 * {
 *   DBusGProxy *proxy;
 *
 *   /<!-- -->* We have found a peer, so let's connect to it with a well-known
 *    <!-- -->* interface.
 *    *<!-- -->/
 *   proxy = dbus_g_proxy_new_for_name_owner (connection,
 *                                            name,
 *                                            "/org/myapp/MyPeerIface",
 *                                            "org.myapp.MyPeerIface",
 *                                            NULL);
 * }
 *
 * </programlisting></informalexample>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <dbus/dbus-glib-lowlevel.h>

#include "dbus-peer.h"

G_DEFINE_TYPE (DbusPeer, dbus_peer, G_TYPE_OBJECT)

#define DBUS_PEER_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE(obj, \
        DBUS_TYPE_PEER, DbusPeerPrivate))

#define _DbusPeerIter GSequenceIter

/**
 * DbusPeerPrivate:
 *
 * Ignore this structure.
 **/
struct _DbusPeerPrivate
{
  DBusGConnection *connection;
  DBusGProxy      *dbus_proxy;

  gchar *swarm_name;
  gchar *peer_name;

  gboolean connected;
};

/* Globals */
enum
{
  PROP_0,
  PROP_SWARM_NAME
};

enum
{
  CONNECTED,
  PEER_FOUND,
  PEER_LOST,

  LAST_SIGNAL
};

static guint32 _peer_signals[LAST_SIGNAL] = { 0 };

/* Forwards */

/* GObject Init */
static void
dbus_peer_finalize (GObject *object)
{
  DbusPeerPrivate *priv;

  priv = DBUS_PEER (object)->priv;

  if (priv->swarm_name)
    {
      g_free (priv->swarm_name);
      priv->swarm_name = NULL;
    }
  if (priv->peer_name)
    {
      g_free (priv->peer_name);
      priv->peer_name = NULL;
    }
  if (priv->dbus_proxy)
    {
      g_object_unref (priv->dbus_proxy);
      priv->dbus_proxy = NULL;
    }
  if (priv->connection)
    {
      dbus_g_connection_unref (priv->connection);
      priv->connection = NULL;
    }

  G_OBJECT_CLASS (dbus_peer_parent_class)->finalize (object);
}

static void
dbus_peer_set_property (GObject       *object,
                        guint          id,
                        const GValue  *value,
                        GParamSpec    *pspec)
{
  switch (id)
    {
    case PROP_SWARM_NAME:
      dbus_peer_set_swarm_name (DBUS_PEER (object), g_value_get_string (value));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, id, pspec);
      break;
    }
}

static void
dbus_peer_get_property (GObject     *object,
                        guint        id,
                        GValue      *value,
                        GParamSpec  *pspec)
{
  switch (id)
    {
    case PROP_SWARM_NAME:
      g_value_set_string (value, DBUS_PEER (object)->priv->swarm_name);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, id, pspec);
      break;
    }
}

static void
dbus_peer_class_init (DbusPeerClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);
  GParamSpec   *pspec;

  obj_class->finalize     = dbus_peer_finalize;
  obj_class->set_property = dbus_peer_set_property;
  obj_class->get_property = dbus_peer_get_property;

  /* Add Signals */
  /**
   * DbusPeer::connected:
   * @self: the #DbusPeer on which the signal is emitted
   * @peer_name: the name of the connection
   *
   * Connect to this signal to be notified when the peer connects to DBus.
   *   Sub-classes should connect to this signal so they register themselves
   *   on the bus at the same time.
   **/
  _peer_signals[CONNECTED] =
    g_signal_new ("connected",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (DbusPeerClass, connected),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1,
                  G_TYPE_STRING);

  /**
   * DbusPeer::peer-found:
   * @self: the #DbusPeer on which the signal is emitted
   * @name: the DBus name of the object found
   *
   * Connect to this signal to be notified of existing and new peers that are
   *   in your swarm.
   **/
  _peer_signals[PEER_FOUND] =
    g_signal_new ("peer-found",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (DbusPeerClass, peer_found),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1,
                  G_TYPE_STRING);

  /**
   * DbusPeer::peer-lost:
   * @self: the #DbusPeer on which the signal is emitted
   * @name: the DBus name of the object that disconnected
   *
   * Connect to this signal to be notified when peers disconnect from the swarm
   **/
  _peer_signals[PEER_LOST] =
    g_signal_new ("peer-lost",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (DbusPeerClass, peer_lost),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1,
                  G_TYPE_STRING);

  /* Add properties */
  /**
   * DbusPeer::swarm-name:
   *
   * The name of the swarm that this peer is connected to.
   **/
  pspec = g_param_spec_string ("swarm-name", "Swarm Name",
                               "Well-known name to find other peers with",
                               NULL,
                               G_PARAM_READWRITE | G_PARAM_CONSTRUCT
                               | G_PARAM_STATIC_STRINGS);
  g_object_class_install_property (obj_class, PROP_SWARM_NAME, pspec);

  /* Add private data */
  g_type_class_add_private (obj_class, sizeof (DbusPeerPrivate));
}

static void
dbus_peer_init (DbusPeer *peer)
{
  DbusPeerPrivate *priv;

  priv = peer->priv = DBUS_PEER_GET_PRIVATE (peer);
  priv->connected = FALSE;
}

/* Private Methods */
static void
on_dbus_name_owner_changed (DBusGProxy  *proxy,
                            const gchar *name,
                            const gchar *prev,
                            const gchar *new,
                            DbusPeer    *self)
{
  DbusPeerPrivate *priv;

  g_return_if_fail (DBUS_IS_PEER (self));
  priv = self->priv;

  /* Not interested in the standard ':1.*' names */
  if (name != NULL && name[0] == ':')
    return;

  if (prev != NULL && prev[0] == '\0')
    {
      if (g_str_has_prefix (name, priv->swarm_name)
          && g_strcmp0 (name, priv->peer_name) != 0)
        {
          g_signal_emit (self, _peer_signals[PEER_FOUND], 0, name);
        }
    }
  else
    {
      if (g_str_has_prefix (name, priv->swarm_name)
          && g_strcmp0 (name, priv->peer_name) != 0)
        {
          g_signal_emit (self, _peer_signals[PEER_LOST], 0, name);
        }
    }
}

static void
on_dbus_client_names_received (DBusGProxy  *proxy,
                               char       **names,
                               GError      *error,
                               void        *data)
{
  DbusPeer        *self = DBUS_PEER (data);
  DbusPeerPrivate *priv;
  guint            i;

  g_return_if_fail (DBUS_IS_PEER (self));
  priv = self->priv;

  if (error)
    {
      g_warning ("%s: Unable to get existing dbus client names: %s",
                 G_STRLOC,
                 error->message);
      return;
    }

  for (i = 0; names[i] != NULL; i++)
    {
      if (names[i] != NULL && names[i][0] == ':')
        continue;

      if (g_str_has_prefix (names[i], priv->swarm_name)
          && g_strcmp0 (names[i], priv->peer_name) != 0)
        {
          g_signal_emit (self, _peer_signals[PEER_FOUND], 0, names[i]);
        }
    }
}

static gboolean
dbus_peer_connect_real (DbusPeer *self)
{
  DbusPeerPrivate *priv;
  GError          *error = NULL;
  guint32          ret = 0;

  g_return_val_if_fail (DBUS_IS_PEER (self), FALSE);
  priv = self->priv;
  g_return_val_if_fail (priv->swarm_name != NULL, FALSE);
  g_return_val_if_fail (priv->connected == FALSE, FALSE);

  /* Get and name the connection if we don't already have it */
  if (!priv->connection)
    {
      DBusError       err;
      DBusConnection *conn;
      gchar          *unique_name;

      //priv->connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
      //dbus_g_connection_unref (priv->connection);

      dbus_error_init (&err);
      conn = dbus_bus_get_private (DBUS_BUS_SESSION, &err);

      if (conn == NULL)
        {
          g_error ("%s: Unable to get session bus: %s",
                   G_STRLOC,
                   err.message);
          dbus_error_free (&err);
          return FALSE;
        }

      dbus_connection_setup_with_g_main (conn,
                                         g_main_context_default ());

      priv->connection = dbus_connection_get_g_connection (conn);

      /* Create a unique name for the connection so other peers can find +
       * it is unique for this connection & swarm
       */
      conn = dbus_g_connection_get_connection (priv->connection);
      unique_name = g_strdelimit (g_strdup (dbus_bus_get_unique_name (conn)),
                                  ":.",
                                  '_');
      priv->peer_name = g_strdup_printf ("%s.%s",
                                         priv->swarm_name,
                                         unique_name);
      g_free (unique_name);
    }

  /* Grab the DBus proxy if we don't already have it */
  if (!G_IS_OBJECT (priv->dbus_proxy))
    {
      priv->dbus_proxy = dbus_g_proxy_new_for_name_owner (priv->connection,
                                                          DBUS_SERVICE_DBUS,
                                                          DBUS_PATH_DBUS,
                                                          DBUS_INTERFACE_DBUS,
                                                          &error);
      if (error)
        {
          g_error ("%s: Unable to get dbus proxy from session bus: %s",
                   G_STRLOC,
                   error->message);
          g_error_free (error);
          return FALSE;
        }
    }

  /* Name the connection */
  if (!org_freedesktop_DBus_request_name (priv->dbus_proxy,
                                          priv->peer_name,
                                          0,
                                          &ret,
                                          &error))
    {
      g_warning ("%s: Unable to request name (%s) from D-Bus: %s",
                 G_STRLOC,
                 priv->peer_name,
                 error->message);
      g_error_free (error);

      return FALSE;
    }

  if (ret != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER)
    {
      gint   i = 0;
      gchar *cur_name;

      ret = 0;

      for (i = 0; i < 1000; i++) /* 1000 is so random it hurts */
        {
          cur_name = g_strdup_printf ("%s_%d", priv->peer_name, i);
          if (org_freedesktop_DBus_request_name (priv->dbus_proxy,
                                                 cur_name,
                                                 0,
                                                 &ret,
                                                 &error))
            {
              if (ret == DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER)
                {
                  break;
                }
              else
                {
                  g_free (cur_name);
                  cur_name = NULL;
                }
            }
          else
            {
              g_free (cur_name);
              cur_name = NULL;
            }
        }

      if (cur_name)
        {
          g_free (priv->peer_name);
          priv->peer_name = g_strdup (cur_name);
        }
      else
        {
          g_warning ("%s: Unable to get unique name for peer: %s",
                     G_STRLOC,
                     error ? error->message : "Unknown");
          return FALSE;
        }
      ret = 0;
    }

  /* Connect to the listening signals on the DBus proxy */
  dbus_g_proxy_add_signal (priv->dbus_proxy, "NameOwnerChanged",
                           G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING,
                           G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (priv->dbus_proxy, "NameOwnerChanged",
                               G_CALLBACK (on_dbus_name_owner_changed), self,
                               NULL);

  /* Grab the list of activate clients on the bus to see if any peers have
   * already started
   */
  org_freedesktop_DBus_list_names_async (priv->dbus_proxy,
                                         on_dbus_client_names_received,
                                         self);

  priv->connected = TRUE;

  g_signal_emit (self, _peer_signals[CONNECTED], 0, priv->peer_name);

  return FALSE;
}


/* Public Methods */
/**
 * dbus_peer_connect:
 * @self: a #DbusPeer
 *
 * Will cause @self to connect to the swarm and begin monitoring peers.
 **/
void
dbus_peer_connect (DbusPeer *self)
{
  g_return_if_fail (DBUS_IS_PEER (self));

  dbus_peer_connect_real (self);
}

/**
 * dbus_peer_get_connection: (skip)
 * @self: a #DbusPeer
 *
 * Gets the #DBusGConnection that @self uses. Peer objects should export
 *   themselves on the same connection
 *
 * This function can only be used after #dbus_peer_connect() has been called.
 *
 * Return value: (transfer full): a #DBusGConnection. Unref with
 *   #dbus_g_connection_unref when done.
 **/
DBusGConnection *
dbus_peer_get_connection (DbusPeer *self)
{
  g_return_val_if_fail (DBUS_PEER (self), NULL);

  return self->priv->connection ? dbus_g_connection_ref (self->priv->connection)
                                : NULL;
}

/**
 * dbus_peer_get_peer_name:
 * @self: a #DbusPeer
 *
 * Gets the unique name for this peer in the swarm. Peer objects should be
 * published to this address.
 *
 * Return value: the unique name for this peer in the swarm
 **/
const gchar *
dbus_peer_get_peer_name (DbusPeer *self)
{
  g_return_val_if_fail (DBUS_PEER (self), NULL);

  return self->priv->peer_name;
}

/**
 * dbus_peer_set_swarm_name:
 * @self: a #DbusPeer
 * @swarm_name: a well-known name for peers of the swarm to find eachother
 *
 * Sets the swarm-name to @swarm_name. This should be a well-known name that
 * peers can use to find each other on the bus.
 *
 * The format should match standard DBus naming policy e.g. org.MyApp.Peers
 **/
void
dbus_peer_set_swarm_name (DbusPeer    *self,
                          const gchar *swarm_name)
{
  DbusPeerPrivate *priv;

  g_return_if_fail (DBUS_IS_PEER (self));
  g_return_if_fail (swarm_name != NULL);
  priv = self->priv;

  if (priv->swarm_name)
    {
      g_warning ("%s: Unable to set previously set swarm_name (%s) to (%s)",
                 G_STRLOC,
                 priv->swarm_name,
                 swarm_name);
    }
  else
    {
      priv->swarm_name = g_strdup (swarm_name);

      g_object_notify (G_OBJECT (self), "swarm-name");
    }
}

/**
 * dbus_peer_get_swarm_name:
 * @self: a #DbusPeer
 *
 * Get the currently set swarm-name, or %NULL
 *
 * Return value: the currently set swarm-name or %NULL
 **/
const gchar *
dbus_peer_get_swarm_name (DbusPeer *self)
{
  g_return_val_if_fail (DBUS_IS_PEER (self), NULL);

  return self->priv->swarm_name;
}
