/*
 * Copyright (C) 2009 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include <glib.h>
#include <glib-object.h>
#include <dbusmodel/dbusmodel.h>

typedef struct
{
  DbusModel *model;

} SignalsFixture;

static void rows_setup    (SignalsFixture *fix, gconstpointer data);
static void rows_teardown (SignalsFixture *fix, gconstpointer data);

static void test_signal_add (SignalsFixture *fix, gconstpointer data);
static void test_signal_remove (SignalsFixture *fix, gconstpointer data);
static void test_signal_changed (SignalsFixture *fix, gconstpointer data);

void
test_model_signals_create_suite (void)
{
#define DOMAIN "/Model/Signals"

  g_test_add (DOMAIN"/Add", SignalsFixture, 0,
              rows_setup, test_signal_add, rows_teardown);
  g_test_add (DOMAIN"/Remove", SignalsFixture, 0,
              rows_setup, test_signal_remove, rows_teardown);
  g_test_add (DOMAIN"/Changed", SignalsFixture, 0,
              rows_setup, test_signal_changed, rows_teardown);
}

static void
rows_setup (SignalsFixture *fix, gconstpointer data)
{
  fix->model = dbus_model_new ("com.canonical.DbusModel.Tests.Signals",
                               2,
                               G_TYPE_INT,
                               G_TYPE_STRING);

  g_assert (DBUS_IS_MODEL (fix->model));
}

static void
rows_teardown (SignalsFixture *fix, gconstpointer data)
{
  g_object_unref (fix->model);
}

static guint n_add_signals = 0;

static void
test_signal_add_callback (DbusModel *model, DbusModelIter *iter)
{
  n_add_signals++;
}

static void
test_signal_add (SignalsFixture *fix, gconstpointer data)
{
  gint i;

  g_signal_connect (fix->model, "row-added",
                    G_CALLBACK (test_signal_add_callback), NULL);

  n_add_signals = 0;
  for (i = 0; i < 10000; i++)
    {
      dbus_model_add (fix->model, 0, i, 1, "Test", -1);
    }
  g_assert_cmpint (n_add_signals, ==, 10000);
}

static guint n_remove_signals = 0;

static void
test_signal_remove_callback (DbusModel *model, DbusModelIter *iter)
{
  n_remove_signals++;
}

static void
test_signal_remove (SignalsFixture *fix, gconstpointer data)
{
  gint           i;
  DbusModelIter *iter;

  g_signal_connect (fix->model, "row-removed",
                    G_CALLBACK (test_signal_remove_callback), NULL);

  for (i = 0; i < 10000; i++)
    {
      dbus_model_add (fix->model, 0, i, 1, "Test", -1);
    }

  n_remove_signals = i = 0;
  iter = dbus_model_get_first_iter (fix->model);
  while (iter != NULL && !dbus_model_is_last (fix->model, iter))
    {
      dbus_model_remove (fix->model, iter);
      iter = dbus_model_get_first_iter (fix->model);
    }
  g_assert_cmpint (n_remove_signals, ==, 10000);
}

static guint n_changed_signals = 0;

static void
test_signal_changed_callback (DbusModel *model, DbusModelIter *iter)
{
  n_changed_signals++;
}

static void
test_signal_changed (SignalsFixture *fix, gconstpointer data)
{
  DbusModelIter *iter;
  gint           i;

  g_signal_connect (fix->model, "row-changed",
                    G_CALLBACK (test_signal_changed_callback), NULL);

  for (i = 0; i < 10000; i++)
    {
      dbus_model_add (fix->model, 0, i, 1, "Test", -1);
    }

  n_changed_signals = 0;
  iter = dbus_model_get_first_iter (fix->model);
  while (iter != NULL && !dbus_model_is_last (fix->model, iter))
    {
      gint   j = 0;
      gchar *k = "ing";

      dbus_model_set (fix->model, iter, 0, j, 1, k, -1);

      iter = dbus_model_next (fix->model, iter);
    }
  g_assert_cmpint (n_changed_signals, ==, 10000);

  n_changed_signals = 0;
  iter = dbus_model_get_first_iter (fix->model);
  while (iter != NULL && !dbus_model_is_last (fix->model, iter))
    {
      GValue value = { 0, };

      g_value_init (&value, G_TYPE_INT);
      g_value_set_int (&value, 10);

      dbus_model_set_value (fix->model, iter, 0, &value);

      g_value_unset (&value);

      iter = dbus_model_next (fix->model, iter);
    }
  g_assert_cmpint (n_changed_signals, ==, 10000);
}
