//  
// Copyright (C) 2009 Robert Dyer
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
// 

using System;
using System.Collections.Generic;
using System.Net;
using System.Threading;
using System.Web;
using System.Xml;

using Gdk;
using Gtk;
using Cairo;
using Mono.Unix;

using Docky.Services;
using Docky.Widgets;

// disable the warning message about System.Net.ServicePointManager.CertificatePolicy being obsolete
#pragma warning disable 618

namespace GMail
{
	public enum GMailState
	{
		Normal,
		Reloading,
		ManualReload,
		Error
	}
	
	// remove when ServicePointManager.ServerCertificateValidationCallback implemented in mono
	class CertHandler : System.Net.ICertificatePolicy
	{
		public bool CheckValidationResult(System.Net.ServicePoint srvPoint, System.Security.Cryptography.X509Certificates.X509Certificate certificate, System.Net.WebRequest request, int certificateProblem)
		{
                return true;
		}
	} 
	
	public struct UnreadMessage
	{
		public string Topic;
		public string From;
		public string FromName;
		public string FromEmail;
		public DateTime SendDate;
		public string Link;
	}
	
	/// <summary>
	/// </summary>
	public class GMailAtom
	{
		static ConfigDialog config;
		
		static event EventHandler ResetNeeded;
		
		public static void SettingsChanged ()
		{
			if (ResetNeeded != null)
				ResetNeeded (null, new EventArgs());
		}
		
		public event EventHandler GMailChecked;
		public event EventHandler GMailChecking;
		public event EventHandler<GMailErrorArgs> GMailFailed;
		
		public GMailState State { get; protected set; }

		public int UnreadCount { get; protected set; }
		public int NewCount { get; protected set; }
		
		public bool HasUnread {
			get {
				return UnreadCount > 0 && State != GMailState.Error;
			}
		}
		
		public GMailAtom (string label)
		{
			CurrentLabel = label;
			State = GMailState.ManualReload;
			DockServices.System.ConnectionStatusChanged += HandleNeedReset;
			// this is not implemented in mono yet
//			ServicePointManager.ServerCertificateValidationCallback +=
//				(sender, cert, chain, errors) => { return true; };
			ResetNeeded += HandleNeedReset;
		}
		
		public void Dispose ()
		{
			DockServices.System.ConnectionStatusChanged -= HandleNeedReset;
			ResetNeeded -= HandleNeedReset;
		}
		
		void HandleNeedReset (object o, EventArgs state)
		{
			ResetTimer ();
		}
		
		List<UnreadMessage> messages = new List<UnreadMessage> ();
		
		public IEnumerable<UnreadMessage> Messages {
			get {
				return messages as IEnumerable<UnreadMessage>;
			}
		}
		
		uint UpdateTimer { get; set; }
		
		public void ResetTimer (bool manual)
		{
			if (manual)
				State = GMailState.ManualReload;
			ResetTimer ();
		}
		
		public void ResetTimer ()
		{
			StopTimer ();
			
			if (!DockServices.System.NetworkConnected)
				return;
			
			CheckGMail ();
			
			UpdateTimer = GLib.Timeout.Add (GMailPreferences.RefreshRate * 60 * 1000, () => { CheckGMail (); return true; });
		}
		
		public void StopTimer ()
		{
			if (UpdateTimer != 0)
				GLib.Source.Remove (UpdateTimer);
			UpdateTimer = 0;
		}
		
		public string CurrentLabel { get; protected set; }
		
		public static bool ValidateCredentials (string username, string password)
		{
			try {
				String[] login = username.Split (new char[] {'@'});
				string domain = login.Length > 1 ? login [1] : "gmail.com";
				string url = "https://mail.google.com/a/" + domain;
				if (domain.Equals ("gmail.com") || domain.Equals ("googlemail.com"))
					url = "https://mail.google.com/mail";
				url += "/feed/atom/";
				
				Log<GMailAtom>.Info ("Fetching Atom feed: " + url);
				HttpWebRequest request = (HttpWebRequest)WebRequest.Create (url);
				request.UserAgent = @"Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.0.10) Gecko/2009042523 Ubuntu/9.04 (jaunty) Firefox/3.0.10";
				request.Credentials = new NetworkCredential (username, password);
				// FIXME: remove when ServicePointManager.ServerCertificateValidationCallback implemented in mono
				System.Net.ServicePointManager.CertificatePolicy = new CertHandler ();
				
				using (HttpWebResponse response = (HttpWebResponse)request.GetResponse ())
					try { } finally {
						response.Close ();
					}
			} catch (WebException e) {
				if (e.Message.IndexOf ("401") != -1) return false;
			} catch (Exception) { }
			
			return true;
		}
		
		void CheckGMail ()
		{
			if (string.IsNullOrEmpty (GMailPreferences.User) || string.IsNullOrEmpty (GMailPreferences.Password)) {
				OnGMailFailed (Catalog.GetString ("Username or Password not set"));
				return;
			}

			DockServices.System.RunOnThread (() => {
				try {
					Gtk.Application.Invoke (delegate { OnGMailChecking (); });

					String[] login = GMailPreferences.User.Split (new char[] {'@'});
					string domain = login.Length > 1 ? login [1] : "gmail.com";
					string url = "https://mail.google.com/a/" + domain;
					if (domain.Equals ("gmail.com") || domain.Equals ("googlemail.com"))
						url = "https://mail.google.com/mail";
					url += "/feed/atom/" + HttpUtility.UrlEncode (CurrentLabel);
					
					Log<GMailAtom>.Info ("Fetching Atom feed: " + url);
					HttpWebRequest request = (HttpWebRequest)WebRequest.Create (url);
					request.UserAgent = @"Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.0.10) Gecko/2009042523 Ubuntu/9.04 (jaunty) Firefox/3.0.10";
					request.Credentials = new NetworkCredential (GMailPreferences.User, GMailPreferences.Password);
					// remove when ServicePointManager.ServerCertificateValidationCallback implemented in mono
					System.Net.ServicePointManager.CertificatePolicy = new CertHandler ();
					
					XmlDocument xml = new XmlDocument ();
					XmlNamespaceManager nsmgr = new XmlNamespaceManager (xml.NameTable);
					nsmgr.AddNamespace ("atom", "http://purl.org/atom/ns#");
					
					using (HttpWebResponse response = (HttpWebResponse)request.GetResponse ())
						try {
							xml.Load (response.GetResponseStream ());
						} finally {
							response.Close ();
						}
					
					List<UnreadMessage> tmp = new List<UnreadMessage> ();
					XmlNodeList nodelist = xml.SelectNodes ("//atom:entry", nsmgr);
					
					for (int i = 0; i < nodelist.Count; i++)
					{
						XmlNode item = nodelist.Item (i);
						
						UnreadMessage message = new UnreadMessage ();
						message.Topic = HttpUtility.HtmlDecode (item.SelectSingleNode ("atom:title", nsmgr).InnerText);
						message.FromName = HttpUtility.HtmlDecode (item.SelectSingleNode ("atom:author/atom:name", nsmgr).InnerText);
						message.FromEmail = item.SelectSingleNode ("atom:author/atom:email", nsmgr).InnerText;
						message.From = message.FromName + " <" + message.FromEmail + ">";
						try {
							message.SendDate = DateTime.Parse (item.SelectSingleNode ("atom:modified", nsmgr).InnerText);
						} catch (Exception) {}
						message.Link = item.SelectSingleNode ("atom:link", nsmgr).Attributes ["href"].InnerText;
						if (message.Topic.Length == 0)
							message.Topic = Catalog.GetString ("(no subject)");
						
						tmp.Add (message);
					}
					
					UnreadCount = Convert.ToInt32 (xml.SelectSingleNode ("//atom:fullcount", nsmgr).InnerText);
					
					NewCount = 0;
					foreach (UnreadMessage message in tmp)
						if (message.SendDate > GMailPreferences.LastChecked)
							NewCount++;
					
					if (GMailPreferences.Notify) {
						if (NewCount > 5)
							Log.Notify (CurrentLabel, "gmail", "You have {0} new, unread messages", NewCount);
						else
							foreach (UnreadMessage message in tmp)
								if (message.SendDate > GMailPreferences.LastChecked)
									Log.Notify (message.Topic, "gmail", Catalog.GetString ("From: {0}"), message.From);
					}
					
					try {
						GMailPreferences.LastChecked = DateTime.Parse (xml.SelectSingleNode ("/atom:feed/atom:modified", nsmgr).InnerText);
					} catch (Exception) { GMailPreferences.LastChecked = DateTime.Now; }
					
					messages = tmp;
					Gtk.Application.Invoke (delegate { OnGMailChecked (); });
				} catch (NullReferenceException) {
					Gtk.Application.Invoke (delegate {
						OnGMailFailed (Catalog.GetString ("Feed Error"));
					});
				} catch (XmlException) {
					Gtk.Application.Invoke (delegate {
						OnGMailFailed (Catalog.GetString ("Feed Error"));
					});
				} catch (WebException e) {
					if (e.Message.IndexOf ("401") != -1)
						Gtk.Application.Invoke (delegate {
							OnGMailFailed (Catalog.GetString ("Invalid Username"));
						});
					else
						Gtk.Application.Invoke (delegate {
							OnGMailFailed (Catalog.GetString ("Network Error"));
						});
				} catch (Exception e) {
					Log<GMailAtom>.Error (e.ToString ());
					Gtk.Application.Invoke (delegate {
						OnGMailFailed (Catalog.GetString ("General Error"));
					});
				}
			});
		}
		
		void OnGMailChecked ()
		{
			State = GMailState.Normal;
			if (GMailChecked != null)
				GMailChecked (null, EventArgs.Empty);
		}
		
		void OnGMailChecking ()
		{
			if (State != GMailState.ManualReload)
				State = GMailState.Reloading;
			if (GMailChecking != null)
				GMailChecking (null, EventArgs.Empty);
		}
		
		void OnGMailFailed (string error)
		{
			State = GMailState.Error;
			if (GMailFailed != null)
				GMailFailed (null, new GMailErrorArgs (error));
		}
		
		public static void ShowConfig ()
		{
			if (config == null) {
				config = new ConfigDialog (Catalog.GetString ("GMail Configuration"),
								new Gtk.Widget [] { new GMailLoginConfig (), new GMailLabelConfig ()});
			}
			config.Show ();
		}
	}
}
