/*
 * Easy Network
 * Copyright (c) <2009>, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
#include "IPSettings.h"

#include <wx/intl.h>
#include <wx/string.h>
#include <wx/msgdlg.h>
#include "ENMApp.h"
#include "ENMUtil.h"

#include "ipv4.h"

const long IPSettings::ID_TCIP = wxNewId();
const long IPSettings::ID_TCSM = wxNewId();
const long IPSettings::ID_TCGA = wxNewId();
const long IPSettings::ID_TCDNS = wxNewId();
const long IPSettings::ID_BUTTONCANCEL = wxNewId();
const long IPSettings::ID_BUTTONOK = wxNewId();

BEGIN_EVENT_TABLE(IPSettings, wxDialog)
	EVT_BUTTON(ID_BUTTONCANCEL, IPSettings::OnButtonCancelClick)
	EVT_BUTTON(ID_BUTTONOK, IPSettings::OnButtonOKClick)
	EVT_INIT_DIALOG(IPSettings::OnInit)
END_EVENT_TABLE()

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
IPSettings::IPSettings(wxWindow* parent)
        : wxDialog(parent, wxID_ANY, _("IP Settings"))
{
    int width = 370, height = 188, margin = 10;
    wxSize stxSize(160, 20);
    wxSize tclSize(180, 24);

	SetClientSize(wxSize(width, height));
	Move(wxDefaultPosition);

    int x = margin;
    int y = margin;
	m_StIP = new wxStaticText(this, wxID_ANY, _("IP:"), wxPoint(x, y + 2), stxSize);
	m_TcIP = new wxTextCtrl(this, ID_TCIP, wxEmptyString, wxPoint(x + stxSize.GetWidth() + margin, y), tclSize);

    y += tclSize.GetHeight() + margin;
	m_StSm = new wxStaticText(this, wxID_ANY, _("Subnet Mask:"), wxPoint(x, y + 2), stxSize);
	m_TcSm = new wxTextCtrl(this, ID_TCSM, wxEmptyString, wxPoint(x + stxSize.GetWidth() + margin, y), tclSize);

    y += tclSize.GetHeight() + margin;
	m_StGa = new wxStaticText(this, wxID_ANY, _("Gateway Address:"), wxPoint(x, y), stxSize);
	m_TcGa = new wxTextCtrl(this, ID_TCGA, wxEmptyString, wxPoint(x + stxSize.GetWidth() + margin, y), tclSize);

    y += tclSize.GetHeight() + margin;
	m_StDns = new wxStaticText(this, wxID_ANY, _("DNS:"), wxPoint(x, y), stxSize);
	m_TcDns = new wxTextCtrl(this, ID_TCDNS, wxEmptyString, wxPoint(x + stxSize.GetWidth() + margin, y), tclSize);

	wxSize btnSize(80, 30);
//	x = width - (btnSize.GetWidth() + margin) * 2;
	x = stxSize.GetWidth() + margin * 2;
	y = height - btnSize.GetHeight() - margin;
#if defined(__WXMSW__)
	m_BtnOK = new wxButton(this, ID_BUTTONOK, _("OK"), wxPoint(x, y), btnSize);
	x += btnSize.GetWidth() + margin;
	m_BtnCancel = new wxButton(this, ID_BUTTONCANCEL, _("Cancel"), wxPoint(x, y), btnSize);
#endif
#if defined(__UNIX__)
	m_BtnCancel = new wxButton(this, ID_BUTTONCANCEL, _("Cancel"), wxPoint(x, y), btnSize);
//	x += btnSize.GetWidth() + margin;
    x = width - (btnSize.GetWidth() + margin);
	m_BtnOK = new wxButton(this, ID_BUTTONOK, _("OK"), wxPoint(x, y), btnSize);
#endif
    m_BtnOK->SetDefault();

    m_TcIP->SetMaxLength(15);
    m_TcSm->SetMaxLength(15);
    m_TcGa->SetMaxLength(15);
    m_TcDns->SetMaxLength(15);
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
IPSettings::~IPSettings()
{
}

IPDNSSet_t IPSettings::GetIPSetting() const
{
    return m_ipSet;
}

void IPSettings::SetIPSetting(const IPDNSSet_t& set)
{
    m_ipSet = set;
}


void IPSettings::OnInit(wxInitDialogEvent& event)
{
    Center();
    m_TcIP->SetValue(m_ipSet.IPvalue);
    m_TcGa->SetValue(m_ipSet.NetGateway);
    m_TcSm->SetValue(m_ipSet.SubnetAddress);
    m_TcDns->SetValue(m_ipSet.DNSvalue);
    m_TcIP->SetFocus();
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
void IPSettings::OnButtonCancelClick(wxCommandEvent& event)
{
    Close();
}

/**************************************************************************************************
* Function Name :                                                                                 *
* Description  :                                                                                  *
* Date   :                                                                                        *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
void IPSettings::OnButtonOKClick(wxCommandEvent& event)
{
    if (m_TcIP->IsEmpty())
    {
        wxMessageDialog msg(this, _("IP address cannot be empty!"), wxEmptyString, wxOK | wxICON_INFORMATION);
        msg.ShowModal();
        m_TcIP->SetFocus();
        return;
    }
    if (m_TcSm->IsEmpty())
    {
        wxMessageDialog msg(this, _("Subnet mask cannot be empty!"), wxEmptyString, wxOK | wxICON_INFORMATION);
        msg.ShowModal();
        m_TcSm->SetFocus();
        return;
    }
    if (m_TcGa->IsEmpty())
    {
        wxMessageDialog msg(this, _("Gateway address cannot be empty!"), wxEmptyString, wxOK | wxICON_INFORMATION);
        msg.ShowModal();
        m_TcGa->SetFocus();
        return;
    }
    if (m_TcDns->IsEmpty())
    {
        wxMessageDialog msg(this, _("DNS cannot be empty!"), wxEmptyString, wxOK | wxICON_INFORMATION);
        msg.ShowModal();
        m_TcDns->SetFocus();
        return;
    }
    if ( !m_TcIP->GetValue().IsAscii() )
    {
        wxMessageDialog msg(this, _("Invalid IP!"), wxEmptyString, wxOK | wxICON_INFORMATION);
        msg.ShowModal();
        m_TcIP->SetFocus();
        return;
    }
    if ( !m_TcSm->GetValue().IsAscii() )
    {
        wxMessageDialog msg(this, _("Invalid subnet mask!"), wxEmptyString, wxOK | wxICON_INFORMATION);
        msg.ShowModal();
        m_TcSm->SetFocus();
        return;
    }
    if ( !m_TcGa->GetValue().IsAscii() )
    {
        wxMessageDialog msg(this, _("Invalid gateway address!"), wxEmptyString, wxOK | wxICON_INFORMATION);
        msg.ShowModal();
        m_TcGa->SetFocus();
        return;
    }
    if ( !m_TcDns->GetValue().IsAscii() )
    {
        wxMessageDialog msg(this, _("Invalid DNS!"), wxEmptyString, wxOK | wxICON_INFORMATION);
        msg.ShowModal();
        m_TcDns->SetFocus();
        return;
    }
    char pszIp[32] = { 0 };
    char pszMask[32] = { 0 };
    char pszGateway[32] = { 0 };
    strcpy(pszIp, m_TcIP->GetValue().mb_str());
    strcpy(pszMask, m_TcSm->GetValue().mb_str());
    strcpy(pszGateway, m_TcGa->GetValue().mb_str());
    wxString message = wxEmptyString;
    wxControl* pToFocus = NULL;
    switch ( IPv4AddressUtil::ValidateAdapterAddress(pszIp, pszMask, pszGateway) )
    {
    case INET_INVALID_ADDRESS:
        message = _("Invalid IP!");
        pToFocus = m_TcIP;
        break;
    case INET_BROADCAST_HOST_ADDRESS:
        message = _("IP address cannot be broadcast address!");
        pToFocus = m_TcIP;
        break;
    case INET_BROADCAST_GATEWAY_ADDRESS:
        message = _("Gateway cannot be broadcast address!");
        pToFocus = m_TcGa;
        break;
    case INET_MULTICAST_ADDRESS:
        message = _("IP address cannot be multicast address!");
        pToFocus = m_TcIP;
        break;
    case INET_RESERVED_ADDRESS:
        message = _("IP address cannot be reserved address!");
        pToFocus = m_TcIP;
        break;
    case INET_NETWORK_HOST_ADDRESS:
        message = _("IP address cannot be the same as the network address!");
        pToFocus = m_TcIP;
        break;
    case INET_NETWORK_GATEWAY_ADDRESS:
        message = _("Gateway address cannot be the same as the network address!");
        pToFocus = m_TcGa;
        break;
    case INET_INVALID_MASK:
        message = _("Invalid subnet mask!");
        pToFocus = m_TcSm;
        break;
    case INET_INVALID_GATEWAY:
        message = _("Invalid gateway address!");
        pToFocus = m_TcGa;
        break;
    case INET_ADDRESS_NOT_MATCH:
        message = _("IP address does not match the gateway!");
        pToFocus = m_TcIP;
        break;
    case INET_OK:
        switch ( IPv4AddressUtil::ValidateHostAddress(m_TcDns->GetValue().mbc_str()) )
        {
        case INET_MULTICAST_ADDRESS:
            message = _("DNS cannot be multicast address!");
            pToFocus = m_TcDns;
            break;
        case INET_RESERVED_ADDRESS:
            message = _("DNS cannot be reserved address!");
            pToFocus = m_TcDns;
            break;
        case INET_OK:
            break;
        default:
            message = _("Invalid DNS!");
            pToFocus = m_TcDns;
            break;
        };
        break;
    default:
        message = _("Invalid IP!");
        pToFocus = m_TcIP;
        break;
    };

    if ( message != wxEmptyString )
    {
        wxMessageDialog msg(this, message, wxEmptyString, wxOK | wxICON_INFORMATION);
        msg.ShowModal();
        pToFocus->SetFocus();
    }
    else
    {
        m_ipSet.IPvalue = m_TcIP->GetValue();
        m_ipSet.NetGateway = m_TcGa->GetValue();
        m_ipSet.SubnetAddress = m_TcSm->GetValue();
        m_ipSet.DNSvalue = m_TcDns->GetValue();
        EndModal(wxID_OK);
    }
}
