# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2007-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampann <benjamin@fluendo.com>


from elisa.plugins.search.searcher import Searcher
from elisa.core.resource_manager import ResourceProviderNotFound
from elisa.core import common

from twisted.trial.unittest import TestCase

# Dummy classes
class Dummy(object): pass

class DummyResourceManager(object):

    def __init__(self):
        self.provider_by_path = {}
        self.last_path = None

    def get_resource_provider_by_path(self, path):
        self.last_path = path
        try:
            return self.provider_by_path[path]
        except KeyError:
            raise ResourceProviderNotFound(path)

class SearcherMixin(object):
    """
    A Mixin class that your test can inherit from to set up the generic searcher
    as self.seacher. Just set the searcher_class variable to your searcher
    class.
    """
    searcher_class = None

    def setUp(self):
        def created(searcher):
            self.searcher = searcher

        dfr = self.searcher_class.create()
        dfr.addCallback(created)
        return dfr

    def tearDown(self):
        return self.searcher.clean()

class TestGenericSearcher(SearcherMixin, TestCase):
    searcher_class = Searcher

    def test_search(self):
        """
        Ensures the API is right.
        """
        dfr = self.searcher.search(None, None)
        self.assertFailure(dfr, NotImplementedError)
        return dfr

class TestProviderSetup(SearcherMixin, TestCase):
    searcher_class = Searcher

    def setUp(self):
        dfr = super(TestProviderSetup, self).setUp()
        self.application = common.application
        
        common.application = Dummy()
        common.application.resource_manager = DummyResourceManager()
        self.manager = common.application.resource_manager
        return dfr

    def tearDown(self):
        common.application = self.application

    def test_right_provider_requested(self):

        self.assertEquals(self.searcher.provider, None)
        self.assertEquals(self.searcher.active, True)

        self.searcher.provider_path = "yepyep"
        self.manager.provider_by_path['yepyep'] = "TRUE!"

        self.assertEquals(self.searcher._query_provider(), "TRUE!")

        self.assertEquals(self.searcher.provider, "TRUE!")
        self.assertEquals(self.searcher.active, True)

    def test_provider_request_fails(self):

        self.assertEquals(self.searcher.provider, None)
        self.assertEquals(self.searcher.active, True)

        self.searcher.provider_path = "false"
        self.manager.provider_by_path = {}

        self.assertEquals(self.searcher._query_provider(), None)

        self.assertEquals(self.searcher.provider, None)
        self.assertEquals(self.searcher.active, False)


