# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>

from elisa.core.metadata_manager import IncompleteMetadataResponse
from elisa.core.utils.i18n import install_translation
_ = install_translation('database')

from twisted.internet import defer, task

from elisa.plugins.poblesec.base.list_switcher import ListSwitcherController
from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.link import Link
from elisa.plugins.base.models.image import ImageModel
from elisa.plugins.base.models import image as image_model

from elisa.plugins.poblesec.base.preview_list import \
    MenuItemPreviewListController, DoubleLineMenuItemPreviewListController
from elisa.plugins.poblesec.base.coverflow import \
    ImageWithReflectionCoverflowController
from elisa.plugins.poblesec.base.grid import GridItemGridController

from elisa.plugins.poblesec.base.list import  GenericListViewMode

from elisa.plugins.database.models import Image, PhotoAlbum, File, \
                                                             PICTURES_SECTION
from elisa.core import common
from elisa.core.media_uri import MediaUri

from elisa.plugins.poblesec.actions import Action
from elisa.plugins.poblesec.actions import ViewSlideshowAction
from elisa.plugins.database.actions import PhotoAddToFavoritesAction
from elisa.plugins.database.actions import PhotoRemoveAction
from elisa.plugins.database.actions import ViewPhotoAlbumSlideshowAction, \
        ViewPhotoDateSlideshowAction

from elisa.plugins.favorites.models import FavoritesItem

import os, datetime
import os.path

from gtk import gdk


def photo_lib_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/database/photo_library'
    link.label = _('Photo Library')
    link.icon = 'elisa.plugins.poblesec.photo_library'
    controller.model.append(link)

    return defer.succeed(None)

def photo_lib_albums_decorator(controller):
    albums = Link()
    albums.controller_path = '/poblesec/database/photo/albums'
    albums.label = _('Albums')
    albums.icon = 'elisa.plugins.poblesec.photo_album'
    controller.model.append(albums)
    return defer.succeed(None)


def photo_lib_images_decorator(controller):
    tracks = Link()
    tracks.controller_path = '/poblesec/database/photo/list'
    tracks.label = _('Photos')
    tracks.icon = 'elisa.plugins.poblesec.photo'
    controller.model.append(tracks)
    return defer.succeed(None)


def photo_lib_month_decorator(controller):
    times = Link()
    times.controller_path = '/poblesec/database/photo/by_date'
    times.label = _('Browse by Date')
    times.icon = 'elisa.plugins.poblesec.by_decade'
    controller.model.append(times)
    return defer.succeed(None)

class ThumbnailMixin(object):

    def __init__(self):
        # FIXME: we need the frontend to get a reference to the gst_metadata
        # instance. This a cheap - UGLY - way to get the frontend without
        # changing a lot of client code. It is really ugly as we assume there
        # is only one frontend, which might not be the case in the future...
        frontend = common.application.interface_controller.frontends.values()[0]
        # Retrieve and store a reference to gst_metadata
        controllers = frontend.retrieve_controllers('/poblesec')
        try:
            self.gst_metadata = controllers[0].gst_metadata
        except AttributeError:
            msg = 'GstMetadata missing: thumbnails will not be generated.'
            self.warning(msg)
            self.gst_metadata = None

    def _updated_thumbnail(self, thumbnail, item):
        # apply the image rotation if there's any
        # FIXME: this is ugly, we get a thumbnail from gst_metadata and
        #        a new one is generated...
        if item.orientation:
            path, ext = os.path.splitext(thumbnail)
            # FIXME: not compliant with XDG thumbnail spec:
            rotated_thumbnail = "%s-%s%s" % (path, item.orientation, ext)
            if not os.path.exists(rotated_thumbnail):

                pixbuf = gdk.pixbuf_new_from_file(thumbnail)
                pixbuf_rotations = {image_model.ROTATED_0: gdk.PIXBUF_ROTATE_NONE,
                                    image_model.ROTATED_90_CCW: gdk.PIXBUF_ROTATE_COUNTERCLOCKWISE,
                                    image_model.ROTATED_180: gdk.PIXBUF_ROTATE_UPSIDEDOWN,
                                    image_model.ROTATED_90_CW:  gdk.PIXBUF_ROTATE_CLOCKWISE}
                rotated = pixbuf.rotate_simple(pixbuf_rotations[item.orientation])
                rotated.save(rotated_thumbnail, ext[1:])

            thumbnail = rotated_thumbnail

        item.thumbnail = thumbnail
        return thumbnail

    def _request_thumbnail(self, path):
        if self.gst_metadata is None:
            msg = 'No gst_metadata.'
            return defer.fail(IncompleteMetadataResponse(msg))

        def got_metadata(metadata):
            return metadata['thumbnail']

        metadata = {'uri': MediaUri("file://%s" % path), 'thumbnail': None,}

        dfr = self.gst_metadata.get_metadata(metadata)
        dfr.addCallback(got_metadata)
        return dfr

class PhotoViewMode(GenericListViewMode, ThumbnailMixin):

    def get_label(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.title)
        return defer.succeed(os.path.basename(item.file_path))

    def get_default_image(self, item):
        if isinstance(item, Action):
            return item.icon
        resource = 'elisa.plugins.poblesec.photo'
        return resource

    def get_image(self, item, theme):
        if isinstance(item, Action):
            return None

        if hasattr(item, 'thumbnail') and item.thumbnail != None:
            return defer.succeed(item.thumbnail)

        dfr = self._request_thumbnail(item.file_path)
        dfr.addCallback(self._updated_thumbnail, item)
        return dfr

    def get_preview_image(self, item, theme):
        if isinstance(item, Action):
            return None

        if hasattr(item, 'thumbnail'):
            return item.thumbnail


class GenericPhotosController(HierarchyController):

    empty_label = _('There are no photos in this section')
    empty_icon = 'elisa.plugins.poblesec.photo'

    def initialize(self, album=None, photo=None, photos=None):
        self.album = album
        self.photo = photo
        self.photos = photos

        dfr = super(GenericPhotosController, self).initialize()
        self.actions.extend(self.make_actions())
        return dfr

    def is_empty(self):
        return super(GenericPhotosController, self).is_empty() and self.photo is None

    def make_actions(self):
        actions = []

        if self.album:
            action = ViewSlideshowAction(self)
            action.title = _('Play Slideshow')
            action.subtitle = _('Slideshow of All Photos')
            actions.append(action)

            action = PhotoAddToFavoritesAction(self)
            action.untoggled_title = _('Add Album To Favorites')
            action.untoggled_subtitle = _("Add '%s' To Your Favorites") % self.album.name
            action.toggled_title = _('Remove Album From Favorites')
            action.toggled_subtitle = _("Remove '%s' From Your Favorites") % self.album.name
            actions.append(action)

            dfr = common.application.store.find(FavoritesItem,
                                                FavoritesItem.foreign_id == self.album.name,
                                                FavoritesItem.foreign_class == u'PhotoAlbum')
            dfr.addCallback(lambda rs: rs.all())
            dfr.addCallback(action.setup)
        elif self.photo:

            action = ViewSlideshowAction(self)
            action.title = _('View Photo')
            action.subtitle = _("View '%s'") % os.path.basename(self.photo.file_path)
            actions.append(action)

            action = PhotoAddToFavoritesAction(self)
            action.untoggled_title = _('Add Photo To Favorites')
            action.untoggled_subtitle = _("Add '%s' To Your Favorites") % os.path.basename(self.photo.file_path)
            action.toggled_title = _('Remove From Favorites')
            action.toggled_subtitle = _("Remove '%s' From Your Favorites") % os.path.basename(self.photo.file_path)
            actions.append(action)

            dfr = common.application.store.find(FavoritesItem,
                                                FavoritesItem.foreign_id == self.photo.file_path,
                                                FavoritesItem.foreign_class == u'Image')
            dfr.addCallback(lambda rs: rs.all())
            dfr.addCallback(action.setup)

            # FIXME: implement this properly instead of hijacking the app config
            app_config = common.application.config
            if int(app_config.get_option('enable_remove',
                                         'poblesec.main:PoblesecController',
                                         default='0')):
                remove_action = PhotoRemoveAction(self)
                remove_action.title = _('Remove Photo')
                remove_action.subtitle = _("Remove '%s'") % os.path.basename(self.photo.file_path)
                actions.append(remove_action)

        else:
            action = ViewSlideshowAction(self)
            action.title = _('Play Slideshow')
            action.sub_title = _('Slideshow of All Photos')
            actions.append(action)

        return actions

    def node_clicked(self, widget, model):
        if isinstance(model, Action):
            model.run()
        else:
            browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
            photos = filter(lambda p: not isinstance(p, Action), self.model)
            title = os.path.basename(model.file_path)
            dfr = browser.history.append_controller('/poblesec/database/photo/photo_actions',
                                                    title,
                                                    photo=model,
                                                    photos=photos)

class PhotosController(GenericPhotosController):

    def initialize(self, album=None, photo=None, photos=None):
        dfr = super(PhotosController, self).initialize(album=album, photo=photo, photos=photos)

        def get_images(result):
            store = common.application.store

            if self.album:
                dfr = store.find(Image, Image.album_name == album.name,
                                 Image.file_path == File.path,
                                 File.hidden == False)
            else:
                dfr = store.find(Image, Image.file_path == File.path,
                                 File.hidden == False,
                                 Image.section == PICTURES_SECTION)

            return dfr

        def got_images(images):
            self.model.extend(images)
            return self

        def sort_images(result_set):
            result_set.order_by(Image.shot_time, Image.file_path)
            return result_set.all()

        if photo is None:
            dfr.addCallback(get_images)
            dfr.addCallback(sort_images)
            dfr.addCallback(got_images)

        return dfr

class PhotosMonthController(GenericPhotosController):

    def initialize(self, month=None, year=None):
        self.date = datetime.datetime(year, month, 1)

        dfr = super(PhotosMonthController, self).initialize()

        def get_images(result):
            store = common.application.store
            try:
                last = datetime.datetime(year, month+1, 1)
            except ValueError:
                # we reached the month 12 + 1, so we have to skip to
                # january next year
                last = datetime.datetime(year+1, 1, 1)

            dfr = store.find(Image,
                             Image.section == PICTURES_SECTION,
                             Image.shot_time >= self.date,
                             Image.shot_time < last,
                             Image.file_path == File.path,
                             File.hidden == False)

            return dfr

        def got_images(images):
            self.model.extend(images)
            return self

        def sort_images(result_set):
            result_set.order_by(Image.shot_time, Image.file_path)
            return result_set.all()

        dfr.addCallback(get_images)
        dfr.addCallback(sort_images)
        dfr.addCallback(got_images)

        return dfr

    def make_actions(self):
        action = ViewSlideshowAction(self)
        action.title = _('Play Slideshow')
        subtitle = self.date.strftime(_("Slideshow of Photos from %B, %Y"))
        action.subtitle = subtitle
        return [action]

class PhotoAlbumViewMode(GenericListViewMode, ThumbnailMixin):

    def get_label(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.title)
        return defer.succeed(item.name)

    def get_default_image(self, item):
        if isinstance(item, Action):
            return item.icon
        resource = 'elisa.plugins.poblesec.photo_album'
        return resource

    def _retrieve_file_path(self, album):
        def got_one(one):
            return one.file_path

        dfr = album.photos.any()
        dfr.addCallback(got_one)
        return dfr

    def _got_thumbnail(self, thumbnail, item):
        item.thumbnail = thumbnail
        return None

    def get_image(self, item, theme):
        if isinstance(item, Action):
            return None

        if hasattr(item, 'thumbnail') and item.thumbnail != None:
            return defer.succeed(item.thumbnail)

        dfr = self._retrieve_file_path(item)
        dfr.addCallback(self._request_thumbnail)
        dfr.addCallback(self._got_thumbnail, item)
        return dfr

    def get_preview_image(self, item, theme):
        if isinstance(item, Action):
            return None
        if hasattr(item, 'thumbnail'):
            return item.thumbnail


class GenericPhotoAlbumsController(HierarchyController):

    empty_label = _('There are no albums in this section')
    empty_icon = 'elisa.plugins.poblesec.photo_album'

    def initialize(self):
        dfr = super(GenericPhotoAlbumsController, self).initialize()
        self.actions.extend(self.make_actions())
        return dfr

    def make_actions(self):
        actions = []

        action = ViewPhotoAlbumSlideshowAction(self)
        action.title = _('Play Slideshow')
        action.sub_title = _('Slideshow of All Photos')
        actions.append(action)

        return actions

     # copy-paste-code again :(
    def node_clicked(self, widget, model):
        if isinstance(model, Action):
            model.run()
        else:
            controllers = self.frontend.retrieve_controllers('/poblesec/browser')
            history = controllers[0].history
            dfr = history.append_controller('/poblesec/database/photo/list',
                                                    model.name, album=model)

class PhotoTimesViewMode(GenericListViewMode):

    def get_label(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.title)

        return defer.succeed(item.strftime(_('%B, %Y')))

    def get_default_image(self, item):
        if isinstance(item, Action):
            return item.icon
        resource = 'elisa.plugins.poblesec.photo_album'
        return resource

    def get_image(self, item, theme):
        return None

    def get_preview_image(self, item, theme):
        if isinstance(item, Action):
            return None
        return None

class PhotosTimesController(HierarchyController):

    query = "SELECT DISTINCT STRFTIME('%m %Y', shot_time)" \
            " from images, files where shot_time not NULL" \
            + (" and images.section = %d" % PICTURES_SECTION) + \
            " and files.path = images.file_path and files.hidden = 0" \
            " order by shot_time;"

    empty_label = _('There are no photos in this section')
    empty_icon = 'elisa.plugins.poblesec.by_decade'

    def initialize(self):
        dfr = super(PhotosTimesController, self).initialize()

        def get_times(result):
            store = common.application.store
            dfr = store.execute(self.query)
            return dfr

        def get_all(result):
            return result.get_all()

        def iterate(rows):
            for row in rows:
                month, year = row[0].split()
                date = datetime.datetime(int(year), int(month), 1)
                self.model.append(date)
                yield date

        def got_times(times):
            return task.coiterate(iterate(times))

        def done(res):
            return self

        dfr.addCallback(get_times)
        dfr.addCallback(get_all)
        dfr.addCallback(got_times)
        dfr.addCallback(done)
        return dfr

    def make_actions(self):
        actions = []

        action = ViewPhotoDateSlideshowAction(self)
        action.title = _('Play Slideshow')
        action.sub_title = _('Slideshow of All Photos')
        actions.append(action)

        return actions

     # copy-paste-code again :(
    def node_clicked(self, widget, model):
        if isinstance(model, Action):
            model.run()
        else:
            controllers = self.frontend.retrieve_controllers('/poblesec/browser')
            history = controllers[0].history
            dfr = history.append_controller('/poblesec/database/photo/by_month',
                    model.strftime(_('%B, %Y')), year=model.year, \
                    month=model.month)



class PhotoAlbumsController(GenericPhotoAlbumsController):

    def initialize(self):
        dfr = super(PhotoAlbumsController, self).initialize()

        def get_albums(self):
            store = common.application.store
            dfr = store.find(PhotoAlbum)
            return dfr

        def got_albums(albums):
            self.model.extend(albums)
            return self

        def sort_albums(result_set):
            result_set.order_by(PhotoAlbum.name)
            return result_set.all()

        dfr.addCallback(get_albums)
        dfr.addCallback(sort_albums)
        dfr.addCallback(got_albums)

        return dfr


# Photo
class PhotoVerticalWithPreview(PhotosController, MenuItemPreviewListController):
    view_mode = PhotoViewMode

class PhotoCoverflow(PhotosController, ImageWithReflectionCoverflowController):
    view_mode = PhotoViewMode

class PhotoGrid(PhotosController, GridItemGridController):
    view_mode = PhotoViewMode

class PhotoListSwitcherController(ListSwitcherController):
    modes = [PhotoVerticalWithPreview,
             PhotoCoverflow,
             PhotoGrid]
    default_mode = PhotoGrid

# Photo by month
class PhotoMonthVerticalWithPreview(PhotosMonthController, MenuItemPreviewListController):
    view_mode = PhotoViewMode

class PhotoMonthCoverflow(PhotosMonthController, ImageWithReflectionCoverflowController):
    view_mode = PhotoViewMode

class PhotoMonthGrid(PhotosMonthController, GridItemGridController):
    view_mode = PhotoViewMode

class PhotoMonthListSwitcherController(ListSwitcherController):
    modes = [PhotoMonthVerticalWithPreview,
             PhotoMonthCoverflow,
             PhotoMonthGrid]
    default_mode = PhotoMonthGrid

# Photo Times Overview
class PhotoTimesVerticalWithPreview(PhotosTimesController, MenuItemPreviewListController):
    view_mode = PhotoTimesViewMode

class PhotoTimesCoverflow(PhotosTimesController, ImageWithReflectionCoverflowController):
    view_mode = PhotoTimesViewMode

class PhotoTimesGrid(PhotosTimesController, GridItemGridController):
    view_mode = PhotoTimesViewMode

class PhotoTimesListSwitcherController(ListSwitcherController):
    modes = [PhotoTimesVerticalWithPreview,
             PhotoTimesCoverflow,
             PhotoTimesGrid]
    default_mode = PhotoTimesVerticalWithPreview

# Albums
class AlbumsVerticalWithPreview(PhotoAlbumsController, MenuItemPreviewListController):
    view_mode = PhotoAlbumViewMode

class AlbumsCoverflow(PhotoAlbumsController, ImageWithReflectionCoverflowController):
    view_mode = PhotoAlbumViewMode

class AlbumsGrid(PhotoAlbumsController, GridItemGridController):
    view_mode = PhotoAlbumViewMode

class AlbumsListSwitcherController(ListSwitcherController):

    modes = [AlbumsVerticalWithPreview,
             AlbumsCoverflow,
             AlbumsGrid]
    default_mode = AlbumsVerticalWithPreview
