# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Philippe Normand <philippe@fluendo.com>

from elisa.plugins.poblesec.widgets.player.control_ribbon import Control
from elisa.plugins.pigment.widgets.const import *
from elisa.plugins.base.models import image

from elisa.core.utils import defer
from elisa.core import common
from elisa.core.utils.i18n import install_translation

try:
    from elisa.plugins.database import models as db_models
except ImportError:
    # database missing
    db_models = None
else:
    DBImage = db_models.Image

_ = install_translation('poblesec')

class RotationControl(Control):
    """
    Control for Image rotation. This requires the database plugin. When it's
    activated, it:

     - updates the orientation field of the Image in db

     - rotates clock-wise the various drawables showing the image on the UI

    When the 90° CCW angle is reached, the next rotation is the
    'original' orientation of the image.

    Its L{slave} passed to the constructor must be of type
    L{elisa.plugins.poblesec.player_slideshow.PlayerController}.
    """

    caption = _("Rotate")
    glyphs = \
            {STATE_NORMAL: 'elisa.plugins.poblesec.player.glyphs.rotate_normal',
             STATE_SELECTED: 'elisa.plugins.poblesec.player.glyphs.rotate_selected',
             STATE_ACTIVE: 'elisa.plugins.poblesec.player.glyphs.rotate_active',
            }

    def activate(self):

        # shouldn't be necessary because the Control is not added in the slideshow
        # control ribbon if the database plugin is missing. But let's remain
        # careful.
        if not db_models:
            return

        player = self.slave.player
        uri = player.playlist[player.current_index].references[-1]

        # clock-wise rotations
        rotations = [image.ROTATED_0, image.ROTATED_90_CW,
                     image.ROTATED_180, image.ROTATED_90_CCW]

        store = common.application.store

        if uri.scheme != 'file' or not store:
            # database only support files
            return

        def update(result):
            if not result:
                return

            player.debug("updating rotation")

            orientation = result.orientation
            if not orientation:
                orientation = image.ROTATED_0

            player.debug("current orientation = %r" % orientation)
            index = rotations.index(orientation) + 1
            try:
                next_orientation = rotations[index]
            except IndexError:
                next_orientation = rotations[0]
                index = 0

            player.debug('next orientation = %r' % next_orientation)
            result.orientation = next_orientation
            store.commit()

            # rotate the background image
            img = player.background._widgets[1].image
            player.apply_orientation(img, next_orientation)

            img = player.background._widgets[1].quick_image
            player.apply_orientation(img, next_orientation)

            # on thumbnail (in status_display)
            thumbnail = self.slave.player_osd.status.preview
            player.apply_pixbuf_orientation(uri, thumbnail, next_orientation)

            # TODO: rotate the image in the list widget too

            # TODO: activate this, maybe!
            # update the caption so that it shows what will be the next rotation
            ## try:
            ##     next_orientation = rotations[index+1]
            ## except IndexError:
            ##     next_orientation = rotations[0]
            ## text = image.IMAGE_ORIENTATIONS[next_orientation][0]
            ## self.caption = text
            ## self.emit('caption-changed')

        dfr = store.get(DBImage, uri.path)
        dfr.addCallback(update)
        return dfr

class ShowNextControl(Control):
    """
    Show next picture of the slideshow, if there's any

    Its L{slave} passed to the constructor must be of type
    L{elisa.plugins.poblesec.player_slideshow.PlayerController}.
    """
    caption = _("Show Next")
    glyphs = \
            {STATE_NORMAL: 'elisa.plugins.poblesec.player.glyphs.next_normal',
             STATE_SELECTED: 'elisa.plugins.poblesec.player.glyphs.next_selected',
             STATE_ACTIVE: 'elisa.plugins.poblesec.player.glyphs.next_active',
            }

    def activate(self):
        self.slave.player.next_image()
        return defer.succeed(None)

class ShowPreviousControl(Control):
    """
    Show the previous picture of the slideshow if the beginning of the
    list hasn't been reached yet.

    Its L{slave} passed to the constructor must be of type
    L{elisa.plugins.poblesec.player_slideshow.PlayerController}.
    """
    caption = _("Show Previous")
    glyphs = \
            {STATE_NORMAL: 'elisa.plugins.poblesec.player.glyphs.previous_normal',
             STATE_SELECTED: 'elisa.plugins.poblesec.player.glyphs.previous_selected',
             STATE_ACTIVE: 'elisa.plugins.poblesec.player.glyphs.previous_active',
            }

    def activate(self):
        self.slave.player.previous_image()
        return defer.succeed(None)
