# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Guillaume Emont <guillaume@fluendo.com>

import gobject
from win32gui import *
from win32con import *
from win32gui_struct import PackDEV_BROADCAST_DEVICEINTERFACE, \
        UnpackDEV_BROADCAST


class WindowsMessageSource(gobject.Source):
    """
    A gobject source to attach to your main loop. It will create an invisible
    window for you and call your window proc(s) when the window receives
    messages.
    """
    # if that's the FD to poll, glib will do some PeekMessage
    G_WIN32_MSG_HANDLE = 19981206
    def __init__(self):
        gobject.Source.__init__(self)
        self._poll_fd = gobject.PollFD(self.G_WIN32_MSG_HANDLE, gobject.IO_IN)
        self.add_poll(self._poll_fd)
        self.can_recurse = True

        self._window_class = WNDCLASS()
        self._window_class.lpszClassName = "elisa_win32_message_window"
        self._window_class.style = CS_GLOBALCLASS | CS_VREDRAW | CS_HREDRAW
        self._window_class.hbrBackground = COLOR_WINDOW+1
        self._window_class.lpfnWndProc = self._main_wnd_proc
        RegisterClass(self._window_class)
        self._hwnds = []
        self._hwnds.append(CreateWindow(self._window_class.lpszClassName,
                "Win32 message window", WS_CAPTION,
                100,100,900,900, 0, 0, 0, None))

        self._hdevs = []
        
    def _main_wnd_proc(self, hwnd, msg, wParam, lParam):
        pass
    
    def add_wndproc(self, window_procs):
        """
        @arg window_procs: a dict of message type:callback
        @type window_procs: C{dict}
        """
        self._hwnds.append(SetWindowLong
            (self._hwnds[len(self._hwnds)-1], GWL_WNDPROC, window_procs)) 

    def destroy(self):
        ret = gobject.Source.destroy(self)
        for hdev in self._hdevs:
            UnregisterDeviceNotification(hdev)
        DestroyWindow(self._hwnds[0])
        UnregisterClass(self._window_class.lpszClassName, None)
        return ret

    def register_device_notification(self, guid):
        """
        Add a guid for which we want to receive device notifications.
        """
        filter = PackDEV_BROADCAST_DEVICEINTERFACE(guid)
        hdev = RegisterDeviceNotification(self._hwnds[0], filter,
                                    DEVICE_NOTIFY_WINDOW_HANDLE)
        self._hdevs.append(hdev)
        return hdev

    def prepare(self):
        ret, msg = PeekMessage(None, 0, 0, PM_NOREMOVE)
        return ret, -1
    
    def check(self):
        ret = False
        if self._poll_fd.revents & gobject.IO_IN:
            ret, msg = PeekMessage(None, 0, 0, PM_NOREMOVE)
        return ret

    def dispatch(self, callback, args):
        # will call the WindowProc with the messages in the queue
        PumpWaitingMessages()
        return True

def main():
    from twisted.internet import glib2reactor
    glib2reactor.install()
    from twisted.internet import reactor
    def on_device_change(hwnd, msg, wParam, lParam):
        # Unpack the 'lParam' into the appropriate DEV_BROADCAST_* structure,
        # using the self-identifying data inside the DEV_BROADCAST_HDR.
        info = UnpackDEV_BROADCAST(lParam)
        print "Device change notification:", wParam, info
        return True

    device_source = WindowsMessageSource()
    device_source.add_wndproc({WM_DEVICECHANGE:on_device_change})
    GUID_DEVINTERFACE_USB_DEVICE = "{A5DCBF10-6530-11D2-901F-00C04FB951ED}"
    device_source.register_device_notification(GUID_DEVINTERFACE_USB_DEVICE)
    device_source.attach()

    print "Waiting for device changes"
    reactor.run()

if __name__ == '__main__':
    import sys
    sys.exit(main())
