/*
 ***********************************************************************
 * Copyright 2005-2010 by Freescale Semiconductor, Inc.
 * All modifications are confidential and proprietary information
 * of Freescale Semiconductor, Inc. ALL RIGHTS RESERVED.
 ***********************************************************************
 */

/***************************************************************************** 
** dut_wmv9_api_vts.c 
** 
** 
** Description: Contains the API functions that are invoked by VTS.
**
** Author:
**     Larry Lou   <b06557@freescale.com>
**      
** Revision History: 
** ----------------- 
** 1.0  07/29/2007  Larry Lou   create this file
*   2.0  08 July 2009       Eagle Zhou                ENGR00114047 : mosaic for stream with intensity-compensation and B frame
*   3.0  06 Aug 2009       Eagle Zhou                ENGR00114778 : add dropping B frames
*****************************************************************************/ 


/***************************************************************************** 
 * <Includes> 
 *****************************************************************************/ 
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include "wmv9mp_dec_api.h"

//#define WMV9_SIMPLE_ONLY
/***************************************************************************** 
 * <Macros> 
 *****************************************************************************/ 
#define PrintStatus(a)
#define Printf(format, ...)
#define DebugAppText(format, ...)
/* local return value definitions, starting from a high value for failure */
#define ERROR_NO_MEM           200
#define ERROR_NO_ARGS          201
#define ERROR_READ_FAILED      202
#define ERROR_CORRUPTED_DATA   203
#define ERROR_UNSUPPORTED_FMT  204
#define ERROR_INVALID_ARGS     205

#ifdef _FSL_VTS
#include "dut_api_vts.h"
#endif

#include "dut_test_api.h"

#define tmp_EXPANDY	 24
#define tmp_EXPANDUV 12

#define BUFFER_NUM 4


/*
** we don't have the macro WRAPPER_API_VERSION in old wrapper api header files, 
** so just define it to 20 (version 2.0) which is less than 21 (version 2.1)
*/
#ifndef WRAPPER_API_VERSION
#define WRAPPER_API_VERSION 20
#endif

/* 
** a switch. 
** set to 1 to download bitstream file from host at the beginning of decoding, 
** and save as a file to local file system. set to 0 to download bitstream when
** decoder needs data. 
*/
#define DOWNLOAD_TESTVECTOR_ONCE    1

/***************************************************************************** 
 * <Typedefs> 
 *****************************************************************************/ 

typedef struct
{
    /* rcv header information, passed around using this structure */
    WMV9D_S32           s32Width;
    WMV9D_S32           s32Height;
    WMV9D_S32           s32BitRate;
    WMV9D_S32           s32FrameRate;
    WMV9D_S32           s32RcvVersion;
    char*               filename;     
    const WMV9D_U8*     pu8CompFmtString;

    /* sequence header information */
    WMV9D_S32           s32SeqDataLen;
    WMV9D_U8*           pu8SeqData;

    WMV9D_S32           s32NumTotalFrames;    /* total number of frames */
    WMV9D_S32           s32NumFramesToDecode; /* number of frames to decode */

    /* information related to reading from file, for ewach frame */
    WMV9D_S32           s32TotalBytes;   /* total bytes in this frame */
    WMV9D_S32           s32BytesLeft;    /* number of bytes left in frame */

    FILE*               pInFile;         /* input file pointer  */
    FILE*               pOutFile;        /* output file pointer  */

    sWmv9DecObjectType* pWmv9Object;     /* WMV9 decoder object pointer */

	/* Remove Global Variable for re-enterable@DQ */
#if (21 > WRAPPER_API_VERSION)
	VTS_PROBES_DUT      sVTSProbes;
#elif (21 == WRAPPER_API_VERSION)
	FuncProbeDut * gpfnVtsProbe ;
#endif

	WMV9D_S32           gs32NumBFrames;
	WMV9D_S32           bFlush1Frame;
    unsigned char*      Buffers[BUFFER_NUM];
    int                 bufferIndex;	

} sInputHandlerType;

typedef struct
{
    sInputHandlerType       sInHandler;
    WMV9D_S32               s32DecodeCount;
    sWmv9DecObjectType      sDecObj;
    int                     status;
	long (*fnPrbDutFunc)(T_PROBE_TYPE prb_type, void *para);
	int totalmemorysize;
} WMV9_DUT_OBJECT;

#define DEC_TIMER_BEGIN() {if(psWMV9DutObj->fnPrbDutFunc) psWMV9DutObj->fnPrbDutFunc(T_DECODE_BEGIN, 0);}
#define DEC_TIMER_END() {if(psWMV9DutObj->fnPrbDutFunc) psWMV9DutObj->fnPrbDutFunc(T_DECODE_END, 0);}


#define STACK_TAG() {if(psWMV9DutObj->fnPrbDutFunc) psWMV9DutObj->fnPrbDutFunc(T_STACK_TAG, 0);}
#define STACK_UPDATE() {if(psWMV9DutObj->fnPrbDutFunc) psWMV9DutObj->fnPrbDutFunc(T_STACK_UPDATE, 0);}
 
#define STACK_CALLBACK_TAG()
#define STACK_CALLBACK_UPDATE()

#define HEAP_INCREASE(size) {if(psWMV9DutObj->fnPrbDutFunc) psWMV9DutObj->fnPrbDutFunc(T_HEAP_INCREASE, (void*)size);}
#define HEAP_DECREASE(size) {if(psWMV9DutObj->fnPrbDutFunc) psWMV9DutObj->fnPrbDutFunc(T_HEAP_DECREASE, size);}

/***************************************************************************** 
 * <Global Variables> 
 *****************************************************************************/
/* probes to store some information for VTS */
//VTS_PROBES_DUT sVTSProbes = { NULL, NULL, NULL, NULL };
/* number of consecutive B frames. it's the NUMBFRAMES in seq header */
//WMV9D_S32 gs32NumBFrames = 0;

/***************************************************************************** 
 * <Local Variables> 
 *****************************************************************************/
/* none */

/***************************************************************************** 
 * <Forward declaretions> 
 *****************************************************************************/
/* none */

/*---------------------------------------------------------------------------*/
/****************************** LOCAL FUNCTIONS ******************************/
/*---------------------------------------------------------------------------*/
WMV9D_Void*
AppAllocateSlowMemory (WMV9D_S32 s32Size)
{
    return malloc (s32Size);
}

WMV9D_Void*
AppAllocateFastMemory (WMV9D_S32 s32Size)
{
    return malloc (s32Size);
}

/* memory allocation according to the requests provided by decoder */
int
AppAllocateMemory (WMV9_DUT_OBJECT * psWMV9DutObj, sWmv9DecMemAllocInfoType* psMemInfo)
{
    WMV9D_S32     s32Count;
    WMV9D_Void*   pvUnalignedBuf;
    WMV9D_Void*   pvOldBuf;
    WMV9D_S32     s32ExtraSize;
    WMV9D_S32     s32Mask;

	psWMV9DutObj->totalmemorysize = 0;
    for (s32Count = 0; s32Count < psMemInfo->s32NumReqs; s32Count++)
    {
        sWmv9DecMemBlockType* psMemBlk = psMemInfo->asMemBlks + s32Count;

        /* Get the extra amount to be allocated for the alignment */
        switch (psMemBlk->eMemAlign)
        {
            case E_WMV9D_ALIGN_NONE:
                s32ExtraSize = 0;
                s32Mask = 0xffffffff;
                break;

            case E_WMV9D_ALIGN_HALF_WORD:
                s32ExtraSize = 1;
                s32Mask = 0xfffffffe;
                break;

            case E_WMV9D_ALIGN_WORD:
                s32ExtraSize = 3;
                s32Mask = 0xfffffffc;
                break;

            case E_WMV9D_ALIGN_DWORD:
                s32ExtraSize = 7;
                s32Mask = 0xfffffff8;
                break;

            case E_WMV9D_ALIGN_QWORD:
                s32ExtraSize = 15;
                s32Mask = 0xfffffff0;
                break;

            case E_WMV9D_ALIGN_OCTAWORD:
                s32ExtraSize = 31;
                s32Mask = 0xffffffe0;
                break;

            default :
                s32ExtraSize = -1;  /* error condition */
                s32Mask = 0x00000000;
                printf ("Memory alignment type %d not supported\n",
                        psMemBlk->eMemAlign);
                break;
        }

        /* Save the old pointer, in case memory is being reallocated */
        pvOldBuf = psMemBlk->pvBuffer;

        /* Allocate new memory, if required */
        if (psMemBlk->s32Size > 0)
        {
            if (WMV9D_IS_SLOW_MEMORY(psMemBlk->s32MemType))
                pvUnalignedBuf = AppAllocateSlowMemory (psMemBlk->s32Size +
                                                        s32ExtraSize);
            else
                pvUnalignedBuf = AppAllocateFastMemory (psMemBlk->s32Size +
                                                        s32ExtraSize);

            psMemBlk->pvBuffer = (WMV9D_Void*)
                    (((WMV9D_S32)pvUnalignedBuf + s32ExtraSize) & s32Mask);

//			HEAP_INCREASE(psMemBlk->s32Size + s32ExtraSize);
			psWMV9DutObj->totalmemorysize += (psMemBlk->s32Size + s32ExtraSize);

        }
        else
        {
            pvUnalignedBuf = NULL;
            psMemBlk->pvBuffer = NULL;
        }

        /* Check if the memory is being reallocated */
        if (WMV9D_IS_SIZE_CHANGED(psMemBlk->s32MemType))
        {
            if (psMemBlk->s32OldSize > 0)
            {
                WMV9D_S32  s32CopySize = psMemBlk->s32OldSize;

                if (psMemBlk->s32Size < s32CopySize)
                    s32CopySize = psMemBlk->s32Size;

                if (WMV9D_NEEDS_COPY_AT_RESIZE(psMemBlk->s32MemType))
                    memcpy (psMemBlk->pvBuffer, pvOldBuf, s32CopySize);

                free (psMemBlk->pvUnalignedBuffer);
            }
        }

        /* Now save the new unaligned buffer pointer */
        psMemBlk->pvUnalignedBuffer = pvUnalignedBuf;

        /* debug message */
        DebugAppText ("[%2d] size %d type %d align %d priority %d buffer %x",
                      s32Count, psMemBlk->s32Size, psMemBlk->eMemAlign,
                      psMemBlk->eMemAlign, psMemBlk->s32Priority,
                      psMemBlk->pvBuffer);
    }

    return 0;
}


WMV9D_Void
AppFreeMemory (sWmv9DecMemAllocInfoType* psMemInfo)
{
    WMV9D_S32     s32Count;

    for (s32Count = 0; s32Count < psMemInfo->s32NumReqs; s32Count++)
    {
        if (psMemInfo->asMemBlks[s32Count].s32Size > 0)
        {
            free (psMemInfo->asMemBlks[s32Count].pvUnalignedBuffer);
            psMemInfo->asMemBlks[s32Count].s32Size = 0;
        }
    }
}


/* various cleaning functions */

void
CleanHandler (sInputHandlerType* pHandler)
{
    if (pHandler == NULL)
        return;

    if (pHandler->pInFile != NULL)
    {
#if ( 21 == WRAPPER_API_VERSION ) && ( 0 == DOWNLOAD_TESTVECTOR_ONCE )
        if ( pHandler->gpfnVtsProbe )
        {
            pHandler->gpfnVtsProbe( E_CLOSE_BITSTREAM, &pHandler->pInFile );
        }
#else
        fclose (pHandler->pInFile);
        pHandler->pInFile = NULL;
#endif
	}

    if (pHandler->pOutFile != NULL)
    {
        fclose (pHandler->pOutFile);
        pHandler->pOutFile = NULL;
	}

    if (pHandler->pu8SeqData != NULL)
    {
        free (pHandler->pu8SeqData);
        //change for free() call
        pHandler->pu8SeqData = NULL;
	}

}


void
CleanAppData (sInputHandlerType* psHandler)
{
    AppFreeMemory (&psHandler->pWmv9Object->sMemInfo);
    CleanHandler (psHandler);
}

/* Reads the data in the provided buffer. It reads s32NumItems,
   each of size u16Size
*/

WMV9D_S32
GetRCVData (WMV9D_U8* pu8Buffer, WMV9D_S32 s32Size, WMV9D_S32 s32NumItems,
            sInputHandlerType* pHandler)
{
    if (pHandler == NULL || pHandler->pInFile == NULL)
        return -1;

#if ( 21 == WRAPPER_API_VERSION ) && ( 0 == DOWNLOAD_TESTVECTOR_ONCE )
    if ( pHandler->gpfnVtsProbe )
    {
        DUT_STREAM_READ sReadStream = { pHandler->pInFile, pu8Buffer, s32Size * s32NumItems };
        printf( "GetRCVData() : Get %d bytes\n", s32Size * s32NumItems );
        return pHandler->gpfnVtsProbe( E_READ_BITSTREAM, &sReadStream );
    }
#else
    return fread (pu8Buffer, s32Size, s32NumItems, pHandler->pInFile);
#endif
}

void
ByteReverse (WMV9D_U8* pu8Buffer, WMV9D_S32 s32Size, WMV9D_S32 s32NumItems)
{
    WMV9D_U8    u8TempByte;
    WMV9D_S32   i, j;

    for (i = 0; i < s32NumItems; i = i + s32Size)
    {
        for (j = 0; j < s32Size/2; j++)
        {
            u8TempByte = pu8Buffer[j];
            pu8Buffer[j] = pu8Buffer[s32Size - j - 1];
            pu8Buffer[s32Size - j - 1] = u8TempByte;
        }
        pu8Buffer += s32Size;
    }
}


/* call back function used to get the part of bitstream */
WMV9D_S32
AppGetBitstream (WMV9D_S32 s32BufLen, WMV9D_U8* pu8Buf, WMV9D_S32* bEndOfFrame,
                 WMV9D_Void* pvAppContext)
{
    sInputHandlerType* pInHandler = (sInputHandlerType*)pvAppContext;
    WMV9D_S32          s32NumRead = -1;

    /* The first call is for the sequence data, return that from the handler */
    if (pInHandler->s32SeqDataLen > 0)
    {
        assert (s32BufLen >= pInHandler->s32SeqDataLen);
        Printf ("Returning sequence layer data of length %d\n",
                pInHandler->s32SeqDataLen);
        
       memcpy (pu8Buf, pInHandler->pu8SeqData, pInHandler->s32SeqDataLen);
        s32NumRead = pInHandler->s32SeqDataLen;
                pInHandler->s32SeqDataLen = 0;
                *bEndOfFrame = 1;
		
        free (pInHandler->pu8SeqData);
        pInHandler->pu8SeqData = NULL;

        return s32NumRead;
    }

    if (pInHandler->s32BytesLeft > s32BufLen)
    {
#if ( 21 == WRAPPER_API_VERSION ) && ( 0 == DOWNLOAD_TESTVECTOR_ONCE )
        if ( pInHandler->gpfnVtsProbe )
        {
            DUT_STREAM_READ sReadStream = { pInHandler->pInFile, pu8Buf, s32BufLen };
            printf( "AppGetBitstream()1 : Get %d bytes\n", s32BufLen );
            s32NumRead = pInHandler->gpfnVtsProbe( E_READ_BITSTREAM, &sReadStream );
        }
#else
        s32NumRead = fread (pu8Buf, 1, s32BufLen, pInHandler->pInFile);
#endif
        *bEndOfFrame = 0;
        pInHandler->s32BytesLeft -= s32BufLen;
    }
    else
    {
#if ( 21 == WRAPPER_API_VERSION ) && ( 0 == DOWNLOAD_TESTVECTOR_ONCE )
        if ( pInHandler->gpfnVtsProbe )
        {
            DUT_STREAM_READ sReadStream = { pInHandler->pInFile, pu8Buf, pInHandler->s32BytesLeft };
            printf( "AppGetBitstream()2 : Get %d bytes\n", pInHandler->s32BytesLeft );
            s32NumRead = pInHandler->gpfnVtsProbe( E_READ_BITSTREAM, &sReadStream );
        }
#else
        s32NumRead = fread (pu8Buf, 1, pInHandler->s32BytesLeft,
                            pInHandler->pInFile);
#endif
        *bEndOfFrame = 1;
        pInHandler->s32BytesLeft -= pInHandler->s32BytesLeft;
    }

#if ( 21 > WRAPPER_API_VERSION )
    //Added for dynamic test
    if( pInHandler->sVTSProbes.pfStoreBitsRmTime != NULL)
        pInHandler->sVTSProbes.pfStoreBitsRmTime();
    if( pInHandler->sVTSProbes.pfStoreBitsUnitLen != NULL)
        pInHandler->sVTSProbes.pfStoreBitsUnitLen(s32NumRead);
#endif
    return s32NumRead;
}


int
ReadHeaderInfo (sInputHandlerType* psInHandler)
{
    WMV9D_S32 s32BytesRead = 0;
    WMV9D_S32 s32Buffer;
    WMV9D_S32 s32CodecVersion, s32HdrExtn;

    /* hdrext : bit[31], rcv_version : bit[30], s32CodecVersion : bit [29:24] */

    s32BytesRead = GetRCVData ((WMV9D_U8*)&s32Buffer, 4, 1, psInHandler);
    if (s32BytesRead <= 0)
    {
        CleanHandler (psInHandler);
        return ERROR_READ_FAILED;
    }

    //ByteReverse ((WMV9D_U8*)&s32Buffer, 4, 1);
//printf( "%s, %d : s32Buffer = 0x%x\n", __FUNCTION__, __LINE__, s32Buffer );

    psInHandler->s32RcvVersion = (s32Buffer >> 30) & 0x1;

    s32CodecVersion = s32Buffer >> 24;

    /* number of frames present is in lower 3 bytes */
    psInHandler->s32NumTotalFrames = s32Buffer & 0xffffff;

    s32HdrExtn = (s32Buffer >> 31) & 0x1;

    Printf ("Initial versions : RCV %d Extn %d Codec %d\n",
            psInHandler->s32RcvVersion, s32HdrExtn,
            s32CodecVersion & 0x3f);

    if (psInHandler->s32RcvVersion == 0) {
        s32CodecVersion &= 0x7f;  // no point, bit 30 is already 0
    } else {
        s32CodecVersion &= 0x3f;
    }
//printf( "%s, %d : s32CodecVersion = %d\n", __FUNCTION__, __LINE__, s32CodecVersion );
    /* Set the compression format */
    if (s32CodecVersion == 0)
    {
        PrintStatus ("Bitstream is encoded in WMV standard, version 1\n");
        /* s32CodecVersion = FOURCC_WMV1_WMV; */
        psInHandler->pu8CompFmtString = "WMV1";
    }
    else if (s32CodecVersion == 1)
    {
        PrintStatus ("Bitstream is encoded in MPEG4 standard, version 3\n");
        /* s32CodecVersion = FOURCC_MP43_WMV; */
        psInHandler->pu8CompFmtString = "MP43";
    }
    else if (s32CodecVersion == 2)
    {
        PrintStatus ("Bitstream is encoded in WMV standard, version 2\n");
        /* s32CodecVersion = FOURCC_WMV2_WMV; */
        psInHandler->pu8CompFmtString = "WMV2";
    }
    else if (s32CodecVersion== 3)
    {
        PrintStatus ("Bitstream is encoded in MPEG4 standard, version 2\n");
        /* s32CodecVersion = FOURCC_MP42_WMV; */
        psInHandler->pu8CompFmtString = "MP42";
    }
    else if (s32CodecVersion == 4)
    {
        PrintStatus ("Bitstream is encoded in MPEG4 standard, "\
                     "MPEG4 with short video header\n");
        /* s32CodecVersion = FOURCC_MP4S_WMV; */
        psInHandler->pu8CompFmtString = "MP4S";
    }
    else if (s32CodecVersion == 5)
    {
        PrintStatus ("Bitstream is encoded in WMV standard, version 3\n");
        /* s32CodecVersion = FOURCC_WMV3_WMV; */
        psInHandler->pu8CompFmtString = "WMV3";
    }
    else
    {
        psInHandler->pu8CompFmtString = NULL;
        CleanHandler (psInHandler);
        return ERROR_UNSUPPORTED_FMT;
    }

    /* Read any extension of the header, if present
     * Actually this is the sequence layer data
     */
    if (s32HdrExtn != 0)
    {
        s32BytesRead = GetRCVData ((WMV9D_U8*)&psInHandler->s32SeqDataLen,
                                   4, 1, psInHandler);
        if (s32BytesRead <= 0)
        {
            CleanHandler (psInHandler);
            return ERROR_READ_FAILED;
        }

        //ByteReverse ((WMV9D_U8*)&psInHandler->s32SeqDataLen, 4, 1);

        Printf ("Sequence data size 0x%04x\n", psInHandler->s32SeqDataLen);

        //psInHandler->pu8SeqData = malloc (psInHandler->s32SeqDataLen);
        //printf("psInHandler->pu8SeqData = 0X%x\n",psInHandler->pu8SeqData);
        //free(psInHandler->pu8SeqData);
        psInHandler->pu8SeqData = malloc (psInHandler->s32SeqDataLen);
        //printf("psInHandler->pu8SeqData = 0X%x\n",psInHandler->pu8SeqData);

        if (psInHandler->pu8SeqData == NULL)
        {
            CleanHandler (psInHandler);
            return ERROR_NO_MEM;
        }

        s32BytesRead = GetRCVData (psInHandler->pu8SeqData, 1,
                                   psInHandler->s32SeqDataLen, psInHandler);
        if (s32BytesRead <= 0)
        {
            CleanHandler (psInHandler);
            return ERROR_READ_FAILED;
        }
    }

    /* Get the height of the frames */
    if (GetRCVData ((WMV9D_U8*)&psInHandler->s32Height, 4, 1, psInHandler) <= 0)
    {
        CleanHandler (psInHandler);
        return ERROR_READ_FAILED;
    }

         //ByteReverse ((WMV9D_U8*)&psInHandler->s32Height, 4, 1);

    Printf ("Frame height 0x%08x\n", psInHandler->s32Height);

    if (psInHandler->s32Height <= 0)
    {
        CleanHandler (psInHandler);
        return ERROR_CORRUPTED_DATA;
    }

    /* Get the width of the frames */
    if (GetRCVData ((WMV9D_U8*)&psInHandler->s32Width, 4, 1, psInHandler) <= 0)
    {
        CleanHandler (psInHandler);
        return ERROR_READ_FAILED;
    }

        //ByteReverse ((WMV9D_U8*)&psInHandler->s32Width, 4, 1);

    Printf ("Frame width 0x%08x\n", psInHandler->s32Width);

    if (psInHandler->s32Width <= 0)
    {
        CleanHandler (psInHandler);
        return ERROR_CORRUPTED_DATA;
    }

    Printf("Frame, Width = %d, Height = %d\n", psInHandler->s32Width,
           psInHandler->s32Height);

    if (psInHandler->s32RcvVersion == 1)
    {
        WMV9D_S32  rcv_additional_header_size, pre_roll;
        // additional header size
        if (GetRCVData ((WMV9D_U8*)&rcv_additional_header_size, 4, 1, psInHandler) <= 0)
        {
            CleanHandler (psInHandler);
            return ERROR_READ_FAILED;
        }

        //ByteReverse ((WMV9D_U8*)&rcv_additional_header_size, 4, 1);

        if (GetRCVData ((WMV9D_U8*)&pre_roll, 4, 1, psInHandler) <= 0) // pre-roll
        {
            CleanHandler (psInHandler);
            return ERROR_READ_FAILED;
        }

        //ByteReverse ((WMV9D_U8*)&pre_roll, 4, 1);

        pre_roll &= 0x0fffffff;
        /* bit rate, 4 bytes */
        if (GetRCVData ((WMV9D_U8*)&psInHandler->s32BitRate, 4, 1, psInHandler) <= 0)
        {
            CleanHandler (psInHandler);
            return ERROR_READ_FAILED;
        }

        //ByteReverse ((WMV9D_U8*)&psInHandler->s32BitRate, 4, 1);

        /* frame rate, 4 bytes */
        if (GetRCVData ((WMV9D_U8*)&psInHandler->s32FrameRate, 4, 1, psInHandler) <= 0)
        {
            CleanHandler (psInHandler);
            return ERROR_READ_FAILED;
        }

        //ByteReverse ((WMV9D_U8*)&psInHandler->s32FrameRate, 4, 1);

    }
    else
    {
        /* Set to some default value */
        psInHandler->s32BitRate = 0;
        psInHandler->s32FrameRate = 0;
    }

    return 0;
}


/********************** MAD ShangHai Changed Start***************************
** Nov 30, -- DingQiang Add Direct Rendering
**
*****************************************************************************/

//#define BUFFER_NUM 4
//unsigned char* Buffers[BUFFER_NUM];
//static int bufferIndex = 0;
void* getBuffer(void * pvAppContext);
void rejectBuffer(void* buffer,void * pvAppContext);

int initBuffers(sInputHandlerType * psInhandler, int size)
{
	int i;
	for (i = 0; i < BUFFER_NUM; i ++)
	{
		if (NULL == (psInhandler->Buffers[i] = malloc(size)))
		{
			printf("Buffers[%d] is NULL: size=%d\n",i,size);
			return 0;
		}
	}
	return 1;
}
void releaseBuffers(sInputHandlerType * psInhandler)
{
	int i;
	for (i = 0; i< BUFFER_NUM; i ++)
	{
		if ( NULL != psInhandler->Buffers[i])
			free(psInhandler->Buffers[i]);
	}
	printf(" total %d buffers are provided to decoder\n",psInhandler->bufferIndex);
}
static int get = 0;
static int rel = 0;
static buf_flag[BUFFER_NUM] = {1,1,1,1};
void* getBuffer(void * pvAppContext)
{
	int i=0;
	sInputHandlerType * psInhandler;
	void* temp;

	psInhandler = (sInputHandlerType *)pvAppContext;
//	if(get==0)
//	{
//		buf_flag[0] = buf_flag[1] = buf_flag[2] = buf_flag[3] = 1;
//		printf("buffer 0  %x\n",psInhandler->Buffers[0]);
//		printf("buffer 1  %x\n",psInhandler->Buffers[1]);
//		printf("buffer 2  %x\n",psInhandler->Buffers[2]);
//		printf("buffer 3  %x\n",psInhandler->Buffers[3]);
//	}
	get++;
	for(i=0;i<BUFFER_NUM;i++){
		temp = psInhandler->Buffers[i];
		if(buf_flag[i])
		{
			buf_flag[i] = 0;
	//		psInhandler->Buffers[psInhandler->bufferIndex%BUFFER_NUM] = 0;
	//		psInhandler->bufferIndex++;
	//		printf("\n>>>>>>>>>>>>>>>>>get buffer %d                   %x\n",get,temp);
			return temp;
		}
	//	psInhandler->bufferIndex++;
	}
	printf("get buffer error!\n");

	//void* temp = Buffers[0];
	//psInhandler->bufferIndex++;
	//printf("buffer count:%d   ",bufferIndex-1);
	//if (5 == bufferIndex) return NULL;
	//printf("Address : %x \n",Buffers[0]);
	//if(bufferIndex==1)
	//return temp;
	//return Buffers[0];
	//else
	//	return NULL;
}
void rejectBuffer(void* buffer,void * pvAppContext)
{
	printf("rejected %x\n", buffer);
}
void releaseBuffer(void* buffer,void* AppContext)
{
	int i;
    sInputHandlerType *psWmv9DutObj = (sInputHandlerType *)AppContext;
	void* temp;
	rel++;
	if(buffer==0)
		return;
	for(i=0;i<BUFFER_NUM;i++){
		temp = psWmv9DutObj->Buffers[i];
		if(temp==buffer)
		{
//			printf("\n**************rel buffer %d  %x\n",rel,buffer);
			buf_flag[i] = 1;
//			if((psWmv9DutObj->s32Width>0)&&(psWmv9DutObj->s32Height>0))
//				memset(((char*)buffer)+psWmv9DutObj->s32Width * (psWmv9DutObj->s32Height+1)/2,0,psWmv9DutObj->s32Width * (psWmv9DutObj->s32Height+1)/2);
			return;
		}
	}
	printf("release buffer error   %x\n",buffer);
}


#if (WRAPPER_API_VERSION == 21)
#define BUFFER_SIZE (1024 * 1024) /* temp buffer for bitstream. 1M size is tradeoff of trans speed and memory size */
/* download bitstream file from the host, and save as a file to local file sytem */
int WMV9D_GetInputFile( char *          strInFileLocal, 
                        char *          strInFileRemote, 
                        FuncProbeDut *  pfProbe )
{
    FILE * fpInFileSrc = NULL, * fpInFileDst = NULL;
    char * p8Buffer = NULL;
    int bRetVal = 1;
    DUT_STREAM_OPEN sStreamOpen;
    DUT_STREAM_READ sStreamRead;
    int iReadSize = BUFFER_SIZE, iWriteSize = BUFFER_SIZE;
    
    /* open file from host */
    sStreamOpen.strBitstream = strInFileRemote;
    sStreamOpen.strMode      = "rb";
    fpInFileSrc = (FILE *)pfProbe( E_OPEN_BITSTREAM, &sStreamOpen );
    /* create and open the local file */
    fpInFileDst = fopen( strInFileLocal, "wb" );
    if ( ( NULL == fpInFileSrc ) || ( NULL == fpInFileDst ) )
    {
        bRetVal = 0;
        goto exit;
    }
    /* allocate the temp bitstream buffer */
    p8Buffer = (char *)malloc( BUFFER_SIZE );
    if ( NULL == p8Buffer )
    {
        bRetVal = 0;
        goto exit;
    }
    /* read bitstream from host and write to local file */
    sStreamRead.hBitstream    = (int)fpInFileSrc;
    sStreamRead.pBitstreamBuf = p8Buffer;
    sStreamRead.iLength       = BUFFER_SIZE;
    while ( BUFFER_SIZE == iReadSize )
    {
        iReadSize = pfProbe( E_READ_BITSTREAM, &sStreamRead );
        if ( 0 >= iReadSize )
        {
            break;
        }
        iWriteSize = fwrite( p8Buffer, 1, iReadSize, fpInFileDst );
        if ( iReadSize != iWriteSize )
        {
            bRetVal = 0;
            goto exit;
        }
    }
    
exit:
    // close file
    pfProbe( E_CLOSE_BITSTREAM, &fpInFileSrc );
    if ( NULL != fpInFileDst )
    {
        fclose( fpInFileDst );
    }
    if ( NULL != p8Buffer )
    {
        free( p8Buffer );
    }

    return bRetVal;
}
#endif


/********************** MAD ShangHai Changed End  ****************************/
/*---------------------------------------------------------------------------*/
/******************************* API FUNCTIONS *******************************/
/*---------------------------------------------------------------------------*/
int GetUserInput(sWmv9DecObjectType   * psDecObj,int argc, char *argv[])
{
#define CASE_FIRST(x)   if (strncmp(argv[0], x, strlen(x)) == 0)
#define CASE(x)         else if (strncmp(argv[0], x, strlen(x)) == 0)
#define DEFAULT         else

	while (argc)
	{
		if (argv[0][0] == '-')
		{
			CASE_FIRST("-b")
			{
				eWMV9DEnableSkipBMode(psDecObj,1);
				printf("enable: drop B frame \r\n");
			}
			//CASE("-i")
			//{
			//	eWMV9DEnableSkipMode(psDecObj);
			//	printf("enable: skip to next I frame \r\n");
			//}
			DEFAULT                             // Has to be last
			{
				printf("Unsupported option %s\n", argv[0]);
			}
		}
		else
		{
			printf("Please input '-' option \r\n");
		}
		argc--;
		argv++;
	}

	return 1;
}

/*!
 ***********************************************************************
 * -Function:
 *    DEC_RETURN_DUT DLL_EXPORTS VideoDecInit_dut( void **     _ppDecObj, 
 *                                             DEC_INIT_CONTXT_DUT * _psInitContxt );
 *
 * -Description:
 *    This function initializes the RealVideo DUT by using the infomation from VTS app.
 *
 * -Input Param
 *    *pFileParseContxt       pointer of structure contain dut init infomation. 
 *                            It should be DEC_INIT_CONTXT_DUT for RealVideo DUT
 *
 * -Output Param
 *    **ppDecObj              pointer of decoder object pointer
 *
 * -Return
 *    E_DEC_INIT_OK_DUT      decoder is initialized successfully
 *    E_DEC_INIT_ERROR_DUT   encounter errer during initializing
 ***********************************************************************
 */
#if ( 21 == WRAPPER_API_VERSION )
static int WMV9DecInit( void **_ppDecObj, DUT_INIT_CONTXT_2_1 *_psInitContxt, T_DEC_CONTXT_DUT *_psDecContxt)
#else
static int WMV9DecInit( void **_ppDecObj, DEC_INIT_CONTXT_DUT *_psInitContxt, T_DEC_CONTXT_DUT *_psDecContxt)
#endif

{
	T_DEC_CONTXT_DUT *psWmv9DecContxt = (T_DEC_CONTXT_DUT *)_psDecContxt;
#if ( 21 == WRAPPER_API_VERSION )
    DUT_INIT_CONTXT_2_1  * psWmv9InitContxt = (DUT_INIT_CONTXT_2_1 *)_psInitContxt;
#else
    DEC_INIT_CONTXT_DUT  * psWmv9InitContxt = (DEC_INIT_CONTXT_DUT *)_psInitContxt;
#endif
    WMV9_DUT_OBJECT       * psWMV9DutObj = NULL;
    sInputHandlerType     * psInHandler;
    WMV9D_S32               s32UseFRate;
    sWmv9DecObjectType    * psDecObj;
    int                     status = E_WMV9D_SUCCESS;

    /* allocate memory for dut object */
	psWMV9DutObj = (WMV9_DUT_OBJECT *)malloc (sizeof(WMV9_DUT_OBJECT));
    if ( psWMV9DutObj == NULL )
    {
        fprintf(stderr,"ERROR -- allocate memory for WMV9_DUT_OBJECT failed!\n");
        return E_DEC_INIT_ERROR_DUT;
    }
    memset( psWMV9DutObj, 0, sizeof(WMV9_DUT_OBJECT) );
    *_ppDecObj = psWMV9DutObj;

    psInHandler = &psWMV9DutObj->sInHandler;
    psDecObj = &psWMV9DutObj->sDecObj;

	/* Init Golbal Variable @dq*/
	psInHandler->bFlush1Frame = 1;

    /* open input file */
	if(psWmv9InitContxt)
	{
#if ( 21 == WRAPPER_API_VERSION )
#if DOWNLOAD_TESTVECTOR_ONCE
        char * strInFile = "wmv9bitstream.bits";
        int iRetVal = WMV9D_GetInputFile( strInFile, 
                                          psWmv9InitContxt->strInFile, 
                                          psWmv9InitContxt->pfProbe );
        psInHandler->gpfnVtsProbe = psWmv9InitContxt->pfProbe;
        if ( 0 == iRetVal )
        {
			fprintf (stderr, "Failed to download bitstream file\n" );
			return E_DEC_INIT_ERROR_DUT;
        }
		psInHandler->pInFile = fopen ( strInFile, "rb" );
#else
        psInHandler->gpfnVtsProbe = psWmv9InitContxt->pfProbe;
        if ( psInHandler->gpfnVtsProbe )
        {
    		DUT_STREAM_OPEN sOpenStream = { psWmv9InitContxt->strInFile, "rb" };
            psInHandler->pInFile = psInHandler->gpfnVtsProbe( E_OPEN_BITSTREAM, &sOpenStream );
        }
#endif
#else
		psInHandler->pInFile = fopen ( psWmv9InitContxt->strInFile, "rb" );
#endif
		if (psInHandler->pInFile == NULL) {
			fprintf (stderr, "Could not open %s for reading\n", 
				psWmv9InitContxt->strInFile );
			return E_DEC_INIT_ERROR_DUT;
		}
	}
	else
	{
		psInHandler->pInFile = fopen ( psWmv9DecContxt->strInFile, "rb" );
		if (psInHandler->pInFile == NULL) {
			fprintf (stderr, "Could not open %s for reading\n", 
				psWmv9DecContxt->strInFile );
			return E_DEC_INIT_ERROR_DUT;
		}
	}
//printf( "%s, %d\n", __FUNCTION__, __LINE__ );
    /* Read the header information from the rcv file */
    status = ReadHeaderInfo (psInHandler);
    if (status != 0)
    {
        CleanHandler (psInHandler);
        return E_DEC_INIT_ERROR_DUT;
    }

    /* Larry 09252007 : Get sequence header info to see if there will be output delay */
    psInHandler->gs32NumBFrames = psInHandler->pu8SeqData[3] & 0x70;
    /* ~Larry 09252007 */


    /* Do query for memory requirement */
	STACK_TAG();
    PrintStatus ("Querying for memory requirement\n");
    status = eWMV9DQuerymem (psDecObj, psInHandler->s32Height,
                             psInHandler->s32Width);
	STACK_UPDATE();

    /* Allocate memory for the decoder */
    if (AppAllocateMemory (psWMV9DutObj,&psDecObj->sMemInfo))
    {
        CleanHandler (psInHandler);
        return E_DEC_INIT_ERROR_DUT;
    }

    /* Save the dec object pointer in the handler, so we that we can access
     * it in the callback, if required.
     */
    psInHandler->pWmv9Object = psDecObj;

    /* Do the initialization of the structure, and call init */
    psDecObj->pfCbkBuffRead = AppGetBitstream;
    psDecObj->pvAppContext = psInHandler;
    psDecObj->sDecParam.eCompressionFormat =
                   eWMV9DCompFormat (psInHandler->pu8CompFmtString);
    psDecObj->sDecParam.s32FrameRate   = 0 ;
    psDecObj->sDecParam.s32BitRate     = 0 ;
    psDecObj->sDecParam.u16FrameWidth  = (WMV9D_U16)psInHandler->s32Width ;
    psDecObj->sDecParam.u16FrameHeight = (WMV9D_U16)psInHandler->s32Height ;
//    psDecObj->sDecParam.sOutputBuffer.tOutputFormat  = IYUV_WMV;  

	// Add for direct rendering@MAD Shanghai DingQiang
	{
		int width,height;
		#if 0
		printf("u16FrameWidth=%d,u16FrameHeight=%d\n",\
			sDecObj.sDecParam.u16FrameWidth, sDecObj.sDecParam.u16FrameHeight );
		#endif
		
		width = ((psDecObj->sDecParam.u16FrameWidth+15)>>4)*16+2*tmp_EXPANDY;
		height = ((psDecObj->sDecParam.u16FrameHeight+15)>>4)*16+2*tmp_EXPANDY;
		
		#if 0
		printf("width=%d, height=%d , width*height*1.5= %d\n",\
			width, height,((int)(width*height*1.5)));
		#endif
		
		initBuffers(psInHandler,width*height*1.5);
	}
#ifdef OUTPUT_BUFFER_CHANGES
    {
		WMV9D_S32 width_pad;
		WMV9D_S32 height_pad;
		WMV9D_S32 EXPANDY = 48;

		width_pad = (psDecObj->sDecParam.u16FrameWidth +15)&(~15);
		height_pad = (psDecObj->sDecParam.u16FrameHeight +15)&(~15);
		width_pad += EXPANDY;
		height_pad += EXPANDY;   
		psDecObj->sDecParam.sOutputBuffer.tOutputFormat  = IYUV_WMV;

		#if 0 //Direct Rendering
		
		psDecObj->sDecParam.sOutputBuffer.pu8YBuf = (WMV9D_U8 *)malloc 
		                             (width_pad * height_pad * sizeof(WMV9D_U8));
		                             
		if(psDecObj->sDecParam.sOutputBuffer.pu8YBuf == NULL)
		{
		    printf ("Unable to allocate memory for Output Buffer pu8YBuf \n");
            return E_DEC_INIT_ERROR_DUT;

		}
		
		psDecObj->sDecParam.sOutputBuffer.pu8CbBuf = (WMV9D_U8 *)malloc 
		                             (width_pad * height_pad * (sizeof(WMV9D_U8))/4);

        if(psDecObj->sDecParam.sOutputBuffer.pu8CbBuf == NULL)
		{
		    printf ("Unable to allocate memory for Output Buffer pu8CbBuf \n");
            return E_DEC_INIT_ERROR_DUT;

		}
		
        
        
        psDecObj->sDecParam.sOutputBuffer.pu8CrBuf = (WMV9D_U8 *)malloc 
		                             (width_pad * height_pad * (sizeof(WMV9D_U8))/4);

        if(psDecObj->sDecParam.sOutputBuffer.pu8CrBuf == NULL)
		{
		    printf ("Unable to allocate memory for Output Buffer pu8CrBuf \n");
            return E_DEC_INIT_ERROR_DUT;

		}
		#endif
    }
#endif

	/* MAD ShangHai -DingQiang Add for Direct Rendering*/
	/*register frame manager*/
	{
		WMV9D_FrameManager manager;
		manager.BfGetter = getBuffer;
		manager.BfRejector = rejectBuffer;
		WMV9DSetBufferManager(psDecObj, &manager);
		//printf("set a frame manager\n");
	}
	STACK_TAG();
    status = eWMV9DInit (psDecObj);
	eWMV9DSetAdditionalCallbackFunction (psDecObj, E_RELEASE_FRAME, (void*)releaseBuffer);
	STACK_UPDATE();

    if (status != E_WMV9D_SUCCESS)
    {
        CleanAppData (psInHandler);
        return E_DEC_INIT_ERROR_DUT;
    }

	if(_psDecContxt)
	{		
		if(!GetUserInput(psDecObj, _psDecContxt->argc,_psDecContxt->argv))
		{
		        CleanAppData (psInHandler);
		        return E_DEC_INIT_ERROR_DUT;
		}		
	}	

    /* Set the frame rate to be used for timestamp printing */
    if ( psDecObj->sDecParam.s32FrameRate > 0)
    {
       s32UseFRate = psDecObj->sDecParam.s32FrameRate;
    }
    else
    {
       s32UseFRate = 30;  /* use default 30fps */
    }

    /* Sanity */
    if (s32UseFRate > 40)
    {
        printf ("Seting frame rate to 30 from %d\n", s32UseFRate);
        s32UseFRate = 30;
    }

    /* Set the number of frames to be decoded */
    if (psInHandler->s32NumFramesToDecode <= 0)
    {
        psInHandler->s32NumFramesToDecode = psInHandler->s32NumTotalFrames;
    }
	

    psWMV9DutObj->status = status;

	if(psWmv9DecContxt)
	{
		psWmv9DecContxt->uiWidth = psDecObj->sDecParam.u16FrameWidth;
		psWmv9DecContxt->uiHeight = psDecObj->sDecParam.u16FrameHeight;
		psWmv9DecContxt->uiBitRate = psDecObj->sDecParam.s32BitRate;
		psWmv9DecContxt->uiFrameRate = psDecObj->sDecParam.s32FrameRate;
	}
    

    return E_DEC_INIT_OK_DUT;
}

/*-------------------------------------------------------------------------------------*/
/******************************* API functions for VTS *********************************/
/*-------------------------------------------------------------------------------------*/
#if ( 21 == WRAPPER_API_VERSION )
DEC_RETURN_DUT DLL_EXPORTS VideoDecInit( void **_ppDecObj, void *_psInitContxt)
#else
DEC_RETURN_DUT DLL_EXPORTS VideoDecInit_dut( void **_ppDecObj, DEC_INIT_CONTXT_DUT *_psInitContxt)
#endif

{
	printf("%s \n", WMV9DCodecVersionInfo());
	return WMV9DecInit(_ppDecObj, _psInitContxt, NULL);
}

T_DEC_RETURN_DUT DLL_EXPORTS VideoTestDecInit_dut( void **_ppDecObj, T_DEC_CONTXT_DUT *_psDecContxt)
{
	DEC_RETURN_DUT eDecRet;
	printf("%s \n", WMV9DCodecVersionInfo());
	eDecRet = WMV9DecInit(_ppDecObj, NULL, _psDecContxt);
	return (eDecRet == E_DEC_INIT_OK_DUT) ? T_DEC_INIT_OK_DUT : T_DEC_INIT_ERROR_DUT;
}

/*!
 ***********************************************************************
 * -Function:
 *    DEC_RETURN_DUT VideoDecFrame_dut( void * pDecObj, 
 *                                       void * pParam )
 *
 * -Description:
 *    This function calls the RealVideo decoder API to decode the video data in stream buffer.
 *    When it returns, it outputs one frame and store it to the buffer array if necessary.
 *
 * -Input Param
 *    *pDecObj                pointer of decoder object
 *    *pParam                 pointer of the structure contains input buffer infomation, 
 *                            or pointer of callback fuction which get video bitstreams.
 *
 *
 * -Output Param
 *    none
 *
 * -Return
 *    E_DEC_FRAME_DUT    finish output one or more frame
 *    E_DEC_FINISH_DUT   finish decoding all the bitstream
 *    E_DED_ALLOUT_DUT   all decoded frames have been output
 *    E_DEC_ERROR_DUT    encounter error during decoding
 ***********************************************************************
 */

//WMV9D_S32            bFlush1Frame = 1;

static DEC_RETURN_DUT WMV9DecFrame(void * _pDecObj, void * _pParam)
{
    WMV9_DUT_OBJECT *       psWMV9DutObj = _pDecObj;
 	T_DEC_CONTXT_DUT * psWmv9DecContxt = (T_DEC_CONTXT_DUT *)_pParam;
   sInputHandlerType     * psInHandler;
    WMV9D_S32               s32Buffer;
    WMV9D_S32               s32BytesRead;
    WMV9D_S32               s32DecodeCount;
    sWmv9DecObjectType    * psDecObj;
    sWmv9DecYCbCrBufferType* psFrameBuffer;
    int                     status = E_WMV9D_SUCCESS;
    DEC_RETURN_DUT         eRetVal = E_DEC_FRAME_DUT;

    assert( psWMV9DutObj != NULL );

    psInHandler = &psWMV9DutObj->sInHandler;
    psDecObj = &psWMV9DutObj->sDecObj;
    s32DecodeCount = psWMV9DutObj->s32DecodeCount;
    status = psWMV9DutObj->status;

    psFrameBuffer = &psDecObj->sDecParam.sOutputBuffer;


    if ((s32DecodeCount >= psInHandler->s32NumFramesToDecode) 
		&& (psInHandler->bFlush1Frame == 0))
    {
        psWMV9DutObj->status = E_WMV9D_SUCCESS;
        return E_DEC_ALLOUT_DUT;
		
    }

    s32BytesRead = GetRCVData ((WMV9D_U8*)&s32Buffer, 1, 4, psInHandler);
    //printf( "%s, %d\n", __FUNCTION__, __LINE__ );
#ifdef WMV9_SIMPLE_ONLY
    if (s32BytesRead <= 0)
    {
        CleanAppData (psInHandler);
        psWMV9DutObj->s32DecodeCount = s32DecodeCount;
        psWMV9DutObj->status = E_WMV9D_FAILED;
        return E_DEC_ERROR_DUT;
    }
#endif
    //ByteReverse ((WMV9D_U8*)&s32Buffer, 4, 1);
#ifndef WMV9_SIMPLE_ONLY
         if (s32BytesRead != 4)
        {
           psInHandler->s32TotalBytes = psInHandler->s32BytesLeft = 0;
           s32Buffer = 0;
        }
		 else
		 {
			if (psInHandler->s32RcvVersion == 1)
			{
				WMV9D_S32 s32TimeStamp;

				DebugAppText ("Key frame status %d", s32Buffer >> 31);
				s32Buffer &= 0x3fffffff;
				s32BytesRead = GetRCVData ((WMV9D_U8*)&s32TimeStamp, 4, 1,
									   psInHandler);
				if (s32BytesRead <= 0)
				{
					CleanAppData (psInHandler);
					psWMV9DutObj->s32DecodeCount = s32DecodeCount;
					psWMV9DutObj->status = E_WMV9D_FAILED;
					return E_DEC_ERROR_DUT;
				}
			}
		 }
#else
    // Read the next frame's extra data
    if (s32Buffer == 0)
    {
        CleanAppData (psInHandler);
        psWMV9DutObj->s32DecodeCount = s32DecodeCount;
        psWMV9DutObj->status = E_WMV9D_FAILED;
        return E_DEC_ERROR_DUT;
    }

    if (psInHandler->s32RcvVersion == 1)
    {
        WMV9D_S32 s32TimeStamp;

        DebugAppText ("Key frame status %d", s32Buffer >> 31);
        s32Buffer &= 0x3fffffff;
        s32BytesRead = GetRCVData ((WMV9D_U8*)&s32TimeStamp, 4, 1,
                               psInHandler);
        if (s32BytesRead <= 0)
        {
            CleanAppData (psInHandler);
            psWMV9DutObj->s32DecodeCount = s32DecodeCount;
            psWMV9DutObj->status = E_WMV9D_FAILED;
            return E_DEC_ERROR_DUT;
        }
    }
#endif
    /* Set the read parameters in the input handler */
    psInHandler->s32TotalBytes = psInHandler->s32BytesLeft = s32Buffer;
		STACK_TAG();
		DEC_TIMER_BEGIN();

#ifdef WMV9_SIMPLE_ONLY
           status =  eWMV9DDecode (psDecObj, s32Buffer);
#else
       if (s32Buffer != 0){
           status =  eWMV9DDecode (psDecObj, s32Buffer);
       }
       else {
           psInHandler->bFlush1Frame = 0;
           status =  eWMV9DDecode (psDecObj, 0);
       }
#endif
        DEC_TIMER_END();
        STACK_UPDATE();

	if(status==E_WMV9D_SKIPPED_FRAME)
	{
		//skip left bytes of current frame
		//printf("skipped %d bytes  \r\n",psInHandler->s32BytesLeft);
		fseek(psInHandler->pInFile,psInHandler->s32BytesLeft,SEEK_CUR);
		goto end_frame;
	}

#ifdef OUTPUT_BUFFER_CHANGES
	 //if(status == E_WMV9D_SUCCESS)
	 {
		 STACK_TAG();
		status = eWMV9DecGetOutputFrame(psDecObj);
		STACK_UPDATE();
	 }
#endif
#ifdef WMV9_SIMPLE_ONLY
    if (status != E_WMV9D_SUCCESS) 
    {
        printf ("Frame number %d failed for decoding, status %d\n",
                s32DecodeCount, status);
        CleanAppData (psInHandler);
        psWMV9DutObj->s32DecodeCount = s32DecodeCount;
        psWMV9DutObj->status = E_WMV9D_FAILED;
        return E_DEC_ERROR_DUT;
    }
#else
    if ((status != E_WMV9D_SUCCESS) && (status != E_WMV9D_NO_OUTPUT))
    {
        printf ("Frame number %d failed for decoding, status %d\n",
                s32DecodeCount, status);
        CleanAppData (psInHandler);
        psWMV9DutObj->s32DecodeCount = s32DecodeCount;
        psWMV9DutObj->status = E_WMV9D_FAILED;
        return E_DEC_ERROR_DUT;
    }
#endif

    /* Larry 09252007 : if the first I frame is output twice, drop the first one */
    if ( ( psInHandler->gs32NumBFrames > 0 ) && ( s32DecodeCount == 0 ) )
    {
        /* do nothing here, just skip the output code */
        psInHandler->gs32NumBFrames = psInHandler->gs32NumBFrames;
    }
    /* ~Larry 09252007 */
    else if  ((s32Buffer != 0) || (status != E_WMV9D_NO_OUTPUT))
    {
        if(psWMV9DutObj->fnPrbDutFunc)
        {
        	T_PROBE_PUT_FRAME prbFrm;

			prbFrm.puchLumY = psFrameBuffer->pu8YBuf;
			prbFrm.puchChrU = psFrameBuffer->pu8CbBuf;
			prbFrm.puchChrV = psFrameBuffer->pu8CrBuf;
			prbFrm.iFrmWidth = psInHandler->s32Width;
			prbFrm.iFrmHeight = psInHandler->s32Height;
			prbFrm.iStrideLX = psFrameBuffer->s32YRowSize;
			prbFrm.iStrideLY = psInHandler->s32Height + tmp_EXPANDY * 2;
			prbFrm.iStrideUV = psFrameBuffer->s32CbRowSize;
			prbFrm.iTopOffset = tmp_EXPANDY;
			prbFrm.iLeftOffset = tmp_EXPANDY;//tmp_EXPANDUV;
			
            psWMV9DutObj->fnPrbDutFunc(T_PUT_FRAME, (void*)&prbFrm);
        }
        /* store output frame here */
#if ( 21 == WRAPPER_API_VERSION )
        if ( psInHandler->gpfnVtsProbe )
        {
            FRAME_COPY_INFO sFrmCopy;
            sFrmCopy.puchLumY = psFrameBuffer->pu8YBuf + ((tmp_EXPANDY * psFrameBuffer->s32YRowSize) + tmp_EXPANDY);
            sFrmCopy.puchChrU = psFrameBuffer->pu8CbBuf + ((tmp_EXPANDUV * psFrameBuffer->s32CbRowSize) + tmp_EXPANDUV);
            sFrmCopy.puchChrV = psFrameBuffer->pu8CrBuf + ((tmp_EXPANDUV * psFrameBuffer->s32CrRowSize) + tmp_EXPANDUV);
            sFrmCopy.iFrmWidth = psInHandler->s32Width;
            sFrmCopy.iFrmHeight = psInHandler->s32Height;
            sFrmCopy.iBufStrideY = psFrameBuffer->s32YRowSize;
            sFrmCopy.iBufStrideUV = psFrameBuffer->s32CbRowSize;
            psInHandler->gpfnVtsProbe( E_OUTPUT_FRAME, &sFrmCopy );
        }
#else
        if ( psInHandler->sVTSProbes.pfStoreDispFrm != NULL )
        {
			char* local_output_bufferY;
			char* local_output_bufferU;
			char* local_output_bufferV;
			
			unsigned int  u32PadSize;        
 			u32PadSize = tmp_EXPANDY;//(psFrameBuffer->s32YRowSize- psDecObj->sDecParam.u16FrameWidth)/2;
			local_output_bufferY= psFrameBuffer->pu8YBuf + ((u32PadSize * psFrameBuffer->s32YRowSize) + u32PadSize);
			
			u32PadSize = tmp_EXPANDUV;//(psFrameBuffer->s32CbRowSize- (psDecObj->sDecParam.u16FrameWidth>> 1))/2;
			local_output_bufferU = psFrameBuffer->pu8CbBuf + ((u32PadSize * psFrameBuffer->s32CbRowSize) + u32PadSize);
			
			u32PadSize = tmp_EXPANDUV;//(psFrameBuffer->s32CrRowSize- (psDecObj->sDecParam.u16FrameWidth>> 1))/2;
			local_output_bufferV =psFrameBuffer->pu8CrBuf + ((u32PadSize * psFrameBuffer->s32CrRowSize) + u32PadSize);

            psInHandler->sVTSProbes.pfStoreDispFrm ( local_output_bufferY, 
                                        local_output_bufferU, 
                                        local_output_bufferV, 
                                        psInHandler->s32Width, 
                                        psInHandler->s32Height, 
                                        psFrameBuffer->s32YRowSize, 
                                        psFrameBuffer->s32CbRowSize );
        }
        /* record output time here */
        if ( psInHandler->sVTSProbes.pfStoreFrmOutTime != NULL )
        {
            psInHandler->sVTSProbes.pfStoreFrmOutTime( );
        }
#endif
    }

end_frame:
	
    s32DecodeCount++;

	if(psWmv9DecContxt)psWmv9DecContxt->uiFrameNum = s32DecodeCount;
   
    psWMV9DutObj->s32DecodeCount = s32DecodeCount;
	if(psWMV9DutObj->s32DecodeCount==1)
	{
		HEAP_INCREASE(psWMV9DutObj->totalmemorysize);
	}
    psWMV9DutObj->status = status;

    return eRetVal;
}

#if ( 21 == WRAPPER_API_VERSION )
DEC_RETURN_DUT DLL_EXPORTS VideoDecRun( void * _pDecObj, void * _pParam)
{
    DEC_RETURN_DUT eRetVal = E_DEC_FRAME_DUT;
    
    while ( E_DEC_FRAME_DUT == eRetVal )
    {
        eRetVal = WMV9DecFrame(_pDecObj, NULL);
    }
    
	return eRetVal;
}
#else
DEC_RETURN_DUT DLL_EXPORTS VideoDecFrame_dut( void * _pDecObj, void * _pParam)
{
	return WMV9DecFrame(_pDecObj, NULL);
}
#endif

T_DEC_RETURN_DUT DLL_EXPORTS VideoTestDecFrame_dut( void * _pDecObj,T_DEC_CONTXT_DUT * _psDecContxt)
{
	DEC_RETURN_DUT eDecRet;

	eDecRet = WMV9DecFrame(_pDecObj, _psDecContxt);
	
	switch(eDecRet)
	{
	case E_DEC_FRAME_DUT:
		return T_DEC_FRAME_DUT;
		break;
	case E_DEC_FINISH_DUT:
		return T_DEC_FINISH_DUT;
		break;
	case E_DEC_ALLOUT_DUT:
		return T_DEC_ALLOUT_DUT;
		break;
	default:
		return T_DEC_FINISH_DUT;
		break;
	}
}

/*!
 ***********************************************************************
 * -Function:
 *  DEC_RETURN_DUT DLL_EXPORTS VideoDecRelease_dut( void * _pDecObj );
 *
 * -Description:
 *    This function releases all the memory used by RealVideo DUT, and exit the decoder.
 *
 * -Input Param
 *    *pDecObj                pointer of decoder object
 *
 * -Output Param
 *    none
 *
 * -Return
 *    none
 ***********************************************************************
 */
static void WMV9DecRelease( void * _pDecObj )
{
    WMV9_DUT_OBJECT *       psWMV9DutObj = _pDecObj;
    sWmv9DecObjectType    * psDecObj;

    assert( psWMV9DutObj != NULL );
    psDecObj = &psWMV9DutObj->sDecObj;


   
    /* Close the decoder */
    if ( psWMV9DutObj->status != E_WMV9D_FAILED )
    {
		STACK_TAG();
        eWMV9DFree ( &psWMV9DutObj->sDecObj );
		STACK_UPDATE();
    }
    
    CleanAppData ( &psWMV9DutObj->sInHandler );

	//MAD Shanghai Direct Rendering
	releaseBuffers(&(psWMV9DutObj->sInHandler));
    
    free( psWMV9DutObj );
}

#if ( 21 == WRAPPER_API_VERSION )
DEC_RETURN_DUT DLL_EXPORTS VideoDecRelease( void * _pDecObj )
#else
DEC_RETURN_DUT DLL_EXPORTS VideoDecRelease_dut( void * _pDecObj )
#endif
{	
	WMV9DecRelease(_pDecObj);
	
    return E_DEC_REL_OK_DUT;
}

T_DEC_RETURN_DUT DLL_EXPORTS VideoTestDecRelease_dut( void * _pDecObj, T_DEC_CONTXT_DUT * _psDecContxt )
{	
	WMV9DecRelease(_pDecObj);
	
    return T_DEC_REL_OK_DUT;
}
/*!
 ***********************************************************************
 * -Function:
 *   DEC_RETURN_DUT DLL_EXPORTS VideoDecSetProbes_dut( void *			_pDecObj, 
												   VTS_PROBES_DUT * _pProbe )
 *
 * -Description:
 *    This function set the callback function pointers used by RealVideo DUT.
 *
 * -Input Param
 *    *pDecObj                pointer of decoder object
 *    *pProbe                 pointer of callback function probes structure
 *                            for RealVideo DUT, it is forced to WMV9_CONFTEST_PROBES_DUT
 *
 * -Output Param
 *    none
 *
 * -Return
 *    none
 ***********************************************************************
 */
#if ( 21 > WRAPPER_API_VERSION )
DEC_RETURN_DUT DLL_EXPORTS VideoDecSetProbes_dut(void *	_pDecObj, 
													   VTS_PROBES_DUT * _pProbe )

{
    WMV9_DUT_OBJECT * psWMV9DutObj =(WMV9_DUT_OBJECT *)_pDecObj;
    memcpy( &(psWMV9DutObj->sInHandler.sVTSProbes), _pProbe, sizeof(VTS_PROBES_DUT) );

	return E_SET_PROB_OK_DUT;
}
#endif

T_DEC_RETURN_DUT DLL_EXPORTS VideoTestDecSetProbes_dut( void * _pDecObj, PROBE_DUT_FUNC * _pProbe )
{
	WMV9_DUT_OBJECT * psWMV9DutObj =(WMV9_DUT_OBJECT *)_pDecObj;
	
	psWMV9DutObj->fnPrbDutFunc = _pProbe;
	
	return T_SET_PROB_OK_DUT;
}

#if ( 21 == WRAPPER_API_VERSION )
DEC_RETURN_DUT DLL_EXPORTS QueryAPIVersion( long * _piAPIVersion )
{
    *_piAPIVersion = WRAPPER_API_VERSION;
    
    return E_GET_VER_OK_DUT;
}
#endif

