// GENPO - the GENeral Purpose Organ
// Copyright (C) 2003,2004,2007 - Steve Merrony 

/* This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/******************************************************************************
 ** midiThread - this thread handles all the MIDI input from the keyboard(s)
 ******************************************************************************/

#include <pthread.h>
#include <alsa/asoundlib.h>

#include <qevent.h>

#include "console.h"
#include "customEvents.h"
#include "midiThread.h"
#include "pistonEventData.h"
#include "subscriberThread.h"
    
#include <iostream>
using namespace std;

MidiThread::MidiThread( volatile Console *shared_console ) {

  sconsole = shared_console;
  if (sconsole->verboseMode) cout << "MIDI Thread starting up\n";

}

MidiThread::~MidiThread() {

  cout << "Destroying MIDI thread\n";
  pthread_exit( NULL );

}

void connectDefaultPorts( midi_seq *seq, midi_port *ip, midi_port *op_a, midi_port *op_b, Console *tconsole ) {
  
  // scan all the other MIDI ports and link all readable ones to our input
  // and first writeable one to our output
  
  snd_seq_client_info_t *cinfo;
  snd_seq_port_info_t *pinfo;
  snd_seq_port_subscribe_t *subs;
  snd_seq_addr_t addr;
  int err;

  snd_seq_client_info_alloca( &cinfo );
  snd_seq_port_info_alloca( &pinfo );
  snd_seq_client_info_set_client( cinfo, -1 );
  while (snd_seq_query_next_client( seq->seq_handle, cinfo ) >= 0) {
    if (snd_seq_client_info_get_client( cinfo ) != seq->my_client) { // ignore our own port!
      /* reset query info */
      snd_seq_port_info_set_client( pinfo, snd_seq_client_info_get_client( cinfo ) );
      snd_seq_port_info_set_port( pinfo, -1 );

      while (snd_seq_query_next_port( seq->seq_handle, pinfo ) >= 0) {

	// Is this an output port we can send MIDI to?

	if ( ((snd_seq_port_info_get_capability( pinfo ) & (SND_SEQ_PORT_CAP_WRITE|SND_SEQ_PORT_CAP_SUBS_WRITE))
		      == (SND_SEQ_PORT_CAP_WRITE|SND_SEQ_PORT_CAP_SUBS_WRITE)) 
		      &&  // only interested in synthesizer devices (hardware or software)
		      ((snd_seq_port_info_get_type( pinfo ) & (SND_SEQ_PORT_TYPE_SYNTHESIZER)) 
		      == (SND_SEQ_PORT_TYPE_SYNTHESIZER))
		      || // or the user has specified this port
		      (tconsole->requestedOutput && 
		       (tconsole->requestedOutClient == snd_seq_client_info_get_client( cinfo )) 
		       && 
		      ((tconsole->requestedOutPort ==  snd_seq_port_info_get_port( pinfo )) ||
		      (tconsole->requestedOutPort + 1 ==  snd_seq_port_info_get_port( pinfo ) ))
		      )
	   ) {
	  if (tconsole->verboseMode) cout << "Output port: "
		<< snd_seq_client_info_get_client( cinfo ) << ":"
		<< snd_seq_port_info_get_port( pinfo ) << " "
		<< snd_seq_port_info_get_name( pinfo )
		<< "(" << snd_seq_port_info_get_midi_channels( pinfo ) << " channels) ";

	  if ((tconsole->num_out_ports < MAX_MIDI_OUTPUT_PORTS)
		                && ((!tconsole->requestedOutput) || 
			            (tconsole->requestedOutput && 
				     (tconsole->requestedOutClient == snd_seq_client_info_get_client( cinfo )) 
				    )
				   )
	     ) {
	    snd_seq_port_subscribe_alloca( &subs );
	    addr.client = seq->my_client;
	    switch (tconsole->num_out_ports) {
	      case 0: addr.port = op_a->my_port; break;
	      case 1: addr.port = op_b->my_port; break;
	    }
	    snd_seq_port_subscribe_set_sender(subs, &addr);
	    addr.client = snd_seq_client_info_get_client( cinfo );
	    addr.port = snd_seq_port_info_get_port( pinfo );
	    snd_seq_port_subscribe_set_dest(subs, &addr);
	    snd_seq_port_subscribe_set_queue(subs, 1);
	    snd_seq_port_subscribe_set_time_update(subs, 1);
	    snd_seq_port_subscribe_set_time_real(subs, 1);
	    if ((err = snd_seq_subscribe_port(seq->seq_handle, subs))<0) {
	      cerr << "Cannot subscribe announce port: " << snd_strerror(err) << endl;
	    }
	    else {
	      tconsole->num_out_ports++;
	      if (tconsole->verboseMode) cout << " - Connecting";
	    }
	  }
	  if (tconsole->verboseMode) cout << endl;
	   }

	// Is this an input port we can read MIDI from?

	   if ( ((snd_seq_port_info_get_capability( pinfo ) & (SND_SEQ_PORT_CAP_READ|SND_SEQ_PORT_CAP_SUBS_READ)) == (SND_SEQ_PORT_CAP_READ|SND_SEQ_PORT_CAP_SUBS_READ))
			 &&  // only interested in real MIDI devices
			 ((snd_seq_port_info_get_type( pinfo ) & (SND_SEQ_PORT_TYPE_MIDI_GENERIC)) == (SND_SEQ_PORT_TYPE_MIDI_GENERIC))
	      ) {
	     if (tconsole->verboseMode) cout << "Input port: "
		   << snd_seq_client_info_get_client( cinfo ) << ":"
		   << snd_seq_port_info_get_port( pinfo ) << " "
		   << snd_seq_port_info_get_name( pinfo )
		   << "(" << snd_seq_port_info_get_midi_channels( pinfo ) << " channels) - Connecting\n";
	  // link all input ports to our port...
	     snd_seq_port_subscribe_alloca( &subs );
	     addr.client = snd_seq_client_info_get_client( cinfo );
	     addr.port = snd_seq_port_info_get_port( pinfo );
	     snd_seq_port_subscribe_set_sender(subs, &addr);
	     addr.client = seq->my_client;
	     addr.port = ip->my_port;
	     snd_seq_port_subscribe_set_dest(subs, &addr);
	     snd_seq_port_subscribe_set_queue(subs, 1);
	     snd_seq_port_subscribe_set_time_update(subs, 1);
	     snd_seq_port_subscribe_set_time_real(subs, 1);
	     if ((err = snd_seq_subscribe_port(seq->seq_handle, subs))<0) {
	       cerr << "Cannot subscribe announce port: " << snd_strerror(err) << endl;
	     }

	      }
      } // end while each port
    } // end-if
  } // end while each client
 
}

static void* run(void* mops) {

  int res;

  Console *tconsole = (Console *)mops;

  midi_seq *seq;
  midi_port *ip, *op_a, *op_b;
  seq = &(tconsole->seq);
  ip = &(tconsole->inp_port);
  op_a = &(tconsole->out_port_a);
  op_b = &(tconsole->out_port_b);
  
  // Create the ALSA input port via which GENPO will receive MIDI 
  
  // open the ALSA device
  res = snd_seq_open(&seq->seq_handle, "hw", SND_SEQ_OPEN_INPUT|SND_SEQ_OPEN_OUTPUT, 0);
  snd_seq_set_client_name(seq->seq_handle, "GENPO");
  // get a port
  seq->my_client = snd_seq_client_id(seq->seq_handle);

  // create a new ALSA midi input port on our device
  ip->my_port = snd_seq_create_simple_port(seq->seq_handle, "GENPO Input",
                                       SND_SEQ_PORT_CAP_WRITE |
                                       SND_SEQ_PORT_CAP_SUBS_WRITE 
                                       // SND_SEQ_PORT_CAP_READ,
					   ,
                                       SND_SEQ_PORT_TYPE_MIDI_GENERIC |
                                       SND_SEQ_PORT_TYPE_APPLICATION); 
  if (ip->my_port < 0) {
    perror("create input port");
    exit(1);
  }

  
  // Create the ALSA output ports thru which GENPO sends all MIDI output
  
    // create a new ALSA midi output port
    op_a->my_port = snd_seq_create_simple_port(seq->seq_handle, "GENPO Output A",
                                       SND_SEQ_PORT_CAP_SUBS_READ |
                                       SND_SEQ_PORT_CAP_WRITE | SND_SEQ_PORT_CAP_READ,
                                       SND_SEQ_PORT_TYPE_MIDI_GENERIC |
                                       SND_SEQ_PORT_TYPE_APPLICATION);
    if (op_a->my_port < 0) {
      perror("create output port");
      exit(1);
    }
    
    // create another new ALSA midi output port
    op_b->my_port = snd_seq_create_simple_port(seq->seq_handle, "GENPO Output B",
					       SND_SEQ_PORT_CAP_SUBS_READ |
						SND_SEQ_PORT_CAP_WRITE | SND_SEQ_PORT_CAP_READ,
						SND_SEQ_PORT_TYPE_MIDI_GENERIC |
						SND_SEQ_PORT_TYPE_APPLICATION);
    if (op_b->my_port < 0) {
      perror("create output port");
      exit(1);
    }
    
  connectDefaultPorts( seq, ip, op_a, op_b, tconsole );
  
  //SubscriberThread *sThread = new SubscriberThread( tconsole );
  //sThread->start();
  
  snd_seq_event_t *ev;
  unsigned char saved_note; // use this to avoid type-conversions to MIDI types
  int src_manual=0, src_chan;
  PistonEventData pe;

  /* Do not set any runtime variables dependent upon the organ definition before the next loop.
     They may change as the organ definition is loaded - this may not have been completed by 
     this point!
  */


  ///////////////////  the main note-handling loop //////////////////
  // REMEMBER - THIS LOOP IS TIME-CRITICAL!!!
  while (TRUE) {

    if (snd_seq_event_input( seq->seq_handle, &ev ) < 0) {
      cout << "Error fetching MIDI event - notes may have been lost\n";
    }
    
    switch (ev->type) {

    // if it's a controller event, but not one of the type we care about,
    // then we just throw it away (otherwise it's processed like a note)
    case SND_SEQ_EVENT_CONTROLLER: // too wide-ranging
      if (!((ev->data.control.param == 4) ||  // foot controller
	    (ev->data.control.param == 7) ||  // Volume
	    (ev->data.control.param == 11)))  // Expression
	break;

    case SND_SEQ_EVENT_NOTEON:
    case SND_SEQ_EVENT_NOTEOFF:
      // case SND_SEQ_EVENT_CHANPRESS:

      snd_seq_ev_set_subs( ev );
      snd_seq_ev_set_direct( ev );

      src_chan = ev->data.note.channel;
      src_manual = tconsole->channel_division_map[src_chan]; 
      if (src_manual != FREE_CHANNEL) {
      // send note to all active voices for this manual
      for ( int v = 0; v < MAX_MIDI_VOICES; v++ ) {
	if ( tconsole->voices[v].active && (tconsole->voices[v].manual_channel==src_chan) ) {
	  // adjust velocity if not a note-off
	  if (ev->data.note.velocity!=0) ev->data.note.velocity = tconsole->voices[v].velocity;
	  if (v < MIDI_CHANNELS_PER_PORT) {  //on first synth
	    ev->data.note.channel = v; 
	    snd_seq_ev_set_source( ev, op_a->my_port );
	  }
	  else { // on second synth
	    ev->data.note.channel = v - MIDI_CHANNELS_PER_PORT; 
	    snd_seq_ev_set_source( ev, op_b->my_port );
	  }
	  snd_seq_event_output_direct( seq->seq_handle, ev );
	}
      }

      // send note to all coupled active voices for this manual
      saved_note = ev->data.note.note;
      for (int cplr = 0; cplr < tconsole->divisions[src_manual].num_couplers; cplr++) {
	if (tconsole->divisions[src_manual].couplers[cplr].active) {
	  for ( int v = 0; v < MAX_MIDI_VOICES; v++ ) {
	    if ( tconsole->voices[v].active && 
		 (tconsole->voices[v].manual_channel==tconsole->divisions[src_manual].couplers[cplr].toChannel) ) {
	      // adjust velocity if not a note-off
	      if (ev->data.note.velocity!=0) ev->data.note.velocity = tconsole->voices[v].velocity;
	      switch (tconsole->divisions[src_manual].couplers[cplr].length) {
	      case 16:
		ev->data.note.note -= 12;
		break;
	      case 4:
		ev->data.note.note += 12;
		break;
		//default:
	      }
	      if (v < MIDI_CHANNELS_PER_PORT) { // first synth
		ev->data.note.channel = v; 
		snd_seq_ev_set_source( ev, op_a->my_port );
	      }
	      else {  // second synth
		ev->data.note.channel = v - MIDI_CHANNELS_PER_PORT; 
		snd_seq_ev_set_source( ev, op_b->my_port );
	      }
	      snd_seq_event_output_direct( seq->seq_handle, ev );
	      ev->data.note.note = saved_note;
	    } //end-if
	  } //end-for
	} //end-if
      } //end-for
      
      } //end-if

      break;

    case SND_SEQ_EVENT_PGMCHANGE:  { // This is intended to handle Preset changes coming from a MIDI keyboard
      // OK - we're not going to reinvent the wheel here, we'll simply send a special event for 
      // the appropriate Preset to the main application
      
      int man = tconsole->channel_division_map[ev->data.control.channel];
      if (tconsole->verboseMode) cout << "PgmChange received from ALSA channel: " << (int) ev->data.control.channel << ", decodes to manual: " << man << endl;
      if (man != FREE_CHANNEL) { // only process if it's from one of our divisions
	if (tconsole->verboseMode) cout << " Piston " << tconsole->divisions[man].pistons[ev->data.control.value].label << endl;

	pe.pe_manual = man;
	pe.pe_piston = ev->data.control.value;
 
	QCustomEvent* me = new QCustomEvent( PRESET_CHANGE_EVENT );
	me->setData( (void *)&pe );
	QApplication::sendEvent( qApp->mainWidget(), me );
      }
      else {
	// check for ToePistons
	int realChannel = ev->data.control.channel + 1; // ALSA channel numbers start at 0...
	for (int tp = 0; tp < tconsole->num_toePistons; tp++) {
	  if (realChannel == tconsole->toePistons[tp].midiChannel) {
	    pe.pe_manual = TOE_PISTON;
	    pe.pe_piston = ev->data.control.value;
	    
	    QCustomEvent* me = new QCustomEvent( PRESET_CHANGE_EVENT );
	    me->setData( (void *)&pe );
	    QApplication::sendEvent( qApp->mainWidget(), me );
	  }
	}
      }
      break;
      }
      case SND_SEQ_EVENT_PORT_SUBSCRIBED: {
	tconsole->num_out_ports++;
	if (tconsole->verboseMode) {
	  int dest = ev->data.connect.dest.client;
	  int port = ev->data.connect.dest.port;
	  cout << "MIDI port subscribed " << dest << ":" << port << endl;
	}
	break;}
      case SND_SEQ_EVENT_PORT_UNSUBSCRIBED: {
	tconsole->num_out_ports--;
	if (tconsole->verboseMode) {
	  int dest = ev->data.connect.dest.client;
	  cout << "MIDI port unsubscribed " << dest << endl;
	}
	break;}
	
      // Everything else is thrown away.
      //default:
      
    } // end-switch

    snd_seq_free_event( ev );
  }

  return 0;
}

void MidiThread::start() {
  
  pthread_attr_t* attributes = 0;

  if (pthread_create(&thread, attributes, ::run, (void *)sconsole ))
    perror("creating thread failed:");
}

void MidiThread::stop(bool force = TRUE) {

  if (force) {
    pthread_cancel(thread);
  }
  
}




