------------------------------------------------------------------------------
--  Thin Ada95 binding to OCI (Oracle Call Interface)                    --
--  Copyright (C) 2000-2004 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-lobs.ads,v 1.9 2004/06/28 04:57:05 vagul Exp $

with Ada.Streams;
with Ada.Strings.Unbounded;
with OCI.Thick.Connections;
with System.Address_To_Access_Conversions;

package OCI.Thick.Lobs is

   use Ada.Streams;
   use Connections;

   type Lob_Type is (Bin, File, Char, NChar);

   type Lob is new Handle_Reference with private;

   -------------------------------
   -- Ada like stream interface --
   -------------------------------

   type Lob_Stream is new Root_Stream_Type with private;

   procedure Stream (Stream : in out Lob_Stream'Class; Loc : in Lob);

   procedure Read
     (Stream : in out Lob_Stream;
      Item   :    out Stream_Element_Array;
      Last   :    out Stream_Element_Offset);

   procedure Write
     (Stream : in out Lob_Stream;
      Item   : in     Stream_Element_Array);

   function End_Of_File (Stream : in Lob_Stream) return Boolean;
   --  Use Last < Item'Last condition after Read call for better performance.

   --------------------------------
   -- Oracle like lob interface. --
   --------------------------------

   function Create_Temporary
     (Connect : in Connection;
      LType   : in Lob_Type := Char)
      return Lob;

   function Create_Empty
     (Connect : in Connection;
      LType   : in Lob_Type := Char)
      return Lob;

   procedure Create_Temporary
     (Loc   : in out Lob;
      Cache : in Boolean := True);

   procedure Free_Temporary (Loc : in out Lob);

   type Count is new Natural;

   subtype Positive_Count is Count range 1 .. Count'Last;

   subtype Raw is Ada.Streams.Stream_Element_Array;
   subtype Raw_Offset is Ada.Streams.Stream_Element_Offset;

   procedure Read
     (Loc    :  in     Lob;
      Offset :  in     Positive_Count;
      Buffer :     out Raw;
      Last   :     out Raw_Offset);

   procedure Read
     (Loc    : in     Lob;
      Offset : in     Positive_Count;
      Buffer :    out String;
      Last   :    out Integer);

   procedure Write
     (Loc    : in out Lob;
      Offset : in     Positive_Count;
      Buffer : in     Raw);

   procedure Write
     (Loc    : in out Lob;
      Offset : in     Positive_Count;
      Buffer : in     String);

   function Length (Loc : in Lob) return Count;

   function To_String (Loc : in Lob) return String;

   function To_Unbounded_String
     (Loc : in Lob)
      return Ada.Strings.Unbounded.Unbounded_String;

   function Is_Init (Value : in Lob) return Boolean;
   function Is_Open (Value : in Lob) return Boolean;

   function Is_Temporary (Value : in Lob) return Boolean;

   procedure Create_Temporary
      (Stream  : in out Lob_Stream;
       Connect : in     Connection;
       LType   : in     Lob_Type := Char);

   procedure Free_Temporary (Stream : in out Lob_Stream);

   function To_String (Loc : in Lob_Stream) return String;

   procedure Reset (Stream : in out Lob_Stream);

   function Get_Lob_Type (Loc : Lob) return Lob_Type;

   function Get_Bind_Address (Loc : Lob) return DVoid;
   function Get_Bind_Indicator (Loc : Lob) return A_Sb2;

private

   use Reference_Finalization;

   pragma Inline (To_String);

   package AC is new System.Address_To_Access_Conversions (OCIHandle);

   type Lob is new Handle_Reference with record
      Connect : Connection;
      LType   : Lob_Type;
      Ind     : A_Sb2;
      Locp    : AC.Object_Pointer := null;
   end record;

   procedure Destroy (Object : in out Lob);

   type Lob_Stream is new Root_Stream_Type with record
     Loc : Lob;
     Position : Positive_Count := Positive_Count'First;
   end record;

end OCI.Thick.Lobs;
