------------------------------------------------------------------------------
--  Thin Ada95 binding to OCI (Oracle Call Interface)                    --
--  Copyright (C) 2000-2004 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: dyn_bind.adb,v 1.6 2004/06/28 04:57:06 vagul Exp $

--  Example/test of using dynamically allocated bind variables.

with Ada.Strings.Fixed;
with Ada.Text_IO;

with OCI.Thick.Connections;
with OCI.Thick.Statements;
with OCI.Thick.Strings;

procedure Dyn_Bind is

   use OCI.Thick;
   use OCI.Thick.Connections;
   use OCI.Thick.Statements;
   use OCI.Thick.Strings;
   use Ada.Strings.Fixed;

   Connect : Connection := Logon ("scott/tiger");
   Stmt : Statement := Prepare (Connect,
       "declare Dummy Varchar2 (64) := :Result;"
       & " begin if Length (Dummy) > 7 then :Result := :Result; end if;"
       & " end;");

   Stmt0 : Statement := Prepare (Connect, "begin :Result := '1234567'; end;");

   Stmt1 : Statement := Prepare (Connect,
       "begin "
       & " :Out1 := :In1;"
       & " :Out2 := :In2;"
       & " :Out3 := :In3;"
       & " end;");

   Stmt2 : Statement := Prepare (Connect,
       "begin "
       & " :Out1 := :In1;"
       & " :Out2 := :In2;"
       & " :Out3 := :In3;"
       & " if :Out3 <> :In3 then"
       & " :In3 := :Out3;"
       & " end if;"
       & " end;");

   Stmt3 : Statement := Prepare (Connect,
       "begin "
       & " :Out1 := :In1;"
       & " :Out2 := :In2;"
       & " :Out3 := :In3;"
       & " if :Out1 <> :In1 then"
       & " :In1 := :Out1;"
       & " end if;"
       & " end;");

   Result, In1, In2, In3, Out1, Out2, Out3, Order : Var_Type;

   procedure Show_Result;

   procedure Show_In_Out;

   procedure Show_Out;

   procedure Show_Out_Null;

   procedure Bind_In_Out (Stmt : in out Statement);

   function Three_Columns (C1, C2, C3 : String) return String;

   -----------------
   -- Bind_In_Out --
   -----------------

   procedure Bind_In_Out (Stmt : in out Statement) is
   begin
      Bind (Stmt, In1, "In1");
      Bind (Stmt, In2, "In2");
      Bind (Stmt, In3, "In3");
      Bind (Stmt, Out1, "Out1");
      Bind (Stmt, Out2, "Out2");
      Bind (Stmt, Out3, "Out3");
   end Bind_In_Out;

   -----------------
   -- Show_Result --
   -----------------

   procedure Show_Result is
   begin
      if Is_Null (Result) then
         Ada.Text_IO.Put_Line ("Result is null");
      else
         Ada.Text_IO.Put_Line ("Result " & Value (Result) & ';');
      end if;
   end Show_Result;

   --------------
   -- Show_Out --
   --------------

   procedure Show_Out is
   begin
      Ada.Text_IO.Put (Value (Out1) & ' ');
      Ada.Text_IO.Put (Value (Out2) & ' ');
      Ada.Text_IO.Put (Value (Out3) & ' ');
      Ada.Text_IO.New_Line;
   end Show_Out;

   procedure Show_Out_Null is
   begin
      Ada.Text_IO.Put (Value (Out1, "(null)") & ' ');
      Ada.Text_IO.Put (Value (Out2, "(null)") & ' ');
      Ada.Text_IO.Put (Value (Out3, "(null)") & ' ');
      Ada.Text_IO.New_Line;
   end Show_Out_Null;

   -----------------
   -- Show_In_Out --
   -----------------

   procedure Show_In_Out is
   begin
      Ada.Text_IO.Put (Value (In1) & ' ');
      Ada.Text_IO.Put (Value (In2) & ' ');
      Ada.Text_IO.Put (Value (In3) & ' ');
      Ada.Text_IO.New_Line;

      Show_Out;
   end Show_In_Out;

   -------------------
   -- Three_Columns --
   -------------------

   function Three_Columns (C1, C2, C3 : String) return String is
   begin
      return "select '"
             & C1 & "' C1, '"
             & C2 & "' C2, '"
             & C3 & "' C3 from dual" & ASCII.Lf;
   end Three_Columns;

   Stmt4 : Statement := Prepare (Connect,
     Three_Columns ("", "1", "22")
     & "union all" & ASCII.Lf
     & Three_Columns ("22", "1", "")
     & "union all" & ASCII.Lf
     & Three_Columns ("333", "22", "1")
     & "union all" & ASCII.Lf
     & Three_Columns ("1", "22", "333")
     & "union all" & ASCII.Lf
     & Three_Columns ((1 .. 64 => '=', 65 => '!'), "", "")
     & "union all" & ASCII.Lf
     & Three_Columns ("", (1 .. 62 => '_', 63 => '^'), "")
     & "union all" & ASCII.Lf
     & Three_Columns ("1", "22", (1 .. 63 => '-', 64 => '.'))
     & "union all" & ASCII.Lf
     & Three_Columns ("aaaa", "bbb", 50 * (1 => ASCII.LF,
                                           2 .. 79 => '~',
                                           80 => '.'))
     );

   Stmt5 : Statement := Prepare (Connect,
       Three_Columns ("first", "second", "third")
     & "union all" & ASCII.Lf
     & Three_Columns ("first", "second", "third")
     & "union all" & ASCII.Lf
     & Three_Columns ("first", "second", "third.")
     & "union all" & ASCII.Lf
     & Three_Columns ("first", "second.", "third")
     & "union all" & ASCII.Lf
     & Three_Columns ("first.", "second", "third")
     & "union all" & ASCII.Lf
     & Three_Columns ("first", "second", "third.")
     & "union all" & ASCII.Lf
     & Three_Columns ("first", "second.", "third")
     & "union all" & ASCII.Lf
     & Three_Columns ("first.", "second", "third"));

begin
   Bind (Stmt0, Result, "Result");
   Execute (Stmt0);
   Show_Result;

   Bind (Stmt, Result, "Result");

   Ada.Text_IO.Put_Line ("binded");

   Set_Value (Result, 2 * "aBcDeFgH." & "=_");

   Ada.Text_IO.Put_Line ("execute");
   Execute (Stmt);
   Ada.Text_IO.Put_Line ("done");
   Show_Result;

   Set_Value (Result, "12345");

   Ada.Text_IO.Put_Line ("execute");
   Execute (Stmt);
   Ada.Text_IO.Put_Line ("done");
   Show_Result;

   Clear_Value (Result);
   Ada.Text_IO.Put_Line ("execute");
   Execute (Stmt);
   Ada.Text_IO.Put_Line ("done");
   Show_Result;

   Bind_In_Out (Stmt1);

   Set_Value (In1, "11");
   Set_Value (In2, "2222");
   Set_Value (In3, "333333");

   Execute (Stmt1);
   Show_In_Out;

   Bind_In_Out (Stmt2);

   Set_Value (In1, "1");
   Set_Value (In2, "22");
   Set_Value (In3, "333");

   Execute (Stmt2);

   Show_In_Out;

   Set_Value (In1, "111111111");
   Set_Value (In2, "222222");
   Set_Value (In3, "333");

   Execute (Stmt2);

   Show_In_Out;

   Bind_In_Out (Stmt3);

   Set_Value (In3, (1 .. 63 => '-', 64 => '!'));
   Execute (Stmt3);
   Show_In_Out;

   Set_Value (In1, (1 .. 63 => '=', 64 => '!'));
   Set_Value (In3, "3");
   Execute (Stmt3);
   Show_In_Out;

   Execute (Stmt4);
   Define (Stmt4, Out1, 1);
   Define (Stmt4, Out2, 2);
   Define (Stmt4, Out3, 3);

   while Fetch (Stmt4) loop
      Show_Out_Null;
   end loop;

   Execute (Stmt5);
   Define (Stmt5, Out1, 1);
   Define (Stmt5, Out2, 2);
   Define (Stmt5, Out3, 3);

   while Fetch (Stmt5) loop
      Show_Out_Null;
   end loop;
end Dyn_Bind;
