/*
 * Copyright (C) 2005-2009 Freescale Semiconductor, Inc. All rights reserved.
 * Based on gstasfdemux.c by Erik Walthinsen <omega@cse.ogi.edu>.
 *
 */
 
/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
 
/*
 * Module Name:    mfw_gst_asf_demuxer_plugin.c
 *
 * Description:    Asf Demuxer plug-in for Gstreamer.
 *
 * Portability:    This code is written for Linux OS and Gstreamer
 */  
 
/*
 * Changelog: 
 *
 */



/*=============================================================================
                            INCLUDE FILES
=============================================================================*/
#include <gst/gst.h>
#include <memory.h>
#include "asf_parser/asf_api.h"
#include "asf_parser/wmfdec.h"
#include "mfw_gst_asfdemuxer_plugin.h"

#include "mfw_gst_utils.h"

/*=============================================================================
                            LOCAL CONSTANTS
=============================================================================*/
/** NONE **/

/*=============================================================================
                LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
=============================================================================*/
static GstStaticPadTemplate mfw_gst_asfdemuxer_sink_template_factory
    = GST_STATIC_PAD_TEMPLATE("sink",
			      GST_PAD_SINK,
			      GST_PAD_ALWAYS,
			      GST_STATIC_CAPS("video/x-ms-asf")
    );

/*=============================================================================
                                        LOCAL MACROS
=============================================================================*/
/* Used for debugging */
#define GST_CAT_DEFAULT mfw_gst_asfdemuxer_debug

#define MAKEFOURCC_WMF(ch0, ch1, ch2, ch3) \
        ((guint)(guchar)(ch0) | ((guint)(guchar)(ch1) << 8) |   \
        ((guint)(guchar)(ch2) << 16) | ((guint)(guchar)(ch3) << 24 ))

#define mmioFOURCC_WMF(ch0, ch1, ch2, ch3)  MAKEFOURCC_WMF(ch0, ch1, ch2, ch3)



/******* video output type guids, in preference order  *****/

#define FOURCC_WMV2     mmioFOURCC_WMF('W','M','V','2')
#define FOURCC_WMV1     mmioFOURCC_WMF('W','M','V','1')
#define FOURCC_WMV3     mmioFOURCC_WMF('W','M','V','3')
#define FOURCC_WVC1     mmioFOURCC_WMF('W','V','C','1')
#define FOURCC_M4S2     mmioFOURCC_WMF('M','4','S','2')
#define FOURCC_MP43     mmioFOURCC_WMF('M','P','4','3')
#define FOURCC_mp43     mmioFOURCC_WMF('m','m','4','3')
#define FOURCC_MP4S     mmioFOURCC_WMF('M','P','4','S')
#define FOURCC_mp4s     mmioFOURCC_WMF('m','p','4','s')
#define FOURCC_MP42     mmioFOURCC_WMF('M','P','4','2')
#define FOURCC_mp42     mmioFOURCC_WMF('m','m','4','2')
#define FOURCC_MSS1     mmioFOURCC_WMF('M','S','S','1')
#define FOURCC_MSS2     mmioFOURCC_WMF('M','S','S','2')

/***********************************************************/

#define	GST_TAG_MFW_ASF_WIDTH		"width"
#define GST_TAG_MFW_ASF_HEIGHT	    "height"

/* ENGR00109002 bug fix */
#define MIN_RESOLUTION_WIDTH_ASF  16
#define MIN_RESOLUTION_HEIGHT_ASF 16

/*ENGR115014 ap support*/
#define  LOW_PROFILE         0
#define  MAIN_PROFILE        1
#define  AP_PROFILE          2

/*=============================================================================
                                      LOCAL VARIABLES
=============================================================================*/
static GstElementClass *parent_class = NULL;
/*=============================================================================
                        LOCAL FUNCTION PROTOTYPES
=============================================================================*/
GST_DEBUG_CATEGORY_STATIC(mfw_gst_asfdemuxer_debug);
static void mfw_gst_asfdemuxer_base_init(gpointer);
static void mfw_gst_asfdemuxer_class_init(MFWGstAsfDemuxClass *);
static void mfw_gst_asfdemuxer_init(MFWGstAsfDemux *);
/* Set and Get Property are not used in this version. */
static void mfw_gst_asfdemuxer_set_property(GObject *, guint,
					    const GValue *, GParamSpec *);
static void mfw_gst_asfdemuxer_get_property(GObject *, guint, GValue *,
					    GParamSpec *);
static gboolean mfw_gst_asfdemuxer_set_caps(GstPad *, GstCaps *);
static gboolean mfw_gst_asfdemuxer_sink_event(GstPad *, GstEvent *);
static gboolean mfw_gst_asfdemuxer_src_event(GstPad *, GstEvent *);
static gboolean mfw_gst_asfdemuxer_activate(GstPad *);
static gboolean mfw_gst_asfdemuxer_activate_pull(GstPad *, gboolean);
static void mfw_gst_asfdemuxer_taskfunc(MFWGstAsfDemux *);
static int mfw_gst_asfdemuxer_pulldata(GstBuffer **);
static GstStateChangeReturn mfw_gst_asfdemuxer_change_state
    (GstElement *, GstStateChange);
static gboolean asfdemuxer_fill_stream_info(MFWGstAsfDemux *, void *);
static GstFlowReturn asf_write_header_object(gchar *, int,
					     MFWGstAsfDemux *);
static GstFlowReturn
asf_push_gst_buf(unsigned char *, int, MFW_ASF_MEDIA_TYPE_T,
		 guint64, MFWGstAsfDemux *, gint is_key_frame);

static asf_error mfw_asf_get_data(void *, char *, unsigned long,
				  unsigned long long, unsigned long *);
static void gst_asfdemuxer_close(MFWGstAsfDemux *);
static gboolean mfw_gst_asfdemuxer_src_query(GstPad *, GstQuery *);
static const GstQueryType *mfw_gst_asfdemuxer_get_src_query_types
    (GstPad *);
static gboolean mfw_gst_asfdemuxer_seek(MFWGstAsfDemux *, GstPad *,
					GstEvent *);

static void mfw_gst_asfdemuxer_send_errormsg
    (MFWGstAsfDemux * demuxer, gchar * debug_string);

/*=============================================================================
                            GLOBAL VARIABLES
=============================================================================*/
/*=============================================================================
                            LOCAL FUNCTIONS
=============================================================================*/
/*=============================================================================
FUNCTION:			mfw_gst_asfdemuxer_send_errormsg

DESCRIPTION:		This function is responsible to send error message.

ARGUMENTS PASSED:	demuxer - The context of the main asf demuxer element.
                    debug_string - Debug string that needs to be passed
                    to application.

RETURN VALUE:		None

PRE-CONDITIONS:     None

POST-CONDITIONS:	None

IMPORTANT NOTES:    None
=============================================================================*/
static void mfw_gst_asfdemuxer_send_errormsg
    (MFWGstAsfDemux * demuxer, gchar * debug_string) {
    gst_element_post_message(GST_ELEMENT(demuxer),
			     gst_message_new_error(GST_OBJECT
						   (GST_ELEMENT(demuxer)),
						   NULL, debug_string));
return;
}

/*=============================================================================
FUNCTION:			mfw_gst_asfdemuxer_taskfunc

DESCRIPTION:		This is the main functions, where calls to ASF Parser 
					are made.

ARGUMENTS PASSED:	demuxer - The context of the main asf demuxer element.


RETURN VALUE:		None

PRE-CONDITIONS:     None

POST-CONDITIONS:	None

IMPORTANT NOTES:    None
=============================================================================*/
static void mfw_gst_asfdemuxer_taskfunc(MFWGstAsfDemux * demuxer)
{

    guint i_strm_cnt = 0;
    guint PacketNo = 0;
    asf_error iResult = ASF_OK;
    guint count = 0;
    gboolean ret_value = TRUE;
    GstFlowReturn result;
    gint count_out = 0;
    /* parsing of packets */
     //g_print("entering mfw_asfdemuxer_taskfunc!\n");
    iResult =
	asf_parser_parse_packet(demuxer->asf_handle, demuxer->op_stream);
    //g_print("iResult:%d!\n",iResult);
    if (ASF_OK != iResult) {
	    GST_DEBUG("\n\nParsing is Over. \n\n");
	    /* sending EOS to the other connected elements */
	    for (i_strm_cnt = 0; i_strm_cnt < demuxer->num_streams;
		 i_strm_cnt++) {
		asf_stream_context *stream_ptr;
        
        /* Check for streams with only Audio or video as its content.
           In this case, we need to point to the 1st media type.
        */

        if(demuxer->audio_present)
		stream_ptr = &(demuxer->asf_stream[i_strm_cnt]);
        else
            stream_ptr = &(demuxer->asf_stream[i_strm_cnt + 1]);
		if (mfw_gst_asfdemuxer_sink_event
		    (stream_ptr->src_pad, gst_event_new_eos())
		    != TRUE) {
		    GST_ERROR("\n Error in pushing the event!\n");
		}

	    }
        if (ASF_PARSING_DONE == iResult) {
	    demuxer->task_paused = 1;
	    ret_value = gst_pad_pause_task(demuxer->sinkpad);
	    if (FALSE == ret_value)
		GST_ERROR("\n There is no task on this sink pad !! \n");
	    goto done;
	} else {
	    GST_ERROR("Error is %d, packets parsed is %d!!\n\n", iResult,
		      PacketNo);
	    goto need_pause;
	}
    }
     //g_print("taskfunc line 245\n");
    for (count_out = 0; count_out < demuxer->op_stream->num_payloads;
	 count_out++) {

    /* push audio data onto the next element */
	if (demuxer->op_stream->raw_streams[count_out].stream_type ==
	    ASF_AUDIO_TYPE) {
	    result =
		asf_push_gst_buf((unsigned char *) demuxer->op_stream->
				 raw_streams[count_out].payload,
				 demuxer->op_stream->
				 raw_streams[count_out].object_size,
				 MFW_ASF_MEDIA_AUDIO_TRACK,
				 demuxer->op_stream->
				 raw_streams[count_out].pres_time,
				 demuxer,0);
	} 
    
    /* push video data onto the next element */
    else if (demuxer->op_stream->raw_streams[count_out].
		   stream_type == ASF_VIDEO_TYPE) {

        if (demuxer->is_seeking == TRUE) {
            if (!demuxer->op_stream->raw_streams[count_out].is_key_frame)
                continue;
            demuxer->is_seeking = FALSE;
        }
 //g_print("taskfunc line 272\n");
	    result =
		asf_push_gst_buf((unsigned char *) demuxer->op_stream->
				 raw_streams[count_out].payload,
				 demuxer->op_stream->
				 raw_streams[count_out].object_size,
				 MFW_ASF_MEDIA_VIDEO_TRACK,
				 demuxer->op_stream->
				 raw_streams[count_out].pres_time,
				 demuxer,
				 demuxer->op_stream->
				 raw_streams[count_out].is_key_frame
            );
 //g_print("taskfunc line 286,%d\n", result);
	}
    else {
        GST_ERROR("Wrong state,Pause the task.\n");
        /* Dexter add: should free all the buffer */
        for (;count_out < demuxer->op_stream->num_payloads;
	        count_out++)
	        free_mem(demuxer->op_stream->raw_streams[count_out].payload);
	    goto need_pause;        
    }

		/*** freeing the media object memory, which was allocated in parser lib ***/
	//g_print("taskfunc line 297\n");
	free_mem(demuxer->op_stream->raw_streams[count_out].payload);

	if (result != GST_FLOW_OK && result != GST_FLOW_NOT_LINKED) {
        GST_ERROR("Push data failed, free resouce and pause the task.\n");
        /* Dexter add: should free all the buffer */
        for (count_out++;count_out < demuxer->op_stream->num_payloads;
	        count_out++)
	        free_mem(demuxer->op_stream->raw_streams[count_out].payload);
	    goto need_pause;
	}
    
    }
 //g_print("taskfunc line 310\n");
  done:
    demuxer->op_stream->num_payloads = 0;
#define DEMUX_WAIT_INTERVAL 300
    /* Keep the balance of demuxer and codec */    
    usleep(DEMUX_WAIT_INTERVAL);
    //g_print("leave mfw_asfdemuxer_taskfunc!\n");
    return;

  need_pause:
    GST_ERROR
	("result of write data is %d mfw_gst_asfdemuxer_taskfunc need pause!\n",
	 result);
    demuxer->op_stream->num_payloads = 0;
    gst_pad_pause_task(demuxer->sinkpad);
    //g_print("leave mfw_asfdemuxer_taskfunc!\n");
    return;
}

/*=============================================================================
FUNCTION:   		mfw_asf_get_data

DESCRIPTION:		This is used to fetch data from file src whenever the 
					asf demuxer needs it.

ARGUMENTS PASSED:	hDecoder - Main structure of parser context.
					pBuffer  - The destination buffer.
					cbSize   - Required data length.
					cbOffset - Byte offset from the beginning of the file.
					*cbRead  - No. of bytes successfully read.

RETURN VALUE:		SUCCESS / FAIL

PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:   
=============================================================================*/
static asf_error
mfw_asf_get_data(void *hDecoder, char *pBuffer, unsigned long cbSize,
		 unsigned long long cbOffset, unsigned long *cbRead)
{

    GstBuffer *tmpbuf = NULL;
    asf_error ret_value = ASF_OK;
    GstFlowReturn result = GST_FLOW_OK;
    *cbRead = 0;
    MFWGstAsfDemux * asf_demuxer=NULL;
    asf_demuxer = (MFWGstAsfDemux *)(((WMFDecoder *)hDecoder)->pvAppContext);
	
	if ((int)cbSize<0){
		return ASF_ERROR;
    }
    result =
	gst_pad_pull_range(asf_demuxer->sinkpad, cbOffset, cbSize,
			   &tmpbuf);

    if (result != GST_FLOW_OK) {
	     GST_ERROR
	    (" FILE_READ: not able to read the data from %d, result is %d\n",
	     cbOffset, result);
	     ret_value = ASF_ERROR;
    } else {
	memcpy(pBuffer, GST_BUFFER_DATA(tmpbuf), cbSize);
	*cbRead = GST_BUFFER_SIZE(tmpbuf);
	gst_buffer_unref(tmpbuf);
    }
    return ret_value;

}

#define GST_BUFFER_FLAG_IS_SYNC (GST_BUFFER_FLAG_LAST<<2)
/*=============================================================================
FUNCTION:   		asf_push_gst_buf

DESCRIPTION:		Pushes the parsed data to respective source pads.

ARGUMENTS PASSED:	*buffer 		- pointer to the data which needs to be 
					pushed to source pad.
					length  		- Size of the data pushed.
					track type  	- Audio / Video track ?
					asf_time_stamp	- Time stamp for audio / video

RETURN VALUE:		None

PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
static GstFlowReturn
asf_push_gst_buf(unsigned char *buffer, int length,
		 MFW_ASF_MEDIA_TYPE_T track_type, guint64 asf_time_stamp,
		 MFWGstAsfDemux * asf_demuxer,gint is_key_frame)
{
    GstBuffer *src_data_buf = NULL;
    GstCaps *src_caps = NULL;
    unsigned char *tmp_buf = NULL;
    GstFlowReturn result = GST_FLOW_OK;
    GstClockTime stream_time = 0;
    int src_buf_size = 0;
    //g_print("entering asf_push_gst_buf!\n");
    asf_stream_context *stream_ptr;

    /* new segement events are pushed onto the sink pads 
       in case of starting of playback and when there is a
       seek indicating discontinuity in the data recieved 
    */

    if (track_type == MFW_ASF_MEDIA_AUDIO_TRACK) {
	stream_ptr = &(asf_demuxer->asf_stream[0]);

    if (stream_ptr->src_pad == NULL) {
        GST_DEBUG("The Audio source pad is not linked.\n");
        return GST_FLOW_OK;
    }

	if (asf_demuxer->for_audio_first_time) {
	    asf_demuxer->for_audio_first_time = 0;
	    gint64 start = asf_demuxer->seek_segment.start;
	    gst_pad_push_event(stream_ptr->src_pad,
			       gst_event_new_new_segment(FALSE, 1.0,
							 GST_FORMAT_TIME,
							 start,
							 GST_CLOCK_TIME_NONE,
							 start));
	}
    } else {
	stream_ptr = &(asf_demuxer->asf_stream[1]);
	if (asf_demuxer->for_video_first_time) {
	    asf_demuxer->for_video_first_time = 0;
	    gint64 start = asf_demuxer->seek_segment.start;
	    gst_pad_push_event(stream_ptr->src_pad,
			       gst_event_new_new_segment(FALSE, 1.0,
							 GST_FORMAT_TIME,
							 start,
							 GST_CLOCK_TIME_NONE,
							 start));
	}

    }

    /* allocation of output buffer with corresponding caps */

    src_caps = GST_PAD_CAPS(stream_ptr->src_pad);
    if (NULL == src_caps) {
	GST_ERROR("\n Caps not Correct state is %d\n", asf_demuxer->state);
	return GST_FLOW_ERROR;
    }
    src_buf_size = length;
    result = gst_pad_alloc_buffer(stream_ptr->src_pad, 0, src_buf_size,
				  src_caps, &src_data_buf);
    if (result != GST_FLOW_OK) {
	GST_ERROR("\n Cannot Create Output Buffer, result is %d", result);
	return result;
    }


    /* data is pushed on to the sinkpads setting appropriate 
     timestamp*/

    stream_time = asf_time_stamp * 1000 * 1000;
    tmp_buf = GST_BUFFER_DATA(src_data_buf);
    memcpy(tmp_buf, buffer, length);
    GST_BUFFER_TIMESTAMP(src_data_buf) = stream_time;

    /* Set the SYNC flag to GstBuffer */
    if (is_key_frame)
    {   
        GST_BUFFER_FLAG_SET(src_data_buf, GST_BUFFER_FLAG_IS_SYNC);
    }
    
    result = gst_pad_push(stream_ptr->src_pad, src_data_buf);
    if (result != GST_FLOW_OK) {
	GST_ERROR("\n Cannot Push Data onto next element, result is %d",
		  result);
	return result;
    }
    //g_print("leave asf_push_gst_buf!\n");
    return GST_FLOW_OK;
}

/*=============================================================================
FUNCTION:   		asf_write_header_object

DESCRIPTION:		Pushes the parsed Header object to audio source pads.

ARGUMENTS PASSED:	*buffer 		- pointer to the data which needs to be 
					pushed to source pad.
					copySize  		- Size of the data pushed.
					
RETURN VALUE:		None

PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
static GstFlowReturn asf_write_header_object(gchar * buffer, int copySize,
					     MFWGstAsfDemux * asf_demuxer)
{
    GstBuffer *tmpbuf = NULL;
    GstFlowReturn result = GST_FLOW_OK;
    if ((result =
	 gst_pad_pull_range(asf_demuxer->sinkpad, 0, copySize, &tmpbuf))
	!= GST_FLOW_OK) {
	GST_ERROR(" FILE_READ: not able to read the data from %d\n", 0);
	return result;
    }
    memcpy(buffer, GST_BUFFER_DATA(tmpbuf), copySize);
    result =
	asf_push_gst_buf((unsigned char *) buffer, copySize,
			 MFW_ASF_MEDIA_AUDIO_TRACK,
			 (unsigned int) GST_CLOCK_TIME_NONE, asf_demuxer,0);
    gst_buffer_unref(tmpbuf);
    return result;
}

/*=============================================================================
FUNCTION:   		video_src_templ

DESCRIPTION:		defines the video source pad template.

ARGUMENTS PASSED:	None
					
RETURN VALUE:		Pointer to GstPadTemplate. 

PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
static GstPadTemplate *video_src_templ(void)
{
    static GstPadTemplate *templ = NULL;
    
    if (!templ) {
    	GstCaps *caps = NULL;
      
/* ENGR00109002 bug fix */
    	caps = gst_caps_new_simple("video/x-wmv",
                    "wmvversion", GST_TYPE_INT_RANGE, 0,4, 
                    "width", GST_TYPE_INT_RANGE, MIN_RESOLUTION_WIDTH_ASF/*MIN_RESOLUTION_WIDTH*/, MAX_RESOLUTION_WIDTH,
                    "height", GST_TYPE_INT_RANGE, MIN_RESOLUTION_HEIGHT_ASF/*MIN_RESOLUTION_HEIGHT*/, MAX_RESOLUTION_HEIGHT,NULL); 

    	templ = gst_pad_template_new(PARSER_VIDEOPAD_TEMPLATE_NAME, GST_PAD_SRC,
                    GST_PAD_SOMETIMES, caps);
    }
    return templ;
}

/*=============================================================================
FUNCTION:   		asf_create_video_caps

DESCRIPTION:		create video caps

ARGUMENTS PASSED:	None
					
RETURN VALUE:		Pointer to caps 

PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
GstCaps* asf_create_video_caps(guint32 fccType, gchar **codec_name)
{
    GstCaps *caps = NULL;
    switch (fccType) {
        case GST_MAKE_FOURCC('W', 'M', 'V', '1'):
	    caps = gst_caps_new_simple("video/x-wmv",
				       "wmvversion", G_TYPE_INT, 1,
				       "wmvprofile", G_TYPE_INT, MAIN_PROFILE,NULL);
            *codec_name = "WMV-1";
	    break;
        case GST_MAKE_FOURCC('W', 'M', 'V', '2'):
            caps = gst_caps_new_simple("video/x-wmv",
				       "wmvversion", G_TYPE_INT, 2,
				       "wmvprofile",G_TYPE_INT,MAIN_PROFILE,NULL);
            *codec_name = "WMV-2";
	    break;
        case GST_MAKE_FOURCC('W', 'M', 'V', '3'):
    	    caps = gst_caps_new_simple("video/x-wmv",
               			       "wmvversion", G_TYPE_INT, 3,
               			       "wmvprofile",G_TYPE_INT,MAIN_PROFILE,NULL);
            *codec_name = "WMV-3";
      break;
        /*caps set to be VC-1 AP*/
        case GST_MAKE_FOURCC('W','V','C','1'):
        	caps = gst_caps_new_simple("video/x-wmv",
        	                 "wmvversion", G_TYPE_INT, 3,
        	                 "wmvprofile",G_TYPE_INT,AP_PROFILE,NULL);
         *codec_name = "WVC-1"; 
	    break;
        default:
            break;
     }

return caps;
}

/*=============================================================================
FUNCTION:   		audio_src_templ

DESCRIPTION:		defines the audio source pad template.

ARGUMENTS PASSED:	None
					
RETURN VALUE:		Pointer to GstPadTemplate. 

PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
static GstPadTemplate *audio_src_templ(void)
{
    static GstPadTemplate *templ = NULL;

    if (!templ) {
	GstCaps *caps = NULL;
	GstStructure *structure;

	caps = gst_caps_new_simple("audio/x-wma",
				   "wmaversion", GST_TYPE_INT_RANGE, 0,4, 
                   "channels",   GST_TYPE_INT_RANGE, 1,8,
				   "rate",       GST_TYPE_INT_RANGE, 0,96000, 
                   "bitrate",    GST_TYPE_INT_RANGE, 0,3000000,
				   "block_align",GST_TYPE_INT_RANGE, 0,2147483647, 
                   "depth",      GST_TYPE_INT_RANGE, 8,32, NULL);

	structure = gst_caps_get_structure(caps, 0);
	templ = gst_pad_template_new(PARSER_AUDIOPAD_TEMPLATE_NAME, GST_PAD_SRC,
				     GST_PAD_SOMETIMES, caps);
    }
    return templ;
}

/*=============================================================================
FUNCTION:   		asfdemuxer_fill_stream_info

DESCRIPTION:		Creates the audio / video pads for and sets the 
					corresponding capabilities.

ARGUMENTS PASSED:	demuxer - The main structure of asf parser plugin context.


RETURN VALUE:		None

PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
static gboolean
asfdemuxer_fill_stream_info(MFWGstAsfDemux * demuxer, void *asfContext)
{

    guint count;
    asf_stream_context *stream_ptr;
    gchar *padname = NULL;
    GstPadTemplate *templ = NULL;
    GstCaps *caps = NULL;
    GstPad *pad = NULL;
    gchar  *codec_name = NULL;
    gchar *text_msg     = NULL;

    GstElementClass *klass = GST_ELEMENT_GET_CLASS(demuxer);
    const gchar *tag_name;
    AudioStreamInfo *a_info = NULL;
    VideoStreamInfo *v_info = NULL;
    asf_error ret_val = ASF_OK;
    guint us16EncodeOpt;

    GstBuffer *extradata = NULL;
    a_info = (AudioStreamInfo *) g_malloc(sizeof(AudioStreamInfo));
    //g_print("entering asfdemuxer_fill_stream_info!\n");

    if (NULL == a_info) {
        text_msg = " Audio stream cannot be malloced ";
	GST_ERROR("\nAudio stream cannot be malloced !!\n");
	goto err;

    }
    v_info = (VideoStreamInfo *) g_malloc(sizeof(VideoStreamInfo));
    if (NULL == v_info) {
        text_msg = " Video stream cannot be malloced ";
	GST_ERROR("\nVideo stream cannot be malloced !!\n");
	goto err;
    }

    ret_val = asf_parse_get_video_stream_info(asfContext, v_info);
    if (ret_val != ASF_OK) 
    {
        text_msg = " ASF asf_parse_get_video_stream_info error ";
	GST_ERROR("ASF asf_parse_get_video_stream_info error!\n");
	goto err;
    }
    else if (v_info->bHasValidVideo)
    {
        /* Added condition check for resolution support*/
        if  (
            (v_info->u32Width > MAX_RESOLUTION_WIDTH) ||
            (v_info->u32Height > MAX_RESOLUTION_HEIGHT)
            )
        {
            text_msg = " unsupported video resolution (supports only till D1)";
	        goto err;
        }
    }
    ret_val = asf_parse_get_audio_stream_info(asfContext, a_info);
    if (ret_val != ASF_OK) 
    {
        text_msg = " ASF asf_parse_get_audio_stream_info error ";
	GST_ERROR("ASF asf_parse_get_audio_stream_info error!\n");
	goto err;
    }


    /*ENGR40805:goto err while both audio and video stream not exist*/
    if(0 == a_info->bHasValidAudio && 0 == v_info->bHasValidVideo)
    {
       text_msg = " No valid audio and video ";
       goto err;
    }		

    /********* Added temporarily for One Audio and One Video stream ***********/
    demuxer->byte_rate = a_info->nAvgBytesPerSec;

    for (count = 1; count <= ASF_MAX_STREAMS; count++) {
	caps = NULL;
	if (count == a_info->bHasValidAudio) {
	    guint16 wmaversionval;
	    /**** Add and create audio pads ****/
	    demuxer->audio_present = 1;

        if(a_info->wFormatTag == 0x0160)
            codec_name = "WMA-1";
        else if(a_info->wFormatTag == 0x0161)
            codec_name = "WMA-2";
        else if((a_info->wFormatTag == 0x0162) || (a_info->wFormatTag == 0x0163))
            codec_name = "WMA-3";
        else
        {
            codec_name = "audio codec format not support";
       	    demuxer->audio_present = 0;

        }
	    stream_ptr = &(demuxer->asf_stream[count - 1]);
	    padname = g_strdup_printf(PARSER_AUDIOPAD_TEMPLATE_NAME, count - 1);
        
	    GST_DEBUG("\npAudio_info->nAvgBytesPerSec - %d",
		      a_info->nAvgBytesPerSec);
	    GST_DEBUG("\npAudio_info->wFormatTag - %d",
		      a_info->wFormatTag);
	    GST_DEBUG("\npAudio_info->nBlockAlign - %d",
		      a_info->nBlockAlign);
	    GST_DEBUG("\npAudio_info->u16Channels - %d",
		      a_info->u16Channels);
	    GST_DEBUG("\npAudio_info->u16BitsPerSample - %d",
		      a_info->u16BitsPerSample);
	    GST_DEBUG("\npAudio_info->u16WMAVersion - %d",
		      a_info->u16WMAVersion);
	    GST_DEBUG("\npAudio_info->u32SamplesPerSec - %d",
		      a_info->u32SamplesPerSec);

        /* addition of audio TAGs */
        {
            GstTagList	*list = gst_tag_list_new();
   
            gst_tag_list_add(list,GST_TAG_MERGE_APPEND,GST_TAG_AUDIO_CODEC,
                codec_name,NULL);
            gst_tag_list_add (list, GST_TAG_MERGE_APPEND, GST_TAG_BITRATE,
                (guint)a_info->u32BitRate,NULL);
            gst_tag_list_add (list, GST_TAG_MERGE_APPEND, GST_TAG_DURATION,
                (guint64)a_info->u32Duration,NULL);
            gst_element_found_tags(GST_ELEMENT(demuxer),list); 
        }

	    extradata =
		gst_buffer_new_and_alloc(a_info->u16AudioExtraDataSize);
	    if (NULL == extradata) {
                text_msg = " extra data cannot be malloced ";
		GST_ERROR("\nextra data cannot be malloced !!\n");
		goto err;

	    }

	    memcpy(GST_BUFFER_DATA(extradata), &a_info->AudioExtraData,
		   a_info->u16AudioExtraDataSize);

	    wmaversionval = a_info->wFormatTag - 0x160 + 0x1;

   	    templ = audio_src_templ();
        if (demuxer->audio_present == 1) {
            caps = gst_caps_new_simple("audio/x-wma",
    				       "wmaversion", G_TYPE_INT,
    				       wmaversionval, "channels",
    				       G_TYPE_INT, a_info->u16Channels,
    				       "rate", G_TYPE_INT,
    				       a_info->u32SamplesPerSec, "bitrate",
    				       G_TYPE_INT,
    				       (a_info->nAvgBytesPerSec * 8),
    				       "block_align", G_TYPE_INT,
    				       a_info->nBlockAlign, "depth",
    				       G_TYPE_INT,
    				       a_info->u16BitsPerSample, NULL);

    	    gst_caps_set_simple(caps, "codec_data", GST_TYPE_BUFFER,
    				extradata, NULL);
	        demuxer->num_streams++;

        }
	    tag_name = GST_TAG_AUDIO_CODEC;



	} else if (count == (v_info->bHasValidVideo + 1)) {
	    gint wmvversionval;
	    demuxer->video_present = 1;
	    stream_ptr = &(demuxer->asf_stream[count - 1]);
            
	    padname = g_strdup_printf(PARSER_VIDEOPAD_TEMPLATE_NAME, count - 1);
	    templ = video_src_templ();
        //g_print("asf video type:%x\n",v_info->u32CompressionType);
        caps = asf_create_video_caps(v_info->u32CompressionType,&codec_name);

            if (caps == NULL)
            {
                text_msg = " video caps is NULL  ";
                goto err;
            }
        /* addding the tag information for the video data */
        {
            GstTagList	*list = gst_tag_list_new();
            gst_tag_list_add(list,GST_TAG_MERGE_APPEND,GST_TAG_VIDEO_CODEC,
                codec_name,NULL);
            gst_tag_list_add (list, GST_TAG_MERGE_APPEND, GST_TAG_MFW_ASF_WIDTH,
                (guint)v_info->u32Width,NULL);
            gst_tag_list_add (list, GST_TAG_MERGE_APPEND, GST_TAG_MFW_ASF_HEIGHT,
                (guint)v_info->u32Height,NULL);
            gst_element_found_tags(GST_ELEMENT(demuxer),list); 
        }

	    extradata = gst_buffer_new_and_alloc(v_info->ExtraDataLength);
	    if (NULL == extradata) {
                text_msg = " extra data cannot be malloced ";
		GST_ERROR("\nextra data cannot be malloced !!\n");
		goto err;

	    }

	    memcpy(GST_BUFFER_DATA(extradata), &v_info->ExtraInfo,
		   v_info->ExtraDataLength);
	    
            gst_caps_set_simple(caps,"height", G_TYPE_INT,
				v_info->u32Height, "width",
				G_TYPE_INT, v_info->u32Width, 
                                "codec_data", GST_TYPE_BUFFER,
				extradata, NULL);

	    tag_name = GST_TAG_VIDEO_CODEC;
	    demuxer->num_streams++;
	}
	/* no caps means no stream */
	if (!caps) {
	    /* unref any mem that may be in use */
	    continue;
	}

	pad = stream_ptr->src_pad =
	    gst_pad_new_from_template(templ, padname);
	g_free(padname);
	gst_pad_use_fixed_caps(pad);

	/* Register the functions to the pad later */
	gst_pad_set_element_private(pad, stream_ptr);

	gst_pad_set_caps(pad, caps);
	if (gst_pad_set_active(pad, TRUE) == FALSE) {
	    GST_ERROR("Error in activation of pad\n");

	}

/* check if the caps represents media formats */
    GstCaps *temp_caps;
    temp_caps = gst_pad_get_caps(pad);
 	if (gst_caps_is_empty(temp_caps))  
 	{    
  		gst_caps_unref(caps);  
        gst_caps_unref(temp_caps);
        caps = NULL; 
        text_msg = " caps is out of its template caps  ";
        goto err;
 	}
    gst_caps_unref(temp_caps); 
      
	gst_element_add_pad(GST_ELEMENT(demuxer), pad);

    GST_LOG_OBJECT(GST_ELEMENT(demuxer), "Added pad %s with caps %p", 
        GST_PAD_NAME(pad), caps);

	if (caps) {
	    gst_caps_unref(caps);
	    caps = NULL;
	}

	/* registering functions for position,duration and seek functionalites */
	if (pad != NULL) {

	    gst_pad_set_query_type_function(pad,
					    GST_DEBUG_FUNCPTR
					    (mfw_gst_asfdemuxer_get_src_query_types));

	    gst_pad_set_query_function(pad,
				       GST_DEBUG_FUNCPTR
				       (mfw_gst_asfdemuxer_src_query));

	    gst_pad_set_event_function(pad,
				       GST_DEBUG_FUNCPTR
				       (mfw_gst_asfdemuxer_src_event));
	}

	if (extradata != NULL) {
	    gst_buffer_unref(extradata);
	    extradata = NULL;
	}
    }

    if (a_info != NULL) {
	g_free(a_info);
	a_info = NULL;
    }

    if (v_info != NULL) {
	g_free(v_info);
	v_info = NULL;
    }
    //g_print("leave asfdemuxer_fill_stream_info!\n");
    return TRUE;

  err:
    if (extradata != NULL) {
	gst_buffer_unref(extradata);
	extradata = NULL;
    }
    if (a_info != NULL) {
	g_free(a_info);
	a_info = NULL;
    }

    if (v_info != NULL) {
	g_free(v_info);
	v_info = NULL;
    }

    Err_Msg:
	{
        GstMessage *message = NULL;
	    GError *gerror = NULL;
        gerror = g_error_new_literal(1, 0, text_msg);
	    message =
	        gst_message_new_error(GST_OBJECT(GST_ELEMENT(demuxer)), gerror,
				      "debug none");
	    gst_element_post_message(GST_ELEMENT(demuxer), message);
	    g_error_free(gerror);
	}
	  //g_print("leave asfdemuxer_fill_stream_info!\n");
    return FALSE;
}

/*=============================================================================
FUNCTION:			mfw_gst_asfdemuxer_activate_pull

DESCRIPTION:		This will start the demuxer task function in "pull" based	
					scheduling model.

ARGUMENTS PASSED:	pad		- sink pad, where the data will be pulled into, from	
					file source.
					active 	- 
 

RETURN VALUE:		
		TRUE      -    if operation was successful
        FALSE     -    if operation was unsuccessful
PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
static gboolean
mfw_gst_asfdemuxer_activate_pull(GstPad * pad, gboolean active)
{
	  //g_print("entering mfw_gst_asfdemuxer_activate_pull!\n");
    gboolean ret_val = TRUE;
    MFWGstAsfDemux *demuxer_info =
	MFW_GST_ASF_DEMUXER(GST_PAD_PARENT(pad));
    if (active) {
	ret_val = gst_pad_start_task(pad,
				     (GstTaskFunction)
				     mfw_gst_asfdemuxer_taskfunc,
				     demuxer_info);

	if (ret_val == FALSE) {
	    GST_ERROR("Task could not be started !!\n");
	    return FALSE;
	}

    } else {
	return gst_pad_stop_task(pad);
    }
    //g_print("leave mfw_gst_asfdemuxer_activate_pull!\n");
    return TRUE;
}

/*=============================================================================
FUNCTION:   	mfw_gst_asfdemuxer_activate

DESCRIPTION:    it will call gst_pad_activate_pull which activates or
                deactivates the given pad in pull mode via dispatching to the
                pad's activepullfunc

ARGUMENTS PASSED:
         pad       -    pointer to GstPad to activate or deactivate

RETURN VALUE:
         TRUE      -    if operation was successful
         FALSE     -    if operation was unsuccessful

PRE-CONDITIONS:
        None

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/
static gboolean mfw_gst_asfdemuxer_activate(GstPad * pad)
{
    if (gst_pad_check_pull_range(pad)) {
	return gst_pad_activate_pull(pad, TRUE);
    } else {
	return FALSE;
    }
    return TRUE;
}

/*=============================================================================
FUNCTION:   mfw_gst_asfdemuxer_get_src_query_types  

DESCRIPTION:    return the query type.    

ARGUMENTS PASSED:
        pad     -   pointer to GstPad
             
            
RETURN VALUE:
        GstQueryType    -   query type

PRE-CONDITIONS:
        None

POST-CONDITIONS:
		None

IMPORTANT NOTES:
        None

==============================================================================*/
static const GstQueryType *mfw_gst_asfdemuxer_get_src_query_types(GstPad *
								  pad)
{

    static const GstQueryType types[] = {
	GST_QUERY_SEEKING,
	GST_QUERY_DURATION,
	0
    };
    return types;
}

/*=============================================================================
FUNCTION:   mfw_gst_asfdemuxer_src_query   

DESCRIPTION:    performs query on src pad.    

ARGUMENTS PASSED:
        pad     -   pointer to GstPad
        query   -   pointer to GstQuery        
            
RETURN VALUE:
        TRUE    -   success
        FALSE   -   failure

PRE-CONDITIONS:
        None

POST-CONDITIONS:
		None

IMPORTANT NOTES:
        None

==============================================================================*/
static gboolean
mfw_gst_asfdemuxer_src_query(GstPad * pad, GstQuery * query)
{

    MFWGstAsfDemux *demux;
    gboolean res = FALSE;
    //g_print("entering mfw_gst_asfdemuxer_src_query!\n");

    demux = MFW_GST_ASF_DEMUXER(gst_pad_get_parent(pad));

    switch (GST_QUERY_TYPE(query)) {
    case GST_QUERY_DURATION:
	{

	    GstFormat format;

	    gst_query_parse_duration(query, &format, NULL);

	    if (demux->total_duration != GST_CLOCK_TIME_NONE) {
		GST_DEBUG("returning duration: %" GST_TIME_FORMAT,
			  GST_TIME_ARGS(demux->total_duration * 1000 *
					1000));

		gst_query_set_duration(query, GST_FORMAT_TIME,
				       demux->total_duration * 1000 *
				       1000);

		res = TRUE;
	    } else {
		GST_DEBUG("duration not known yet");

	    }

	    break;
	}

    case GST_QUERY_SEEKING:
	{
        g_print("query seeking %d\n", demux->seekable);
		gst_query_set_seeking(query, GST_FORMAT_TIME, demux->seekable, 0, (demux->total_duration) * 1000 * 1000);
		res = TRUE;

	    break;
	}
    default:
	res = gst_pad_query_default(pad, query);
	break;
    }
    gst_object_unref(demux);
    //g_print("leave mfw_gst_asfdemuxer_src_query!\n");
    return res;
}




/*=============================================================================
FUNCTION:   	mfw_gst_asfdemuxer_seek

DESCRIPTION:    This function is called when seek event occurs. This will reset
				the packet offset in the ASF parser library.

ARGUMENTS PASSED:
		 demuxer	-	 pointer to ASF parser plugin structure.
         pad       	-    pointer to GstPad to activate or deactivate
         event      -    pointer to event

RETURN VALUE:
         TRUE      -    if operation was successful
         FALSE     -    if operation was unsuccessful

PRE-CONDITIONS:
        None

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/
static gboolean
mfw_gst_asfdemuxer_seek(MFWGstAsfDemux * demuxer, GstPad * pad,
			GstEvent * event)
{
    GstEvent *seek_event = NULL;
    gdouble rate = 0;
    GstFormat format;
    GstSeekFlags flags;
    GstSeekType cur_type, stop_type;
    gint64 cur = 0;
    gint64 stop = 0, seek_time_in_ns = 0;
    gint64 time_cur = 0, time_stop = 0;
    gint64 bytes_cur, bytes_stop = 0;
    guint32 offset_table = 0;
    guint64 packet_num = 0;
    gboolean ret = TRUE;
    asf_error iResult = ASF_OK;
    gboolean only_need_update;
    simple_index_object *ind_test = NULL;
    asf_stream_context *stream_ptr = NULL;
    guint64 byte_offset = 0;
    guint packetSCR;



    demuxer = MFW_GST_ASF_DEMUXER(GST_OBJECT_PARENT(pad));
    stream_ptr = &(demuxer->asf_stream[0]);

    gst_event_parse_seek(event, &rate, &format, &flags,
			 &cur_type, &cur, &stop_type, &stop);


    demuxer->seeked_time = cur;
    if (demuxer->video_present) {
	ind_test = &demuxer->asf_index_object;

	    /** convert cur to 100's of nano secs **/
	seek_time_in_ns = cur / 100 + demuxer->asf_file_prop.pre_roll*10000;

	/** Get the Table offset using **/
	offset_table = (seek_time_in_ns) / ind_test->index_interval;

  //offset_table = offset_table>(MAX_INDEX-1)?(MAX_INDEX-1):offset_table;
  offset_table = offset_table>(ind_test->index_enties_count-1)?(ind_test->index_enties_count-1):offset_table;

	    /** Get Packet number from Index Object table **/
	packet_num = ind_test->index_element[offset_table].packet_number;

    } else {

	byte_offset = gst_util_uint64_scale(demuxer->seeked_time,
					    demuxer->byte_rate,
					    GST_SECOND);

	packet_num = (byte_offset) / demuxer->packet_size;
	
        asf_parse_seek(demuxer->asf_handle, packet_num, &packetSCR);

	if(packetSCR < demuxer->seeked_time/(1000*1000))
	{
		while(packetSCR < demuxer->seeked_time/(1000*1000))
		{
                packet_num++;
                iResult = asf_parse_seek(demuxer->asf_handle, packet_num, &packetSCR);
                if ( iResult != ASF_OK)
                    break;
                     
		}
	}
	else
	{
		while(packetSCR > demuxer->seeked_time/(1000*1000))
		{
                packet_num--;
                iResult = asf_parse_seek(demuxer->asf_handle, packet_num, &packetSCR);
                if ( iResult != ASF_OK)
                    break;
		}
	}
    }

    GST_DEBUG(" seeked packet number  = %lld \n", packet_num);
    /** Send the packet number to ASF parser, to reset the Packet Offset **/
    iResult = asf_parse_seek(demuxer->asf_handle, packet_num, &packetSCR);
    if (0 == iResult) {
	ret = TRUE;
    } else {
	GST_ERROR(" parser is not able to seek \n");
	return FALSE;

    }
    cur = (gint64) (packetSCR) * 1000 * 1000;

    format = GST_FORMAT_TIME;
    /* fixme, when stop is other format */
    gst_segment_set_seek(&demuxer->seek_segment, rate, format, flags,
			 cur_type, cur, stop_type, stop, NULL);
    demuxer->for_audio_first_time = TRUE;
    demuxer->for_video_first_time = TRUE;
    demuxer->is_seeking = TRUE;

    return TRUE;
}

/*=============================================================================
FUNCTION:   	mfw_gst_asfdemuxer_src_event

DESCRIPTION:	Handles an event on the source pad.

ARGUMENTS PASSED:
        pad        -    pointer to pad
        event      -    pointer to event
RETURN VALUE:
        TRUE       -	if event is sent to source properly
	    FALSE	   -	if event is not sent to source properly

PRE-CONDITIONS:    None

POST-CONDITIONS:   None

IMPORTANT NOTES:   None
=============================================================================*/
static gboolean
mfw_gst_asfdemuxer_src_event(GstPad * src_pad, GstEvent * event)
{
	  //g_print("entering mfw_gst_asfdemuxer_src_event!\n");
    gboolean res = TRUE;
    MFWGstAsfDemux *demuxer_info =
	MFW_GST_ASF_DEMUXER(GST_PAD_PARENT(src_pad));
    switch (GST_EVENT_TYPE(event)) {
    case GST_EVENT_SEEK:
	{


	    GstSeekFlags flags;
	    gboolean flush;
        GstSeekType cur_type, stop_type;
        gint64 cur = 0;
        gint64 stop = 0;
        gdouble rate = 0;
        GstFormat format;

        guint64 totalduration;

        if (demuxer_info->seekable == FALSE) {
            GST_WARNING("This stream is not seekable.\n");
            return FALSE;
        }

	  /*  gst_event_ref(event); */

        gst_event_parse_seek(event, &rate, &format, &flags,
			 &cur_type, &cur, &stop_type, &stop);

        /* ENGR62699: Demuxer cannot do seek operation if format is not GST_FORMAT_TIME */
        if (format != GST_FORMAT_TIME)
        {
            GST_WARNING("SEEK event not TIME format.\n");
            gst_event_unref (event);
            return TRUE;

        }

        totalduration = (demuxer_info->total_duration) * 1000 * 1000;

        if(cur > totalduration){
            GST_WARNING("SEEK event exceed the maximum duration.\n");
            cur = totalduration;
        }
	    flush = flags & GST_SEEK_FLAG_FLUSH;
	    /* send flush start */
	    if (flush) {
		guint i_strm_cnt = 0;
		res =
		    gst_pad_push_event(demuxer_info->sinkpad,
				       gst_event_new_flush_start());
		if (!res) {
		    GST_WARNING_OBJECT(demuxer_info,
				       "Failed to push event upstream!");
		}

		for (i_strm_cnt = 0;
		     i_strm_cnt < demuxer_info->num_streams;
		     i_strm_cnt++) {
		    asf_stream_context *stream_ptr;

              /* Check for streams with only Audio or video as its content.
       		    In this case, we need to point to the 1st media type.
       		 */

       		 if(demuxer_info->audio_present)
	    		stream_ptr = &(demuxer_info->asf_stream[i_strm_cnt]);
        	 else
            		stream_ptr = &(demuxer_info->asf_stream[i_strm_cnt + 1]);

		    res =
			gst_pad_push_event(stream_ptr->src_pad,
					   gst_event_new_flush_start());
		    if (!res) {
			GST_WARNING_OBJECT(demuxer_info,
					   "Failed to push event downstream!");
		    }
		}
	    } 

        /* for seek event, pause the sinkpad task */
		gst_pad_pause_task(demuxer_info->sinkpad);
	    

	    GST_PAD_STREAM_LOCK(demuxer_info->sinkpad);
	    /* we need to stop flushing on the sinkpad as we're going to use it
	     * next. We can do this as we have the STREAM lock now. */
	    gst_pad_push_event(demuxer_info->sinkpad,
			       gst_event_new_flush_stop());
	    res = mfw_gst_asfdemuxer_seek(demuxer_info, src_pad, event);
	    if (res == FALSE) {
		GST_ERROR("Failed in demuxer seek !!\n");
	    }
	    /* send flush stop */
	    if (flush) {
		guint i_strm_cnt = 0;
		for (i_strm_cnt = 0;
		     i_strm_cnt < demuxer_info->num_streams;
		     i_strm_cnt++) {
		    asf_stream_context *stream_ptr;

		/* Check for streams with only Audio or video as its content.
           	In this case, we need to point to the 1st media type.
        	*/

        	if(demuxer_info->audio_present)
	    		stream_ptr = &(demuxer_info->asf_stream[i_strm_cnt]);
        	else
                stream_ptr = &(demuxer_info->asf_stream[i_strm_cnt + 1]);


            if (stream_ptr->src_pad != NULL)
		    res =
			gst_pad_push_event(stream_ptr->src_pad,
					   gst_event_new_flush_stop());
		    if (!res) {
			GST_WARNING_OBJECT(demuxer_info,
					   "Failed to push event downstream!");
		    }
		}
	    }

	    GST_PAD_STREAM_UNLOCK(demuxer_info->sinkpad);
	    /* streaming can continue now */
	    gst_pad_start_task(demuxer_info->sinkpad,
			       (GstTaskFunction)
			       mfw_gst_asfdemuxer_taskfunc, demuxer_info);

	    break;
	}
    default:
	res = FALSE;
	break;
    }
    gst_event_unref(event);
    //g_print("leave mfw_gst_asfdemuxer_src_event!\n");
    return TRUE;
}

/*=============================================================================
FUNCTION:   	mfw_gst_asfdemuxer_sink_event

DESCRIPTION:	Handles an event on the sink pad.

ARGUMENTS PASSED:
        pad        -    pointer to pad
        event      -    pointer to event
RETURN VALUE:
        TRUE       -	if event is sent to sink properly
	    FALSE	   -	if event is not sent to sink properly

PRE-CONDITIONS:    None

POST-CONDITIONS:   None

IMPORTANT NOTES:   None
=============================================================================*/
static gboolean
mfw_gst_asfdemuxer_sink_event(GstPad * pad, GstEvent * event)
{

    gboolean result = TRUE;
    //g_print("entering mfw_gst_asfdemuxer_sink_event!\n");

    switch (GST_EVENT_TYPE(event)) {
    case GST_EVENT_NEWSEGMENT:
	{
	    GstFormat format;
	    gst_event_parse_new_segment(event, NULL, NULL, &format, NULL,
					NULL, NULL);
	    if (format == GST_FORMAT_TIME) {

	    } else {
		GST_DEBUG("dropping newsegment event in format %s",
			  gst_format_get_name(format));
		gst_event_unref(event);
		result = TRUE;
	    }
	    break;
	}

    case GST_EVENT_EOS:
	{

	    GST_DEBUG("\nDemuxer: Sending EOS to Decoders\n");
	    result = gst_pad_push_event(pad, event);
	    if (result != TRUE) {
		GST_ERROR("\n Error in pushing the event, result is %d\n",
			  result);

	    }
	    break;
	}

    default:
	{

	    result = gst_pad_event_default(pad, event);
	    break;
	}

    }
    //g_print("leave mfw_gst_asfdemuxer_sink_event!\n");
    return result;
}

/*=============================================================================
FUNCTION:   mfw_gst_asfdemuxer_set_caps

DESCRIPTION:    this function handles the link with other plug-ins and used for
                capability negotiation  between pads

ARGUMENTS PASSED:
        pad        -    pointer to GstPad
        caps       -    pointer to GstCaps

RETURN VALUE:
        TRUE       -    if capabilities are set properly
        FALSE      -    if capabilities are not set properly
PRE-CONDITIONS:
        None

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/
static gboolean mfw_gst_asfdemuxer_set_caps(GstPad * pad, GstCaps * caps)
{

    MFWGstAsfDemux *demuxer_info;
    //g_print("entering mfw_gst_asfdemuxer_set_caps!\n");
    GstStructure *structure = gst_caps_get_structure(caps, 0);
    demuxer_info = MFW_GST_ASF_DEMUXER(GST_OBJECT_PARENT(pad));
    //g_print("leave mfw_gst_asfdemuxer_set_caps!\n");
    return TRUE;
}

/*=============================================================================
FUNCTION:               mfw_gst_asfdemuxer_set_property

DESCRIPTION:            Sets the Property for the element.

ARGUMENTS PASSED:
                        object  -> pointer on which property is set
                        prop_id -> ID.
                        value   -> Value to be set.
                        pspec   -> Parameters to be set
RETURN VALUE:
                        None

PRE-CONDITIONS:
                        None

POST-CONDITIONS:
                        None

IMPORTANT NOTES:
                        None
=============================================================================*/
static void
mfw_gst_asfdemuxer_set_property(GObject * object, guint prop_id,
				const GValue * value, GParamSpec * pspec)
{

}

/*=============================================================================
FUNCTION:               mfw_gst_asfdemuxer_get_property

DESCRIPTION:            Gets the Property of the element.

ARGUMENTS PASSED:
                        object  -> pointer on which property is to be obtained.
                        prop_id -> ID.
                        value   -> Value to be get.
                        pspec   -> Parameters to be get.

RETURN VALUE:
                        None

PRE-CONDITIONS:
                        None

POST-CONDITIONS:
                        None

IMPORTANT NOTES:
                        None
=============================================================================*/
static void
mfw_gst_asfdemuxer_get_property(GObject * object, guint prop_id,
				GValue * value, GParamSpec * pspec)
{

}

/*=============================================================================
FUNCTION:   		mfw_gst_asfdemuxer_init

DESCRIPTION:    	create the pad template that has been registered with the
	                element class in the _base_init and do library table
	                initialization

ARGUMENTS PASSED:	demuxer_info - Main asf demuxer plugin structure.

RETURN VALUE:       None

PRE-CONDITIONS:     _base_init and _class_init are called

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
static void mfw_gst_asfdemuxer_init(MFWGstAsfDemux * demux)
{


    GstPadTemplate *template = NULL;
    GstElementClass *klass = GST_ELEMENT_GET_CLASS(demux);
    gboolean ret_val = TRUE;
    //g_print("entering mfw_gst_asfdemuxer_init!\n");

    /* Create and add the sink pad --
       Pad through which data comes into the demuxer element */
    demux->sinkpad =
	gst_pad_new_from_template(gst_element_class_get_pad_template
				  (klass, "sink"), "sink");

	/*** Initializing values ***/
    demux->init_parser = FALSE;
    demux->state = GST_STATE_NULL;
    demux->for_audio_first_time = 1;
    demux->for_video_first_time = 1;
    demux->audio_present = 0;
    demux->video_present = 0;
    demux->asf_handle = NULL;
    demux->is_seeking = FALSE;

    gst_pad_set_activate_function(demux->sinkpad,
				  mfw_gst_asfdemuxer_activate);
    gst_pad_set_activatepull_function(demux->sinkpad,
				      mfw_gst_asfdemuxer_activate_pull);
    ret_val = gst_element_add_pad(GST_ELEMENT(demux), demux->sinkpad);

    if (ret_val == FALSE) {
	GST_ERROR("Sink pad could not be added !!\n");

    }

    gst_pad_set_setcaps_function(demux->sinkpad,
				 mfw_gst_asfdemuxer_set_caps);

    gst_pad_set_event_function(demux->sinkpad,
			       GST_DEBUG_FUNCPTR
			       (mfw_gst_asfdemuxer_sink_event));

#define MFW_GST_ASF_PLUGIN VERSION
    PRINT_CORE_VERSION(asf_parser_version_info());
    PRINT_PLUGIN_VERSION(MFW_GST_ASF_PLUGIN);
    
    //g_print("leave mfw_gst_asfdemuxer_init!\n");
    return;
}



/***********************************************************************/
static GstStateChangeReturn
mfw_gst_asfdemuxer_handle_readytopaused(MFWGstAsfDemux * asf_demuxer_info,
                                        GstElement * element)
{
    //g_print("enter mfw_gst_asfdemuxer_handle_readytopaused!\n");
    GstPad *my_peer_pad = NULL;
    guint count = 0;
    asf_parser_file_properties* asf_file_prop;
    simple_index_object *ind_test = NULL;
    asf_error iResult = ASF_OK;
    asf_parser_content_descriptor meta_info;
    // Vinay TLSbo82983
    int error_code = WMPERR_OK;
    asf_file_prop = &(asf_demuxer_info->asf_file_prop);

    memset(&meta_info,0,sizeof(asf_parser_content_descriptor));
    if (asf_demuxer_info->init_parser == FALSE) {

	/*** Initializing ASF Parser -- Header parsing ***/
	if (gst_pad_check_pull_range(asf_demuxer_info->sinkpad)) {
	    my_peer_pad = gst_pad_get_peer(asf_demuxer_info->sinkpad);
	    gst_pad_activate_pull(my_peer_pad, TRUE);
	} else {

	    GST_ERROR("Can not active src peer in pull mode!\n");
	    return GST_STATE_CHANGE_FAILURE;
	}
   
    asf_demuxer_info->asf_handle =
	    asf_parser_init(mfw_asf_get_data, &error_code,(void *)asf_demuxer_info);
    
	if (NULL == asf_demuxer_info->asf_handle) {
	    /* In case of corrupted bit-stream, this function fails, the error
            reporting needs to be done properly.
        */

        GST_ERROR("Error in opening the ASF Demuxer:Corrupted bit-stream\n");
		
		goto Err_Msg;

	    if (error_code == WMPERR_DRMINITFAIL) {
		mfw_gst_asfdemuxer_send_errormsg(asf_demuxer_info,
						 "DRM Initialization failed");
		GST_ERROR("\nDRM Initialization failed\n");
		return GST_STATE_CHANGE_FAILURE;
	    }
	    if (!gst_pad_pause_task(asf_demuxer_info->sinkpad)) {
		GST_ERROR("\nCould not pause the task\n");
		return GST_STATE_CHANGE_FAILURE;
	    }
	    return GST_STATE_CHANGE_FAILURE;
	}

    /* meta data parsing for content description */
    iResult = asf_parse_get_content_descriptors(
        asf_demuxer_info->asf_handle,&meta_info);
    if(iResult == ASF_OK){
        if(meta_info.meta_data_present){
            GstTagList	*list = gst_tag_list_new();
            gchar           *tag_name;
            tag_name = GST_TAG_TITLE;
            gst_tag_list_add(list,GST_TAG_MERGE_APPEND,tag_name,
                meta_info.Title,NULL);
            
            tag_name = GST_TAG_ARTIST;
            gst_tag_list_add(list,GST_TAG_MERGE_APPEND,tag_name,
                meta_info.Author,NULL);
            
            tag_name = GST_TAG_COPYRIGHT;
            gst_tag_list_add(list,GST_TAG_MERGE_APPEND,tag_name,
                meta_info.Copyright,NULL);
            
            tag_name = GST_TAG_DESCRIPTION;
            gst_tag_list_add(list,GST_TAG_MERGE_APPEND,tag_name,
                meta_info.Description,NULL);
            
            gst_element_found_tags(element,list);   
        }
    }
    else {
        GST_ERROR(" Error in parsing the meta data \n");
    }

	/* init demuxer's segment and set segment duration */
	gst_segment_init(&asf_demuxer_info->seek_segment,
			 GST_FORMAT_UNDEFINED);
	gst_segment_init(&asf_demuxer_info->segment, GST_FORMAT_TIME);
	iResult =
	    asf_parse_get_file_properties(asf_demuxer_info->asf_handle,
					  asf_file_prop);
	if (iResult != ASF_OK) {

	    GST_ERROR("Get ASF file properties returned ERROR !!\n");
	    return GST_STATE_CHANGE_FAILURE;

	}

	asf_demuxer_info->packet_size = asf_file_prop->packet_size;

/*Consider preroll for correct duration and seek position,real duration should sub pre_roll time*/
	asf_demuxer_info->total_duration = asf_file_prop->ms_duration - asf_file_prop->pre_roll;

	gst_segment_set_duration(&asf_demuxer_info->segment,
				 GST_FORMAT_TIME,
				 (gint64) (asf_file_prop->ms_duration -
					   asf_file_prop->pre_roll) *
				 1000000);


	ind_test = &asf_demuxer_info->asf_index_object;
	/** Parsing the Index Object **/
	iResult = asf_parse_index(asf_demuxer_info->asf_handle, ind_test);


	GST_DEBUG("ASF Demuxer#####simple index index_interval %lld max_packet_count %d \
            index_enties_count %d\n", ind_test->index_interval,
		  ind_test->max_packet_count, ind_test->index_enties_count);

	GST_DEBUG("ASF Demuxer#####simple index index_element[7] packet_number \
            %d packet_count %d\n", ind_test->index_element[7].
		  packet_number, ind_test->index_element[7].packet_count);


	asf_demuxer_info->seekable = TRUE;

	 /*** Presently one audio & one video streams ***/
	asf_demuxer_info->num_streams = 0;

	/**** Setting Pads for the streams present ****/
	if (asfdemuxer_fill_stream_info(asf_demuxer_info,
					asf_demuxer_info->asf_handle) ==
	    FALSE)
	    return GST_STATE_CHANGE_FAILURE;

        if (asf_demuxer_info->video_present) {
            /*did not find simple index object */
            if (iResult != ASF_OK) {
                asf_demuxer_info->seekable = FALSE;

            } else
                asf_demuxer_info->seekable = TRUE;
        }

        /* Signalling no more pads to the application. */
	GST_DEBUG("signaling no more pads from mfw_asfdemuxer");
	gst_element_no_more_pads (GST_ELEMENT (asf_demuxer_info));

	/************** After Initializaion -- Data Packet Parsing ****************/
	asf_demuxer_info->op_stream =
	    (asf_parser_stream *) g_malloc(sizeof(asf_parser_stream));
	if (asf_demuxer_info->op_stream == NULL) {
	    GST_ERROR("\nCannot Malloc - op_stream !!\n");
	    return GST_STATE_CHANGE_FAILURE;

	}

	for (count = 0; count < ASF_MAX_PAYLOADS; count++) {
	    asf_demuxer_info->op_stream->raw_streams[count].payload = NULL;
	}

	asf_demuxer_info->op_stream->num_payloads = 0;
	asf_demuxer_info->init_parser = TRUE;
	gst_pad_activate_pull(my_peer_pad, FALSE);
	gst_object_unref(my_peer_pad);

    }

    /*send a segment info start */
    {
	GstFormat format;
	GstSeekFlags flags;
	GstSeekType cur_type, stop_type;
	gint64 cur, stop;
	guint i_strm_cnt = 0;

	flags = GST_SEEK_FLAG_FLUSH | GST_SEEK_FLAG_KEY_UNIT;
	cur_type = GST_SEEK_TYPE_SET;
	stop_type = GST_SEEK_TYPE_NONE;
	cur = 0;
	stop = GST_CLOCK_TIME_NONE;
	format = GST_FORMAT_TIME;
	gst_segment_set_seek(&asf_demuxer_info->seek_segment, 1.0, format,
			     flags, cur_type, cur, stop_type, stop, NULL);
	for (i_strm_cnt = 0; i_strm_cnt < asf_demuxer_info->num_streams;
	     i_strm_cnt++) {
	    GstEvent *event = gst_event_new_new_segment(FALSE,
							asf_demuxer_info->
							seek_segment.rate,
							asf_demuxer_info->
							seek_segment.
							format,
							asf_demuxer_info->
							seek_segment.start,
							asf_demuxer_info->
							seek_segment.stop,
							asf_demuxer_info->
							seek_segment.
							start);
	    asf_stream_context *stream_ptr;
        
        /* Check for streams with only Audio or video as its content.
           In this case, we need to point to the 1st media type.
        */

        if(asf_demuxer_info->audio_present)
	    stream_ptr = &(asf_demuxer_info->asf_stream[i_strm_cnt]);
        else
            stream_ptr = &(asf_demuxer_info->asf_stream[i_strm_cnt + 1]);

    	    if (gst_pad_push_event(stream_ptr->src_pad, event) != TRUE)
    		    GST_ERROR("ASF push new segment event failed!!\n");
	}

    }
    //g_print("leave mfw_gst_asfdemuxer_handle_readytopaused!\n");
    return GST_STATE_CHANGE_SUCCESS;

	/* send an error message for an unrecoverable error */
Err_Msg:
	{
        GstMessage *message = NULL;
        GError *gerror = NULL;
        gerror = g_error_new_literal(1, 0, "ASF parser Error:Enable DEBBUGER to see details!");

        message = gst_message_new_error(GST_OBJECT(GST_ELEMENT(asf_demuxer_info)), gerror,
		          "debug none");
        gst_element_post_message(GST_ELEMENT(asf_demuxer_info), message);
        g_error_free(gerror);
        return GST_STATE_CHANGE_FAILURE;
	}
}

/***********************************************************************/

/*=============================================================================
FUNCTION:   	mfw_gst_asfdemuxer_change_state
	
DESCRIPTION:	This function keeps track of different states of pipeline.

ARGUMENTS PASSED:	element 	- pointer to the asf demuxer element.
					transition	- state of the pipeline.
     
RETURN VALUE:
        None

PRE-CONDITIONS:
        None

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/
static GstStateChangeReturn
mfw_gst_asfdemuxer_change_state(GstElement * element,
				GstStateChange transition)
{
    //g_print("enter mfw_gst_asfdemuxer_change_state!\n");
    GstStateChangeReturn ret;
    gboolean result;
    MFWGstAsfDemux *asf_demuxer_info = MFW_GST_ASF_DEMUXER(element);

    switch (transition) {
    case GST_STATE_CHANGE_NULL_TO_READY:
	{

        /* The parser initialisation is enabled by reseting This flag */
        asf_demuxer_info->init_parser = FALSE;
	    break;
	}
    case GST_STATE_CHANGE_READY_TO_PAUSED:
	{
        gst_tag_register (GST_TAG_MFW_ASF_WIDTH, GST_TAG_FLAG_DECODED,G_TYPE_UINT,
        "image width","image width(pixel)", NULL);
        gst_tag_register (GST_TAG_MFW_ASF_HEIGHT, GST_TAG_FLAG_DECODED,G_TYPE_UINT,
        "image height","image height(pixel)", NULL); 

	    asf_demuxer_info->task_paused = FALSE;
	    ret =
		mfw_gst_asfdemuxer_handle_readytopaused(asf_demuxer_info,element);
	    if (ret != GST_STATE_CHANGE_SUCCESS) {
		return GST_STATE_CHANGE_FAILURE;
	    }
	    break;
	}

    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
	{

	    break;
	}
    default:
	break;
    }
    ret =
	GST_ELEMENT_CLASS(parent_class)->change_state(element, transition);
    switch (transition) {
    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
	{
	    break;
	}

    case GST_STATE_CHANGE_PAUSED_TO_READY:
	{
	    asf_demuxer_info->task_paused = TRUE;
	    gst_asfdemuxer_close(asf_demuxer_info);
	    asf_demuxer_info->init_parser = FALSE;
	    break;
	}

    case GST_STATE_CHANGE_READY_TO_NULL:
	{
	    break;
	}
    default:
	{
	    break;
	}
    }

    //g_print("leave mfw_gst_asfdemuxer_change_state!\n");
    return ret;
}

/*=============================================================================
FUNCTION:   		gst_asfdemuxer_close

DESCRIPTION:		cloes the demuxer and free all the pads .

ARGUMENTS PASSED: 	demuxer - pointer to asf demuxer plugin structure.

RETURN VALUE:       None

PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
static void gst_asfdemuxer_close(MFWGstAsfDemux * demuxer)
{
	  //g_print("enter mfw_gst_asfdemuxer_close!\n");
    asf_stream_context *stream_ptr;

    /* close the demuxer */
    if (demuxer->asf_handle != NULL) {
	asf_parser_close(demuxer->asf_handle);
	demuxer->asf_handle = NULL;

    }

    if (demuxer->video_present) {
	demuxer->video_present = 0;
	stream_ptr = &(demuxer->asf_stream[1]);
	gst_element_remove_pad(GST_ELEMENT(demuxer), stream_ptr->src_pad);
    }
    if (demuxer->audio_present) {
	demuxer->audio_present = 0;
	stream_ptr = &(demuxer->asf_stream[0]);
	gst_element_remove_pad(GST_ELEMENT(demuxer), stream_ptr->src_pad);
    }
    demuxer->init_parser = FALSE;
    demuxer->state = GST_STATE_NULL;
    
    //g_print("leave mfw_gst_asfdemuxer_close!\n");
    return;
}

/*=============================================================================
FUNCTION:   		mfw_gst_asfdemuxer_class_init

DESCRIPTION:		Initialise the class only once (specifying what arguments
					and virtual functions the class has and setting global
					state)

ARGUMENTS PASSED: 	klass   -	MFWGstAsfDemuxClass

RETURN VALUE:       None

PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
static void mfw_gst_asfdemuxer_class_init(MFWGstAsfDemuxClass * klass)
{


    GstElementClass *gstelement_class = GST_ELEMENT_CLASS(klass);
    parent_class = g_type_class_peek_parent(klass);
    gstelement_class->change_state = mfw_gst_asfdemuxer_change_state;
    
    return;
}

/*=============================================================================
FUNCTION:  			mfw_gst_asfdemuxer_base_init

DESCRIPTION:		Elements details are registered with the plugin base_init,
					This function will initialize the class and child class 
					properties during each new child class creation.

ARGUMENTS PASSED:   Klass   -   void pointer

RETURN VALUE:       None

PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
static void mfw_gst_asfdemuxer_base_init(gpointer klass)
{
    static GstElementDetails mfw_gst_asfdemuxer_details =
	GST_ELEMENT_DETAILS("FSL Asf Demuxer",
			    "Codec/Demuxer",
			    "Demultiplex an asf file into audio and video",
			    FSL_GST_MM_PLUGIN_AUTHOR);

    GstElementClass *element_class = GST_ELEMENT_CLASS(klass);
    GstPadTemplate *videosrctempl = NULL;
    GstPadTemplate *audiosrctempl = NULL;
    GstCaps *audcaps = NULL;
    GstCaps *vidcaps = NULL;

    /* Create audio source pad template */
    gst_element_class_add_pad_template(element_class, audio_src_templ());
    gst_element_class_add_pad_template(element_class, video_src_templ());

    gst_element_class_add_pad_template(element_class,
				       gst_static_pad_template_get
				       (&mfw_gst_asfdemuxer_sink_template_factory));

    gst_element_class_set_details(element_class,
				  &mfw_gst_asfdemuxer_details);
    return;
}

/*=============================================================================
FUNCTION:			mfw_gst_type_asfdemuxer_get_type

DESCRIPTION:    	intefaces are initiated in this function.you can register one
	                or more interfaces  after hasfng registered the type itself.

ARGUMENTS PASSED:   None

RETURN VALUE:		A numerical value, which represents the unique identifier of
					this element (asf demuxer)

PRE-CONDITIONS:     None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
GType mfw_gst_type_asfdemuxer_get_type(void)
{
    static GType asfdemuxer_type = 0;
    //g_print("enter mfw_gst_asfdemuxer_get_type!\n");
    if (!asfdemuxer_type) {
	static const GTypeInfo asfdemuxer_info = {
	    sizeof(MFWGstAsfDemuxClass),
	    mfw_gst_asfdemuxer_base_init,
	    NULL,
	    (GClassInitFunc) mfw_gst_asfdemuxer_class_init,
	    NULL,
	    NULL,
	    sizeof(MFWGstAsfDemux),
	    0,
	    (GInstanceInitFunc) mfw_gst_asfdemuxer_init,
	};
	asfdemuxer_type = g_type_register_static(GST_TYPE_ELEMENT,
						 "mfw_asfdemuxer",
						 &asfdemuxer_info, 0);
    }

    GST_DEBUG_CATEGORY_INIT(mfw_gst_asfdemuxer_debug, "mfw_asfdemuxer",
			    0, "Asf demuxer");
		//g_print("leave mfw_gst_asfdemuxer_get_type!\n");
    return asfdemuxer_type;
}


/*=============================================================================
FUNCTION:   plugin_init

DESCRIPTION:    special function , which is called as soon as the plugin or
                element is loaded and information returned by this function
                will be cached in central registry

ARGUMENTS PASSED:
        plugin     -    pointer to container that contains features loaded
                        from shared object module

RETURN VALUE:
        return TRUE or FALSE depending on whether it loaded initialized any
        dependency correctly

PRE-CONDITIONS:		None

POST-CONDITIONS:    None

IMPORTANT NOTES:    None
=============================================================================*/
static gboolean plugin_init(GstPlugin * plugin)
{
    return gst_element_register(plugin, "mfw_asfdemuxer",
				GST_RANK_PRIMARY,
				MFW_GST_TYPE_ASF_DEMUXER);
}

/*===========================================================================*/


/*****************************************************************************/
/*    This is used to define the entry point and meta data of plugin         */
/*****************************************************************************/
GST_PLUGIN_DEFINE(GST_VERSION_MAJOR,	/*   major version of Gstreamer   */
		  GST_VERSION_MINOR,	/*   minor version of Gstreamer   */
		  "mfw_asfdemuxer",	/*   name of the plugin           */
		  " Demuxes the audio video asf streams ",	/* what plugin actually does */
		  plugin_init,	/*   first function to be called  */
		  VERSION,
		  GST_LICENSE_UNKNOWN,
		  FSL_GST_MM_PLUGIN_PACKAGE_NAME, FSL_GST_MM_PLUGIN_PACKAGE_ORIG)