#include <clutter-gst/clutter-gst.h>
#include <clutter/x11/clutter-x11.h>
#include <X11/extensions/XTest.h>
#include <X11/keysym.h>
#include "hrn-iface-player.h"
#include "hrn-video-player.h"
#include "hrn.h"

enum
{
  PROP_0,
};

struct _HrnVideoPlayerPrivate
{
  ClutterActor *video_player;
  guint32       progress_id;
};

static void hrn_player_init (HrnIfacePlayerClass *iface);
#define GET_PRIVATE(obj)  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
                                                        HRN_TYPE_VIDEO_PLAYER, \
                                                        HrnVideoPlayerPrivate))
G_DEFINE_TYPE_WITH_CODE (HrnVideoPlayer, hrn_video_player, CLUTTER_TYPE_ACTOR,
                         G_IMPLEMENT_INTERFACE (HRN_TYPE_IFACE_PLAYER,
                                                hrn_player_init));

static void
hrn_video_player_finalize (GObject *object)
{
  G_OBJECT_CLASS (hrn_video_player_parent_class)->finalize (object);
}

static void
hrn_video_player_dispose (GObject *object)
{
  HrnVideoPlayer        *player = (HrnVideoPlayer *) object;
  HrnVideoPlayerPrivate *priv   = player->priv;

  if (priv->video_player)
    {
      g_signal_handler_disconnect (priv->video_player, priv->progress_id);
      priv->progress_id = 0;

      clutter_media_set_playing ((ClutterMedia *) priv->video_player, FALSE);
      clutter_actor_destroy (priv->video_player);
      priv->video_player = NULL;
    }

  G_OBJECT_CLASS (hrn_video_player_parent_class)->dispose (object);
}

static void
hrn_video_player_set_property (GObject *object, guint prop_id,
                               const GValue *value,
                               GParamSpec   *pspec)
{
  switch (prop_id)
    {
      default:
        break;
    }
}

static void
hrn_video_player_get_property (GObject *object, guint prop_id,
                               GValue     *value,
                               GParamSpec *pspec)
{
  switch (prop_id)
    {
      default:
        break;
    }
}

static void
hrn_video_player_allocate (ClutterActor *actor, const ClutterActorBox *box,
                           ClutterAllocationFlags flags)
{
  HrnVideoPlayer        *player = (HrnVideoPlayer *) actor;
  HrnVideoPlayerPrivate *priv   = player->priv;

  CLUTTER_ACTOR_CLASS (hrn_video_player_parent_class)->allocate
                             (actor, box, flags);

  if (priv->video_player)
    {
      clutter_actor_allocate_preferred_size (priv->video_player,
                                             flags);
    }
}

static void
hrn_video_player_paint (ClutterActor *actor)
{
  HrnVideoPlayer        *player = (HrnVideoPlayer *) actor;
  HrnVideoPlayerPrivate *priv   = player->priv;

  if (priv->video_player && CLUTTER_ACTOR_IS_VISIBLE (priv->video_player))
    {
      clutter_actor_paint (priv->video_player);
    }
}

static void
hrn_video_player_pick (ClutterActor *actor, const ClutterColor *color)
{
  hrn_video_player_paint (actor);
}


static void
hrn_video_player_map (ClutterActor *self)
{
  HrnVideoPlayerPrivate *priv = HRN_VIDEO_PLAYER (self)->priv;

  CLUTTER_ACTOR_CLASS (hrn_video_player_parent_class)->map (self);

  if (priv->video_player)
    clutter_actor_map (CLUTTER_ACTOR (priv->video_player));
}

static void
hrn_video_player_unmap (ClutterActor *self)
{
  HrnVideoPlayerPrivate *priv = HRN_VIDEO_PLAYER (self)->priv;

  CLUTTER_ACTOR_CLASS (hrn_video_player_parent_class)->unmap (self);

  if (priv->video_player)
    clutter_actor_unmap (CLUTTER_ACTOR (priv->video_player));
}


static void
hrn_video_player_class_init (HrnVideoPlayerClass *klass)
{
  GObjectClass      *o_class = (GObjectClass *) klass;
  ClutterActorClass *a_class = (ClutterActorClass *) klass;

  o_class->dispose      = hrn_video_player_dispose;
  o_class->finalize     = hrn_video_player_finalize;
  o_class->set_property = hrn_video_player_set_property;
  o_class->get_property = hrn_video_player_get_property;

  a_class->allocate = hrn_video_player_allocate;
  a_class->paint    = hrn_video_player_paint;
  a_class->pick     = hrn_video_player_pick;
  a_class->map      = hrn_video_player_map;
  a_class->unmap    = hrn_video_player_unmap;

  g_type_class_add_private (klass, sizeof (HrnVideoPlayerPrivate));
}

static void
video_size_change (ClutterTexture *texture,
                   gfloat          width,
                   gfloat          height,
                   gpointer        userdata)
{
  gfloat new_x, new_y, new_width, new_height;

  new_height = (height * CLUTTER_STAGE_WIDTH ()) / width;

  if (new_height <= CLUTTER_STAGE_HEIGHT ())
    {
      new_width = CLUTTER_STAGE_WIDTH ();

      new_x = 0;
      new_y = (CLUTTER_STAGE_HEIGHT () - new_height) / 2;
    }
  else
    {
      new_width  = (width * CLUTTER_STAGE_HEIGHT ()) / height;
      new_height = CLUTTER_STAGE_HEIGHT ();

      new_x = (CLUTTER_STAGE_WIDTH () - new_width) / 2;
      new_y = 0;
    }


  clutter_actor_set_position ((ClutterActor *) texture, new_x, new_y);
  clutter_actor_set_size ((ClutterActor *) texture, new_width, new_height);
}

static void
emit_fake_key_event (void)
{
  Display *display = clutter_x11_get_default_display ();
  XLockDisplay (display);
  XTestFakeKeyEvent (display, XKeysymToKeycode (display, XK_Shift_L), True, CurrentTime);
  XTestFakeKeyEvent (display, XKeysymToKeycode (display, XK_Shift_L), False, CurrentTime);
  XUnlockDisplay (display);
}

static void
video_progress_changed (GObject *object, GParamSpec     *pspec,
                        HrnVideoPlayer *player)
{
  HrnVideoPlayerPrivate *priv = player->priv;
  double                 position;

  position = clutter_media_get_progress ((ClutterMedia *) priv->video_player);
  hrn_iface_player_emit_position_changed ((HrnIfacePlayer *) player,
                                          position);
  hrn_store_state ();

  emit_fake_key_event ();
}

static void
video_eos (ClutterMedia *media, HrnVideoPlayer *player)
{
  hrn_iface_player_emit_eos ((HrnIfacePlayer *) player);
}

static void
player_set_playing (HrnIfacePlayer *player, gboolean playing)
{
  HrnVideoPlayer        *video = (HrnVideoPlayer *) player;
  HrnVideoPlayerPrivate *priv  = video->priv;

  clutter_media_set_playing ((ClutterMedia *) priv->video_player, playing);
}

static void
player_set_uri (HrnIfacePlayer *player, const char     *uri)
{
  HrnVideoPlayer        *video = (HrnVideoPlayer *) player;
  HrnVideoPlayerPrivate *priv  = video->priv;

  clutter_media_set_uri ((ClutterMedia *) priv->video_player, uri);
}

static void
player_set_position (HrnIfacePlayer *player, double position)
{
  HrnVideoPlayer        *video = (HrnVideoPlayer *) player;
  HrnVideoPlayerPrivate *priv  = video->priv;

  clutter_media_set_progress ((ClutterMedia *) priv->video_player, position);
}

static void
hrn_player_init (HrnIfacePlayerClass *iface)
{
  iface->set_playing  = player_set_playing;
  iface->set_uri      = player_set_uri;
  iface->set_position = player_set_position;
}

static void
hrn_clutter_media_error (ClutterMedia *media,
                         GError       *error,
                         gpointer      user_data)
{
  NotifyNotification *notification;

  if (error->domain == g_quark_from_string ("gst-stream-error-quark") &&
      error->code == 6 /* GST_STREAM_ERROR_CODEC_NOT_FOUND */)
    {
      GFile *file;
      gchar *basename;
      gchar *uri = NULL;
      gchar *message;
 
      g_object_get (media, "uri", &uri, NULL);
      file = g_file_new_for_uri (uri);
      basename = g_file_get_basename (file);

      message = g_strdup_printf (_("Sorry, we can't play %s, as we don't have the correct plugin. You could try searching for ❝gstreamer codecs❞ on the web."), basename);
      
      notification = notify_notification_new (_("We can't play this"),
                                            message,
                                            NULL, NULL);
      notify_notification_show (notification,   NULL);

      g_object_unref (notification);
      g_object_unref (file);
      g_free (basename);
      g_free (uri);
      g_free (message);
    }
}

static void
hrn_video_player_init (HrnVideoPlayer *self)
{
  HrnVideoPlayerPrivate *priv = GET_PRIVATE (self);

  self->priv         = priv;
  priv->video_player = clutter_gst_video_texture_new ();

  /* Don't let the underlying pixbuf dictate size */
  g_object_set (G_OBJECT (priv->video_player),
                "sync-size", FALSE,
                NULL);

  /* Handle it ourselves so can scale up for fullscreen better */
  g_signal_connect (priv->video_player, "size-change",
                    G_CALLBACK (video_size_change), NULL);

  g_signal_connect (priv->video_player, "error",
                    G_CALLBACK (hrn_clutter_media_error), NULL);

  priv->progress_id = g_signal_connect (priv->video_player,
                                        "notify::progress",
                                        G_CALLBACK (video_progress_changed),
                                        self);
  g_signal_connect (priv->video_player, "eos",
                    G_CALLBACK (video_eos), self);
  clutter_actor_set_parent (priv->video_player, CLUTTER_ACTOR (self));
  clutter_actor_show (priv->video_player);
}

