#include <string.h>
#include <stdlib.h>

#include <glib.h>
#include <bickley/bkl-item.h>
#include <bickley/bkl-item-audio.h>
#include <bickley/bkl-item-image.h>
#include <bickley/bkl-item-video.h>

#include "hrn-cluster-node.h"
#include "hrn-cluster-tree.h"

typedef struct _ClusterItem {
    BklItem *item;
    HrnClusterNode *node;
} ClusterItem;

#define MAX_ARTISTS_IN_NAME 3
static char *
make_artist_name (GPtrArray *artists)
{
    int i;
    GString *builder;
    char *str;

    if (artists == NULL) {
        return g_strdup ("Unknown");
    }

    builder = g_string_new (artists->pdata[0]);
    for (i = 1; i < MAX_ARTISTS_IN_NAME && i < artists->len; i++) {
        g_string_append (builder, ", ");
        g_string_append (builder, artists->pdata[i]);
    }

    if (artists->len > MAX_ARTISTS_IN_NAME) {
        g_string_append_printf (builder, " and %d more",
                                artists->len - MAX_ARTISTS_IN_NAME);
    }

    str = builder->str;
    g_string_free (builder, FALSE);

    return str;
}

static int
sort_tracks (gconstpointer a,
             gconstpointer b,
             gpointer      userdata)
{
    HrnClusterNode *node_a, *node_b;
    TrackCluster *item_a, *item_b;
    guint track_a, track_b;

    node_a = (HrnClusterNode *) a;
    node_b = (HrnClusterNode *) b;

    item_a = (TrackCluster *) node_a->data;
    item_b = (TrackCluster *) node_b->data;

    track_a = bkl_item_audio_get_track ((BklItemAudio *) item_a->item);
    track_b = bkl_item_audio_get_track ((BklItemAudio *) item_b->item);

    if (track_a == 0 && track_b > 0) {
        return -1;
    } else if (track_a > 0 && track_b == 0) {
        return 1;
    } else if (track_a > 0 && track_b > 0) {
        return track_a - track_b;
    } else {
        const char *name_a, *name_b;

        name_a = bkl_item_audio_get_title ((BklItemAudio *) item_a->item);
        name_b = bkl_item_audio_get_title ((BklItemAudio *) item_b->item);

        if (name_a == NULL) {
            return -1;
        } else if (name_b == NULL) {
            return 1;
        }

        return strcmp (name_a, name_b);
    }

    /* should never get here */
    return 0;
}

static const char *
make_album_name (const char *name)
{
    if (name == NULL) {
        return NULL;
    }

    return g_strstrip ((char *) name);
}

/* merge @src into @dest without duplicating */
static void
merge_artists (GPtrArray *dest,
               GPtrArray *src)
{
    int i;

    if (src == NULL) {
        return;
    }

    /* FIXME: This could be speeded up for large cases by using a hashtable.
       --- a) Add all entries in @src into hashtable
           b) Remove all entries in @dest from hashtable
           c) Add the entries left in hashtable to dest

           Thing is, large numbers of items in @dest and @src
           are not very common, so it might not be worth the extra overhead
    */
    for (i = 0; i < src->len; i++) {
        int j;

        for (j = 0; j < dest->len; j++) {
            if (g_ascii_strcasecmp (src->pdata[i], dest->pdata[j]) == 0) {
                /* Found match, skip */
                goto next_src;
            }
        }

        g_ptr_array_add (dest, g_strdup (src->pdata[i]));

      next_src:
        ; /* No-op */
    }
}

static int
sort_albums (gconstpointer a,
             gconstpointer b,
             gpointer      userdata)
{
    HrnClusterNode *node_a, *node_b;
    AlbumCluster *album_a, *album_b;
    const char *title_a, *title_b;
    int year_a, year_b;

    node_a = (HrnClusterNode *) a;
    node_b = (HrnClusterNode *) b;

    album_a = (AlbumCluster *) node_a->data;
    album_b = (AlbumCluster *) node_b->data;

    year_a = album_a->year;
    year_b = album_b->year;

    if (year_a > 0 && year_b > 0) {
        return year_a - year_b;
    } else if (year_a > 0) {
        return 1;
    } else if (year_b > 0) {
        return -1;
    }

    /* if neither has a year, then sort by name */
    title_a = node_a->canonical_name;
    title_b = node_b->canonical_name;

    if (title_a == NULL) {
        return -1;
    } else if (title_b == NULL) {
        return 1;
    } else {
        return strcmp (title_a, title_b);
    }

    /* Shouldn't get here */
    return 0;
}

static char *
make_canonical (const char *name)
{
    char *canonical;

    canonical = g_ascii_strdown (name, -1);
    return g_strstrip (canonical);
}

static HrnClusterNode *
get_artist (HrnClusterNode *root,
            const char     *name)
{
    AudioCluster *audio_cluster = (AudioCluster *) root->data;
    HrnClusterNode *artist;
    char *canonical;
    ArtistCluster *artist_cluster;

    canonical = make_canonical (name);
    artist = g_hash_table_lookup (audio_cluster->name_to_artist, canonical);
    if (artist) {
        g_free (canonical);
        return artist;
    }

    artist = hrn_cluster_node_new (HRN_CLUSTER_NODE_TYPE_ARTIST, sort_albums);
    artist->name = g_strdup (name);
    artist->canonical_name = canonical;

    artist_cluster = g_slice_new0 (ArtistCluster);
    artist->data = (gpointer) artist_cluster;

    hrn_cluster_node_add_child (root, artist);

    g_hash_table_insert (audio_cluster->name_to_artist,
                         artist->canonical_name, artist);
    return artist;
}

static void
free_artist (HrnClusterNode *root,
             HrnClusterNode *artist)
{
    AudioCluster *audio_cluster = (AudioCluster *) root->data;

    g_hash_table_remove (audio_cluster->name_to_artist, artist->canonical_name);

    hrn_cluster_node_remove_child (root, artist);
    g_object_unref (artist);
}

static void
remove_album_from_artist (HrnClusterNode *root,
                          HrnClusterNode *artist,
                          HrnClusterNode *album)
{
    /* FIXME: Should remove thumbnail uri too? */
    hrn_cluster_node_remove_child (artist, album);

    if (g_sequence_get_length (artist->children) == 0) {
        free_artist (root, artist);
    }
}

static void
add_audio_item (HrnClusterTree *tree,
                BklItemAudio   *audio)
{
    HrnClusterNode *root = tree->audio;
    AudioCluster *cluster = root->data;
    ArtistCluster *artist_cluster;
    AlbumCluster *album_cluster;
    TrackCluster *track_cluster;
    HrnClusterNode *track;
    HrnClusterNode *artist;
    HrnClusterNode *album;
    ClusterItem *ci;
    const char *album_name;
    char *canonical_album;
    int artist_count;

    /* No sorter for the item as it has no children */
    track = hrn_cluster_node_new (HRN_CLUSTER_NODE_TYPE_TRACK, NULL);

    track->name = g_strdup (bkl_item_audio_get_title (audio));
    track_cluster = g_slice_new (TrackCluster);
    track_cluster->item = (BklItem *) audio;
    track->data = (gpointer) track_cluster;

    g_hash_table_insert (tree->uri_to_item,
                         (char *) bkl_item_get_uri ((BklItem *) audio), track);
    ci = g_slice_new (ClusterItem);
    ci->item = (BklItem *) audio;
    ci->node = track;
    g_ptr_array_add (tree->items, ci);

    /* Each item can only be in one album */
    album_name = bkl_item_audio_get_album (audio);
    if (album_name == NULL) {
        album_name = "Unknown";
        canonical_album = g_strdup ("Unknown");
    } else {
        make_album_name (album_name);
        canonical_album = make_canonical (album_name);
    }

    album = g_hash_table_lookup (cluster->name_to_album, canonical_album);

    if (album == NULL) {
        album = hrn_cluster_node_new (HRN_CLUSTER_NODE_TYPE_ALBUM, sort_tracks);
        album->canonical_name = canonical_album;
        album->name = g_strdup (album_name);

        g_hash_table_insert (cluster->name_to_album,
                             album->canonical_name, album);

        album_cluster = g_slice_new0 (AlbumCluster);
        album_cluster->artists = g_ptr_array_new ();
        album->data = (gpointer) album_cluster;
    } else {
        album_cluster = (AlbumCluster *) album->data;
    }

    if (album_cluster->thumbnail_uri == NULL) {
        const char *thumb;

        thumb = bkl_item_extended_get_thumbnail ((BklItemExtended *) audio);
        if (thumb) {
            album_cluster->thumbnail_uri = g_strdup (thumb);
        }
    }

    /* Update the year if we need to */
    if (album_cluster->year == 0) {
        const char *year = bkl_item_audio_get_year (audio);

        if (year) {
            album_cluster->year = atoi (year);
        }
    }

    /* Add the item into the album */
    hrn_cluster_node_add_child (album, track);

    /* Now work out what artist this album belongs to */
    artist_count = album_cluster->artists->len;
    merge_artists (album_cluster->artists, bkl_item_audio_get_artists (audio));

    if (album_cluster->artists->len != artist_count) {
        char *canonical;

        artist = album->parent;
        if (artist == NULL) {
            /* The album has never been owned before,
               so create a new artist for it */
            canonical = make_artist_name (album_cluster->artists);
            artist = get_artist (root, canonical);
            g_free (canonical);
        } else {
            /* Album was originally owned by an old artist, but the artists
               have changed */
            remove_album_from_artist (root, artist, album);

            /* Now get the new artist, creating it if it doesn't exist */
            canonical = make_artist_name (album_cluster->artists);
            artist = get_artist (root, canonical);

            g_free (canonical);
        }

        hrn_cluster_node_add_child (artist, album);
    } else {
        char *canonical;

        if (album->parent) {
            artist = album->parent;
        } else {
            if (album_cluster->artists->len == 0) {
                canonical = g_strdup ("UNKNOWN");
            } else {
                /* This should never actually happen, because if the album has
                   > 0 artists, then it'll have a parent set */
                g_warning ("Unowned album with %d artists found",
                           album_cluster->artists->len);
                canonical = make_artist_name (album_cluster->artists);
            }

            artist = get_artist (root, canonical);
            g_free (canonical);

            hrn_cluster_node_add_child (artist, album);
        }
    }

    if (artist) { /* Should never be NULL, but just to check */
        artist_cluster = (ArtistCluster *) artist->data;

        /* FIXME: Should allow max 5 thumbnails */
        if (artist_cluster->thumbnail_uri == NULL) {
            const char *thumb;

            thumb = bkl_item_extended_get_thumbnail ((BklItemExtended *) audio);
            if (thumb) {
                artist_cluster->thumbnail_uri = g_strdup (thumb);
            }
        }
    }
}

static int
sort_artists (gconstpointer a,
              gconstpointer b,
              gpointer      userdata)
{
    HrnClusterNode *node_a, *node_b;

    node_a = (HrnClusterNode *) a;
    node_b = (HrnClusterNode *) b;

    g_assert (node_a->canonical_name);
    g_assert (node_b->canonical_name);

    return strcmp (node_a->canonical_name, node_b->canonical_name);
}

#if 0
static void
dump_tracks (HrnClusterNode *node)
{
    GSequenceIter *iter = g_sequence_get_begin_iter (node->children);

    while (g_sequence_iter_is_end (iter) == FALSE) {
        HrnClusterNode *child = g_sequence_get (iter);
        TrackCluster *track = (TrackCluster *) child->data;

        g_print ("      (%d) %s\n",
                 bkl_item_audio_get_track ((BklItemAudio *) track->item),
                 bkl_item_audio_get_title ((BklItemAudio *) track->item));

        iter = g_sequence_iter_next (iter);
    }
}

static void
dump_albums (HrnClusterNode *node)
{
    GSequenceIter *iter = g_sequence_get_begin_iter (node->children);

    while (g_sequence_iter_is_end (iter) == FALSE) {
        HrnClusterNode *child = g_sequence_get (iter);
        AlbumCluster *album = (AlbumCluster *) child->data;

        g_print ("   %s (%d)\n", child->name, album->year);
        dump_tracks (child);

        iter = g_sequence_iter_next (iter);
    }
}

static void
dump_artists (HrnClusterNode *root)
{
    GSequenceIter *iter = g_sequence_get_begin_iter (root->children);

    g_print ("Dumping %d artists\n", g_sequence_get_length (root->children));
    while (g_sequence_iter_is_end (iter) == FALSE) {
        HrnClusterNode *child = g_sequence_get (iter);

        g_print ("%s\n", child->name);

        dump_albums (child);

        iter = g_sequence_iter_next (iter);
    }
}
#endif

static HrnClusterNode *
create_audio (void)
{
    HrnClusterNode *audio;
    AudioCluster *audio_cluster;

    audio = hrn_cluster_node_new (HRN_CLUSTER_NODE_TYPE_AUDIO_ROOT,
                                  sort_artists);
    audio->name = g_strdup ("audio-root");
    audio->canonical_name = g_strdup ("audio-root");
    audio_cluster = g_slice_new (AudioCluster);
    audio_cluster->name_to_artist = g_hash_table_new (g_str_hash, g_str_equal);
    audio_cluster->name_to_album = g_hash_table_new (g_str_hash, g_str_equal);
    audio->data = audio_cluster;

    return audio;
}

static int
sort_images (gconstpointer a,
             gconstpointer b,
             gpointer      userdata)
{
    HrnClusterNode *node_a, *node_b;

    node_a = (HrnClusterNode *) a;
    node_b = (HrnClusterNode *) b;

    return strcmp (node_a->canonical_name, node_b->canonical_name);
}

static int
sort_years (gconstpointer a,
            gconstpointer b,
            gpointer      userdata)
{
    HrnClusterNode *node_a, *node_b;
    YearCluster *cluster_a, *cluster_b;
    int year_a, year_b;

    node_a = (HrnClusterNode *) a;
    node_b = (HrnClusterNode *) b;

    cluster_a = (YearCluster *) node_a->data;
    cluster_b = (YearCluster *) node_b->data;

    year_a = cluster_a->year;
    year_b = cluster_b->year;

    if (year_a > 0 && year_b > 0) {
        return year_a - year_b;
    } else if (year_a > 0) {
        /* If a > 0 and b == 0 we want to return that a comes first */
        return -1;
    } else if (year_b > 0) {
        return 1;
    }

    return 0;
}

static int
sort_months (gconstpointer a,
             gconstpointer b,
             gpointer      userdata)
{
    HrnClusterNode *node_a, *node_b;
    MonthCluster *cluster_a, *cluster_b;
    int month_a, month_b;

    node_a = (HrnClusterNode *) a;
    node_b = (HrnClusterNode *) b;

    cluster_a = (MonthCluster *) node_a->data;
    cluster_b = (MonthCluster *) node_b->data;

    month_a = cluster_a->month;
    month_b = cluster_b->month;

    if (month_a > 0 && month_b > 0) {
        return month_a - month_b;
    } else if (month_a > 0) {
        /* If a > 0 and b == 0 we want to return that a comes first */
        return -1;
    } else if (month_b > 0) {
        return 1;
    }

    return 0;
}

static char *months[12] = {
    "January", "Feburary", "March", "April", "May", "June", "July",
    "August", "September", "October", "November", "December"
};

static void
add_image_item (HrnClusterTree *tree,
                BklItemImage   *image)
{
    HrnClusterNode *root = tree->image;
    ImageCluster *cluster = root->data;
    HrnClusterNode *year_node;
    HrnClusterNode *month_node;
    HrnClusterNode *picture_node;
    YearCluster *year_cluster;
    MonthCluster *month_cluster;
    PictureCluster *picture_cluster;
    ClusterItem *ci;
    const char *date_taken;
    int m, y;
    char *month, *year, *canonical, *title;

    date_taken = bkl_item_image_get_time_original (image);
    if (date_taken == NULL || *date_taken == '\0') {
        m = 13;
        y = 0;
        month = g_strdup ("Unknown");
        year = g_strdup ("Unknown");
    } else {
        m = atoi (date_taken + 5);
        y = atoi (date_taken);

        if (m == 0) {
            month = g_strdup ("Unknown");
        } else {
            month = g_strdup (months[m - 1]);
        }

        if (y == 0) {
            year = g_strdup ("Unknown");
        } else {
            year = g_strndup (date_taken, 4);
        }

#if 0
        g_print ("%s: %s %s (%s)\n", date_taken, month, year,
                 bkl_item_get_uri (image));
#endif
    }

    canonical = g_strdup_printf ("%s - %s", year, month);
    year_node = g_hash_table_lookup (cluster->year_to_cluster, year);
    if (year_node == NULL) {
        year_cluster = g_slice_new0 (YearCluster);

        year_node = hrn_cluster_node_new (HRN_CLUSTER_NODE_TYPE_YEAR,
                                          sort_months);
        year_node->canonical_name = year;
        year_node->name = g_strdup (year);

        year_cluster->year = y;
        year_node->data = year_cluster;

        g_hash_table_insert (cluster->year_to_cluster,
                             year_node->canonical_name, year_node);
        hrn_cluster_node_add_child (root, year_node);
    } else {
        year_cluster = (YearCluster *) year_node->data;
        g_free (year);
    }

    if (year_cluster->thumbnail_uri == NULL) {
        const char *thumb;

        thumb = bkl_item_extended_get_thumbnail ((BklItemExtended *) image);
        if (thumb) {
            year_cluster->thumbnail_uri = g_strdup (thumb);
        }
    }

    month_node = g_hash_table_lookup (cluster->month_to_cluster, canonical);
    if (month_node == NULL) {
        month_cluster = g_slice_new0 (MonthCluster);

        month_node = hrn_cluster_node_new (HRN_CLUSTER_NODE_TYPE_MONTH,
                                           sort_images);
        month_node->canonical_name = canonical;
        month_node->name = month;

        month_cluster->month = m;
        month_node->data = month_cluster;

        hrn_cluster_node_add_child (year_node, month_node);
        g_hash_table_insert (cluster->month_to_cluster,
                             month_node->canonical_name, month_node);
    } else {
        month_cluster = (MonthCluster *) month_node->data;
        g_free (canonical);
        g_free (month);
    }

    if (month_cluster->thumbnail_uri == NULL) {
        const char *thumb;

        thumb = bkl_item_extended_get_thumbnail ((BklItemExtended *) image);
        if (thumb) {
            month_cluster->thumbnail_uri = g_strdup (thumb);
        }
    }

    picture_node = hrn_cluster_node_new (HRN_CLUSTER_NODE_TYPE_IMAGE, NULL);
    title = (char *) bkl_item_image_get_title (image);
    if (title == NULL) {
        const char *uri = bkl_item_get_uri ((BklItem *) image);

        /* FIXME: Remove extension */
        title = g_path_get_basename (uri);
    } else {
        title = g_strdup (title);
    }

    picture_node->name = title;
    picture_node->canonical_name = g_strdup (title);

    picture_cluster = g_slice_new (PictureCluster);
    picture_cluster->item = (BklItem *) image;
    picture_node->data = (gpointer) picture_cluster;

    hrn_cluster_node_add_child (month_node, picture_node);

    g_hash_table_insert (tree->uri_to_item,
                         (char *) bkl_item_get_uri ((BklItem *) image),
                         picture_node);

    ci = g_slice_new (ClusterItem);
    ci->item = (BklItem *) image;
    ci->node = picture_node;
    g_ptr_array_add (tree->items, ci);
}

static void
add_video_item (HrnClusterTree *tree,
                BklItemVideo   *video)
{
    HrnClusterNode *root = tree->video;
    HrnClusterNode *video_node;
    VideoCluster *video_cluster;
    ClusterItem *ci;

    video_node = hrn_cluster_node_new (HRN_CLUSTER_NODE_TYPE_VIDEO, NULL);
    video_node->name = g_path_get_basename (bkl_item_get_uri ((BklItem *) video));
    video_node->canonical_name = g_strdup (video_node->name);

    video_cluster = g_slice_new (VideoCluster);
    video_cluster->item = (BklItem *) video;
    video_node->data = (gpointer) video_cluster;

    g_hash_table_insert (tree->uri_to_item,
                         (char *) bkl_item_get_uri ((BklItem *) video),
                         video_node);

    ci = g_slice_new (ClusterItem);
    ci->item = (BklItem *) video;
    ci->node = video_node;
    g_ptr_array_add (tree->items, ci);

    hrn_cluster_node_add_child (root, video_node);
}

static HrnClusterNode *
create_image (void)
{
    HrnClusterNode *image;
    ImageCluster *image_cluster;

    image = hrn_cluster_node_new (HRN_CLUSTER_NODE_TYPE_IMAGE_ROOT,
                                  sort_years);
    image->name = g_strdup ("image-root");
    image->canonical_name = g_strdup ("image-root");

    image_cluster = g_slice_new (ImageCluster);
    image_cluster->year_to_cluster = g_hash_table_new (g_str_hash, g_str_equal);
    image_cluster->month_to_cluster = g_hash_table_new (g_str_hash, g_str_equal);
    image->data = image_cluster;

    return image;
}

static int
sort_videos (gconstpointer a,
             gconstpointer b,
             gpointer      userdata)
{
    HrnClusterNode *node_a, *node_b;

    node_a = (HrnClusterNode *) a;
    node_b = (HrnClusterNode *) b;

    return strcmp (node_a->canonical_name, node_b->canonical_name);
}

static HrnClusterNode *
create_video (void)
{
    HrnClusterNode *video;

    video = hrn_cluster_node_new (HRN_CLUSTER_NODE_TYPE_VIDEO_ROOT,
                                  sort_videos);
    video->name = g_strdup ("video-root");
    video->canonical_name = g_strdup ("video-root");

    return video;
}

HrnClusterTree *
hrn_cluster_tree_new (void)
{
    HrnClusterTree *cluster_tree;

    cluster_tree = g_new0 (HrnClusterTree, 1);
    cluster_tree->audio = create_audio ();
    cluster_tree->image = create_image ();
    cluster_tree->video = create_video ();
    cluster_tree->uri_to_item = g_hash_table_new (g_str_hash, g_str_equal);
    cluster_tree->items = g_ptr_array_new ();

    return cluster_tree;
}

void
hrn_cluster_tree_add_item (HrnClusterTree *tree,
                           BklItem        *item)
{
    switch (bkl_item_get_item_type (item)) {
    case BKL_ITEM_TYPE_AUDIO:
        add_audio_item (tree, (BklItemAudio *) item);
        break;

    case BKL_ITEM_TYPE_IMAGE:
        add_image_item (tree, (BklItemImage *) item);
        break;

    case BKL_ITEM_TYPE_VIDEO:
        add_video_item (tree, (BklItemVideo *) item);
        break;

    default:
        break;
    }
}

static void
hrn_cluster_tree_dump_node (HrnClusterNode *node,
                            int             indent)
{
    int i;
    for (i = 0; i < indent; i++) {
        g_print (" ");
    }

    g_print ("Node->name: %s\n", node->name);
    if (node->parent) {
        hrn_cluster_tree_dump_node (node->parent, indent + 3);
    }
}

void
hrn_cluster_tree_dump_items (HrnClusterTree *tree)
{
    int i;

    for (i = 0; i < tree->items->len; i++) {
        ClusterItem *ci = tree->items->pdata[i];

        g_print ("%s\n", bkl_item_get_uri (ci->item));
        hrn_cluster_tree_dump_node (ci->node, 3);
    }
}

void
hrn_cluster_tree_filter_items (HrnClusterTree *tree,
                               GHashTable     *filter)
{
    int i;

    for (i = 0; i < tree->items->len; i++) {
        ClusterItem *ci = tree->items->pdata[i];
        const char *uri = bkl_item_get_uri (ci->item);

        /* If filter is NULL then we want to show all results */
        if (filter == NULL || g_hash_table_lookup (filter, uri)) {
            hrn_cluster_node_set_hidden (ci->node, FALSE);
        } else {
            hrn_cluster_node_set_hidden (ci->node, TRUE);
        }
    }
}

static gboolean
filter_node_name (const char *name,
                  char      **search_terms)
{
    int i;
    char *up_name;

    up_name = g_ascii_strup (name, -1);

    for (i = 0; search_terms[i]; i++) {
        if (strstr (up_name, search_terms[i]) == FALSE) {
            g_free (up_name);
            return FALSE;
        }
    }

    g_free (up_name);

    return TRUE;
}

static void
filter_node (HrnClusterNode *node,
             char          **search_terms)
{
    if (node->type == HRN_CLUSTER_NODE_TYPE_TRACK ||
        node->type == HRN_CLUSTER_NODE_TYPE_IMAGE) {
        return;
    }

    if (node->children) {
        GSequenceIter *iter = g_sequence_get_begin_iter (node->children);

        while (g_sequence_iter_is_end (iter) == FALSE) {
            HrnClusterNode *child_node = g_sequence_get (iter);

            filter_node (child_node, search_terms);
            iter = g_sequence_iter_next (iter);
        }
    }

    if (node->type == HRN_CLUSTER_NODE_TYPE_AUDIO_ROOT ||
        node->type == HRN_CLUSTER_NODE_TYPE_IMAGE_ROOT) {
        return;
    }

    if (filter_node_name (node->name, search_terms)) {
        hrn_cluster_node_set_children_hidden (node, FALSE);
    }
}

void
hrn_cluster_tree_filter_nodes (HrnClusterTree *tree,
                               const char     *text)
{
    char **search_terms, *search_text;

    search_text  = g_ascii_strup (text, -1);
    /* FIXME: Should we just remove all punctuation? */
    search_terms = g_strsplit_set (search_text, " :➜", 0);
    g_free (search_text);

    filter_node (tree->audio, search_terms);
    filter_node (tree->image, search_terms);
    /* Don't need to filter video because there are no nodes
       and all video items will be filtered by filter_items */

    g_strfreev (search_terms);
}

void
hrn_cluster_tree_free (HrnClusterTree *tree)
{
    int i;

    for (i = 0; i < tree->items->len; i++) {
        ClusterItem *ci = tree->items->pdata[i];
        g_slice_free (ClusterItem, ci);
    }
    g_ptr_array_free (tree->items, TRUE);

    g_hash_table_destroy (tree->uri_to_item);

    g_object_unref (tree->audio);
    g_object_unref (tree->image);
    g_object_unref (tree->video);

    g_free (tree);
}

static void
remove_child_from_parent (HrnClusterNode *child,
                          HrnClusterNode *parent)
{
    hrn_cluster_node_remove_child (parent, child);
    g_object_unref (child);

    /* Check if the parent has any children left,
       it might need removed too */
    if (g_sequence_get_length (parent->children) == 0) {
        if (parent->parent) {
            remove_child_from_parent (parent, parent->parent);
        }
    }
}

void
hrn_cluster_tree_remove_item (HrnClusterTree *tree,
                              BklItem        *item)
{
    HrnClusterNode *node = NULL;
    int i, idx = -1;

    /* Find the ClusterItem */
    for (i = 0; i < tree->items->len; i++) {
        ClusterItem *ci = tree->items->pdata[i];

        if (ci->item == item) {
            node = ci->node;
            g_slice_free (ClusterItem, ci);
            idx = i;
            break;
        }
    }

    if (node == NULL) {
        g_warning ("%s is not contained in the tree", bkl_item_get_uri (item));
        return;
    }

    if (idx > -1) {
        g_ptr_array_remove_index_fast (tree->items, idx);
    }

    g_hash_table_remove (tree->uri_to_item, bkl_item_get_uri (item));

    if (node->parent) {
        remove_child_from_parent (node, node->parent);
    }
}
