/*
 * Hornsey - Moblin Media Player.
 * Copyright © 2009 Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */


#define ROW_OFFSET    5
#define ROW_HEIGHT    40

#include <clutter/clutter.h>
#include <cogl/cogl.h>

#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "hrn.h"
#include "hrn-controls.h"

enum {
    LEAVE_THEATRE,
    REQUEST_NEXT,
    REQUEST_PREVIOUS,
    PLAYING_CHANGED,
    POSITION_CHANGED,
    LAST_SIGNAL
};

#define    QUEUE_TIP_HEIGHT    47  /* used to determine drops on tip */

G_DEFINE_TYPE (HrnControls, hrn_controls, NBTK_TYPE_BIN);

#define HRN_CONTROLS_GET_PRIVATE(obj)                 \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
                                HRN_TYPE_CONTROLS, \
                                HrnControlsPrivate))

static guint32 signals[LAST_SIGNAL] = {0, };
struct _HrnControlsPrivate
{
    HrnTheatre *theatre;

    NbtkWidget *header;
    NbtkWidget *next;
    NbtkWidget *back;
    NbtkWidget *playpause;
    NbtkWidget *leave;
    NbtkWidget *progress_table;
    NbtkWidget *timebar;
    NbtkWidget *time;
    NbtkWidget *length;
    NbtkAdjustment *progress;

    guint32 visual_progress_id;
    guint32 play_pause_id;
    gboolean playing;

    NbtkWidget *title;

    guint64 duration;

    /* FIXME: This is a hack because g_signal_handler_block
       is apparently not working */
    gboolean ignore_position_change;
};

static GObject * hrn_controls_constructor (GType                  type,
                                           guint                  n_params,
                                           GObjectConstructParam *params);
static void hrn_controls_dispose          (GObject               *object);
static void hrn_controls_allocate         (ClutterActor          *self,
                                           const ClutterActorBox *box,
                                           ClutterAllocationFlags flags);

static void
hrn_controls_class_init (HrnControlsClass *klass)
{
  GObjectClass      *gobject_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class   = CLUTTER_ACTOR_CLASS (klass);

  gobject_class->dispose      = hrn_controls_dispose;
  gobject_class->constructor  = hrn_controls_constructor;

  actor_class->allocate       = hrn_controls_allocate;

  g_type_class_add_private (gobject_class, sizeof (HrnControlsPrivate));

  signals[LEAVE_THEATRE] = g_signal_new ("leave-theatre",
                                         G_TYPE_FROM_CLASS (gobject_class),
                                         G_SIGNAL_RUN_FIRST |
                                         G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                         g_cclosure_marshal_VOID__VOID,
                                         G_TYPE_NONE, 0);
  signals[REQUEST_NEXT] = g_signal_new ("request-next",
                                        G_TYPE_FROM_CLASS (gobject_class),
                                        G_SIGNAL_RUN_FIRST |
                                        G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                        g_cclosure_marshal_VOID__VOID,
                                        G_TYPE_NONE, 0);
  signals[REQUEST_PREVIOUS] = g_signal_new ("request-previous",
                                            G_TYPE_FROM_CLASS (gobject_class),
                                            G_SIGNAL_RUN_FIRST |
                                            G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                            g_cclosure_marshal_VOID__VOID,
                                            G_TYPE_NONE, 0);
  signals[PLAYING_CHANGED] = g_signal_new ("request-playing-changed",
                                           G_TYPE_FROM_CLASS (gobject_class),
                                           G_SIGNAL_RUN_FIRST |
                                           G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                           g_cclosure_marshal_VOID__BOOLEAN,
                                           G_TYPE_NONE, 1, G_TYPE_BOOLEAN);
  signals[POSITION_CHANGED] = g_signal_new ("position-changed",
                                            G_TYPE_FROM_CLASS (gobject_class),
                                            G_SIGNAL_RUN_FIRST |
                                            G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                            g_cclosure_marshal_VOID__DOUBLE,
                                            G_TYPE_NONE, 1, G_TYPE_DOUBLE);
}

static void
hrn_controls_init (HrnControls *self)
{
    HrnControlsPrivate *priv;

    priv = self->priv = HRN_CONTROLS_GET_PRIVATE (self);

    priv->ignore_position_change = FALSE;
}

static void
playpause_clicked_cb (NbtkButton  *button,
                      HrnControls *controls)
{
    HrnControlsPrivate *priv = controls->priv;

    hrn_controls_set_playing (controls, !priv->playing);
    g_signal_emit (controls, signals[PLAYING_CHANGED], 0, priv->playing);
}


static void
next_clicked_cb (NbtkButton   *button,
                 HrnControls  *controls)
{
    g_signal_emit (controls, signals[REQUEST_NEXT], 0);
}

static void
back_clicked_cb (NbtkButton   *button,
                 HrnControls  *controls)
{
    g_signal_emit (controls, signals[REQUEST_PREVIOUS], 0);
}

#define SECONDS_PER_MINUTE 60
#define SECONDS_PER_HOUR 3600

static char *
nanosecs_to_time (guint64 duration)
{
    guint dsec = duration;
    guint seconds = 0, minutes = 0, hours = 0;

    if (dsec > SECONDS_PER_HOUR) {
        hours = dsec / SECONDS_PER_HOUR;
        dsec = dsec % SECONDS_PER_HOUR;
    }

    if (dsec > SECONDS_PER_MINUTE) {
        minutes = dsec / SECONDS_PER_MINUTE;
        dsec = dsec % SECONDS_PER_MINUTE;
    }

    seconds = dsec;

    /* FIXME: Would be nice to use a static string to save constant
       reallocations */
    return g_strdup_printf ("%02d:%02d.%02d", hours, minutes, seconds);
}

static void
visual_progress_notify (NbtkAdjustment *adjustment,
                        GParamSpec     *pspec,
                        HrnControls    *controls)
{
    HrnControlsPrivate *priv = controls->priv;
    double position = nbtk_adjustment_get_value (adjustment);
    char *dstr;

    dstr = nanosecs_to_time (priv->duration * position);
    nbtk_label_set_text ((NbtkLabel *) priv->time, dstr);
    g_free (dstr);

    if (priv->ignore_position_change == FALSE) {
        g_signal_emit (controls, signals[POSITION_CHANGED], 0, position);
    }
    priv->ignore_position_change = FALSE;
}

static GObject *
hrn_controls_constructor (GType type, guint n_params,
                          GObjectConstructParam *params)
{
  HrnControlsPrivate *priv;
  GObject            *object;
  HrnControls        *controls;

  object = G_OBJECT_CLASS (hrn_controls_parent_class)->constructor (
    type, n_params, params);

  controls = HRN_CONTROLS (object);
  priv = controls->priv;

  priv->header = nbtk_table_new ();
  nbtk_widget_set_style_class_name (priv->header, "HrnControls");
  clutter_container_add_actor ((ClutterContainer *) controls,
                               (ClutterActor *) priv->header);

  priv->playpause = nbtk_button_new ();
  priv->playing = FALSE;
  nbtk_widget_set_style_class_name (priv->playpause, "HrnControlsPlay");
  clutter_actor_set_size (CLUTTER_ACTOR (priv->playpause), 60, 60);
  priv->play_pause_id = g_signal_connect (priv->playpause, "clicked",
                                          G_CALLBACK (playpause_clicked_cb),
                                          controls);
  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->header),
                                        CLUTTER_ACTOR (priv->playpause), 0, 0,
                                        "row-span", 2,
                                        "x-align", 0.0,
                                        "y-align", 0.0,
                                        "x-fill", FALSE,
                                        "x-expand", FALSE,
                                        NULL);


  priv->back = nbtk_button_new ();
  clutter_actor_set_size (CLUTTER_ACTOR (priv->back), 60, 60);
  nbtk_widget_set_style_class_name (priv->back, "HrnControlsBack");
  g_signal_connect (priv->back, "clicked",
                    G_CALLBACK (back_clicked_cb), controls);
  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->header),
                                        CLUTTER_ACTOR (priv->back), 0, 1,
                                        "row-span", 2,
                                        "x-align", 0.0,
                                        "y-align", 0.0,
                                        "x-fill", FALSE,
                                        "x-expand", FALSE,
                                        NULL);

  priv->next = nbtk_button_new ();
  clutter_actor_set_size (CLUTTER_ACTOR (priv->next), 60, 60);
  nbtk_widget_set_style_class_name (priv->next, "HrnControlsNext");
  g_signal_connect (priv->next, "clicked",
                    G_CALLBACK (next_clicked_cb), controls);
  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->header),
                                        CLUTTER_ACTOR (priv->next), 0, 2,
                                        "row-span", 2,
                                        "x-align", 0.0,
                                        "y-align", 0.0,
                                        "x-fill", FALSE,
                                        "x-expand", FALSE,
                                        NULL);

  priv->title = nbtk_label_new ("");
  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->header),
                                        CLUTTER_ACTOR (priv->title),
                                        0, 3,
                                        "x-align", 0.0,
                                        "y-align", 0.5,
                                        "x-fill", TRUE,
                                        "x-expand", TRUE,
                                        "y-fill", FALSE,
                                        NULL);

  priv->progress_table = nbtk_table_new ();
  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->header),
                                        CLUTTER_ACTOR (priv->progress_table),
                                        1, 3,
                                        "x-align", 0.0,
                                        "y-align", 0.5,
                                        "x-fill", TRUE,
                                        "x-expand", TRUE,
                                        "y-fill", FALSE,
                                        NULL);

  priv->time = nbtk_label_new ("0:00");
  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->progress_table),
                                        CLUTTER_ACTOR (priv->time), 0, 0,
                                        "x-align", 0.0,
                                        "y-align", 0.5,
                                        "x-fill", FALSE,
                                        "x-expand", FALSE,
                                        "y-fill", FALSE,
                                        "y-expand", FALSE,
                                        NULL);

  priv->progress = nbtk_adjustment_new (0.0, 0.0, 1.0, 0, 0, 0);
  priv->visual_progress_id = g_signal_connect
      (priv->progress, "notify::value",
       G_CALLBACK (visual_progress_notify), controls);

  priv->timebar = NBTK_WIDGET (nbtk_scroll_bar_new (priv->progress));
  clutter_actor_set_height ((ClutterActor *) priv->timebar, 14);
  nbtk_widget_set_style_class_name (priv->timebar, "HrnProgressBar");
  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->progress_table),
                                        CLUTTER_ACTOR (priv->timebar), 0, 1,
                                        "x-align", 0.0,
                                        "y-align", 0.5,
                                        "y-fill", FALSE,
                                        "x-expand", TRUE,
                                        "x-fill", TRUE,
                                        NULL);

  priv->length = nbtk_label_new ("1:00");
  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->progress_table),
                                        CLUTTER_ACTOR (priv->length), 0, 2,
                                        "x-align", 0.0,
                                        "y-align", 0.5,
                                        "x-fill", FALSE,
                                        "x-expand", FALSE,
                                        "y-fill", FALSE,
                                        NULL);

  clutter_actor_set_reactive (CLUTTER_ACTOR (object), TRUE);

  /* hrn_controls_hide_video_progress (controls); */

  return object;
}

static void
hrn_controls_allocate (ClutterActor          *self,
                       const ClutterActorBox *box,
                       ClutterAllocationFlags flags)
{
  HrnControlsPrivate   *priv = HRN_CONTROLS_GET_PRIVATE (self);
  ClutterActorClass *parent_class;
  ClutterActorBox    actor_box;
  NbtkPadding padding;

  parent_class = CLUTTER_ACTOR_CLASS (hrn_controls_parent_class);

  parent_class->allocate (self, box, flags);

  nbtk_widget_get_padding ((NbtkWidget *) self, &padding);

  actor_box.x1 = padding.left;
  actor_box.x2 = (box->x2 - box->x1) - (padding.left + padding.right);
  actor_box.y1 = padding.top;
  actor_box.y2 = (box->y2 - box->y1) - (padding.top + padding.bottom);

  clutter_actor_allocate (CLUTTER_ACTOR (priv->header), &actor_box, flags);
}

static void
hrn_controls_dispose (GObject *object)
{
  G_OBJECT_CLASS (hrn_controls_parent_class)->dispose (object);
}

HrnControls *
hrn_controls_new (void)
{
  HrnControls *queue = g_object_new (HRN_TYPE_CONTROLS, NULL);

  return queue;
}

void
hrn_controls_set_playing (HrnControls *controls,
                          gboolean     playing)
{
    HrnControlsPrivate *priv = controls->priv;

    priv->playing = playing;
    if (playing)
        nbtk_widget_set_style_class_name (priv->playpause, "HrnControlsPause");
    else
        nbtk_widget_set_style_class_name (priv->playpause, "HrnControlsPlay");
}

void
hrn_controls_set_title (HrnControls *controls,
                        const char  *title)
{
    HrnControlsPrivate *priv = controls->priv;

    nbtk_label_set_text ((NbtkLabel *) priv->title, title);
}

void
hrn_controls_show_progress (HrnControls *controls,
                            gboolean     shown)
{
    HrnControlsPrivate *priv = controls->priv;

    if (shown) {
        clutter_actor_show (CLUTTER_ACTOR (priv->progress_table));
    } else {
        clutter_actor_hide (CLUTTER_ACTOR (priv->progress_table));
    }
}

void
hrn_controls_set_position (HrnControls *controls,
                           double       position)
{
    HrnControlsPrivate *priv = controls->priv;

    priv->ignore_position_change = TRUE;
    nbtk_adjustment_set_value (priv->progress, position);
}

void
hrn_controls_set_duration (HrnControls *controls,
                           guint64      duration)
{
    HrnControlsPrivate *priv = controls->priv;
    char *dstr;

    priv->duration = duration;
    dstr = nanosecs_to_time (duration);

    nbtk_label_set_text ((NbtkLabel *) priv->length, dstr);
    g_free (dstr);
}
