#include <bickley/bkl.h>

#include "hrn.h"
#include "hrn-iface-player.h"
#include "hrn-image-player.h"

enum
{
  PROP_0,
};

struct _HrnImagePlayerPrivate
{
    ClutterActor *image_player;
    guint32 slideshow_timeout_id;
};

#define SLIDESHOW_TIMEOUT 5 /*seconds */

static void hrn_player_init (HrnIfacePlayerClass *iface);
#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
                                                       HRN_TYPE_IMAGE_PLAYER, \
                                                       HrnImagePlayerPrivate))
G_DEFINE_TYPE_WITH_CODE (HrnImagePlayer, hrn_image_player, CLUTTER_TYPE_GROUP,
                         G_IMPLEMENT_INTERFACE (HRN_TYPE_IFACE_PLAYER,
                                                hrn_player_init));

static void
hrn_image_player_finalize (GObject *object)
{
  G_OBJECT_CLASS (hrn_image_player_parent_class)->finalize (object);
}

static void
hrn_image_player_dispose (GObject *object)
{
    HrnImagePlayer *player = (HrnImagePlayer *) object;
    HrnImagePlayerPrivate *priv = player->priv;

    if (priv->slideshow_timeout_id > 0) {
        g_source_remove (priv->slideshow_timeout_id);
        priv->slideshow_timeout_id = 0;
    }

    G_OBJECT_CLASS (hrn_image_player_parent_class)->dispose (object);
}

static void
hrn_image_player_set_property (GObject      *object,
                               guint         prop_id,
                               const GValue *value,
                               GParamSpec   *pspec)
{
    switch (prop_id) {
    default:
        break;
    }
}

static void
hrn_image_player_get_property (GObject *object, guint prop_id,
                               GValue     *value,
                               GParamSpec *pspec)
{
  switch (prop_id)
    {
      default:
        break;
    }
}

static void
hrn_image_player_allocate (ClutterActor          *actor,
                           const ClutterActorBox *box,
                           ClutterAllocationFlags flags)
{
  HrnImagePlayer        *player = (HrnImagePlayer *) actor;
  HrnImagePlayerPrivate *priv   = player->priv;

  CLUTTER_ACTOR_CLASS (hrn_image_player_parent_class)->allocate
                             (actor, box, flags);

  clutter_actor_allocate_preferred_size (priv->image_player, flags);
}

static void
hrn_image_player_paint (ClutterActor *actor)
{
  HrnImagePlayer        *player = (HrnImagePlayer *) actor;
  HrnImagePlayerPrivate *priv   = player->priv;

  if (priv->image_player && CLUTTER_ACTOR_IS_VISIBLE (priv->image_player)) {
      clutter_actor_paint (priv->image_player);
  }
}

static void
hrn_image_player_pick (ClutterActor *actor, const ClutterColor *color)
{
    hrn_image_player_paint (actor);
}

static void
hrn_image_player_map (ClutterActor *self)
{
    HrnImagePlayerPrivate *priv = HRN_IMAGE_PLAYER (self)->priv;

    CLUTTER_ACTOR_CLASS (hrn_image_player_parent_class)->map (self);

    if (priv->image_player) {
        clutter_actor_map (CLUTTER_ACTOR (priv->image_player));
    }
}

static void
hrn_image_player_unmap (ClutterActor *self)
{
    HrnImagePlayerPrivate *priv = HRN_IMAGE_PLAYER (self)->priv;

    CLUTTER_ACTOR_CLASS (hrn_image_player_parent_class)->unmap (self);

    if (priv->image_player) {
        clutter_actor_unmap (CLUTTER_ACTOR (priv->image_player));
    }
}


static void
hrn_image_player_class_init (HrnImagePlayerClass *klass)
{
    GObjectClass      *o_class = (GObjectClass *) klass;
    ClutterActorClass *a_class = (ClutterActorClass *) klass;

    o_class->dispose      = hrn_image_player_dispose;
    o_class->finalize     = hrn_image_player_finalize;
    o_class->set_property = hrn_image_player_set_property;
    o_class->get_property = hrn_image_player_get_property;

    a_class->allocate = hrn_image_player_allocate;
#if 0
    a_class->paint    = hrn_image_player_paint;
    a_class->pick     = hrn_image_player_pick;
    a_class->map      = hrn_image_player_map;
    a_class->unmap    = hrn_image_player_unmap;
#endif

    g_type_class_add_private (klass, sizeof (HrnImagePlayerPrivate));
}

static gboolean
picture_finished_cb (gpointer data)
{
    HrnImagePlayer *player = (HrnImagePlayer *) data;
    HrnImagePlayerPrivate *priv = player->priv;

    priv->slideshow_timeout_id = 0;
    hrn_iface_player_emit_eos ((HrnIfacePlayer *) player);
    return FALSE;
}

static void
player_set_playing (HrnIfacePlayer *player,
                    gboolean        playing)
{
    HrnImagePlayer *image = (HrnImagePlayer *) player;
    HrnImagePlayerPrivate *priv = image->priv;

    if (playing == FALSE) {
        if (priv->slideshow_timeout_id > 0) {
            g_source_remove (priv->slideshow_timeout_id);
            priv->slideshow_timeout_id = 0;
        }
    } else {
        if (priv->slideshow_timeout_id > 0) {
            g_source_remove (priv->slideshow_timeout_id);
        }

        priv->slideshow_timeout_id = g_timeout_add_seconds
            (SLIDESHOW_TIMEOUT, picture_finished_cb, player);
    }
}


static void
player_set (HrnIfacePlayer *player,
            BklItem        *item,
            const char     *uri)
{
  HrnImagePlayer        *image = (HrnImagePlayer *) player;
  HrnImagePlayerPrivate *priv  = image->priv;
  const gchar           *orient;
  gfloat                 w, h;
  gdouble                xs, ys, scale;
  gint                   x, y, sw, sh;

  if (uri == NULL && item == NULL)
    {
      g_debug ("Image player got NULL");
      /* FIXME: Should blank the image? */
      return;
    }

  /* handle http / UPNP directly with gdkpixbuf and GIO */
  if (g_str_has_prefix (uri, "http"))
    {
      GdkPixbuf *pixbuf = NULL;
      GFile *file;
      GFileInputStream *stream;
      GError *error = NULL;

      file = g_file_new_for_uri (uri);
      stream = g_file_read (file, NULL, NULL);
      if (stream)
        {
          pixbuf = gdk_pixbuf_new_from_stream (G_INPUT_STREAM (stream), NULL, NULL);
          g_object_unref (stream);
          g_object_unref (file);

          if (pixbuf)
            {
              clutter_texture_set_from_rgb_data ((ClutterTexture *) priv->image_player,
                                              gdk_pixbuf_get_pixels (pixbuf),
                                              gdk_pixbuf_get_has_alpha (pixbuf),
                                              gdk_pixbuf_get_width (pixbuf),
                                              gdk_pixbuf_get_height (pixbuf),
                                              gdk_pixbuf_get_rowstride (pixbuf),
                                              gdk_pixbuf_get_has_alpha (pixbuf) ? 4 : 3,
                                              0,
                                              &error);
              if (error)
                {
                  g_warning ("Unable to set pixbuf : %s", error->message);
                  g_error_free (error);
                }
              g_object_unref (pixbuf);
            }
        }

      if (pixbuf == NULL)
        {
          guchar px[8]="";
          /* we get a crash if we set no data at all */
          clutter_texture_set_from_rgb_data ((ClutterTexture *) priv->image_player, px, FALSE,
                                              1, 1, 3, 3, 0, NULL);
          return;
        }
    }
  else {
      GError *error = NULL;
      gchar *path;

      path = g_filename_from_uri (uri, NULL, NULL);

      if (path == NULL) {
          return;
      }

      clutter_texture_set_from_file ((ClutterTexture *) priv->image_player,
                                     path, &error);
      if (error != NULL) {
          g_warning ("Error loading %s: %s", uri, error->message);
          g_error_free (error);
      }
    }

  clutter_actor_get_size (priv->image_player, &w, &h);
  if (item) {
      orient = bkl_item_image_get_orientation (BKL_ITEM_IMAGE (item));

      if (orient) {
          if (g_str_equal (orient, "right - top")) {
              float tmp;

              clutter_actor_set_anchor_point_from_gravity
                  (priv->image_player, CLUTTER_GRAVITY_CENTER);

              clutter_actor_set_rotation (priv->image_player,
                                          CLUTTER_Z_AXIS, 90,
                                          0, 0, 0);

              clutter_actor_set_anchor_point_from_gravity
                  (priv->image_player, CLUTTER_GRAVITY_SOUTH_WEST);

              /* Now that we're rotated, and we've moved the gravity
                 the width/height need to be swapped */
              tmp = w;
              w = h;
              h = tmp;
          }
      }
  }

  gtk_window_get_size (GTK_WINDOW (window), &sw, &sh);

  xs = ((double) sw) / (double) w;
  ys = ((double) sh) / (double) h;

  if (xs > ys) {
      x     = (((float) sw) - (w * ys)) / 2;
      y     = 0;
      scale = ys;
  } else {
      x     = 0;
      y     = (((float ) sh) - (h * xs)) / 2;
      scale = xs;
  }

  clutter_actor_set_position (priv->image_player, x, y);
  clutter_actor_set_scale (priv->image_player, scale, scale);
}

static void
player_set_uri (HrnIfacePlayer *player,
                const char     *uri)
{
    player_set (player, NULL, uri);
}

static void
player_set_item (HrnIfacePlayer *player,
                 BklItem        *item)
{
    player_set (player, item, bkl_item_get_uri (item));
}

static void
player_set_position (HrnIfacePlayer *player, double position)
{
  /* This function intentionally left blank */
}

static guint64
player_get_duration (HrnIfacePlayer *player)
{
    return 0;
}

static void
hrn_player_init (HrnIfacePlayerClass *iface)
{
  iface->set_playing  = player_set_playing;
  iface->set_uri      = player_set_uri;
  iface->set_item     = player_set_item;
  iface->set_position = player_set_position;
  iface->get_duration = player_get_duration;
}

static void
hrn_image_player_init (HrnImagePlayer *self)
{
  HrnImagePlayerPrivate *priv = GET_PRIVATE (self);
  ClutterActor *magic_rect;
  ClutterColor red = {0xff, 0x00, 0x00, 0xff};

  self->priv = priv;

  priv->image_player = clutter_texture_new ();
  clutter_actor_set_reactive (priv->image_player, TRUE);
  clutter_texture_set_filter_quality (CLUTTER_TEXTURE (priv->image_player),
                                      CLUTTER_TEXTURE_QUALITY_HIGH);
  clutter_container_add_actor (CLUTTER_CONTAINER (self), priv->image_player);
  clutter_actor_show (priv->image_player);
}

