#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
from Queue import Queue
import thread
import sys

from twisted.internet.defer import Deferred, DeferredQueue, maybeDeferred
from twisted.python.failure import Failure


class ServiceError(Exception):
    """Base class for all service errors."""

    def __init__(self, message=None):
        # This attribute actually exists today, but it is obsolete, so
        # we actually create our own for posterity.
        self._message = message

    def _get_message(self):
        return self._message

    def _set_message(self, message):
        self._message = message

    message = property(_get_message, _set_message)

    def __str__(self):
        return self.message

    def __repr__(self):
        return "<%s message='%s'>" % (self.__class__.__name__,
                                      self.message)


class UnknownTaskError(ServiceError):
    """Raised when a task is added to a service which has no handler for it."""


class NotStartedError(ServiceError):
    """Raised when performing a run-time action on a stopped service."""


def taskHandler(taskType):
    def decorator(method):
        handlers = sys._getframe(1).f_locals.setdefault("_classHandlers", {})
        handlers[taskType] = method
        return method
    return decorator


# It does exactly the same thing.  This is just a documentation aid to
# remind people that the decorated method is executed inside a thread.
taskHandlerInThread = taskHandler


class ServiceTask(object):

    def __new__(cls, *args, **kwargs):
        task = object.__new__(cls)
        task._deferred = Deferred()
        return task

    def execute(self, callable, *args, **kwargs):
        deferred = maybeDeferred(callable, self, *args, **kwargs)
        deferred.chainDeferred(self._deferred)

    def fromThreadExecute(self, reactor, callable, *args, **kwargs):
        try:
            result = callable(self, *args, **kwargs)
        except:
            result = Failure()
        return reactor.callFromThread(self._callFromThread, result)

    def _callFromThread(self, result):
        if isinstance(result, Deferred):
            result.chainDeferred(self._deferred)
        elif isinstance(result, Failure):
            self._deferred.errback(result)
        else:
            self._deferred.callback(result)

    def getDeferred(self):
        return self._deferred


class ServiceBase(object):

    _classHandlers = {}

    def __init__(self):
        self._handlers = {}
        self._running = False
        for taskType, handler in self._classHandlers.iteritems():
            def wrappedHandler(task, self=self, handler=handler):
                # Handlers are usually defined in a class context, and
                # include the usual 'self' variable as a first argument.
                # That said, the 'handler' method which we get at class
                # definition time is the pure function, unwrapped, so we
                # have to pass 'self' explicitly to the method.
                return handler(self, task)
            self.addHandler(taskType, wrappedHandler)
        self.initializeHandlers()

    def initializeHandlers(self):
        """Override this method to register your task handlers."""

    def addHandler(self, taskType, handler, *args, **kwargs):
        self._handlers[taskType] = (handler, args, kwargs)

    def addTask(self, task):
        if type(task) not in self._handlers:
            raise UnknownTaskError("Unknown task: %s" % task.__class__.__name__)
        self._add(task)
        return task.getDeferred()

    def start(self):
        self._running = True
        self._start()

    def stop(self):
        if not self._running:
            raise NotStartedError("Service %r not started" % self)
        self._running = False
        task = _StopServiceTask()
        self._tasks.put(task)
        return task.getDeferred()


class Service(ServiceBase):

    def __init__(self):
        self._tasks = DeferredQueue()
        ServiceBase.__init__(self)

    def _add(self, task):
        self._tasks.put(task)

    def _start(self):
        deferred = self._tasks.get()
        deferred.addCallback(self._executeTask)

    def _executeTask(self, task):
        if type(task) is _StopServiceTask:
            task.execute(lambda task: None)
        else:
            (handler, args, kwargs) = self._handlers[type(task)]
            task.execute(handler, *args, **kwargs)
            deferred = self._tasks.get()
            deferred.addCallback(self._executeTask)


class ThreadedService(ServiceBase):

    def __init__(self, reactor):
        self._reactor = reactor
        self._tasks = Queue()
        ServiceBase.__init__(self)

    def initializeThread(self):
        """Override this method to perform your thread initialization.

        Note that this method will be run *inside* the thread.
        """

    def destructThread(self):
        """Override this method to perform your thread deinitialization.

        Note that this method will be run *inside* the thread.
        """

    def _add(self, task):
        self._tasks.put(task)

    def _start(self):
        thread.start_new_thread(self._threadLoop, ())

    def _threadLoop(self):
        self.initializeThread()
        while True:
            task = self._tasks.get()
            if type(task) is _StopServiceTask:
                try:
                    self.destructThread()
                finally:
                    task.fromThreadExecute(self._reactor, lambda task: None)
                    break
            else:
                (handler, args, kwargs) = self._handlers[type(task)]
                task.fromThreadExecute(self._reactor, handler, *args, **kwargs)


class _StopServiceTask(ServiceTask):
    """Marker task to help the service implementation to stop.
    
    DO NOT USE THIS OUTSIDE THIS IMPLEMENTATION.
    """


class ServiceHub(object):

    def __init__(self):
        self._services = []

    def addService(self, service):
        self._services.append(service)

    def getService(self, serviceType):
        for service in self._services:
            if type(service) is serviceType:
                return service
        raise ServiceError("%s not found in service hub" %
                           (serviceType.__name__,))

    def start(self):
        for service in self._services:
            service.start()

    def stop(self):
        return [service.stop() for service in self._services]

    def addTask(self, task):
        for service in self._services:
            try:
                return service.addTask(task)
            except UnknownTaskError:
                pass
        raise UnknownTaskError("%s rejected by all registered services."
                               % task.__class__.__name__)
