#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
from imagestore.lib.jsonobject import JSONObject, MissingValueError
from imagestore.lib.tests import TestCase



class JSONObjectTest(TestCase):

    def testIsDict(self):
        obj = JSONObject()
        obj["key"] = "value"
        self.assertEquals(obj["key"], "value")
        self.assertTrue(isinstance(obj, dict))

    def testCopyDictConstructor(self):
        obj = JSONObject({"key": "value"})
        self.assertEquals(obj["key"], "value")

    def testParseJSONConstructor(self):
        obj = JSONObject('{"key": "value"}')
        self.assertEquals(obj["key"], "value")

    def testParseJSONConstructorWithUnicode(self):
        obj = JSONObject(u'{"key": "value"}')
        self.assertEquals(obj["key"], "value")

    def testParseDictionaryOnly(self):
        # That's a valid JSON object, but not a valid JSONObject. :-)
        self.assertRaises(ValueError, JSONObject, '"a"')

    def testRequiredFields(self):
        class MyObject(JSONObject):
            required = ["uri"]
        obj = MyObject(uri="http://...")
        self.assertEquals(obj["uri"], "http://...")
        self.assertRaises(MissingValueError, MyObject, a=1)

    def testSubObjects(self):
        class MySubObject(JSONObject):
            pass
        class MyObject(JSONObject):
            subtypes = {"key": MySubObject}
        obj = MyObject({"key": {"subkey": "subvalue"}})
        self.assertEquals(type(obj["key"]), MySubObject)

    def testListSubObjects(self):
        class MySubObject(JSONObject):
            pass
        class MyObject(JSONObject):
            subtypes = {"key": MySubObject}
        obj = MyObject({"key": [{"subkey": "subvalue"}]})
        self.assertEquals(type(obj["key"][0]), MySubObject)

    def testCopyMethod(self):
        class MySubObject(JSONObject):
            pass
        class MyObject(JSONObject):
            subtypes = {"key": MySubObject}
        obj = MyObject({"key": {"subkey": "subvalue"}})
        objCopy = obj.copy()
        self.assertEquals(objCopy, obj)
        self.assertNotIdentical(objCopy, obj)

        self.assertEquals(objCopy["key"], obj["key"])
        self.assertNotIdentical(objCopy["key"], obj["key"])

    def testToJSON(self):
        obj = JSONObject({"key": "value"})
        self.assertEquals(str(obj), '{"key": "value"}')

    def testToJSONWithSubObjects(self):
        class MySubObject(JSONObject):
            pass
        class MyObject(JSONObject):
            subtypes = {"key": MySubObject}
        obj = MyObject({"key": {"subkey": "subvalue"}})
        self.assertEquals(str(obj), '{"key": {"subkey": "subvalue"}}')

    def testInitializationSingleValuePreservesIdentity(self):
        class MySubObject(JSONObject):
            pass
        class MyObject(JSONObject):
            subtypes = {"key": MySubObject}
        subobj = MySubObject({"subkey": "subvalue"})
        obj = MyObject({"key": subobj})
        self.assertIdentical(obj["key"], subobj)

    def testInitializationListValuePreservesIdentity(self):
        class MySubObject(JSONObject):
            pass
        class MyObject(JSONObject):
            subtypes = {"key": MySubObject}
        subobj1 = MySubObject({"subkey": "subvalue1"})
        subobj2 = MySubObject({"subkey": "subvalue2"})
        obj = MyObject({"key": [subobj1, subobj2]})
        self.assertIdentical(obj["key"][0], subobj1)
        self.assertIdentical(obj["key"][1], subobj2)

    def testEqualityChecksSameType(self):
        class MyObject(JSONObject):
            pass
        self.assertTrue(JSONObject({"Hello": "there!"}) ==
                        JSONObject({"Hello": "there!"}))
        self.assertFalse(JSONObject({"Hello": "there!"}) ==
                         {"Hello": "there!"})
        self.assertFalse(JSONObject({"Hello": "there!"}) ==
                         MyObject({"Hello": "there!"}))
        self.assertFalse(MyObject({"Hello": "there!"}) ==
                         JSONObject({"Hello": "there!"}))

    def testToDictWithSubObjects(self):
        class MyObject(JSONObject):
            pass
        class MySubObject(JSONObject):
            pass
        class MySubSubObject(JSONObject):
            pass
        obj = MyObject({"key":
                        [MySubObject({"subkey":
                                      MySubSubObject({"subsubkey":
                                                      "subsubvalue"})})]})
        self.assertEquals(obj.toDict(),
                          {"key": [{"subkey": {"subsubkey": "subsubvalue"}}]})
