/*
 * indicator-network
 * Copyright 2010-2012 Canonical Ltd.
 *
 * Authors:
 * Antti Kaijanmäki <antti.kaijanmaki@canonical.com>
 * Kalle Valo       <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "network-menu.h"

#include <glib/gi18n.h>

#include <libdbusmenu-glib/client.h>

#include "service.h"
#include "service-manager.h"
#include "dbus-shared-names.h"
#include "manager.h"

#include "wireless-connect-dialog.h"
#include "connection-information-dialog.h"

static void wifi_enabled_changed_cb(gpointer instance,
				    gboolean value,
				    gpointer user_data);

static void airplane_mode_changed_cb(gpointer instance,
				     gboolean value,
				     gpointer user_data);

static void mobile_data_enabled_changed_cb(gpointer instance,
					   gboolean value,
					   gpointer user_data);

G_DEFINE_TYPE (NetworkMenu, network_menu, DBUSMENU_TYPE_MENUITEM)

#define GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), TYPE_NETWORK_MENU, NetworkMenuPrivate))

typedef struct _NetworkMenuPrivate NetworkMenuPrivate;

struct _NetworkMenuPrivate {

  DbusmenuMenuitem *offline;
  DbusmenuMenuitem *offline_separator;
  DbusmenuMenuitem *wired_header;
  DbusmenuMenuitem *wired_separator;
  DbusmenuMenuitem *wireless_header;
  DbusmenuMenuitem *wireless_list_top;
  DbusmenuMenuitem *wireless_other;
  DbusmenuMenuitem *wireless_separator;
  DbusmenuMenuitem *cellular_header;
  DbusmenuMenuitem *operator_item;
  DbusmenuMenuitem *cellular_separator;
  DbusmenuMenuitem *network_settings;
  DbusmenuMenuitem *connection_information;

  DbusmenuMenuitem *disabled_menuitem;

  GList *wired_services;
  GList *wireless_services;
  GList *cellular_services;

  gboolean wired_enabled;
  gboolean wireless_enabled;
  gboolean cellular_enabled;
  gboolean offline_enabled;

  gboolean wired_shown;
  gboolean wireless_shown;
  gboolean cellular_shown;

  gboolean operator_shown;

  gboolean menu_enabled;

  Manager *network_service;
  AndroidService *android_service;
};

static void
network_menu_dispose (GObject *object)
{
  NetworkMenu *self = NETWORK_MENU(object);
  NetworkMenuPrivate *priv = GET_PRIVATE(self);

  g_debug("%s", __func__);

  if (priv->offline != NULL) {
    g_object_unref(priv->offline);
    priv->offline = NULL;
  }

  if (priv->offline_separator != NULL) {
    g_object_unref(priv->offline_separator);
    priv->offline_separator = NULL;
  }

  if (priv->wired_header != NULL) {
    g_object_unref(priv->wired_header);
    priv->wired_header = NULL;
  }

  if (priv->wired_separator != NULL) {
    g_object_unref(priv->wired_separator);
    priv->wired_separator = NULL;
  }

  if (priv->wireless_header != NULL) {
    g_object_unref(priv->wireless_header);
    priv->wireless_header = NULL;
  }
  
  if (priv->wireless_list_top != NULL) {
    g_object_unref(priv->wireless_list_top);
    priv->wireless_list_top = NULL;
  }

  if (priv->wireless_other != NULL) {
    g_object_unref(priv->wireless_other);
    priv->wireless_other = NULL;
  }

  if (priv->wireless_separator != NULL) {
    g_object_unref(priv->wireless_separator);
    priv->wireless_separator = NULL;
  }

  if (priv->cellular_header != NULL) {
    g_object_unref(priv->cellular_header);
    priv->cellular_header = NULL;
  }

  if (priv->operator_item != NULL) {
    g_object_unref(priv->operator_item);
    priv->operator_item = NULL;
  }

  if (priv->cellular_separator != NULL) {
    g_object_unref(priv->cellular_separator);
    priv->cellular_separator = NULL;
  }

  if (priv->network_settings != NULL) {
    g_object_unref(priv->network_settings);
    priv->network_settings = NULL;
  }

  if (priv->connection_information != NULL) {
    g_object_unref(priv->connection_information);
    priv->connection_information = NULL;
  }

  if (priv->disabled_menuitem != NULL) {
    g_object_unref(priv->disabled_menuitem);
    priv->disabled_menuitem = NULL;
  }

  /// @todo check that these are freed correctly
#if 0
  GList *wired_services;
  GList *wireless_services;
  GList *cellular_services;
#endif

  if (priv->android_service != NULL) {
    g_object_unref(priv->android_service);
    priv->android_service = NULL;
  }

  G_OBJECT_CLASS (network_menu_parent_class)->dispose (object);
}

static void
network_menu_finalize (GObject *object)
{
  G_OBJECT_CLASS (network_menu_parent_class)->finalize (object);
}

static void
network_menu_class_init (NetworkMenuClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (klass, sizeof (NetworkMenuPrivate));

  object_class->dispose = network_menu_dispose;
  object_class->finalize = network_menu_finalize;
}

static void
network_menu_init (NetworkMenu *self)
{
}

void network_menu_disable(NetworkMenu *self)
{
  NetworkMenuPrivate *priv = GET_PRIVATE(self);
  Service *service;
  GList *iter;

  if (!priv->menu_enabled)
	  return;

  while ((iter = g_list_first(priv->wired_services)) != NULL) {
    service = iter->data;
    dbusmenu_menuitem_child_delete(DBUSMENU_MENUITEM(self),
				   DBUSMENU_MENUITEM(service));
    priv->wired_services = g_list_delete_link(priv->wired_services,
						 iter);
  }

  while ((iter = g_list_first(priv->wireless_services)) != NULL) {
    service = iter->data;
    dbusmenu_menuitem_child_delete(DBUSMENU_MENUITEM(self),
				   DBUSMENU_MENUITEM(service));
    priv->wireless_services = g_list_delete_link(priv->wireless_services,
						 iter);
  }

  while ((iter = g_list_first(priv->cellular_services)) != NULL) {
    service = iter->data;
    dbusmenu_menuitem_child_delete(DBUSMENU_MENUITEM(self),
				   DBUSMENU_MENUITEM(service));
    priv->cellular_services = g_list_delete_link(priv->cellular_services,
						 iter);
  }

  /* hide all menuitems */
  dbusmenu_menuitem_property_set_bool(priv->offline,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_property_set_bool(priv->offline_separator,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_property_set_bool(priv->wired_separator,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_property_set_bool(priv->wireless_header,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_property_set_bool(priv->wireless_list_top,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_property_set_bool(priv->wireless_other,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_property_set_bool(priv->wireless_separator,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_property_set_bool(priv->cellular_header,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_property_set_bool(priv->operator_item,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_property_set_bool(priv->cellular_separator,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_property_set_bool(priv->network_settings,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_property_set_bool(priv->connection_information,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);

  dbusmenu_menuitem_property_set_bool(priv->disabled_menuitem,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, TRUE);

  priv->menu_enabled = FALSE;
}

static void network_menu_update_wireless(NetworkMenu *self)
{
  NetworkMenuPrivate *priv = GET_PRIVATE(self);
  ServiceManager *service_manager;
  Service *service;
  GList *iter;
  guint pos;
  guint limit;
  
  /** @todo this function has to be modified to reuse existing menuitems
            as much as possible by reordering them when necessary.

	    simply deleting and recreating the items each time there is
	    a change in the list of networks generates a lot of unnecessary
	    D-Bus traffic.
  */

  /* remove old services */
  while ((iter = g_list_first(priv->wireless_services)) != NULL) {
    service = iter->data;
    dbusmenu_menuitem_child_delete(DBUSMENU_MENUITEM(self),
				   DBUSMENU_MENUITEM(service));
    priv->wireless_services = g_list_delete_link(priv->wireless_services,
						 iter);
  }

  /* get new wireless services */
  service_manager = manager_get_service_manager(priv->network_service);
  priv->wireless_services = service_manager_get_wireless(service_manager);

  /*
   * Even there are now wireless services, there are other menuitems and
   * separator is still needed.
   */
  priv->wireless_shown = TRUE;

  if (priv->wireless_services == NULL) {
    /* no wireless services to show */
    dbusmenu_menuitem_property_set_bool(priv->wireless_list_top,
					DBUSMENU_MENUITEM_PROP_VISIBLE,
					TRUE);
    dbusmenu_menuitem_property_set_bool(DBUSMENU_MENUITEM(priv->wireless_other),
					DBUSMENU_MENUITEM_PROP_VISIBLE,
					FALSE);
    return;
  }

  /* turn off the "No network detected" item */
  dbusmenu_menuitem_property_set_bool(priv->wireless_list_top,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);


  pos = dbusmenu_menuitem_get_position(priv->wireless_list_top,
					     DBUSMENU_MENUITEM(self));

  limit = pos + 5;
  for (iter = priv->wireless_services; iter != NULL; iter = iter->next) {
    service = iter->data;
    dbusmenu_menuitem_child_add_position(DBUSMENU_MENUITEM(self),
					 DBUSMENU_MENUITEM(service), pos);

    /* show only certain number of wireless networks */
    if (pos >= limit) {
      // adding anyway as otherwise dbusmenu_menuitem_child_delete() will complain.
      /// @todo fix the update logic, see above.
      dbusmenu_menuitem_property_set_bool(DBUSMENU_MENUITEM(service),
					  DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
    } else {
      dbusmenu_menuitem_property_set_bool(DBUSMENU_MENUITEM(service),
					  DBUSMENU_MENUITEM_PROP_VISIBLE, TRUE);
    }

    pos++;
  }

  dbusmenu_menuitem_property_set_bool(DBUSMENU_MENUITEM(priv->wireless_other),
				      DBUSMENU_MENUITEM_PROP_VISIBLE,
				      pos >= limit);
}

static void network_menu_update_cellular(NetworkMenu *self)
{
  NetworkMenuPrivate *priv = GET_PRIVATE(self);
  ServiceManager *service_manager;
  Service *service;
  GList *iter;
  guint pos;

  /* remove old services */
  while ((iter = g_list_first(priv->cellular_services)) != NULL) {
    service = iter->data;
    dbusmenu_menuitem_child_delete(DBUSMENU_MENUITEM(self),
				   DBUSMENU_MENUITEM(service));
    priv->cellular_services = g_list_delete_link(priv->cellular_services,
						 iter);
  }

  /* get new cellular services */
  service_manager = manager_get_service_manager(priv->network_service);
  priv->cellular_services = service_manager_get_cellular(service_manager);

  if (priv->cellular_services == NULL) {
    /* no cellular services to show */
    priv->cellular_shown = FALSE;
    return;
  }

  pos = dbusmenu_menuitem_get_position(priv->cellular_header,
				       DBUSMENU_MENUITEM(self));
  pos++;

  for (iter = priv->cellular_services; iter != NULL; iter = iter->next) {
    service = iter->data;
    dbusmenu_menuitem_child_add_position(DBUSMENU_MENUITEM(self),
					 DBUSMENU_MENUITEM(service), pos);
    pos++;
  }

  priv->cellular_shown = TRUE;
}

static void update_services(NetworkMenu *self)
{
  NetworkMenuPrivate *priv = GET_PRIVATE(self);
  gboolean show;

  g_return_if_fail(priv != NULL);

  network_menu_update_wireless(self);
  network_menu_update_cellular(self);

  show = priv->wired_shown && (priv->wireless_shown || priv->cellular_shown);
  dbusmenu_menuitem_property_set_bool(priv->wired_separator,
				      DBUSMENU_MENUITEM_PROP_VISIBLE,
				      show);

  // wireless separator is always shown
  dbusmenu_menuitem_property_set_bool(priv->wireless_separator,
				      DBUSMENU_MENUITEM_PROP_VISIBLE,
				      TRUE);

  /* cellular_separator is always shown */
  dbusmenu_menuitem_property_set_bool(priv->cellular_separator,
  				      DBUSMENU_MENUITEM_PROP_VISIBLE,
  				      TRUE);
}

void network_menu_enable(NetworkMenu *self)
{
  NetworkMenuPrivate *priv = GET_PRIVATE(self);

  if (priv->menu_enabled)
    return;

  /* show all menuitems which should be always visible */

  dbusmenu_menuitem_property_set_bool(priv->offline,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_property_set_bool(priv->offline_separator,
  				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);

  dbusmenu_menuitem_property_set_bool(priv->wireless_header,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, TRUE);
  dbusmenu_menuitem_property_set_bool(priv->wireless_list_top,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, TRUE);
  dbusmenu_menuitem_property_set_bool(priv->wireless_other,
  				      DBUSMENU_MENUITEM_PROP_VISIBLE, TRUE);

  dbusmenu_menuitem_property_set_bool(priv->cellular_header,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, TRUE);
  dbusmenu_menuitem_property_set_bool(priv->operator_item,
				      DBUSMENU_MENUITEM_PROP_VISIBLE,
				      priv->operator_shown);

  /// @todo enable when settings are implemented.
  //dbusmenu_menuitem_property_set_bool(priv->network_settings,
  //				      DBUSMENU_MENUITEM_PROP_VISIBLE, TRUE);

  dbusmenu_menuitem_property_set_bool(priv->connection_information,
  				      DBUSMENU_MENUITEM_PROP_VISIBLE, TRUE);

  dbusmenu_menuitem_property_set_bool(priv->disabled_menuitem,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);

  update_services(self);

  priv->menu_enabled = TRUE;
}

static void services_updated(ServiceManager *sm, gpointer user_data)
{
  NetworkMenu *self = NETWORK_MENU(user_data);
  NetworkMenuPrivate *priv = GET_PRIVATE(self);

  g_return_if_fail(IS_NETWORK_MENU(self));
  g_return_if_fail(priv != NULL);

  /* don't update services if menu is disabled */
  if (!priv->menu_enabled)
    return;

  update_services(self);
}

static void connect_state_changed(ServiceManager *sm, gpointer user_data)
{
  NetworkMenu *self = NETWORK_MENU(user_data);

  g_return_if_fail(IS_NETWORK_MENU(self));
  
}

static void toggle_wireless_cb(DbusmenuMenuitem *mi, guint timestamp,
			       gpointer user_data)
{
  NetworkMenu *self = NETWORK_MENU(user_data);
  NetworkMenuPrivate *priv = GET_PRIVATE(self);

  android_service_set_wifi_enabled(priv->android_service,
				   !priv->wireless_enabled);
}

static void toggle_offline_cb(DbusmenuMenuitem *mi, guint timestamp,
			       gpointer user_data)
{
  NetworkMenu *self = NETWORK_MENU(user_data);
  NetworkMenuPrivate *priv = GET_PRIVATE(self);

  g_return_if_fail(priv != NULL);
  g_return_if_fail(priv->android_service != NULL);

  android_service_set_airplane_mode(priv->android_service,
				    !priv->offline_enabled);
}

static void toggle_cellular_cb(DbusmenuMenuitem *mi, guint timestamp,
			       gpointer user_data)
{
  NetworkMenu *self = NETWORK_MENU(user_data);
  NetworkMenuPrivate *priv = GET_PRIVATE(self);

  g_return_if_fail(priv != NULL);
  g_return_if_fail(priv->android_service != NULL);

  android_service_set_mobile_data_enabled(priv->android_service,
					  !priv->cellular_enabled);
}


static void network_settings_activated(DbusmenuMenuitem *mi, guint timestamp,
				       gpointer user_data)
{
  gchar *argv[6];

  argv[0] = "/usr/bin/ubuntu-android-launcher";
  argv[1] = "--action"; 
  argv[2] = "android.intent.action.MAIN"; 
  argv[3] = "--cmp"; 
  argv[4] = "com.android.settings/.WirelessSettings";
  argv[5] = NULL;

  g_debug("%s(): starting %s", __func__, argv[0]);
  g_spawn_async(NULL, argv, NULL, 0, NULL, NULL, NULL, NULL);
}

static void dialog_response(GtkDialog *dialog,
			    gint response_id,
			    gpointer user_data)
{
  NetworkMenu *self = user_data;
  NetworkMenuPrivate *priv = GET_PRIVATE(self);

  AndroidService *android_service;
  const gchar *identifier;

  WirelessConnectDialog *wireless_dialog = WIRELESS_CONNECT_DIALOG(dialog);

  gtk_widget_hide(GTK_WIDGET(wireless_dialog));

  if (response_id == GTK_RESPONSE_ACCEPT) {
    android_service = manager_get_android_service(priv->network_service);
    identifier = wireless_connect_dialog_get_selected(wireless_dialog);

    android_service_enable_network_with_identifier(android_service,
						   identifier);
  }

  /// @todo fix me, the dialog can't really perform a harakiri here, right?
  g_object_unref(wireless_dialog);
}

static void wireless_other_activated(DbusmenuMenuitem *mi, guint timestamp,
				       gpointer user_data)
{
  NetworkMenu *self = user_data;
  NetworkMenuPrivate *priv = GET_PRIVATE(self);
  Manager *ns = priv->network_service;

  WirelessConnectDialog *dialog;
  dialog = wireless_connect_dialog_new();
  wireless_connect_dialog_show(dialog, priv->wireless_services);

  g_signal_connect(G_OBJECT(dialog), "response",
		   G_CALLBACK(dialog_response),
		   self);
}

static void connection_information_dialog_response(GtkDialog *dialog,
						   gint response_id,
						   gpointer user_data)
{
  /// @todo fix me, the dialog can't really perform a harakiri here, right?
  g_object_unref(dialog);
}

static void
connection_information_activated(DbusmenuMenuitem *mi,
				 guint timestamp,
				 gpointer user_data)
{
  NetworkMenu *self = user_data;
  NetworkMenuPrivate *priv = GET_PRIVATE(self);
  GHashTable *info;

  ConnectionInformationDialog *dialog;

  info = android_service_get_connection_information(priv->android_service);
  
  dialog = connection_information_dialog_new();
  g_signal_connect(G_OBJECT(dialog), "response",
		   G_CALLBACK(connection_information_dialog_response),
		   self);
  connection_information_dialog_show(dialog, info);
}

NetworkMenu *network_menu_new(Manager *ns)
{
  NetworkMenu *self;
  NetworkMenuPrivate *priv;
  ServiceManager *sm;

  self = g_object_new(TYPE_NETWORK_MENU, NULL);
  priv = GET_PRIVATE(self);

  priv->network_service = ns;
  g_return_val_if_fail(manager_get_android_service(ns) != NULL, NULL);
  priv->android_service = g_object_ref(manager_get_android_service(ns));

  g_signal_connect(priv->android_service,
		   "airplane-mode-changed",
		   G_CALLBACK(airplane_mode_changed_cb),
		   self);
  g_signal_connect(priv->android_service,
		   "mobile-data-enabled-changed",
		   G_CALLBACK(mobile_data_enabled_changed_cb),
		   self);
  g_signal_connect(priv->android_service,
		   "wifi-enabled-changed",
		   G_CALLBACK(wifi_enabled_changed_cb),
		   self);

  priv->wireless_enabled = android_service_get_wifi_enabled(priv->android_service);
  priv->offline_enabled = android_service_get_airplane_mode(priv->android_service);
  priv->cellular_enabled = android_service_get_mobile_data_enabled(priv->android_service);

  sm = manager_get_service_manager(ns);
  g_signal_connect(G_OBJECT(sm), "services-updated",
		   G_CALLBACK(services_updated), self);
  g_signal_connect(G_OBJECT(sm), "state-changed",
		   G_CALLBACK(connect_state_changed), self);

  /* offline mode */
  priv->offline = dbusmenu_menuitem_new ();
  dbusmenu_menuitem_property_set(priv->offline,
				 DBUSMENU_MENUITEM_PROP_TYPE,
				 TECH_MENUITEM_NAME);
  dbusmenu_menuitem_property_set(priv->offline,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 _("Flight Mode"));
  dbusmenu_menuitem_property_set_int(priv->offline,
				     DBUSMENU_MENUITEM_PROP_TOGGLE_STATE,
				     priv->offline_enabled);
  dbusmenu_menuitem_child_append(DBUSMENU_MENUITEM(self), priv->offline);
  g_signal_connect(G_OBJECT(priv->offline),
		   DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED,
		   G_CALLBACK(toggle_offline_cb), self);

  priv->offline_separator = dbusmenu_menuitem_new();
  dbusmenu_menuitem_property_set(priv->offline_separator,
				 DBUSMENU_MENUITEM_PROP_TYPE,
				 DBUSMENU_CLIENT_TYPES_SEPARATOR);
  dbusmenu_menuitem_child_append(DBUSMENU_MENUITEM(self),
				 priv->offline_separator);

  /* Wired section of the menu */
  priv->wired_header = dbusmenu_menuitem_new ();
  dbusmenu_menuitem_property_set_bool(priv->wired_header,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_child_append(DBUSMENU_MENUITEM(self), priv->wired_header);

  priv->wired_separator = dbusmenu_menuitem_new();
  dbusmenu_menuitem_property_set(priv->wired_separator,
				 DBUSMENU_MENUITEM_PROP_TYPE,
				 DBUSMENU_CLIENT_TYPES_SEPARATOR);
  dbusmenu_menuitem_child_append(DBUSMENU_MENUITEM(self), priv->wired_separator);

  /* Wireless section of the menu */
  priv->wireless_header = dbusmenu_menuitem_new ();
  dbusmenu_menuitem_property_set(priv->wireless_header,
				 DBUSMENU_MENUITEM_PROP_TYPE,
				 TECH_MENUITEM_NAME);
  dbusmenu_menuitem_property_set(priv->wireless_header,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 _("Wi-Fi"));
  dbusmenu_menuitem_child_append(DBUSMENU_MENUITEM(self), priv->wireless_header);
  dbusmenu_menuitem_property_set_int(priv->wireless_header,
				     DBUSMENU_MENUITEM_PROP_TOGGLE_STATE,
				     priv->wireless_enabled);

  g_signal_connect(G_OBJECT(priv->wireless_header),
		   DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED,
		   G_CALLBACK(toggle_wireless_cb), self);

  priv->wireless_list_top = dbusmenu_menuitem_new ();
  dbusmenu_menuitem_property_set(priv->wireless_list_top,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 _("No network detected"));
  dbusmenu_menuitem_property_set_bool(priv->wireless_list_top,
				      DBUSMENU_MENUITEM_PROP_ENABLED, FALSE);
  dbusmenu_menuitem_child_append(DBUSMENU_MENUITEM(self),
				 priv->wireless_list_top);

  priv->wireless_other = dbusmenu_menuitem_new ();
  dbusmenu_menuitem_property_set(priv->wireless_other,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 _("More Networks..."));
  g_signal_connect(G_OBJECT(priv->wireless_other),
		   DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED,
		   G_CALLBACK(wireless_other_activated), self);
  dbusmenu_menuitem_child_append(DBUSMENU_MENUITEM(self),
				 priv->wireless_other);

  priv->wireless_separator = dbusmenu_menuitem_new ();
  dbusmenu_menuitem_property_set(priv->wireless_separator,
				 DBUSMENU_MENUITEM_PROP_TYPE,
				 DBUSMENU_CLIENT_TYPES_SEPARATOR);
  dbusmenu_menuitem_child_append(DBUSMENU_MENUITEM(self),
				 priv->wireless_separator);

  /* Mobile section */
  priv->cellular_header = dbusmenu_menuitem_new ();
  dbusmenu_menuitem_property_set(priv->cellular_header,
				 DBUSMENU_MENUITEM_PROP_TYPE,
				 TECH_MENUITEM_NAME);
  dbusmenu_menuitem_property_set(priv->cellular_header,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 _("Mobile Data"));
  dbusmenu_menuitem_child_append(DBUSMENU_MENUITEM(self),
				 priv->cellular_header);

  priv->operator_item = dbusmenu_menuitem_new ();
  dbusmenu_menuitem_property_set_bool(priv->operator_item,
				      DBUSMENU_MENUITEM_PROP_ENABLED,
				      FALSE);
  dbusmenu_menuitem_child_append(DBUSMENU_MENUITEM(self),
				 priv->operator_item);

  priv->cellular_separator = dbusmenu_menuitem_new();
  dbusmenu_menuitem_property_set(priv->cellular_separator,
				 DBUSMENU_MENUITEM_PROP_TYPE,
				 DBUSMENU_CLIENT_TYPES_SEPARATOR);
  dbusmenu_menuitem_child_append(DBUSMENU_MENUITEM(self),
				 priv->cellular_separator);

  g_signal_connect(G_OBJECT(priv->cellular_header),
		   DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED,
		   G_CALLBACK(toggle_cellular_cb), self);


  /* Network settings */
  priv->network_settings = dbusmenu_menuitem_new ();
  dbusmenu_menuitem_property_set(priv->network_settings,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 _("Network Settings..."));
  g_signal_connect(G_OBJECT(priv->network_settings),
		   DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED,
		   G_CALLBACK(network_settings_activated), self);
  dbusmenu_menuitem_child_append(DBUSMENU_MENUITEM(self),
				 priv->network_settings);

  /* Connection Information */
  priv->connection_information = dbusmenu_menuitem_new ();
  dbusmenu_menuitem_property_set(priv->connection_information,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 _("Connection Information..."));
  g_signal_connect(G_OBJECT(priv->connection_information),
		   DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED,
		   G_CALLBACK(connection_information_activated), self);
  dbusmenu_menuitem_child_append(DBUSMENU_MENUITEM(self),
				 priv->connection_information);

  /* "androidservice not available" message */
  priv->disabled_menuitem = dbusmenu_menuitem_new();
  dbusmenu_menuitem_property_set(priv->disabled_menuitem,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 _("Android Service not available"));
  dbusmenu_menuitem_property_set_bool(priv->disabled_menuitem,
				      DBUSMENU_MENUITEM_PROP_ENABLED, FALSE);
  dbusmenu_menuitem_property_set_bool(priv->disabled_menuitem,
				      DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_child_append(DBUSMENU_MENUITEM(self),
				 priv->disabled_menuitem);

  /*
   * first start menu disabled, Manager will enable it when everything
   * is ready
   */
  priv->menu_enabled = TRUE;
  network_menu_disable(self);

  return self;
}

static void
wifi_enabled_changed_cb(gpointer instance, gboolean value, gpointer user_data)
{
  NetworkMenu *self = NETWORK_MENU(user_data);
  NetworkMenuPrivate *priv = GET_PRIVATE(self);

  g_return_if_fail(IS_NETWORK_MENU(self));
  g_return_if_fail(priv != NULL);
  g_return_if_fail(priv->wireless_header != NULL);

  priv->wireless_enabled = value;

  dbusmenu_menuitem_property_set_int(priv->wireless_header,
				     DBUSMENU_MENUITEM_PROP_TOGGLE_STATE,
				     priv->wireless_enabled);
}

static void
airplane_mode_changed_cb(gpointer instance, gboolean value, gpointer user_data)
{
  NetworkMenu *self = NETWORK_MENU(user_data);
  NetworkMenuPrivate *priv = GET_PRIVATE(self);
  gboolean mobile_mode = FALSE;
  gboolean mobile_toggle_enable = FALSE;

  g_return_if_fail(IS_NETWORK_MENU(self));
  g_return_if_fail(priv != NULL);
  g_return_if_fail(priv->android_service != NULL);
  g_return_if_fail(priv->offline != NULL);

  priv->offline_enabled = value;

  dbusmenu_menuitem_property_set_int(priv->offline,
				     DBUSMENU_MENUITEM_PROP_TOGGLE_STATE,
				     priv->offline_enabled);
  if (priv->offline_enabled) {
    mobile_toggle_enable = FALSE;
    mobile_mode = FALSE;
  } else {
    mobile_toggle_enable = TRUE;
    mobile_mode = priv->cellular_enabled;
  }
  dbusmenu_menuitem_property_set_int(priv->cellular_header,
				     DBUSMENU_MENUITEM_PROP_ENABLED,
				     mobile_toggle_enable);
  dbusmenu_menuitem_property_set_int(priv->cellular_header,
				     DBUSMENU_MENUITEM_PROP_TOGGLE_STATE,
				     mobile_mode);
}

static void
mobile_data_enabled_changed_cb(gpointer instance,
			       gboolean value,
			       gpointer user_data)
{
  NetworkMenu *self = NETWORK_MENU(user_data);
  NetworkMenuPrivate *priv = GET_PRIVATE(self);
  gboolean mobile_mode = FALSE;

  g_return_if_fail(IS_NETWORK_MENU(self));
  g_return_if_fail(priv != NULL);
  g_return_if_fail(priv->android_service != NULL);
  g_return_if_fail(priv->offline != NULL);

  priv->cellular_enabled = value;

  if (priv->offline_enabled) {
    mobile_mode = FALSE;
  } else {
    mobile_mode = priv->cellular_enabled;
  }
  dbusmenu_menuitem_property_set_int(priv->cellular_header,
				     DBUSMENU_MENUITEM_PROP_TOGGLE_STATE,
				     mobile_mode);
}

void
network_menu_set_operator(NetworkMenu *self, const gchar *value)
{
  NetworkMenuPrivate *priv = GET_PRIVATE(self);

  if (g_strcmp0(value, "") == 0) {
    // if operator name is empty, hide the menuitem
    priv->operator_shown = FALSE;
    dbusmenu_menuitem_property_set_bool(priv->operator_item,
					DBUSMENU_MENUITEM_PROP_VISIBLE,
					priv->operator_shown);
    return;
  } else {
    priv->operator_shown = TRUE;
  }

  dbusmenu_menuitem_property_set(priv->operator_item,
				 DBUSMENU_MENUITEM_PROP_LABEL,
				 value);
  if (priv->menu_enabled) {
    // set visible only if the menu is enabled
    dbusmenu_menuitem_property_set_bool(priv->operator_item,
					DBUSMENU_MENUITEM_PROP_VISIBLE, TRUE);
  }
}
