# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###

import  gtk
import gobject

import utils
import time
from helper import Dispatcher
from library import ListenDB
from player import Player

from audioscrobbler_manager import AudioScrobblerManager

class SongTimer(gtk.HBox):
    __gsignals__ = {
        "play-end" : (gobject.SIGNAL_RUN_LAST,gobject.TYPE_NONE,())
    }
    def __init__(self):
        super(SongTimer,self).__init__(spacing=6)

        self.label_time = gtk.Label("<span size=\"small\">00:00 "+_("of")+" 00:00</span>")
        self.label_time.set_alignment(1,1)
        self.label_time.set_use_markup(True)

        self.bar = gtk.HScale()
        self.bar.set_draw_value(False)
        self.bar.set_range(0,1000)
        self.bar.set_value(0)
        self.bar.set_increments(1, 20)
        self.bar.connect("button_press_event", self.on_bar_press)
        self.bar.connect("button_release_event", self.on_bar_release)
        self.__value_changed_id = self.bar.connect("value-changed", self.on_bar_value_changed)
        self.bar.handler_block(self.__value_changed_id)
        self.bar.set_size_request(250,-1)
        self.pack_start(self.bar,True,True)
        #self.pack_start(self.label_time,False, False)
        self.update_bar = 1
        self.duration = 0
        self.__idle_release_id = None
        self.delete = False
        self.__need_report = False

        Player.connect("instant-new-song",self.set_duration)

        Player.bin.connect("tick",self.on_tick)
        Player.connect("seeked",self.on_seek)
        Player.connect("stopped",self.set_duration)

    def get_label(self):
        return self.label_time
    
    def stop(self):
        self.delete = True

    def set_duration(self,player,song=None):
        if not song:
            if player.song: song = player.song
            else: return 
        if player.current_stream_seeked():
            self.__need_report = False
        else:
            self.__need_report = True
        
        self.duration = song.get("#duration") / 1000
        self.set_current_time(0, self.duration)

    def on_tick(self,bin,pos,duration):
        self.duration = duration
        if self.update_bar == 1:
            self.set_current_time(pos,duration)
        return not self.delete

    def set_current_time(self,pos, duration):
        if self.update_bar == 1:
            if duration == 0 or pos >= duration:
                self.bar.set_range(0,1000)
                self.bar.set_value(0)
                self.bar.set_sensitive(False)
            else :
                self.bar.set_sensitive(True)
                self.bar.set_range(0,duration)
                self.bar.set_value(pos)

        if pos >= duration:
            text = utils.duration_to_string(pos*1000,"00:00")
        else:
            text = utils.duration_to_string(pos*1000,"00:00")+" "+_("of")+" "+utils.duration_to_string(duration*1000,"00:00")
        self.label_time.set_label("<span size=\"small\">"+text+"</span>")
        if self.__need_report and duration>10 and pos >= min(duration/2, 240*1000):
            self.__need_report = False
            s = Player.song
            if s:
                if s.get("#playcount"):
                    ListenDB.set_property(s,{"#playcount":s.get("#playcount")+1})
                else:
                    ListenDB.set_property(s,{"#playcount":1})
                ListenDB.set_property(s,{"#lastplayed":time.time()})
                stamp = str(int(time.time()) - duration)
                Player.set_song_reported()
                AudioScrobblerManager.report_song(Player.song, stamp)

    def on_seek(self, *args, **kwargs):
        self.__need_report = False
    
    def on_bar_value_changed(self, widget):
        pos = self.bar.get_value()
        self.set_current_time(pos, self.duration)

    def on_bar_press(self,widget,event):
        if self.__idle_release_id:
            gobject.source_remove(self.__idle_release_id)
            self.__idle_release_id = None
        self.update_bar = 0
        self.__need_report = False

        self.bar.handler_unblock(self.__value_changed_id)

    def __idle_release(self):
        self.update_bar = 1
        self.__idle_release_id = None

    def on_bar_release(self,widget,event):
        self.bar.handler_block(self.__value_changed_id)

        if Player.song!=None and Player.song.get_type() in [ "webradio","volatile_webradio"]:
            return
        Player.seek(self.bar.get_value())

        # wait a bit that the player are really seek to update the progress bar
        if not self.__idle_release_id:
            self.__idle_release_id = gobject.idle_add(self.__idle_release)
