/*
 * Copyright 2009 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of either or both of the following licenses:
 *
 * 1) the GNU Lesser General Public License version 3, as published by the
 * Free Software Foundation; and/or
 * 2) the GNU Lesser General Public License version 2.1, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the applicable version of the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of both the GNU Lesser General Public
 * License version 3 and version 2.1 along with this program.  If not, see
 * <http://www.gnu.org/licenses/>
 *
 * Authored by: Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "gadget-proxy.h"

G_DEFINE_ABSTRACT_TYPE (GadgetProxy, gadget_proxy, G_TYPE_OBJECT);

#define GADGET_PROXY_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  GADGET_TYPE_PROXY, \
  GadgetProxyPrivate))

struct _GadgetProxyPrivate
{
  gchar *path;
  gchar *uid;
  gchar *name;
  gint   width;
  gint   height;
};

enum
{
  PROP_0,

  PROP_PATH,
  PROP_UID,
  PROP_NAME,
  PROP_WIDTH,
  PROP_HEIGHT
};

enum
{
  QUEUE_REDRAW,
  QUEUE_RESIZE,
  OPEN_URL,
  BEGIN_RESIZE,
  BEGIN_DRAG_MOVE,
  REMOVE_ME,

  BUTTON_PRESS_EVENT,
  BUTTON_RELEASE_EVENT,
  ENTER_EVENT,
  LEAVE_EVENT,
  KEY_PRESS_EVENT,
  KEY_RELEASE_EVENT,
  SCROLL_EVENT,
  MOTION_EVENT,
  FOCUS_IN_EVENT,
  FOCUS_OUT_EVENT,

  LAST_SIGNAL
};

/* Globals */
static guint32 _gadget_proxy_signals[LAST_SIGNAL] = { 0 };

/* Forwards */


/* GObject stuff */
static void
gadget_proxy_finalize (GObject *object)
{
  GadgetProxyPrivate *priv = GADGET_PROXY_GET_PRIVATE (object);

  if (priv->path)
    {
      g_free (priv->path);
      priv->path = NULL;
    }
  if (priv->uid)
    {
      g_free (priv->uid);
      priv->uid = NULL;
    }
  if (priv->name)
    {
      g_free (priv->name);
      priv->name = NULL;
    }

  G_OBJECT_CLASS (gadget_proxy_parent_class)->finalize (object);
}

static void
gadget_proxy_set_property (GObject      *object,
                           guint         prop_id,
                           const GValue *value,
                           GParamSpec   *pspec)
{
  GadgetProxy *self = GADGET_PROXY (object);

  switch (prop_id)
    {
    case PROP_PATH:
      gadget_proxy_set_path (self, g_value_get_string (value));
      break;

    case PROP_UID:
      gadget_proxy_set_uid (self, g_value_get_string (value));
      break;

    case PROP_NAME:
      gadget_proxy_set_name (self, g_value_get_string (value));
      break;

    case PROP_WIDTH:
      gadget_proxy_set_width (self, g_value_get_int (value));
      break;

    case PROP_HEIGHT:
      gadget_proxy_set_height (self, g_value_get_int (value));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
gadget_proxy_get_property (GObject    *object,
                           guint       prop_id,
                           GValue     *value,
                           GParamSpec *pspec)
{
  GadgetProxy *self = GADGET_PROXY (object);

  switch (prop_id)
    {
    case PROP_PATH:
      g_value_set_string (value, gadget_proxy_get_path (self));
      break;

    case PROP_UID:
      g_value_set_string (value, gadget_proxy_get_uid (self));
      break;

    case PROP_NAME:
      g_value_set_string (value, gadget_proxy_get_name (self));
      break;

    case PROP_WIDTH:
      g_value_set_int (value, gadget_proxy_get_width (self));
      break;

    case PROP_HEIGHT:
      g_value_set_int (value, gadget_proxy_get_height (self));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
gadget_proxy_class_init (GadgetProxyClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);
  GParamSpec   *pspec;

  /* Overrides */
  obj_class->finalize        = gadget_proxy_finalize;
  obj_class->set_property    = gadget_proxy_set_property;
  obj_class->get_property    = gadget_proxy_get_property;

  /* Install Properties */
  pspec = g_param_spec_string ("path", "path",
                               "Path to gadget_proxy",
                               NULL, G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_PATH, pspec);

  pspec = g_param_spec_string ("uid", "uid",
                               "Unique identifier of gadget_proxy",
                               NULL, G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_UID, pspec);

  pspec = g_param_spec_string ("name", "name",
                               "Name of gadget_proxy",
                               NULL, G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_NAME, pspec);

  pspec = g_param_spec_int ("width", "width", "Preferred width of gadget_proxy",
                            0, G_MAXINT, 1,
                            G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_WIDTH, pspec);

  pspec = g_param_spec_int ("height", "height",
                            "Preferred height of gadget_proxy",
                            0, G_MAXINT, 1,
                            G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_HEIGHT, pspec);

  /* Add signals */
  _gadget_proxy_signals[QUEUE_REDRAW] =
    g_signal_new ("queue-redraw",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (GadgetProxyClass, queue_redraw),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  _gadget_proxy_signals[QUEUE_RESIZE] =
    g_signal_new ("queue-resize",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (GadgetProxyClass, queue_resize),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  _gadget_proxy_signals[OPEN_URL] =
    g_signal_new ("open-url",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (GadgetProxyClass, open_url),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1, G_TYPE_STRING);

  _gadget_proxy_signals[REMOVE_ME] =
    g_signal_new ("remove-me",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (GadgetProxyClass, remove_me),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  _gadget_proxy_signals[BEGIN_RESIZE] =
    g_signal_new ("begin-resize",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (GadgetProxyClass, begin_resize),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__INT,
                  G_TYPE_NONE, 1, G_TYPE_INT);

  _gadget_proxy_signals[BEGIN_DRAG_MOVE] =
    g_signal_new ("begin-drag-move",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (GadgetProxyClass, begin_drag_move),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  _gadget_proxy_signals[BUTTON_PRESS_EVENT] =
    g_signal_new ("button-press-event",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (GadgetProxyClass, button_press_event),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE, 1, G_TYPE_POINTER);

  _gadget_proxy_signals[BUTTON_RELEASE_EVENT] =
    g_signal_new ("button-release-event",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (GadgetProxyClass, button_release_event),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE, 1, G_TYPE_POINTER);

  _gadget_proxy_signals[ENTER_EVENT] =
    g_signal_new ("enter-event",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (GadgetProxyClass, enter_event),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE, 1, G_TYPE_POINTER);

  _gadget_proxy_signals[LEAVE_EVENT] =
    g_signal_new ("leave-event",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (GadgetProxyClass, leave_event),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE, 1, G_TYPE_POINTER);

  _gadget_proxy_signals[KEY_PRESS_EVENT] =
    g_signal_new ("key-press-event",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (GadgetProxyClass, key_press_event),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE, 1, G_TYPE_POINTER);

  _gadget_proxy_signals[KEY_RELEASE_EVENT] =
    g_signal_new ("key-release-event",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (GadgetProxyClass, key_release_event),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE, 1, G_TYPE_POINTER);

  _gadget_proxy_signals[SCROLL_EVENT] =
    g_signal_new ("scroll-event",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (GadgetProxyClass, scroll_event),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE, 1, G_TYPE_POINTER);

  _gadget_proxy_signals[MOTION_EVENT] =
    g_signal_new ("motion-event",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (GadgetProxyClass, motion_event),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE, 1, G_TYPE_POINTER);

  _gadget_proxy_signals[FOCUS_IN_EVENT] =
    g_signal_new ("focus-in-event",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (GadgetProxyClass, focus_in_event),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  _gadget_proxy_signals[FOCUS_OUT_EVENT]=
    g_signal_new ("focus-out-event",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (GadgetProxyClass, focus_out_event),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);


  /* Add Private struct */
  g_type_class_add_private (obj_class, sizeof (GadgetProxyPrivate));
}

static void
gadget_proxy_init (GadgetProxy *self)
{
  GadgetProxyPrivate *priv;

  priv = self->priv = GADGET_PROXY_GET_PRIVATE (self);
}

/*
 * Private methods
 */

/*
 * Public methods
 */
const gchar *
gadget_proxy_get_path(GadgetProxy      *self)
{
  g_return_val_if_fail (GADGET_IS_PROXY (self), NULL);
  return self->priv->path;
}

void
gadget_proxy_set_path (GadgetProxy      *self,
                       const gchar *path)
{
  g_return_if_fail (GADGET_IS_PROXY (self));

  if (self->priv->path)
    g_free (self->priv->path);

  self->priv->path = NULL;

  if (path)
    self->priv->path = g_strdup (path);

  g_object_notify (G_OBJECT (self), "path");
}

const gchar *
gadget_proxy_get_uid (GadgetProxy *self)
{
  g_return_val_if_fail (GADGET_IS_PROXY (self), NULL);
  return self->priv->uid;
}

void
gadget_proxy_set_uid (GadgetProxy      *self,
                      const gchar *uid)
{
  g_return_if_fail (GADGET_IS_PROXY (self));

  if (self->priv->uid)
    g_free (self->priv->uid);

  self->priv->uid = NULL;

  if (uid)
    self->priv->uid = g_strdup (uid);

  g_object_notify (G_OBJECT (self), "uid");
}

const gchar *
gadget_proxy_get_name (GadgetProxy      *self)
{
  g_return_val_if_fail (GADGET_IS_PROXY (self), NULL);
  return self->priv->name;
}

void
gadget_proxy_set_name (GadgetProxy      *self,
                       const gchar *name)
{
  g_return_if_fail (GADGET_IS_PROXY (self));

  if (self->priv->name)
    g_free (self->priv->name);

  self->priv->name = NULL;

  if (name)
    self->priv->name = g_strdup (name);

  g_object_notify (G_OBJECT (self), "name");
}

gint
gadget_proxy_get_width (GadgetProxy      *self)
{
  g_return_val_if_fail (GADGET_IS_PROXY (self), 0);
  return self->priv->width;
}

void
gadget_proxy_set_width (GadgetProxy      *self,
                        gint         width)
{
  g_return_if_fail (GADGET_IS_PROXY (self));

  self->priv->width = width;

  g_object_notify (G_OBJECT (self), "width");
}


gint
gadget_proxy_get_height (GadgetProxy      *self)
{
  g_return_val_if_fail (GADGET_IS_PROXY (self), 0);
  return self->priv->height;
}

void
gadget_proxy_set_height (GadgetProxy      *self,
                         gint         height)
{
  g_return_if_fail (GADGET_IS_PROXY (self));

  self->priv->height = height;

  g_object_notify (G_OBJECT (self), "height");
}

void
gadget_proxy_event (GadgetProxy *self,
                    GdkEvent    *event)
{
  gint nsignal;

  g_return_if_fail (GADGET_IS_PROXY (self));
  g_return_if_fail (event);

  nsignal = 0;

  switch (event->type)
    {
    case GDK_BUTTON_PRESS:
      nsignal = BUTTON_PRESS_EVENT;
      break;

    case GDK_BUTTON_RELEASE:
      nsignal = BUTTON_RELEASE_EVENT;
      break;

    case GDK_ENTER_NOTIFY:
      nsignal = ENTER_EVENT;
      break;

    case GDK_LEAVE_NOTIFY:
      nsignal = LEAVE_EVENT;
      break;

    case GDK_KEY_PRESS:
      nsignal = KEY_PRESS_EVENT;
      break;

    case GDK_KEY_RELEASE:
      nsignal = KEY_RELEASE_EVENT;
      break;

    case GDK_SCROLL:
      nsignal = SCROLL_EVENT;
      break;

    case GDK_MOTION_NOTIFY:
      nsignal = MOTION_EVENT;
      break;

    default:
      nsignal = 0;
      break;
    }

  if (nsignal)
    g_signal_emit (self, _gadget_proxy_signals[nsignal], 0, event);
}

void
gadget_proxy_set_focused (GadgetProxy *self,
                          gboolean     is_focused)
{
  g_return_if_fail (GADGET_IS_PROXY (self));

  g_signal_emit (self,
                 _gadget_proxy_signals[is_focused ? FOCUS_IN_EVENT
                                       : FOCUS_OUT_EVENT],
                 0);
}

GdkPixbuf *
gadget_proxy_get_icon (GadgetProxy *self)
{
  GadgetProxyClass *klass;

  g_return_val_if_fail (GADGET_IS_PROXY (self), NULL);
  klass = GADGET_PROXY_GET_CLASS (self);

  if (klass->get_icon)
    return klass->get_icon (self);

  return NULL;
}

void
gadget_proxy_removed (GadgetProxy *self)
{
  GadgetProxyClass *klass;

  g_return_if_fail (GADGET_IS_PROXY (self));
  klass = GADGET_PROXY_GET_CLASS (self);

  if (klass->removed)
    klass->removed (self);
}
