/*
 * Copyright 2009 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of either or both of the following licenses:
 *
 * 1) the GNU Lesser General Public License version 3, as published by the
 * Free Software Foundation; and/or
 * 2) the GNU Lesser General Public License version 2.1, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the applicable version of the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of both the GNU Lesser General Public
 * License version 3 and version 2.1 along with this program.  If not, see
 * <http://www.gnu.org/licenses/>
 *
 * Authored by: Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "gadget-source.h"

G_DEFINE_ABSTRACT_TYPE (GadgetSource, gadget_source, G_TYPE_OBJECT);

#define GADGET_SOURCE_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  GADGET_TYPE_SOURCE, \
  GadgetSourcePrivate))

struct _GadgetSourcePrivate
{
  gchar *name;
  gchar *icon;
};

enum
{
  PROP_0,

  PROP_NAME,
  PROP_ICON
};

#if 0
enum
{
  LAST_SIGNAL
};

/* Globals */
static guint32 _gadget_source_signals[LAST_SIGNAL] = { 0 };
#endif
/* Forwards */


/* GObject stuff */
static void
gadget_source_finalize (GObject *object)
{
  GadgetSourcePrivate *priv = GADGET_SOURCE_GET_PRIVATE (object);

  if (priv->name)
    {
      g_free (priv->name);
      priv->name = NULL;
    }
  if (priv->icon)
    {
      g_free (priv->icon);
      priv->icon = NULL;
    }

  G_OBJECT_CLASS (gadget_source_parent_class)->finalize (object);
}

static void
gadget_source_set_property (GObject      *object,
                            guint         prop_id,
                            const GValue *value,
                            GParamSpec   *pspec)
{
  GadgetSource *self = GADGET_SOURCE (object);

  switch (prop_id)
    {
    case PROP_NAME:
      gadget_source_set_name (self, g_value_get_string (value));
      break;

    case PROP_ICON:
      gadget_source_set_icon (self, g_value_get_string (value));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
gadget_source_get_property (GObject    *object,
                            guint       prop_id,
                            GValue     *value,
                            GParamSpec *pspec)
{
  GadgetSource *self = GADGET_SOURCE (object);

  switch (prop_id)
    {
    case PROP_NAME:
      g_value_set_string (value, gadget_source_get_name (self));
      break;

    case PROP_ICON:
      g_value_set_string (value, gadget_source_get_icon (self));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
gadget_source_class_init (GadgetSourceClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);
  GParamSpec   *pspec;

  /* Overrides */
  obj_class->finalize        = gadget_source_finalize;
  obj_class->set_property    = gadget_source_set_property;
  obj_class->get_property    = gadget_source_get_property;

  /* Install Properties */
  pspec = g_param_spec_string ("name", "name", "Name of gadget source",
                               NULL, G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_NAME, pspec);

  pspec = g_param_spec_string ("icon", "icon",
                               "Icon representing gadget source", NULL,
                               G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_ICON, pspec);

  /* Add signals */

  /* Add Private struct */
  g_type_class_add_private (obj_class, sizeof (GadgetSourcePrivate));
}

static void
gadget_source_init (GadgetSource *self)
{
  GadgetSourcePrivate *priv;

  priv = self->priv = GADGET_SOURCE_GET_PRIVATE (self);
}

/*
 * Private methods
 */

/*
 * Public methods
 */
const gchar *
gadget_source_get_name (GadgetSource *self)
{
  g_return_val_if_fail (GADGET_IS_SOURCE (self), NULL);
  return self->priv->name;
}

void
gadget_source_set_name (GadgetSource *self,
                        const gchar  *name)
{
  g_return_if_fail (GADGET_IS_SOURCE (self));

  if (self->priv->name)
    g_free (self->priv->name);

  self->priv->name = NULL;

  if (name)
    self->priv->name = g_strdup (name);

  g_object_notify (G_OBJECT (self), "name");
}

const gchar *
gadget_source_get_icon (GadgetSource *self)
{
  g_return_val_if_fail (GADGET_IS_SOURCE (self), NULL);
  return self->priv->icon;
}

void
gadget_source_set_icon (GadgetSource *self,
                        const gchar  *icon)
{
  g_return_if_fail (GADGET_IS_SOURCE (self));

  if (self->priv->icon)
    g_free (self->priv->icon);

  self->priv->icon = NULL;

  if (icon)
    self->priv->icon = g_strdup (icon);

  g_object_notify (G_OBJECT (self), "name");
}

gboolean
gadget_source_can_handle_path (GadgetSource *self,
                               const gchar  *path)
{
  GadgetSourceClass *klass;

  g_return_val_if_fail (GADGET_IS_SOURCE (self), FALSE);
  g_return_val_if_fail (path, FALSE);
  klass = GADGET_SOURCE_GET_CLASS (self);

  if (klass->can_handle_path)
    return klass->can_handle_path (self, path);

  return FALSE;
}

GadgetProxy *
gadget_source_load_gadget (GadgetSource *self,
                           const gchar  *path,
                           const gchar  *uid)
{
  GadgetSourceClass *klass;

  g_return_val_if_fail (GADGET_IS_SOURCE (self), NULL);
  g_return_val_if_fail (path, NULL);
  g_return_val_if_fail (uid, NULL);
  klass = GADGET_SOURCE_GET_CLASS (self);

  if (klass->load_gadget)
    return klass->load_gadget (self, path, uid);

  return NULL;
}

GList *
gadget_source_get_available_gadgets (GadgetSource *self)
{
  GadgetSourceClass *klass;

  g_return_val_if_fail (GADGET_IS_SOURCE (self), NULL);
  klass = GADGET_SOURCE_GET_CLASS (self);

  if (klass->get_available_gadgets)
    return klass->get_available_gadgets (self);

  return NULL;
}
