/*
 * Copyright (C) 2009 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "launcher-application.h"

#include <gio/gdesktopappinfo.h>

struct _LauncherApplication
{
  gint ref_count;

  WnckApplication *wnckapp;

  gchar *name;
  gchar *match_name;
  gchar *comment;
  gchar *icon_name;
  gchar *exec_path;
  gchar *desktop_file;
};

GType
launcher_application_get_type (void)
{
  static GType _launcher_application_type = 0;

  if (G_UNLIKELY (_launcher_application_type == 0))
    {
      _launcher_application_type =
        g_boxed_type_register_static ("LauncherApplication",
                                      (GBoxedCopyFunc) launcher_application_copy,
                                      (GBoxedFreeFunc) launcher_application_free);

    }

  return _launcher_application_type;
}

/*
 * Public Methods
 */
LauncherApplication *
launcher_application_new_from_info (const gchar  *name,
                                    const gchar  *comment,
                                    const gchar  *icon_name,
                                    const gchar  *exec_path,
                                    const gchar  *desktop_file)
{
  LauncherApplication *application;

  application = g_slice_new0 (LauncherApplication);

  if (application != NULL)
    {
      application->ref_count = 1;

      application->name = g_strdup (name);
      application->comment = g_strdup (comment);
      application->icon_name = g_strdup (icon_name);
      application->exec_path = g_strdup (exec_path);
      application->desktop_file = g_strdup (desktop_file);
    }


  return application;
}

LauncherApplication *
launcher_application_new_from_wnck_app (WnckApplication *app)
{
  LauncherApplication *application;

  application = g_slice_new0 (LauncherApplication);

  if (application != NULL)
    {
      application->ref_count = 1;

      application->wnckapp = app;
      application->name = g_strdup (wnck_application_get_name (app));
      application->icon_name = g_strdup (wnck_application_get_icon_name (app));
    }

  return application;
}

LauncherApplication *
launcher_application_copy (const LauncherApplication *application)
{
  if (G_LIKELY (application != NULL))
    {
      LauncherApplication *ret = NULL;

      ret = g_slice_new0 (LauncherApplication);
      /*FIXME: Update for real copy */
      return ret;
    }
  return NULL;
}

void
launcher_application_free (LauncherApplication *application)
{
  if (G_LIKELY (application != NULL))
    {
      g_free (application->name);
      g_free (application->match_name);
      g_free (application->comment);
      g_free (application->icon_name);
      g_free (application->exec_path);
      g_free (application->desktop_file);

      g_slice_free (LauncherApplication, application);
      application = NULL;
    }
}

LauncherApplication *
launcher_application_ref (LauncherApplication *application)
{
  g_return_val_if_fail (application, NULL);
  g_return_val_if_fail (application->ref_count > 0, NULL);

  g_atomic_int_inc (&application->ref_count);

  return application;
}

void
launcher_application_unref (LauncherApplication *application)
{
  g_return_if_fail (application);
  g_return_if_fail (application->ref_count > 0);

  if (G_UNLIKELY (g_atomic_int_dec_and_test (&application->ref_count)))
    {
      launcher_application_free (application);
    }
}

void
launcher_application_set_wnckapp (LauncherApplication *application,
                                  WnckApplication *wnckapp)
{
  g_return_if_fail (application);
  application->wnckapp = wnckapp;
}

WnckApplication *
launcher_application_get_wnckapp (LauncherApplication *application)
{
  g_return_val_if_fail (application, NULL);
  return application->wnckapp;
}

const gchar *
launcher_application_get_name (LauncherApplication *application)
{
  g_return_val_if_fail (application, NULL);
  return application->name;
}

const gchar *
launcher_application_get_comment (LauncherApplication *application)
{
  g_return_val_if_fail (application, NULL);
  return application->comment;
}

const gchar *
launcher_application_get_icon_name (LauncherApplication *application)
{
  g_return_val_if_fail (application, NULL);
  return application->icon_name;
}

const gchar *
launcher_application_get_exec_string (LauncherApplication *application)
{
  g_return_val_if_fail (application, NULL);
  return application->exec_path;
}

const gchar *
launcher_application_get_desktop_file (LauncherApplication *application)
{
  g_return_val_if_fail (application, NULL);
  return application->desktop_file;
}

const gchar *
launcher_application_get_match_name (LauncherApplication *application)
{
  g_return_val_if_fail (application, NULL);

  if (!application->match_name)
    application->match_name = g_utf8_strdown (application->name, -1);

  return application->match_name;
}

void
launcher_application_launch (LauncherApplication *app, GError **error)
{
  GDesktopAppInfo     *info;
  GdkAppLaunchContext *context;

  g_return_if_fail (app);
  g_return_if_fail (*error == NULL);

  info = g_desktop_app_info_new_from_filename (app->desktop_file);

  if (!info)
    {
      static GQuark quark;
      if (!quark)
        quark = g_quark_from_static_string ("launcher_application_launch_error");

      *error = g_error_new (quark, 1, "Unable to load GDesktopAppInfo for %s",
                            app->desktop_file);
      return;
    }

  context = gdk_app_launch_context_new ();
  gdk_app_launch_context_set_screen (context, gdk_screen_get_default ());
  gdk_app_launch_context_set_timestamp (context, GDK_CURRENT_TIME);
  gdk_app_launch_context_set_icon_name (context, app->icon_name);

  g_app_info_launch ((GAppInfo *)info, NULL, (GAppLaunchContext*)context,
                     error);

  g_object_unref (context);
  g_object_unref (info);
}
