/*
 * Copyright (c) 2015 Intel Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <poll.h>
#include <pthread.h>
#include "sensors.h"
#include "libsensorhub.h"
#include "psh_data_dumper.h"

#define CMD_LIST_SENSORS  (1)
#define CMD_ACTIVE_SENSOR (2)
#define DATA_FORMAT_PSH (1)
#define DATA_FORMAT_ISH (2)
#define DATA_FORMAT_IIO (3)

static struct sensor_t const* android_list = NULL;
static struct sensor_t const* aware_list = NULL;
static int android_sensors_count = 0;
static int aware_sensors_count = 0;
static int data_format;
static int streaming_sensor_type;

static size_t (*dump_sensors_events)(void *data, size_t size);

static void help(const char *program_name)
{
        printf("Usage: %s --command=cmd [options]\n", program_name);
        printf("Options:\n");

        printf("\t --command=cmd, -c cmd\n");
        printf("\t\tlist: list all sensors with the interface type(default is Android).\n");
        printf("\t\tactive: active specified sensor(handle must be specified).\n");
        printf("\n");

        printf("\t --interface_type=type, -i type\n");
        printf("\t\tandroid: using Android standard session category, data structures are defined by Android, all sensors events using the same interface.\n");
        printf("\t\taware: using Aware format for and will report raw data, each sensor has its own interface.\n");
        printf("\n");

        printf("\t --format=fmt, -f fmt\n");
        printf("\t\tOnly using for Aware interface type, will specify the data structure of the sensors.\n");
        printf("\t\tpsh: sensors from PSH\n");
        printf("\t\tish: sensors from ISH\n");
        printf("\t\tiio: sensors from IIO\n");
        printf("\n");

        printf("\t --handle=sensor_handle, -H sensor_handle\n");
        printf("\t\tA handle is used to identify a sensor, it will be displayed when you list all sensors.\n");
        printf("\n");

        printf("\t --sampling_period=period, -p period\n");
        printf("\t\tSpecify the sampling period, default unit is millisecond\n");
        printf("\n");

        printf("\t --max_report_latency=latency, -l latency\n");
        printf("\t\tSpecify the maximum report latency in batch mode, default unit is millisecond\n");
        printf("\n");

        printf("\t --time_unit=unit, -u unit\n");
        printf("\t\tThe unit for sampling_period and max_report_latency if you specify one(or both) of them.\n");
        printf("\t\tms|millisecond\n");
        printf("\t\tus|microsecond\n");
        printf("\t\tns|nanosecond\n");
        printf("\n");

        printf("\t --timeout=timeout, -t timeout\n");
        printf("\t\tWhen you specify this, the program will stop dump sensor data and exit when timeout reached.\n");
        printf("\n");

        printf("\t --help, -h\n");
        printf("\t\tDisplay this help and exit\n");
        printf("\n");
}

static void dump_sensor_info(const struct sensor_t * device)
{
        printf("************************%s************************\n", device->name);
        printf("name: %s\n", device->name);
        printf("vendor: %s\n", device->vendor);
        printf("version: %d\n", device->version);
        printf("handle: %d\n", device->handle);
        printf("type: %d\n", device->type);
        printf("maxRange: %f\n", device->maxRange);
        printf("resolution: %f\n", device->resolution);
        printf("power: %f\n", device->power);
        printf("minDelay: %d\n", device->minDelay);
        printf("fifoReservedEventCount: %u\n", device->fifoReservedEventCount);
        printf("fifoMaxEventCount: %u\n", device->fifoMaxEventCount);
        printf("stringType: %s\n", device->stringType);
        printf("requiredPermission: %s\n", device->requiredPermission);
#ifdef __LP64__
        printf("maxDelay: %lld\n", device->maxDelay);
        printf("flags: 0x%llx\n", device->flags);
#else
        printf("maxDelay: %d\n", device->maxDelay);
        printf("flags: 0x%x\n", device->flags);
#endif
        printf("Reporting Mode: ");
        switch (device->flags & ~SENSOR_FLAG_WAKE_UP) {
        case SENSOR_FLAG_CONTINUOUS_MODE:
                printf("continous\n");
                break;
        case SENSOR_FLAG_ON_CHANGE_MODE:
                printf("on-change\n");
                break;
        case SENSOR_FLAG_ONE_SHOT_MODE:
                printf("one-shot\n");
                break;
        case SENSOR_FLAG_SPECIAL_REPORTING_MODE:
                printf("special-reporting\n");
                break;
        default:
                printf("unknown\n");
                break;
        }
        printf("Wakeup: %s\n", device->flags & SENSOR_FLAG_WAKE_UP ? "true" : "false");
        printf("************************%s************************\n", device->name);
}

static const char* get_sensor_name_by_handle(int handle, session_category_t category)
{
        int i;
        const char *name = NULL;

        if (category == AWARE_SERVICE) {
                for (i = 0; i < aware_sensors_count; i++) {
                        if (aware_list[i].handle == handle) {
                                name = aware_list[i].name;
                                break;
                        }
                }
        } else {
                for (i = 0; i < android_sensors_count; i++) {
                        if (android_list[i].handle == handle) {
                                name = android_list[i].name;
                                break;
                        }
                }
        }

        return name;
}

static int get_sensor_type_by_handle(int handle, session_category_t category)
{
        int i;
        int type = 0;

        if (category == AWARE_SERVICE) {
                for (i = 0; i < aware_sensors_count; i++) {
                        if (aware_list[i].handle == handle) {
                                type = aware_list[i].type;
                                break;
                        }
                }
        } else {
                for (i = 0; i < android_sensors_count; i++) {
                        if (android_list[i].handle == handle) {
                                type = android_list[i].type;
                                break;
                        }
                }
        }

        return type;
}

static size_t dump_android_sensors_events(void *data, size_t size)
{
        sensors_event_t *events = (sensors_event_t *)data;
        size_t rest = size % sizeof(sensors_event_t);
        int count = size / sizeof(sensors_event_t);
        int i;

        for (i = 0; i < count; i++) {
                if (events[i].type == SENSOR_TYPE_META_DATA)
                        printf("%s: flush complete.\n", get_sensor_name_by_handle(events[i].meta_data.sensor, ANDROID_STANDARD));
                else if (events[i].type == SENSOR_TYPE_STEP_COUNTER)
                        printf("%s: [%llu] timestamp=%lld\n", get_sensor_name_by_handle(events[i].sensor, ANDROID_STANDARD), events[i].u64.step_counter, events[i].timestamp);
                else if (events[i].type == SENSOR_TYPE_ROTATION_VECTOR || events[i].type == SENSOR_TYPE_GAME_ROTATION_VECTOR || events[i].type == SENSOR_TYPE_GEOMAGNETIC_ROTATION_VECTOR)
                        printf("%s: [%f, %f, %f, %f] timestamp=%lld\n", get_sensor_name_by_handle(events[i].sensor, ANDROID_STANDARD), events[i].data[0], events[i].data[1], events[i].data[2], events[i].data[3], events[i].timestamp);
                else
						printf("%s: [x= %f, y= %f, z= %f] timestamp=%lld\n",
								get_sensor_name_by_handle(events[i].sensor, ANDROID_STANDARD),
								events[i].data[0], events[i].data[1], events[i].data[2],
								events[i].timestamp);
		}

        return rest;
}

static size_t dump_aware_sensors_events(void *data, size_t size)
{
        if (data_format == DATA_FORMAT_PSH)
                return dump_psh_data(streaming_sensor_type, data, size);
        else if (data_format == DATA_FORMAT_ISH)
                printf("Currently ISH not supported\n");
        else if (data_format == DATA_FORMAT_IIO)
                printf("Currently IIO not supported\n");
        return 0;
}

#define MAX_BUF_SIZE  4096
#define MAX_CMD_SIZE  15
static void *sensor_events_listener(void * arg)
{
        struct pollfd *pollfds = (struct pollfd *)arg;
        char buf[MAX_BUF_SIZE];
        char cmd[MAX_CMD_SIZE + 1];
        int num;
        ssize_t size;
        size_t rest = 0;

        while (1) {
                num = poll(pollfds, 2, -1);
                if (num <= 0) {
                        if (errno == EINTR)
                                continue;
                        fprintf(stderr, "%s line: %d poll error: %s\n", __FUNCTION__, __LINE__, strerror(errno));
                        break;
                }

                if (pollfds[0].revents == POLLIN) {
                        if (rest > 0) {
                                memcpy(buf, buf + (MAX_BUF_SIZE - rest), rest);
                        }
                        size = read(pollfds[0].fd, buf + rest, MAX_BUF_SIZE - rest);
                        if (size < 0) {
                                fprintf(stderr, "%s line: %d read error: %s\n", __FUNCTION__, __LINE__, strerror(errno));
                                break;
                        }
                        rest = dump_sensors_events(buf, size + rest);
                } else if (pollfds[1].revents == POLLIN) {
                        size = read(pollfds[1].fd, cmd, MAX_CMD_SIZE);
                        if (size < 0) {
                                fprintf(stderr, "%s line: %d read error: %s\n", __FUNCTION__, __LINE__, strerror(errno));
                                break;
                        }
                        if (strncmp(cmd, "terminal", 9) == 0)
                                break;
                }
                pollfds[0].revents == 0;
                pollfds[1].revents == 0;
        }

        return NULL;
}

void sig_handler(int signo)
{
        if (signo == SIGINT) {
                /* do nothing just wake up from sleep and let the program continue and finish */
                printf("received SIGINT\n");
        }
}

int main(int argc, char *argv[])
{
        static struct option long_options[] = {
                {"command", required_argument, NULL, 'c'},
                {"interface_type", required_argument, NULL, 'i'},
                {"format", required_argument, NULL, 'f'},
                {"handle", required_argument, NULL, 'H',},
                {"sampling_period", required_argument, NULL, 'p',},
                {"max_report_latency", required_argument, NULL, 'l',},
                {"time_unit", required_argument, NULL, 'u',},
                {"timeout", required_argument, NULL, 't',},
                {"help", no_argument, NULL, 'h'},
                {NULL, 0, NULL, 0}
        };

        int command = 0;
        sensorhub_handle_t sensorhub_handle = NULL;
        int handle = 0;
        int64_t sampling_period_ns = 66666667;
        int64_t max_report_latency_ns = 0;
        int64_t unit = 1000000;
        int sampling_period_changed = 0;
        int max_report_latency_changed = 0;
        sensor_list_flag_t list_type = FLAG_ANDROID_LIST;
        session_category_t category = ANDROID_STANDARD;
        int timeout = 0x7FFFFFFF;

        int c;

        while (1) {
                int option_index = 0;

                c = getopt_long (argc, argv, "c:i:f:H:p:l:u:t:h", long_options, &option_index);

                if (c == -1)
                        break;

                switch (c) {
                case 'c':
                        if (strcmp(optarg, "list") == 0) {
                                printf("command is list sensors.\n");
                                command = CMD_LIST_SENSORS;
                        } else if (strcmp(optarg, "active") == 0) {
                                printf("command is active sensors.\n");
                                command = CMD_ACTIVE_SENSOR;
                        } else {
                                fprintf(stderr, "unknown command %s\n", optarg);
                                help(argv[0]);
                                return -1;
                        }
                        break;
                case 'i':
                        if (strcmp(optarg, "Aware") == 0 || strcmp(optarg, "aware") == 0) {
                                list_type = FLAG_AWARE_LIST;
                                category = AWARE_SERVICE;
                        } else if (strcmp(optarg, "Android") != 0 && strcmp(optarg, "android") != 0) {
                                fprintf(stderr, "Unknown interface type: %s . Use android as default.\n", optarg);
                        }
                        break;
                case 'f':
                        if (strcmp(optarg, "PSH") == 0 || strcmp(optarg, "psh") == 0) {
                                data_format = DATA_FORMAT_PSH;
                        } else if (strcmp(optarg, "ISH") == 0 || strcmp(optarg, "ish") == 0) {
                                data_format = DATA_FORMAT_ISH;
                        } else if (strcmp(optarg, "IIO") == 0 || strcmp(optarg, "iio") == 0) {
                                data_format = DATA_FORMAT_IIO;
                        } else {
                                fprintf(stderr, "Error: Invaild data format type: %s\n", optarg);
                                return -1;
                        }
                        break;
                case 'H':
                        handle = atoi(optarg);
                        break;
                case 'p':
                        sscanf(optarg, "%lld", &sampling_period_ns);
                        sampling_period_changed = 1;
                        break;
                case 'l':
                        sscanf(optarg, "%lld", &max_report_latency_ns);
                        max_report_latency_changed = 1;
                        break;
                case 'u':
                        if (strcmp(optarg, "ms") == 0 || strcmp(optarg, "millisecond") == 0) {
                                /* default value */
                                unit = 1000000;
                        } else if (strcmp(optarg, "us") == 0 || strcmp(optarg, "microsecond") == 0) {
                                unit = 1000;
                        } else if (strcmp(optarg, "ns") == 0 || strcmp(optarg, "nanosecond") == 0) {
                                unit = 1;
                        } else {
                                fprintf(stderr, "Unknown unit: %s\n", optarg);
                        }
                        break;
                case 't':
                        timeout = atoi(optarg);
                        break;
                case 'h':
                        help(argv[0]);
                        return 0;
                default:
                        break;
                }
        }

        if (command == 0) {
                fprintf(stderr, "Please specify a command!\n");
                help(argv[0]);
                return -1;
        }

        int i, ret = 0;

        if (android_list == NULL) {
                android_sensors_count = sensorhub_get_sensors_list(FLAG_ANDROID_LIST, &android_list);
        }
        if (aware_list == NULL) {
                aware_sensors_count = sensorhub_get_sensors_list(FLAG_AWARE_LIST, &aware_list);
        }

        printf("interface type set to: %s\n", category == AWARE_SERVICE ? "Aware Service" : "Android Standard");

        if (command == CMD_LIST_SENSORS) {
                int i, sensors_count;
                struct sensor_t const *list;
                if (list_type == FLAG_AWARE_LIST) {
                        sensors_count = aware_sensors_count;
                        list = aware_list;
                } else {
                        sensors_count = android_sensors_count;
                        list = android_list;
                }
                for (i = 0; i < sensors_count; i++)
                        dump_sensor_info(list + i);
        } else {
                const char *name = get_sensor_name_by_handle(handle, category);
                if (name == NULL) {
                        fprintf(stderr, "Cannot find sensor with handle(%d)\n", handle);
                        ret = -1;
                        goto error;
                }
                printf("Sensor set to: %s(handle: %d)\n", name, handle);

                if (category == AWARE_SERVICE) {
                        dump_sensors_events = dump_aware_sensors_events;
                } else {
                        dump_sensors_events = dump_android_sensors_events;
                }
                streaming_sensor_type = get_sensor_type_by_handle(handle, category);

                if (sampling_period_changed) {
                        sampling_period_ns *= unit;
                }

                if (max_report_latency_changed) {
                        max_report_latency_ns *= unit;
                }

                printf("sampling period set to: %lld nanoseconds; maximum report latency set to: %lld nanoseconds\n", sampling_period_ns, max_report_latency_ns);

                if (timeout < 0) {
                        fprintf(stderr, "Error timeout: %d\n", timeout);
                        ret = -1;
                        goto error;
                } else {
                        printf("timeout set to: %d seconds\n", timeout);
                }


                sensorhub_handle = sensorhub_open(category);
                if (sensorhub_handle == NULL) {
                        fprintf(stderr, "Connect to sensorhub error!\n");
                        ret = -1;
                        goto error;
                }

                struct pollfd pollfds[2];
                pollfds[0].fd = sensorhub_get_data_fd(sensorhub_handle);
                pollfds[0].events = POLLIN;
                pollfds[0].revents = 0;
                if (pollfds[0].fd < 0) {
                        fprintf(stderr, "Cannot get data file descriptor from sensorhub!\n");
                        ret = -1;
                        goto error;
                }

                int  wakeup[2];
                ret = pipe(wakeup);
                if (ret) {
                        fprintf(stderr, "pipe error: %s!\n", strerror(errno));
                        goto error;
                }
                pollfds[1].fd = wakeup[0];
                pollfds[1].events = POLLIN;
                pollfds[1].revents = 0;

                if (signal(SIGINT, sig_handler) == SIG_ERR)
                        fprintf(stderr, "cannot catch SIGINT\n");

                pthread_t tid;
                int err;
                err = pthread_create(&tid, NULL, sensor_events_listener, pollfds);
                if (err) {
                        fprintf(stderr, "Thread create error: %s", strerror(err));
                        ret = -1;
                        goto error;
                }

                ret = sensorhub_start_streaming(sensorhub_handle, handle, (sampling_period_ns + 500) / 1000, (max_report_latency_ns + 500) / 1000);
                if (ret) {
                        fprintf(stderr, "sensorhub_start_streaming error!\n");
                } else {
                        sleep(timeout);
                        sensorhub_stop_streaming(sensorhub_handle, handle);
                }

                write(wakeup[1], "terminal", 9);
                sensorhub_close(sensorhub_handle);

                pthread_join(tid, NULL);
        }

error:
        printf("%s: program finished.\n", argv[0]);
        sensorhub_release_sensors_list();
        return ret;
}
