/*
 * Copyright (c) 2015 Intel Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <stddef.h>
#include <stdio.h>
#include "sensors.h"

#define SENSOR_TYPE_FACE_DETECTION       (SENSOR_TYPE_DEVICE_PRIVATE_BASE + 0x1001)
#define SENSOR_TYPE_CONTINUOUS_PROXIMITY (SENSOR_TYPE_DEVICE_PRIVATE_BASE + 0x1002)

struct accel_data {
        int64_t ts;
        short x;
        short y;
        short z;
        unsigned char reserved[2];
} __attribute__ ((packed));

struct gyro_raw_data {
        int64_t ts;
        short x;
        short y;
        short z;
        unsigned char reserved[2];
} __attribute__ ((packed));

struct compass_raw_data {
        int64_t ts;
        short x;
        short y;
        short z;
        unsigned char reserved[2];
} __attribute__ ((packed));

struct baro_raw_data {
        int64_t ts;
        int32_t p;
} __attribute__ ((packed));

struct als_raw_data {
        int64_t ts;
        unsigned short lux;
        unsigned short reserved;
} __attribute__ ((packed));

struct ps_phy_data {
        int64_t ts;
        unsigned short near;
        unsigned char reserved[2];
} __attribute__ ((packed));

struct shaking_data {
        int64_t ts;
        short shaking;
} __attribute__ ((packed));

struct stap_data {
        int64_t ts;
        short stap;
} __attribute__ ((packed));

struct pickup_data {
        int64_t ts;
        short pickup;
} __attribute__ ((packed));

struct tilt_detector_data {
        int64_t ts;
        short tiltd;
} __attribute__ ((packed));

struct rotation_vector_data {
        int64_t ts;
        int x;
        int y;
        int z;
        int w;
} __attribute__ ((packed));

struct game_rotation_vector_data {
        int64_t ts;
        int x;
        int y;
        int z;
        int w;
} __attribute__ ((packed));

struct geomagnetic_rotation_vector_data {
        int64_t ts;
        int x;
        int y;
        int z;
        int w;
} __attribute__ ((packed));

struct gravity_data {
        int64_t ts;
        int x;
        int y;
        int z;
} __attribute__ ((packed));

struct linear_accel_data {
        int64_t ts;
        int x;
        int y;
        int z;
} __attribute__ ((packed));

struct orientation_data {
        int64_t ts;
        int azimuth;
        int pitch;
        int roll;
} __attribute__ ((packed));

struct sm_data {
        int64_t ts;
        short state;
} __attribute__ ((packed));

struct stepcounter_data {
        int64_t ts;
        int num;
} __attribute__ ((packed));

struct stepdetector_data {
        int64_t ts;
        int state;
} __attribute__ ((packed));

typedef struct _SFacePosition
{
        // The normalized (0 to 1) horizontal position in the scene of the upper-left corner of the first face rectangle.
        float left;
        // The normalized (0 to 1) vertical position in the scene of the upper-left corner of the first face rectangle.
        float top;
        // The normalized (0 to 1) horizontal position in the scene of the lower-right corner of the first face rectangle.
        float right;
        // The normalized (0 to 1) vertical position in the scene of the lower-right of the first face rectangle.
        float bottom;
}SFacePosition;

struct fd_data
{
        // The time in nanosecond at which the frame was captured
        uint64_t timestamp;
        // Number of faces which were detected
        uint32_t faces_nr;
        // Array which holds the position of the faces detected
        SFacePosition faces[0];
}__attribute__ ((packed));

size_t dump_psh_data(int type, void *data, size_t size)
{
        size_t rest = 0;
        int i, count = 0;
        /* for face detection */
        size_t remaining_size = size;
        struct fd_data *faced = NULL;

        switch (type) {
        case SENSOR_TYPE_ACCELEROMETER:
                count = size / sizeof(struct accel_data);
                rest = size % sizeof(struct accel_data);
                for (i = 0; i < count; i++)
                        printf("[RAW]Accelerometer: [%hd, %hd, %hd] timestamp: %lld reserved data: [0x%x 0x%x]\n",
                               ((struct accel_data *)data)[i].x, ((struct accel_data *)data)[i].y,
                               ((struct accel_data *)data)[i].z, ((struct accel_data *)data)[i].ts,
                               ((struct accel_data *)data)[i].reserved[0], ((struct accel_data *)data)[i].reserved[1]);
                break;
        case SENSOR_TYPE_MAGNETIC_FIELD:
                count = size / sizeof(struct compass_raw_data);
                rest = size % sizeof(struct compass_raw_data);
                for (i = 0; i < count; i++)
                        printf("[RAW]Magnetic Field: [%hd, %hd, %hd] timestamp: %lld reserved data: [0x%x 0x%x]\n",
                               ((struct compass_raw_data *)data)[i].x, ((struct compass_raw_data *)data)[i].y,
                               ((struct compass_raw_data *)data)[i].z, ((struct compass_raw_data *)data)[i].ts,
                               ((struct compass_raw_data *)data)[i].reserved[0], ((struct compass_raw_data *)data)[i].reserved[1]);
                break;
        case SENSOR_TYPE_ORIENTATION:
                count = size / sizeof(struct orientation_data);
                rest = size % sizeof(struct orientation_data);
                for (i = 0; i < count; i++)
                        printf("[RAW]Orientation: [%d, %d, %d] timestamp: %lld\n",
                               ((struct orientation_data *)data)[i].azimuth, ((struct orientation_data *)data)[i].pitch,
                               ((struct orientation_data *)data)[i].roll, ((struct orientation_data *)data)[i].ts);
                break;
        case SENSOR_TYPE_GYROSCOPE:
                count = size / sizeof(struct gyro_raw_data);
                rest = size % sizeof(struct gyro_raw_data);
                for (i = 0; i < count; i++)
                        printf("[RAW]Gyroscope: [%hd, %hd, %hd] timestamp: %lld reserved data: [0x%x 0x%x]\n",
                               ((struct gyro_raw_data *)data)[i].x, ((struct gyro_raw_data *)data)[i].y,
                               ((struct gyro_raw_data *)data)[i].z, ((struct gyro_raw_data *)data)[i].ts,
                               ((struct gyro_raw_data *)data)[i].reserved[0], ((struct gyro_raw_data *)data)[i].reserved[1]);
                break;
        case SENSOR_TYPE_LIGHT:
                count = size / sizeof(struct als_raw_data);
                rest = size % sizeof(struct als_raw_data);
                for (i = 0; i < count; i++)
                        printf("[RAW]Light: [%hu] timestamp: %lld\n",
                               ((struct als_raw_data *)data)[i].lux, ((struct als_raw_data *)data)[i].ts);
                break;
        case SENSOR_TYPE_PRESSURE:
                count = size / sizeof(struct baro_raw_data);
                rest = size % sizeof(struct baro_raw_data);
                for (i = 0; i < count; i++)
                        printf("[RAW]Pressure: [%d] timestamp: %lld\n",
                               ((struct baro_raw_data *)data)[i].p, ((struct baro_raw_data *)data)[i].ts);
                break;
        case SENSOR_TYPE_TEMPERATURE:
                break;
        case SENSOR_TYPE_PROXIMITY:
        case SENSOR_TYPE_CONTINUOUS_PROXIMITY:
                count = size / sizeof(struct ps_phy_data);
                rest = size % sizeof(struct ps_phy_data);
                for (i = 0; i < count; i++)
                        printf("[RAW]Proximity: [%hu] timestamp: %lld reserved data: [0x%x 0x%x]\n",
                               ((struct ps_phy_data *)data)[i].near, ((struct ps_phy_data *)data)[i].ts,
                               ((struct ps_phy_data *)data)[i].reserved[0], ((struct ps_phy_data *)data)[i].reserved[1]);
                break;
        case SENSOR_TYPE_GRAVITY:
                count = size / sizeof(struct gravity_data);
                rest = size % sizeof(struct gravity_data);
                for (i = 0; i < count; i++)
                        printf("[RAW]Gravity: [%d, %d, %d] timestamp: %lld\n",
                               ((struct gravity_data *)data)[i].x, ((struct gravity_data *)data)[i].y,
                               ((struct gravity_data *)data)[i].z, ((struct gravity_data *)data)[i].ts);
                break;
        case SENSOR_TYPE_LINEAR_ACCELERATION:
                count = size / sizeof(struct linear_accel_data);
                rest = size % sizeof(struct linear_accel_data);
                for (i = 0; i < count; i++)
                        printf("[RAW]Linear Acceleration: [%d, %d, %d] timestamp: %lld\n",
                               ((struct linear_accel_data *)data)[i].x, ((struct linear_accel_data *)data)[i].y,
                               ((struct linear_accel_data *)data)[i].z, ((struct linear_accel_data *)data)[i].ts);
                break;
        case SENSOR_TYPE_ROTATION_VECTOR:
                count = size / sizeof(struct rotation_vector_data);
                rest = size % sizeof(struct rotation_vector_data);
                for (i = 0; i < count; i++)
                        printf("[RAW]Rotation Vector: [%d, %d, %d, %d] timestamp: %lld\n",
                               ((struct rotation_vector_data *)data)[i].x, ((struct rotation_vector_data *)data)[i].y,
                               ((struct rotation_vector_data *)data)[i].z, ((struct rotation_vector_data *)data)[i].w,
                               ((struct rotation_vector_data *)data)[i].ts);
                break;
        case SENSOR_TYPE_RELATIVE_HUMIDITY:
                break;
        case SENSOR_TYPE_AMBIENT_TEMPERATURE:
                break;
        case SENSOR_TYPE_MAGNETIC_FIELD_UNCALIBRATED:
                break;
        case SENSOR_TYPE_GAME_ROTATION_VECTOR:
                count = size / sizeof(struct game_rotation_vector_data);
                rest = size % sizeof(struct game_rotation_vector_data);
                for (i = 0; i < count; i++)
                        printf("[RAW]Game Rotation Vector: [%d, %d, %d, %d] timestamp: %lld\n",
                               ((struct game_rotation_vector_data *)data)[i].x, ((struct game_rotation_vector_data *)data)[i].y,
                               ((struct game_rotation_vector_data *)data)[i].z, ((struct game_rotation_vector_data *)data)[i].w,
                               ((struct game_rotation_vector_data *)data)[i].ts);
                break;
        case SENSOR_TYPE_GYROSCOPE_UNCALIBRATED:
                break;
        case SENSOR_TYPE_SIGNIFICANT_MOTION:
                count = size / sizeof(struct sm_data);
                rest = size % sizeof(struct sm_data);
                for (i = 0; i < count; i++)
                        printf("[RAW]Significant Motion: [%hd] timestamp: %lld\n",
                               ((struct sm_data *)data)[i].state, ((struct sm_data *)data)[i].ts);
                break;
        case SENSOR_TYPE_STEP_DETECTOR:
                count = size / sizeof(struct stepdetector_data);
                rest = size % sizeof(struct stepdetector_data);
                for (i = 0; i < count; i++)
                        printf("[RAW]Step Detector: [%d] timestamp: %lld\n",
                               ((struct stepdetector_data *)data)[i].state, ((struct stepdetector_data *)data)[i].ts);
                break;
        case SENSOR_TYPE_STEP_COUNTER:
                count = size / sizeof(struct stepcounter_data);
                rest = size % sizeof(struct stepcounter_data);
                for (i = 0; i < count; i++)
                        printf("[RAW]Step Countor: [%d] timestamp: %lld\n",
                               ((struct stepcounter_data *)data)[i].num, ((struct stepcounter_data *)data)[i].ts);
                break;
        case SENSOR_TYPE_GEOMAGNETIC_ROTATION_VECTOR:
                count = size / sizeof(struct geomagnetic_rotation_vector_data);
                rest = size % sizeof(struct geomagnetic_rotation_vector_data);
                for (i = 0; i < count; i++)
                        printf("[RAW]Geomagnetic Rotation Vector: [%d, %d, %d, %d] timestamp: %lld\n",
                               ((struct geomagnetic_rotation_vector_data *)data)[i].x, ((struct geomagnetic_rotation_vector_data *)data)[i].y,
                               ((struct geomagnetic_rotation_vector_data *)data)[i].z, ((struct geomagnetic_rotation_vector_data *)data)[i].w,
                               ((struct geomagnetic_rotation_vector_data *)data)[i].ts);
                break;
        case SENSOR_TYPE_HEART_RATE:
                break;
        case SENSOR_TYPE_TILT_DETECTOR:
                count = size / sizeof(struct tilt_detector_data);
                rest = size % sizeof(struct tilt_detector_data);
                for (i = 0; i < count; i++)
                        printf("[RAW]Tilt Detector: [%hd] timestamp: %lld\n",
                               ((struct tilt_detector_data *)data)[i].tiltd, ((struct tilt_detector_data *)data)[i].ts);
                break;
        case SENSOR_TYPE_WAKE_GESTURE:
                count = size / sizeof(struct shaking_data);
                rest = size % sizeof(struct shaking_data);
                for (i = 0; i < count; i++)
                        printf("[RAW]Wake Gesture: [%hd] timestamp: %lld\n",
                               ((struct shaking_data *)data)[i].shaking, ((struct shaking_data *)data)[i].ts);
                break;
        case SENSOR_TYPE_GLANCE_GESTURE:
                count = size / sizeof(struct stap_data);
                rest = size % sizeof(struct stap_data);
                for (i = 0; i < count; i++)
                        printf("[RAW]Glance Gesture: [%hd] timestamp: %lld\n",
                               ((struct stap_data *)data)[i].stap, ((struct stap_data *)data)[i].ts);
                break;
        case SENSOR_TYPE_PICK_UP_GESTURE:
                count = size / sizeof(struct pickup_data);
                rest = size % sizeof(struct pickup_data);
                for (i = 0; i < count; i++)
                        printf("[RAW]Pick Up Gesture: [%hd] timestamp: %lld\n",
                               ((struct pickup_data *)data)[i].pickup, ((struct pickup_data *)data)[i].ts);
                break;
        case SENSOR_TYPE_FACE_DETECTION:
                while (remaining_size >= sizeof(struct fd_data)) {
                        faced = (struct fd_data *)(data + (size - remaining_size));
                        printf("[RAW]Face Detection: timestamp: %lld face number: %u\n", faced->timestamp, faced->faces_nr);
                        remaining_size -= sizeof(struct fd_data);
                        if (remaining_size < faced->faces_nr * sizeof(SFacePosition)) {
                                printf("Error: remaining data size is invalid: %u Not enough for %u faces: %u\n", remaining_size, faced->faces_nr);
                                break;
                        } else {
                                for (i = 0; i < faced->faces_nr; i++) {
                                        printf("    Faces[%d]: left: %f top: %f right: %f bottom: %f\n", i,
                                               faced->faces[i].left, faced->faces[i].top,
                                               faced->faces[i].right, faced->faces[i].bottom);
                                }
                        }
                        remaining_size -= faced->faces_nr * sizeof(SFacePosition);
                }
                break;
        default:
                break;
        }

        return rest;
}
