/*
 * Copyright (C) 2009 Gnome Unity
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU  General Public License
 * version 2.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License version 2.0 for more details.
 *
 * You should have received a copy of the GNU  General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Taken from the Gnome Unity Project (src/unity-tray-manager.[h|c])
 */

#include <clutter/clutter.h>
#include <clutter/x11/clutter-x11.h>
#include <gtk/gtk.h>

#include "unity-tray-manager.h"
#include "na-tray-manager.h"

#include "shell-gtk-embed.h"
#include "shell-embedded-window.h"

struct _UnityTrayManagerPrivate {
  NaTrayManager *na_manager;
  ClutterStage *stage;
  ClutterColor bg_color;

  GHashTable *icons;
};

typedef struct {
  UnityTrayManager *manager;
  GtkWidget *socket;
  GtkWidget *window;
  ClutterActor *actor;
} UnityTrayManagerChild;

enum {
  PROP_0,

  PROP_BG_COLOR
};

/* Signals */
enum
{
  TRAY_ICON_ADDED,
  TRAY_ICON_REMOVED,
  LAST_SIGNAL
};

G_DEFINE_TYPE (UnityTrayManager, unity_tray_manager, G_TYPE_OBJECT);

static guint unity_tray_manager_signals [LAST_SIGNAL] = { 0 };

/* Sea Green - obnoxious to force people to set the background color */
static const ClutterColor default_color = { 0xbb, 0xff, 0xaa };

static void na_tray_icon_added (NaTrayManager *na_manager, GtkWidget *child, gpointer manager);
static void na_tray_icon_removed (NaTrayManager *na_manager, GtkWidget *child, gpointer manager);

static void
free_tray_icon (gpointer data)
{
  UnityTrayManagerChild *child = data;

  gtk_widget_hide (child->window);
  gtk_widget_destroy (child->window);
  g_signal_handlers_disconnect_matched (child->actor, G_SIGNAL_MATCH_DATA,
                                        0, 0, NULL, NULL, child);
  g_object_unref (child->actor);
  g_slice_free (UnityTrayManagerChild, child);
}

static void
unity_tray_manager_set_property(GObject         *object,
                                guint            prop_id,
                                const GValue    *value,
                                GParamSpec      *pspec)
{
  UnityTrayManager *manager = UNITY_TRAY_MANAGER (object);

  switch (prop_id)
    {
    case PROP_BG_COLOR:
      {
        ClutterColor *color = g_value_get_boxed (value);
        if (color)
          manager->priv->bg_color = *color;
        else
          manager->priv->bg_color = default_color;
      }
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
unity_tray_manager_get_property(GObject         *object,
                                guint            prop_id,
                                GValue          *value,
                                GParamSpec      *pspec)
{
  UnityTrayManager *manager = UNITY_TRAY_MANAGER (object);

  switch (prop_id)
    {
    case PROP_BG_COLOR:
      g_value_set_boxed (value, &manager->priv->bg_color);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
unity_tray_manager_init (UnityTrayManager *manager)
{
  manager->priv = G_TYPE_INSTANCE_GET_PRIVATE (manager, UNITY_TYPE_TRAY_MANAGER,
                                               UnityTrayManagerPrivate);
  manager->priv->na_manager = na_tray_manager_new ();

  manager->priv->icons = g_hash_table_new_full (NULL, NULL,
                                                NULL, free_tray_icon);
  manager->priv->bg_color = default_color;

  g_signal_connect (manager->priv->na_manager, "tray-icon-added",
                    G_CALLBACK (na_tray_icon_added), manager);
  g_signal_connect (manager->priv->na_manager, "tray-icon-removed",
                    G_CALLBACK (na_tray_icon_removed), manager);
}

static void
unity_tray_manager_finalize (GObject *object)
{
  UnityTrayManager *manager = UNITY_TRAY_MANAGER (object);

  g_object_unref (manager->priv->na_manager);
  g_object_unref (manager->priv->stage);
  g_hash_table_destroy (manager->priv->icons);

  G_OBJECT_CLASS (unity_tray_manager_parent_class)->finalize (object);
}

static void
unity_tray_manager_class_init (UnityTrayManagerClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (klass, sizeof (UnityTrayManagerPrivate));

  gobject_class->finalize = unity_tray_manager_finalize;
  gobject_class->set_property = unity_tray_manager_set_property;
  gobject_class->get_property = unity_tray_manager_get_property;

  unity_tray_manager_signals[TRAY_ICON_ADDED] =
    g_signal_new ("tray-icon-added",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (UnityTrayManagerClass, tray_icon_added),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__OBJECT,
		  G_TYPE_NONE, 1,
                  CLUTTER_TYPE_ACTOR);
  unity_tray_manager_signals[TRAY_ICON_REMOVED] =
    g_signal_new ("tray-icon-removed",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (UnityTrayManagerClass, tray_icon_removed),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__OBJECT,
		  G_TYPE_NONE, 1,
                  CLUTTER_TYPE_ACTOR);

  /* Lifting the CONSTRUCT_ONLY here isn't hard; you just need to
   * iterate through the icons, reset the background pixmap, and
   * call na_tray_child_force_redraw()
   */
  g_object_class_install_property (gobject_class,
                                   PROP_BG_COLOR,
                                   g_param_spec_boxed ("bg-color",
                                                       "BG Color",
                                                       "Background color (only if we don't have transparency)",
                                                       CLUTTER_TYPE_COLOR,
                                                       G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
}

UnityTrayManager *
unity_tray_manager_new (void)
{
  return g_object_new (UNITY_TYPE_TRAY_MANAGER, NULL);
}

void
unity_tray_manager_manage_stage (UnityTrayManager *manager,
                                 ClutterStage     *stage)
{
  Window stage_xwindow;
  GdkWindow *stage_window;
  GdkScreen *screen;

  g_return_if_fail (manager->priv->stage == NULL);

  manager->priv->stage = g_object_ref (stage);

  stage_xwindow = clutter_x11_get_stage_window (stage);

  /* This is a pretty ugly way to get the GdkScreen for the stage; it
   *  will normally go through the foreign_new() case with a
   *  round-trip to the X server, it might be nicer to pass the screen
   *  in in some way. (The Clutter/Mutter combo is currently incapable
   *  of multi-screen operation, so alternatively we could just assume
   *  that clutter_x11_get_default_screen() gives us the right
   *  screen.)
   */
  stage_window = gdk_window_lookup (stage_xwindow);
  if (stage_window)
    g_object_ref (stage_window);
  else
    stage_window = gdk_window_foreign_new (stage_xwindow);

  screen = gdk_drawable_get_screen (stage_window);

  g_object_unref (stage_window);

  na_tray_manager_manage_screen (manager->priv->na_manager, screen);
}

static GdkPixmap *
create_bg_pixmap (GdkColormap  *colormap,
                  ClutterColor *color)
{
  GdkScreen *screen = gdk_colormap_get_screen (colormap);
  GdkVisual *visual = gdk_colormap_get_visual (colormap);
  GdkPixmap *pixmap = gdk_pixmap_new (gdk_screen_get_root_window (screen),
                                      1, 1,
                                      visual->depth);
  cairo_t *cr;

  gdk_drawable_set_colormap (pixmap, colormap);

  cr = gdk_cairo_create (pixmap);
  cairo_set_source_rgb (cr,
                        color->red / 255.,
                        color->green / 255.,
                        color->blue / 255.);
  cairo_paint (cr);
  cairo_destroy (cr);

  return pixmap;
}

static void
unity_tray_manager_child_on_realize (GtkWidget             *widget,
                                     UnityTrayManagerChild *child)
{
  GdkPixmap *bg_pixmap;

  /* If the tray child is using an RGBA colormap (and so we have real
   * transparency), we don't need to worry about the background. If
   * not, we obey the bg-color property by creating a 1x1 pixmap of
   * that color and setting it as our background. Then "parent-relative"
   * background on the socket and the plug within that will cause
   * the icons contents to appear on top of our background color.
   */
  if (!na_tray_child_has_alpha (NA_TRAY_CHILD (child->socket)))
    {
      bg_pixmap = create_bg_pixmap (gtk_widget_get_colormap (widget),
                                    &child->manager->priv->bg_color);
      gdk_window_set_back_pixmap (widget->window, bg_pixmap, FALSE);
      g_object_unref (bg_pixmap);
    }
}

static void
na_tray_icon_added (NaTrayManager *na_manager, GtkWidget *socket,
                    gpointer user_data)
{
  UnityTrayManager *manager = user_data;
  GtkWidget *win;
  ClutterActor *icon;
  UnityTrayManagerChild *child;

  /* We don't need the NaTrayIcon to be composited on the window we
   * put it in: the window is the same size as the tray icon
   * and transparent. We can just use the default X handling of
   * subwindows as mode of SOURCE (replace the parent with the
   * child) and then composite the parent onto the stage.
   */
  na_tray_child_set_composited (NA_TRAY_CHILD (socket), FALSE);

  win = shell_embedded_window_new ();
  gtk_container_add (GTK_CONTAINER (win), socket);

  /* The colormap of the socket matches that of its contents; make
   * the window we put it in match that as well */
  gtk_widget_set_colormap (win, gtk_widget_get_colormap (socket));

  child = g_slice_new (UnityTrayManagerChild);
  child->manager = manager;
  child->window = win;
  child->socket = socket;

  g_signal_connect (win, "realize",
                    G_CALLBACK (unity_tray_manager_child_on_realize), child);

  gtk_widget_show_all (win);

  icon = shell_gtk_embed_new (SHELL_EMBEDDED_WINDOW (win));

  child->actor = g_object_ref (icon);
  g_hash_table_insert (manager->priv->icons, socket, child);

  g_signal_emit (manager,
                 unity_tray_manager_signals[TRAY_ICON_ADDED], 0,
                 icon);
}

static void
na_tray_icon_removed (NaTrayManager *na_manager, GtkWidget *socket,
                      gpointer user_data)
{
  UnityTrayManager *manager = user_data;
  UnityTrayManagerChild *child;

  child = g_hash_table_lookup (manager->priv->icons, socket);
  g_return_if_fail (child != NULL);

  g_signal_emit (manager,
                 unity_tray_manager_signals[TRAY_ICON_REMOVED], 0,
                 child->actor);
  g_hash_table_remove (manager->priv->icons, socket);
}
