/*
 * Copyright (C) 2010 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by
 *             Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 */

#include "zeitgeist-eggdbusconversions.h"
#include <stdlib.h> /* For strtol() */

ZeitgeistEvent*
_egg_zeitgeist_event_to_zeitgeist_event (EggZeitgeistEvent *event)
{
  ZeitgeistEvent   *result;
  ZeitgeistSubject *subject;
  EggDBusArraySeq  *subjects, *payload_data;
  GByteArray       *payload;
  gchar            **event_data, **subject_data, *val;
  guint32           id;
  gint64            timestamp;
  gint              num_subjects, payload_size, i;
  guint8            *payload_byte;

  result = zeitgeist_event_new ();
  event_data = egg_zeitgeist_event_get_event_data (event);
  subjects = egg_zeitgeist_event_get_subjects (event);
  payload_data = egg_zeitgeist_event_get_payload (event);

  /* Set event metadata */
  g_return_val_if_fail (g_strv_length (event_data) > EGG_ZEITGEIST_EVENT_DATA_OFFSET_ACTOR, NULL);
  
  val = event_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ID];
  id = g_ascii_strtoull (val, NULL, 0);
  zeitgeist_event_set_id (result, id);
  
  val = event_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_TIMESTAMP];
  timestamp = g_ascii_strtoll (val, NULL, 0);
  zeitgeist_event_set_timestamp (result, timestamp);

  val = event_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_INTERPRETATION];
  zeitgeist_event_set_interpretation (result, val);

  val = event_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_MANIFESTATION];
  zeitgeist_event_set_manifestation (result, val);

  val = event_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ACTOR];
  zeitgeist_event_set_actor (result, val);

  /* Build the list of subjects */
  num_subjects = egg_dbus_array_seq_get_size (subjects);
  for (i = 0; i < num_subjects; i++)
    {
      subject_data = egg_dbus_array_seq_get (subjects, i);
      g_return_val_if_fail (g_strv_length (subject_data) > EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_STORAGE, NULL);
      
      subject = g_object_new (ZEITGEIST_TYPE_SUBJECT, NULL);

      val = subject_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_URI];
      zeitgeist_subject_set_uri (subject, val);

      val = subject_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_INTERPRETATION];
      zeitgeist_subject_set_interpretation (subject, val[0] == '\0' ? NULL : val);

      val = subject_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_MANIFESTATION];
      zeitgeist_subject_set_manifestation (subject, val[0] == '\0' ? NULL : val);

      val = subject_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_MIMETYPE];
      zeitgeist_subject_set_mimetype (subject, val[0] == '\0' ? NULL : val);

      val = subject_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_ORIGIN];
      zeitgeist_subject_set_origin (subject, val[0] == '\0' ? NULL : val);

      val = subject_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_TEXT];
      zeitgeist_subject_set_text (subject, val[0] == '\0' ? NULL : val);

      val = subject_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_STORAGE];
      zeitgeist_subject_set_storage (subject, val[0] == '\0' ? NULL : val);

      zeitgeist_event_add_subject (result, subject);
    }

  /* Construct the event payload if necessary */
  payload_size = egg_dbus_array_seq_get_size (payload_data);
  if (payload_size > 0)
    {
      payload = g_byte_array_sized_new (payload_size);
      for (i = 0; i < payload_size; i++)
        {
          // FIXME: Implicit guchar <-> uint8 conversion..?
          payload_byte = egg_dbus_array_seq_get (payload_data, i);
          g_byte_array_append (payload, payload_byte, 1);
        }
      zeitgeist_event_set_payload (result, payload);
    }
  
  return result;
}

EggZeitgeistEvent*
_zeitgeist_event_to_egg_zeitgeist_event (ZeitgeistEvent *event)
{
  EggZeitgeistEvent *result;
  ZeitgeistSubject  *subject;
  EggDBusArraySeq   *subjects_data, *payload_data;
  GByteArray        *payload;
  gchar            **event_data, **subject_data;
  const gchar       *val;
  gint               num_subjects, i;

  /* Event metadata */
  event_data = g_new0 (gchar*, EGG_ZEITGEIST_EVENT_DATA_OFFSET_ACTOR + 2);
  
  event_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ID] =
      zeitgeist_event_get_id (event) == 0 ? g_strdup("") :
          g_strdup_printf ("%"G_GUINT32_FORMAT, zeitgeist_event_get_id (event));

  event_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_TIMESTAMP] =
   zeitgeist_event_get_timestamp (event) == 0 ? g_strdup ("") :
    g_strdup_printf ("%"G_GINT64_FORMAT, zeitgeist_event_get_timestamp (event));

  val = zeitgeist_event_get_interpretation (event);
  event_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_INTERPRETATION] =
          g_strdup (val ? val : "");

  val = zeitgeist_event_get_manifestation (event);
  event_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_MANIFESTATION] =
          g_strdup (val ? val : "");

  val = zeitgeist_event_get_actor (event);
  event_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ACTOR] =
          g_strdup (val ? val : "");

  /* Subjects */
  subjects_data = egg_dbus_array_seq_new (G_TYPE_STRV,
                                          (GDestroyNotify) g_strfreev,
                                          (GBoxedCopyFunc) g_strdupv,
                                          NULL);
  num_subjects = zeitgeist_event_num_subjects (event);
  for (i = 0; i < num_subjects; i++)
    {
      subject = zeitgeist_event_get_subject (event, i);
      subject_data = g_new0 (gchar*,
                             EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_STORAGE + 2);

      val = zeitgeist_subject_get_uri (subject);
      subject_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_URI] =
        g_strdup (val ? val : "");

      val = zeitgeist_subject_get_interpretation (subject);
      subject_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_INTERPRETATION] =
        g_strdup (val ? val : "");

      val = zeitgeist_subject_get_manifestation (subject);
      subject_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_MANIFESTATION] =
        g_strdup (val ? val : "");

      val = zeitgeist_subject_get_mimetype (subject);
      subject_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_MIMETYPE] =
        g_strdup (val ? val : "");

      val = zeitgeist_subject_get_origin (subject);
      subject_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_ORIGIN] =
        g_strdup (val ? val : "");

      val = zeitgeist_subject_get_text (subject);
      subject_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_TEXT] =
        g_strdup (val ? val : "");

      val = zeitgeist_subject_get_storage (subject);
      subject_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_STORAGE] =
        g_strdup (val ? val : "");

      /* The array takes ownership of subject_data */
      egg_dbus_array_seq_add (subjects_data, subject_data);
    }

  /* Payload */
  payload = zeitgeist_event_get_payload (event);
  payload_data = egg_dbus_array_seq_new (G_TYPE_UCHAR, NULL, NULL, NULL);
  if (payload)
    {      
      for (i = 0; i < payload->len; i++)
        {
          /* FIXME: Implicit guchar <-> guint8 conversion..? */
          egg_dbus_array_seq_add_fixed (payload_data, payload->data[i]);
        }
    }

  result = egg_zeitgeist_event_new (event_data, subjects_data, payload_data);

  g_strfreev (event_data);
  g_object_unref (subjects_data);
  g_object_unref (payload_data);

  return result;
}

GPtrArray*
_egg_zeitgeist_events_to_zeitgeist_events (EggDBusArraySeq *events)
{
  GPtrArray         *result;
  EggZeitgeistEvent *event;
  gint               i, num_events;

  g_return_val_if_fail (EGG_DBUS_IS_ARRAY_SEQ (events), NULL);
  //g_return_val_if_fail (egg_dbus_array_seq_get_element_type (events) == EGG_ZEITGEIST_TYPE_EVENT,
  //                      NULL);

  num_events = egg_dbus_array_seq_get_size (events);
  result = g_ptr_array_sized_new (num_events);  
  for (i = 0; i < num_events; i++)
    {
      event = EGG_ZEITGEIST_EVENT (egg_dbus_array_seq_get (events, i));
      g_ptr_array_add (result, _egg_zeitgeist_event_to_zeitgeist_event(event));
    }

  return result;
}

EggDBusArraySeq*
_zeitgeist_events_to_egg_zeitgeist_events (GPtrArray *events)
{
  EggDBusArraySeq *result;
  ZeitgeistEvent  *event;
  gint             i;

  g_return_val_if_fail (events != NULL, NULL);

  result = egg_dbus_array_seq_new (EGG_ZEITGEIST_TYPE_EVENT,
                                   (GDestroyNotify) g_object_unref,
                                   (GBoxedCopyFunc) g_object_ref,
                                   NULL);

  for (i = 0; i < events->len; i++)
    {
      event = ZEITGEIST_EVENT (g_ptr_array_index (events, i));
      egg_dbus_array_seq_add (result,
                              _zeitgeist_event_to_egg_zeitgeist_event (event));
    }

  return result;
}

GPtrArray*
_zeitgeist_events_from_valist (va_list    events)
{
  ZeitgeistEvent *event = NULL;
  GPtrArray *result;

  result = g_ptr_array_new_with_free_func ((GDestroyNotify) g_object_unref);

  event = va_arg (events, ZeitgeistEvent*);
  while (event != NULL)
    {
      g_return_val_if_fail (ZEITGEIST_IS_EVENT (event), NULL);
      g_ptr_array_add (result, event);
      event = va_arg (events, ZeitgeistEvent*);
    }
  
  return result;
}

EggZeitgeistTimeRange*
_zeitgeist_time_range_to_egg_zeitgeist_time_range (ZeitgeistTimeRange *time_range)
{
  EggZeitgeistTimeRange *result;

  g_return_val_if_fail (ZEITGEIST_IS_TIME_RANGE (time_range), NULL);

  result = egg_zeitgeist_time_range_new (zeitgeist_time_range_get_start (time_range),
                                         zeitgeist_time_range_get_end (time_range));

  return result;
}

ZeitgeistTimeRange*
_egg_zeitgeist_time_range_to_zeitgeist_time_range (EggZeitgeistTimeRange *time_range)
{
  ZeitgeistTimeRange *result;

  g_return_val_if_fail (EGG_ZEITGEIST_IS_TIME_RANGE (time_range), NULL);

  result = zeitgeist_time_range_new (egg_zeitgeist_time_range_get_start (time_range),
                                     egg_zeitgeist_time_range_get_end (time_range));

  return result;
}

ZeitgeistDataSource*
_egg_zeitgeist_data_source_to_zeitgeist_data_source (EggZeitgeistDataSource *data_source)
{
  ZeitgeistDataSource *result;
  GPtrArray           *templates;
  gboolean             running;
  gboolean             enabled;
  gint64               timestamp;

  g_return_val_if_fail (EGG_ZEITGEIST_IS_DATA_SOURCE (data_source), NULL);

  templates = _egg_zeitgeist_events_to_zeitgeist_events (egg_zeitgeist_data_source_get_event_templates (data_source));

  result = zeitgeist_data_source_new_full (
      egg_zeitgeist_data_source_get_unique_id (data_source),
      egg_zeitgeist_data_source_get_name (data_source),
      egg_zeitgeist_data_source_get_description (data_source),
      templates);

  running = egg_zeitgeist_data_source_get_running (data_source);
  enabled = egg_zeitgeist_data_source_get_enabled (data_source);
  timestamp = egg_zeitgeist_data_source_get_last_seen (data_source);

  zeitgeist_data_source_set_running (result, running);
  zeitgeist_data_source_set_timestamp (result, timestamp);
  zeitgeist_data_source_set_enabled (result, enabled);

  return result;
}

GPtrArray*
_egg_zeitgeist_data_sources_to_zeitgeist_data_sources (EggDBusArraySeq *data_sources)
{
  GPtrArray              *result;
  EggZeitgeistDataSource *source;
  gint                    i, num_sources;

  g_return_val_if_fail (EGG_DBUS_IS_ARRAY_SEQ (data_sources), NULL);

  num_sources = egg_dbus_array_seq_get_size (data_sources);
  result = g_ptr_array_sized_new (num_sources);
  for (i = 0; i < num_sources; i++)
    {
      source = EGG_ZEITGEIST_DATA_SOURCE (egg_dbus_array_seq_get (data_sources, i));
      g_ptr_array_add (result, _egg_zeitgeist_data_source_to_zeitgeist_data_source (source));
    }

  return result;
}

