/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <mhs/mhs.h>
#include <nsCOMArray.h>
#include <nsWeakPtr.h>
#include <nsIWeakReferenceUtils.h>
#include <nsICryptoHash.h>
#include <nsDirectoryServiceDefs.h>
#include <nsDirectoryServiceUtils.h>
#include <nsNetUtil.h>
#include <dbus/dbus-glib.h>
#include <nbtk/nbtk.h>
#include <libintl.h>
#include "mwb-pages-service.h"

#define MWB_FALLBACK_THUMBNAIL "chrome://mwbpages/content/fallback-page.png"

NS_IMPL_ISUPPORTS1(MwbFavorite, MwbIFavorite)

MwbFavorite::MwbFavorite (const gchar *a_url, const gchar *a_title)
: url (g_strdup (a_url)),
  title (g_strdup (a_title))
{
}

MwbFavorite::~MwbFavorite ()
{
  g_free (url);
  g_free (title);
}

NS_IMETHODIMP
MwbFavorite::GetUrl (nsACString &a_url)
{
  a_url.Assign (url);

  return NS_OK;
}

NS_IMETHODIMP
MwbFavorite::GetTitle (nsACString &a_title)
{
  a_title.Assign (title);

  return NS_OK;
}

NS_INTERFACE_MAP_BEGIN(MwbPagesService)
  NS_INTERFACE_MAP_ENTRY_AMBIGUOUS(nsISupports, MwbIPagesService)
  NS_INTERFACE_MAP_ENTRY(MwbIPagesService)
NS_INTERFACE_MAP_END

MwbPagesService *MwbPagesService::pages_service = nsnull;

MwbPagesService *
MwbPagesService::GetSingleton(void)
{
  if (!pages_service)
    pages_service = new MwbPagesService ();

  return pages_service;
}

MwbPagesService::MwbPagesService ()
{
  history = mhs_history_new ();

  g_signal_connect (history, "favorites-received",
                    G_CALLBACK (StaticFavoritesReceivedCb),
                    this);
  g_signal_connect (history, "link-visited",
                    G_CALLBACK (StaticLinkVisitedCb),
                    this);
  g_signal_connect (history, "pinned-page",
                    G_CALLBACK (StaticPinnedPageCb),
                    this);
  g_signal_connect (history, "unpinned-page",
                    G_CALLBACK (StaticUnpinnedPageCb),
                    this);

  cookies = mhs_cookies_new ();
  prefs = mhs_prefs_new ();
  lms = mhs_login_manager_storage_new ();
  pm = mhs_permission_manager_new ();

  bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");

  /* Start fetching the initial pinned pages list */
  mhs_history_get_pinned_pages (history);
}

MwbPagesService::~MwbPagesService ()
{
  g_signal_handlers_disconnect_by_func (history,
                                        (gpointer) StaticFavoritesReceivedCb,
                                        this);
  g_object_unref (history);
  g_object_unref (cookies);
  g_object_unref (prefs);
  g_object_unref (lms);
  g_object_unref (pm);
}

NS_IMETHODIMP_(nsrefcnt)
MwbPagesService::AddRef ()
{
  return 1;
}

NS_IMETHODIMP_(nsrefcnt)
MwbPagesService::Release ()
{
  return 1;
}

nsresult
MwbPagesService::GetFavorites ()
{
  mhs_history_get_favorites (history);

  return NS_OK;
}

/* readonly attribute long numPinnedPages; */
nsresult
MwbPagesService::GetNumPinnedPages (PRInt32 *aNumPinnedPages)
{
  *aNumPinnedPages = pinned_pages_by_title.Count ();
  return NS_OK;
}

/* MwbIPinnedPage getPinnedPageByTitle (in long index); */
nsresult
MwbPagesService::GetPinnedPageByTitle (PRInt32 index,
                                       MwbIPinnedPage **_retval NS_OUTPARAM)
{
  if (index < 0 || index >= pinned_pages_by_title.Count ())
    return NS_ERROR_UNEXPECTED;

  *_retval = pinned_pages_by_title[index];
  NS_ADDREF (*_retval);

  return NS_OK;
}

/* MwbIPinnedPage getPinnedPageByTime (in long index); */
nsresult
MwbPagesService::GetPinnedPageByTime (PRInt32 index,
                                      MwbIPinnedPage **_retval NS_OUTPARAM)
{
  if (index < 0 || index >= pinned_pages_by_time.Count ())
    return NS_ERROR_UNEXPECTED;

  *_retval = pinned_pages_by_time[index];
  NS_ADDREF (*_retval);

  return NS_OK;
}

/* void unpinPage (in AUTF8String url); */
nsresult
MwbPagesService::UnpinPage (const nsACString &url)
{
  mhs_history_unpin_page (history, PromiseFlatCString (url).get ());

  return NS_OK;
}

/* void removeFavorite (in AUTF8String url); */
nsresult
MwbPagesService::RemoveFavorite (const nsACString &url)
{
  GError *error = NULL;

  if (!mhs_history_remove_favorite (history,
                                    PromiseFlatCString (url).get (),
                                    &error))
    {
      nsresult rv = mhs_error_to_nsresult (error);
      g_error_free (error);
      return rv;
    }

  return NS_OK;
}

nsresult
MwbPagesService::StartPrivateBrowsing ()
{
  DBusGConnection *connection;
  DBusGProxy *proxy;
  GError *error = NULL;
  nsresult rv = NS_OK;

  connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
  if (connection == NULL)
    {
      g_warning ("Failed to connect to session bus: %s", error->message);
      g_error_free (error);
      rv = NS_ERROR_UNEXPECTED;
    }
  else
    {
      proxy = dbus_g_proxy_new_for_name (connection,
                                         "org.moblin.MoblinWebBrowser",
                                         "/org/moblin/MoblinWebBrowser",
                                         "org.moblin.MoblinWebBrowser");

      if (!dbus_g_proxy_call (proxy, "StartPrivateBrowsing", &error,
                              G_TYPE_INVALID, G_TYPE_INVALID))
        {
          g_warning ("Failed to start private browsing: %s", error->message);
          g_error_free (error);
          rv = NS_ERROR_UNEXPECTED;
        }

      g_object_unref (proxy);
    }

  return rv;
}

nsresult
MwbPagesService::AddPagesObserver (MwbIPagesObserver *observer)
{
  nsresult rv;

  nsCOMPtr<nsIWeakReference> weak_ptr
    = getter_AddRefs (NS_GetWeakReference (observer, &rv));
  if (NS_FAILED (rv))
    return rv;
  favorites_observers.AppendObject (weak_ptr);

  return NS_OK;
}

nsresult
MwbPagesService::RemovePagesObserver (MwbIPagesObserver *observer)
{
  PRInt32 i;

  for (i = 0; i < favorites_observers.Count (); i++)
    {
      nsWeakPtr weak_ptr = favorites_observers[i];
      nsCOMPtr<MwbIPagesObserver> strong_ptr = do_QueryReferent (weak_ptr);
      if (strong_ptr == observer)
        {
          favorites_observers.RemoveObjectAt(i);
          break;
        }
    }

  return NS_OK;
}

nsresult
MwbPagesService::FormatTime (PRInt32 timestamp,
                             nsACString &retval NS_OUTPARAM)
{
  GTimeVal time_val;
  gchar *time_str;

  time_val.tv_sec = timestamp;
  time_val.tv_usec = 0;

  time_str = nbtk_utils_format_time (&time_val);
  retval = nsDependentCString (time_str);
  g_free (time_str);

  return NS_OK;
}

void
MwbPagesService::FavoritesReceivedCb (gchar **urls, gchar **titles)
{
  int n_favorites = 0;
  gchar **url_p, **title_p;
  MwbIFavorite **favorites;
  int i = 0;

  /* Count the number of favorites */
  for (url_p = urls, title_p = titles; *url_p && *title_p; url_p++, title_p++)
    n_favorites++;

  favorites = (MwbIFavorite **) g_malloc (n_favorites
                                          * sizeof (MwbIFavorite *));

  for (url_p = urls, title_p = titles; *url_p && *title_p; url_p++, title_p++)
    {
      favorites[i] = new MwbFavorite (*url_p, *title_p);
      favorites[i++]->AddRef ();
    }

  for (i = 0; i < favorites_observers.Count ();)
    {
      nsWeakPtr weak_ptr = favorites_observers[i];
      nsCOMPtr<MwbIPagesObserver> strong_ptr = do_QueryReferent (weak_ptr);
      if (strong_ptr)
        {
          strong_ptr->FavoritesReceived (favorites, n_favorites);
          i++;
        }
      else
        /* Observer has gone away, remove it */
        favorites_observers.RemoveObjectAt (i);
    }

  for (i = 0; i < n_favorites; i++)
    favorites[i]->Release ();

  g_free (favorites);
}

void
MwbPagesService::InsertPinnedPageByVisitTime (MwbPinnedPage *pinned_page)
{
  int i;

  /* Insert the page at the right place so that the array will remain
     sorted by visit time */
  for (i = 0;
       i < pinned_pages_by_time.Count () &&
         (((MwbPinnedPage *) pinned_pages_by_time[i])->visit_time >
          pinned_page->visit_time);
       i++);
  pinned_pages_by_time.InsertObjectAt (pinned_page, i);
}

void
MwbPagesService::LinkVisitedCb (const gchar *uri,
                                gint visit_time)
{
  int i;
  PRBool is_pinned = PR_FALSE;

  /* Look for a pinned page with the same uri */
  for (i = 0; i < pinned_pages_by_time.Count (); i++)
    if (((MwbPinnedPage *) pinned_pages_by_time[i])->uri.Equals (uri))
      {
        nsCOMPtr<MwbIPinnedPage> pinned_page = pinned_pages_by_time[i];
        /* Update the visit time */
        ((MwbPinnedPage *) pinned_page.get ())->visit_time = visit_time;
        /* Remove the page */
        pinned_pages_by_time.RemoveObjectAt(i);
        /* Reinsert it at the right place */
        InsertPinnedPageByVisitTime((MwbPinnedPage *) pinned_page.get ());

        is_pinned = PR_TRUE;
        break;
      }

  /* Notify observers */
  for (i = 0; i < favorites_observers.Count ();)
    {
      nsWeakPtr weak_ptr = favorites_observers[i];
      nsCOMPtr<MwbIPagesObserver> strong_ptr = do_QueryReferent (weak_ptr);
      if (strong_ptr)
        {
          strong_ptr->LinkVisited (nsDependentCString (uri),
                                   visit_time,
                                   is_pinned);
          i++;
        }
      else
        /* Observer has gone away, remove it */
        favorites_observers.RemoveObjectAt (i);
    }
}

void
MwbPagesService::PinnedPageCb (const gchar *title, const gchar *uri,
                               gint visit_time, gboolean more_pending)
{
  int i;
  nsCOMPtr<MwbIPinnedPage> pinned_page;
  gchar *collation_key = g_utf8_collate_key (title, -1);

  /* Look for an existing entry with the same uri */
  for (i = 0; i < pinned_pages_by_title.Count (); i++)
    if (((MwbPinnedPage *) pinned_pages_by_title[i])->uri.Equals (uri))
      {
        pinned_page = pinned_pages_by_title[i];
        /* Update the fields */
        ((MwbPinnedPage *) pinned_page.get ())->title = title;
        ((MwbPinnedPage *) pinned_page.get ())->visit_time = visit_time;
        /* Remove the page so it can be reinserted */
        pinned_pages_by_title.RemoveObjectAt(i);
        pinned_pages_by_time.RemoveObject(pinned_page);
        break;
      }
  /* If we didn't find one then add a new entry */
  if (!pinned_page)
    pinned_page = new MwbPinnedPage (title, uri, visit_time);

  /* Insert at the right place to remain sorted by title */
  for (i = 0;
       i < pinned_pages_by_title.Count () &&
         strcmp (((MwbPinnedPage *) pinned_pages_by_title[i])->
                 collation_key.get (), collation_key) < 0;
       i++);
  pinned_pages_by_title.InsertObjectAt (pinned_page, i);

  /* Insert at the right place to remain sorted by visit time */
  InsertPinnedPageByVisitTime ((MwbPinnedPage *) pinned_page.get ());

  g_free (collation_key);

  /* Unless there are more pages to come then notify listeners that
     the pinned pages list has changed */
  if (!more_pending)
    for (i = 0; i < favorites_observers.Count ();)
      {
        nsWeakPtr weak_ptr = favorites_observers[i];
        nsCOMPtr<MwbIPagesObserver> strong_ptr = do_QueryReferent (weak_ptr);
        if (strong_ptr)
          {
            strong_ptr->PinnedPagesChanged ();
            i++;
          }
        else
          /* Observer has gone away, remove it */
          favorites_observers.RemoveObjectAt (i);
      }
}

void
MwbPagesService::UnpinnedPageCb (const gchar *uri)
{
  int i;

  /* Look for an existing entry with the same uri */
  for (i = 0; i < pinned_pages_by_title.Count (); i++)
    if (((MwbPinnedPage *) pinned_pages_by_title[i])->uri.Equals (uri))
      {
        /* Remove the page */
        pinned_pages_by_time.RemoveObject(pinned_pages_by_title[i]);
        pinned_pages_by_title.RemoveObjectAt(i);
        break;
      }

  for (i = 0; i < favorites_observers.Count ();)
    {
      nsWeakPtr weak_ptr = favorites_observers[i];
      nsCOMPtr<MwbIPagesObserver> strong_ptr = do_QueryReferent (weak_ptr);
      if (strong_ptr)
        {
          strong_ptr->PinnedPagesChanged ();
          i++;
        }
      else
        /* Observer has gone away, remove it */
        favorites_observers.RemoveObjectAt (i);
    }
}

void
MwbPagesService::StaticFavoritesReceivedCb (MhsHistory *history,
                                            gchar **urls,
                                            gchar **titles,
                                            MwbPagesService *self)
{
  self->FavoritesReceivedCb (urls, titles);
}

void
MwbPagesService::StaticLinkVisitedCb (MhsHistory *history,
                                      const gchar *uri,
                                      gint visit_time,
                                      MwbPagesService *self)
{
  self->LinkVisitedCb (uri, visit_time);
}

void
MwbPagesService::StaticPinnedPageCb (MhsHistory *history,
                                     const gchar *title,
                                     const gchar *uri,
                                     gint visit_time,
                                     gboolean more_pending,
                                     MwbPagesService *self)
{
  self->PinnedPageCb (title, uri, visit_time, more_pending);
}

void
MwbPagesService::StaticUnpinnedPageCb (MhsHistory *history,
                                       const gchar *uri,
                                       MwbPagesService *self)
{
  self->UnpinnedPageCb (uri);
}

NS_IMETHODIMP
MwbPagesService::ClearHistory ()
{
  GError *error = NULL;
  DBusGConnection *connection;
  DBusGProxy *proxy;

  if (!mhs_history_clear_history (history, &error))
    {
      g_warning ("Failed to clear history: %s", error->message);
      g_clear_error (&error);
    }

  mhs_history_unpin_all_pages (history);

  if (!mhs_cookies_remove_all (cookies, &error))
    {
      g_warning ("Failed to clear cookies: %s", error->message);
      g_clear_error (&error);
    }

  if (!mhs_prefs_reset_user (prefs, &error))
    {
      g_warning ("Failed to clear prefs: %s", error->message);
      g_clear_error (&error);
    }

  if (!mhs_lms_remove_all_logins (lms, &error))
    {
      g_warning ("Failed to clear prefs: %s", error->message);
      g_clear_error (&error);
    }

  if (!mhs_pm_remove_all (pm, &error))
    {
      g_warning ("Failed to clear permissions: %s", error->message);
      g_clear_error (&error);
    }

  /* Ask the browser to clear the session history in all of the tabs */
  connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
  if (connection == NULL)
    {
      g_warning ("Failed to connect to session bus: %s", error->message);
      g_clear_error (&error);
    }
  else
    {
      proxy = dbus_g_proxy_new_for_name (connection,
                                         "org.moblin.MoblinWebBrowser",
                                         "/org/moblin/MoblinWebBrowser",
                                         "org.moblin.MoblinWebBrowser");

      if (!dbus_g_proxy_call (proxy, "PurgeSessionHistory", &error,
                              G_TYPE_INVALID, G_TYPE_INVALID))
        {
          g_warning ("Failed to purge session history: %s", error->message);
          g_clear_error (&error);
        }

      g_object_unref (proxy);
    }

  /* Request the pinned pages and favorites so that any tabs on the
     start page will be redrawn */
  mhs_history_get_favorites (history);
  mhs_history_get_pinned_pages (history);

  return NS_OK;
}

/* AUTF8String gettext (in AUTF8String str); */
NS_IMETHODIMP
MwbPagesService::Gettext (const nsACString &str, nsACString &retval NS_OUTPARAM)
{
  const PromiseFlatCString flat_str (str);
  retval.Assign (nsDependentCString (dgettext (GETTEXT_PACKAGE,
                                               flat_str.get ())));

  return NS_OK;
}

/* AUTF8String ngettext (in AUTF8String singular,
                         in AUTF8String plural,
                         in unsigned long count); */
NS_IMETHODIMP
MwbPagesService::Ngettext (const nsACString &singular,
                           const nsACString &plural,
                           PRUint32 count,
                           nsACString & retval NS_OUTPARAM)
{
  const PromiseFlatCString flat_singular (singular);
  const PromiseFlatCString flat_plural (plural);
  retval.Assign (nsDependentCString (dngettext (GETTEXT_PACKAGE,
                                                flat_singular.get (),
                                                flat_plural.get (),
                                                count)));

  return NS_OK;
}

/* nsIURI getThumbnailURL (in ACString uri); */
NS_IMETHODIMP
MwbPagesService::GetThumbnailURL (const nsACString &uri,
                                  nsIURI **_retval NS_OUTPARAM)
{
  nsresult rv;
  PRUint32 i, j;

  // Get an nsIFile for ~/.thumbnails/large
  nsCOMPtr<nsIFile> thumbnailFile;
  rv = NS_GetSpecialDirectory (NS_OS_HOME_DIR, getter_AddRefs (thumbnailFile));
  NS_ENSURE_SUCCESS (rv, rv);
  rv = thumbnailFile->Append (NS_LITERAL_STRING (".thumbnails"));
  NS_ENSURE_SUCCESS (rv, rv);
  rv = thumbnailFile->Append (NS_LITERAL_STRING ("large"));
  NS_ENSURE_SUCCESS (rv, rv);

  nsCOMPtr<nsICryptoHash> crypto_hash =
    do_GetService ("@mozilla.org/security/hash;1", &rv);
  NS_ENSURE_SUCCESS (rv, rv);

  // Get an MD5 sum of the URI
  rv = crypto_hash->Init (nsICryptoHash::MD5);
  NS_ENSURE_SUCCESS (rv, rv);
  rv = crypto_hash->Update (reinterpret_cast<const PRUint8 *>
                            (uri.BeginReading ()), uri.Length ());
  NS_ENSURE_SUCCESS (rv, rv);
  nsCAutoString md5_sum;
  rv = crypto_hash->Finish (PR_FALSE, md5_sum);
  NS_ENSURE_SUCCESS (rv, rv);

  // Convert the MD5 sum to hex ASCII
  nsAutoString filename;
  for (i = 0; i < md5_sum.Length (); i++)
    {
      unsigned char ch = md5_sum[i];
      for (j = 0; j < 2; j++)
        {
          filename += ((ch >> 4) >= 10 ?
                       (ch >> 4) - 10 + 'a' :
                       (ch >> 4) + '0');

          ch <<= 4;
        }
    }

  filename += NS_LITERAL_STRING (".png");

  rv = thumbnailFile->Append (filename);
  NS_ENSURE_SUCCESS (rv, rv);

  // Check if the file exists. If not we will use a fallback instead
  PRBool exists;
  rv = thumbnailFile->Exists (&exists);
  NS_ENSURE_SUCCESS (rv, rv);
  if (exists)
    return NS_NewFileURI (_retval, thumbnailFile);
  else
    return NS_NewURI (_retval, NS_LITERAL_STRING (MWB_FALLBACK_THUMBNAIL));
}

NS_IMPL_ISUPPORTS1(MwbPinnedPage, MwbIPinnedPage)

MwbPinnedPage::MwbPinnedPage (const gchar *title_arg,
                              const gchar *uri_arg,
                              PRInt32 visit_time_arg)
{
  SetTitle (title_arg);

  uri = uri_arg;
  visit_time = visit_time_arg;
}

/* readonly attribute AUTF8String title; */
NS_IMETHODIMP
MwbPinnedPage::GetTitle (nsACString &aTitle)
{
  aTitle = title;
  return NS_OK;
}

void
MwbPinnedPage::SetTitle (const gchar *title_arg)
{
  gchar *collation_key_val = g_utf8_collate_key (title_arg, -1);

  title = title_arg;
  collation_key = collation_key_val;
  g_free (collation_key_val);
}

/* readonly attribute AUTF8String uri; */
NS_IMETHODIMP
MwbPinnedPage::GetUri (nsACString &aUri)
{
  aUri = uri;
  return NS_OK;
}

/* readonly attribute long visitTime; */
NS_IMETHODIMP
MwbPinnedPage::GetVisitTime (PRInt32 *aVisitTime)
{
  *aVisitTime = visit_time;
  return NS_OK;
}
