/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mwb-toolbar.h"
#include "mwb-radical-bar.h"
#include "mwb-utils.h"

G_DEFINE_TYPE (MwbToolbar, mwb_toolbar, NBTK_TYPE_TABLE)

#define TOOLBAR_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), MWB_TYPE_TOOLBAR, MwbToolbarPrivate))

enum
{
  TRAVEL,
  SETTINGS,
  QUIT,

  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0, };

struct _MwbToolbarPrivate
{
  NbtkWidget   *back_button;
  NbtkWidget   *forward_button;
  NbtkWidget   *radical_bar;
  NbtkWidget   *settings_button;
  NbtkWidget   *close_button;
};

static void
mwb_toolbar_get_property (GObject *object, guint property_id,
                              GValue *value, GParamSpec *pspec)
{
  switch (property_id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_toolbar_set_property (GObject *object, guint property_id,
                              const GValue *value, GParamSpec *pspec)
{
  switch (property_id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_toolbar_dispose (GObject *object)
{
  G_OBJECT_CLASS (mwb_toolbar_parent_class)->dispose (object);
}

static void
mwb_toolbar_finalize (GObject *object)
{
  G_OBJECT_CLASS (mwb_toolbar_parent_class)->finalize (object);
}

static void
mwb_toolbar_class_init (MwbToolbarClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (klass, sizeof (MwbToolbarPrivate));

  object_class->get_property = mwb_toolbar_get_property;
  object_class->set_property = mwb_toolbar_set_property;
  object_class->dispose = mwb_toolbar_dispose;
  object_class->finalize = mwb_toolbar_finalize;

  signals[TRAVEL] =
    g_signal_new ("travel",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MwbToolbarClass, travel),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__INT,
                  G_TYPE_NONE, 1, G_TYPE_INT);

  signals[SETTINGS] =
    g_signal_new ("settings",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MwbToolbarClass, settings),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  signals[QUIT] =
    g_signal_new ("quit",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MwbToolbarClass, quit),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);
}

static void
mwb_toolbar_back_clicked_cb (NbtkButton *button, MwbToolbar *self)
{
  g_signal_emit (self, signals[TRAVEL], 0, -1);
}

static void
mwb_toolbar_forward_clicked_cb (NbtkButton *button, MwbToolbar *self)
{
  g_signal_emit (self, signals[TRAVEL], 0, 1);
}

static void
mwb_toolbar_settings_clicked_cb (NbtkButton *button, MwbToolbar *self)
{
  g_signal_emit (self, signals[SETTINGS], 0);
}

static void
mwb_toolbar_quit_clicked_cb (NbtkButton *button, MwbToolbar *self)
{
  g_signal_emit (self, signals[QUIT], 0);
}

static void
mwb_toolbar_init (MwbToolbar *self)
{
  NbtkTable *table;
  ClutterActor *separator;

  MwbToolbarPrivate *priv = self->priv = TOOLBAR_PRIVATE (self);

  clutter_actor_set_name (CLUTTER_ACTOR (self), "browser-toolbar");

  priv->back_button = nbtk_button_new ();
  clutter_actor_set_name (CLUTTER_ACTOR (priv->back_button),
                          "back-button");
  priv->forward_button = nbtk_button_new ();
  clutter_actor_set_name (CLUTTER_ACTOR (priv->forward_button),
                          "forward-button");
  priv->radical_bar = mwb_radical_bar_new ();
  priv->settings_button = nbtk_button_new ();
  clutter_actor_set_name (CLUTTER_ACTOR (priv->settings_button),
                          "settings-button");
  priv->close_button = nbtk_button_new ();
  clutter_actor_set_name (CLUTTER_ACTOR (priv->close_button),
                          "close-button");

  g_signal_connect (priv->back_button, "clicked",
                    G_CALLBACK (mwb_toolbar_back_clicked_cb), self);
  g_signal_connect (priv->forward_button, "clicked",
                    G_CALLBACK (mwb_toolbar_forward_clicked_cb), self);
  g_signal_connect (priv->settings_button, "clicked",
                    G_CALLBACK (mwb_toolbar_settings_clicked_cb), self);
  g_signal_connect (priv->close_button, "clicked",
                    G_CALLBACK (mwb_toolbar_quit_clicked_cb), self);

  table = NBTK_TABLE (self);

  nbtk_table_set_col_spacing (table, 6);

  mwb_utils_table_add (table, CLUTTER_ACTOR (priv->back_button), 0, 0,
                       FALSE, FALSE, FALSE, FALSE);
  separator = clutter_texture_new_from_file (PKGDATADIR"/toolbar-separator.png",
                                             NULL);
  mwb_utils_table_add (table, separator, 0, 1, FALSE, FALSE, FALSE, FALSE);
  mwb_utils_table_add (table, CLUTTER_ACTOR (priv->forward_button), 0, 2,
                       FALSE, FALSE, FALSE, FALSE);
  mwb_utils_table_add (table, CLUTTER_ACTOR (priv->radical_bar), 0, 3,
                       TRUE, TRUE, FALSE, FALSE);
  mwb_utils_table_add (table, CLUTTER_ACTOR (priv->settings_button), 0, 4,
                       FALSE, FALSE, FALSE, FALSE);
  separator = clutter_clone_new (separator);
  mwb_utils_table_add (table, separator, 0, 5, FALSE, FALSE, FALSE, FALSE);
  mwb_utils_table_add (table, CLUTTER_ACTOR (priv->close_button), 0, 6,
                       FALSE, FALSE, FALSE, FALSE);

  mwb_toolbar_set_can_go_back (self, FALSE);
  mwb_toolbar_set_can_go_forward (self, FALSE);
}

NbtkWidget*
mwb_toolbar_new (void)
{
  return g_object_new (MWB_TYPE_TOOLBAR, NULL);
}

MwbRadicalBar *
mwb_toolbar_get_radical_bar (MwbToolbar *toolbar)
{
  return MWB_RADICAL_BAR (toolbar->priv->radical_bar);
}

void
mwb_toolbar_set_can_go_back (MwbToolbar *toolbar, gboolean enable)
{
  MwbToolbarPrivate *priv = toolbar->priv;
  clutter_actor_set_reactive (CLUTTER_ACTOR (priv->back_button), enable);

  if (!enable)
    nbtk_widget_set_style_pseudo_class (priv->back_button, "disabled");
  else
    nbtk_widget_set_style_pseudo_class (priv->back_button, NULL);
}

void
mwb_toolbar_set_can_go_forward (MwbToolbar *toolbar, gboolean enable)
{
  MwbToolbarPrivate *priv = toolbar->priv;
  clutter_actor_set_reactive (CLUTTER_ACTOR (priv->forward_button), enable);
  if (!enable)
    nbtk_widget_set_style_pseudo_class (priv->forward_button, "disabled");
  else
    nbtk_widget_set_style_pseudo_class (priv->forward_button, NULL);
}

