/* netbook-launcher-efl: nice desktop launcher targeted at netbooks.
 *
 * Copyright 2009 Canonical Limited.  All rights reserved.
 * This software is subject to the terms of your agreement with Canonical.
 *
 * Author: Gustavo Sverzut Barbieri <gustavo.barbieri@canonical.com>
 */

/**
 * This file takes care of gluing two main loops: Ecore and GLib.
 *
 * It does so by using one thread per loop and exchanging functions to
 * be called on the other thread. This is done asynchronously, so
 * either use locks around your data or copy it.
 *
 * Do not call GLib functions from the Ecore thread and vice-versa!
 *
 * The communication is LOCKLESS except by queues, that use locks
 * internally. Do not add useless locks around parts of the system,
 * try to use given primitives to avoid UI blocking on backend actions
 * and vice versa.
 *
 * Pipes are used to wakeup the other main loop, functions
 * gstuff_glib_run() and gstuff_ecore_run() will take care of internals to
 * add function context to queue and then wakeup the thread.
 */

#include "netbook-launcher.h"
#include <Ecore.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <pthread.h>

struct gstuff_com
{
  struct {
    int in, out;
    pthread_t thread;
    GAsyncQueue *queue;
    Ecore_Fd_Handler *fd_handler;
  } master;
  struct {
    int in, out;
    pthread_t thread;
    GAsyncQueue *queue;
  } slave;
};

static struct gstuff_com com = {
  {-1, -1, 0, NULL, NULL},
  {-1, -1, 0, NULL}
};

static Eina_Bool
_gstuff_read_and_dispatch(int fd, GAsyncQueue *queue, const char *dbgname)
{
  ssize_t ret, n;
  char buf[128];
  struct func_ctxt *ctxt;

  ret = read(fd, buf, sizeof(buf));
  if (ret < 0)
  {
    if ((errno != EAGAIN) && (errno != EINTR))
    {
      fprintf(stderr,
              "ERR: could not read from %s (%d) file descriptor: %s. Exit.\n",
              dbgname, fd, strerror(errno));
      return 0;
    }
  }

  n = 0;
  while ((n < ret) && ((ctxt = g_async_queue_try_pop(queue)) != NULL))
  {
    n++;
    ctxt->func(ctxt->data);
    ctxt->free(ctxt);
  }

  if (n != ret)
    fprintf(stderr,
            "WARN (%s/%d): "
            "number of queue %p elements smaller than bytes read? "
            "(bytes: %zd < queue: %zd/%d)\n",
            dbgname, fd, queue, ret, n, g_async_queue_length(queue));

  return 1;
}

static int
_gstuff_master_read(void *data __UNUSED__, Ecore_Fd_Handler *fd_handler)
{
  if (ecore_main_fd_handler_active_get(fd_handler, ECORE_FD_ERROR))
  {
    ERR("com.master.in (%d) file descriptor has errors. Exit.\n",
        com.master.in);
    goto error;
  }

  if (!_gstuff_read_and_dispatch
      (com.master.in, com.master.queue, "com.master.in"))
    goto error;

  return 1;

error:
  ERR("error on Ecore thread, exit.\n");
  ecore_main_loop_quit();
  com.master.fd_handler = NULL;
  return 0;
}

static gboolean
_gstuff_slave_read(GIOChannel *source __UNUSED__, GIOCondition cond, gpointer data __UNUSED__)
{
  if ((cond == G_IO_ERR) || (cond == G_IO_HUP) || (cond == G_IO_NVAL))
  {
    g_critical("com.slave.in (%d) file descriptor has errors. Exit.\n",
               com.slave.in);
    goto error;
  }

  if (!_gstuff_read_and_dispatch
      (com.slave.in, com.slave.queue, "com.slave.in"))
    goto error;

  return 1;

error:
  g_critical("error on GLib thread, exit.");
  gtk_main_quit();
  return 0;
}

static void *
_gstuff_thread(void *data __UNUSED__)
{
  GIOChannel *ioc = NULL;
  guint io_watch = 0;
  struct func_ctxt *ctxt;
  ssize_t ret;
  char c = 0;

  ioc = g_io_channel_unix_new(com.slave.in);
  if (!ioc)
    goto end;
  io_watch = g_io_add_watch(ioc, G_IO_IN, _gstuff_slave_read, NULL);
  if (!io_watch)
    goto end;

  g_debug("tell main thread we're ready.");
  while ((ret = write(com.master.out, &c, 1)) != 1)
  {
    if ((errno != EAGAIN) && (errno != EINTR))
    {
      g_warning("could not inform main thread: %s", strerror(errno));
      goto end;
    }
  }

  g_debug("enter gtk main loop.");
  gtk_main();
  g_debug("exit gtk main loop.");

  while ((ctxt = g_async_queue_try_pop(com.slave.queue)) != NULL)
  {
    ctxt->func(ctxt->data);
    ctxt->free(ctxt);
  }

end:
  if (io_watch)
    g_source_remove(io_watch);
  if (ioc)
    g_io_channel_unref(ioc);

  gstuff_ecore_exit();

  return NULL;
}

void
gstuff_shutdown(void)
{
  if (com.slave.thread)
  {
    void *data = NULL;

    gstuff_glib_exit();

    DBG("join thread %zd...\n", com.slave.thread);
    pthread_join(com.slave.thread, &data);
    DBG("joined thread %zd: %p\n", com.slave.thread, data);
    if (data)
      WRN("thread returned unexpected value: %p\n", data);
    com.slave.thread = 0;
  }

  if (com.master.queue)
  {
    struct func_ctxt *ctxt;
    while ((ctxt =  g_async_queue_try_pop(com.master.queue)) != NULL)
      ctxt->free(ctxt);
    g_async_queue_unref(com.master.queue);
    com.master.queue = NULL;
  }

  if (com.slave.queue)
  {
    struct func_ctxt *ctxt;
    while ((ctxt =  g_async_queue_try_pop(com.slave.queue)) != NULL)
      ctxt->free(ctxt);
    g_async_queue_unref(com.slave.queue);
    com.slave.queue = NULL;
  }

  if (com.master.fd_handler)
  {
    ecore_main_fd_handler_del(com.master.fd_handler);
    com.master.fd_handler = NULL;
  }

  if (com.master.in >= 0)
  {
    close(com.master.in);
    com.master.in = -1;
  }

  if (com.master.out >= 0)
  {
    close(com.master.out);
    com.master.out = -1;
  }

  if (com.slave.in >= 0)
  {
    close(com.slave.in);
    com.slave.in = -1;
  }

  if (com.slave.out >= 0)
  {
    close(com.slave.out);
    com.slave.out = -1;
  }
}

Eina_Bool
gstuff_init(void)
{
  int fds[2];
  char c = 0;
  ssize_t ret;

  if (com.master.in >= 0)
  {
    DBG("gstuff already initialized\n");
    return 1;
  }

  if (pipe(fds) == -1)
  {
    ERR("could not create pipe: %s\n", strerror(errno));
    return 0;
  }
  com.master.in = fds[0];
  com.master.out = fds[1];

  if (pipe(fds) == -1)
  {
    ERR("could not create pipe: %s\n", strerror(errno));
    gstuff_shutdown();
    return 0;
  }
  com.slave.in = fds[0];
  com.slave.out = fds[1];

  com.master.thread = pthread_self();
  com.master.fd_handler = ecore_main_fd_handler_add
    (com.master.in, ECORE_FD_READ | ECORE_FD_ERROR, _gstuff_master_read, NULL,
     NULL, NULL);
  if (!com.master.fd_handler)
  {
    ERR("could not create master fd handler\n");
    gstuff_shutdown();
    return 0;
  }

  g_thread_init(NULL);
  g_set_application_name(PACKAGE);
  if (pthread_create(&com.slave.thread, NULL, _gstuff_thread, NULL) != 0)
  {
    ERR("could not create thread: %s\n", strerror(errno));
    gstuff_shutdown();
    return 0;
  }

  DBG("sync with slave thread %zd.\n", com.slave.thread);
  while ((ret = read(com.master.in, &c, 1)) != 1)
    if ((errno != EAGAIN) && (errno != EINTR))
    {
      ERR("could not sync to slave thread %zd: %s. Exit.\n",
          com.slave.thread, strerror(errno));
      pthread_kill(com.slave.thread, SIGTERM);
      com.slave.thread = 0;
      gstuff_shutdown();
      return 0;
    }
  DBG("slave thread %zd is ready.\n", com.slave.thread);

  if (fcntl(com.slave.in, F_SETFL, O_NONBLOCK) != 0)
  {
    ERR("could not set com.slave.in to non-blocking.\n");
    gstuff_shutdown();
    return 0;
  }

  if (fcntl(com.master.in, F_SETFL, O_NONBLOCK) != 0)
  {
    ERR("could not set com.master.in to non-blocking.\n");
    gstuff_shutdown();
    return 0;
  }

  com.master.queue = g_async_queue_new();
  if (!com.master.queue)
  {
    ERR("could not create GAsyncQueue for master.\n");
    gstuff_shutdown();
    return 0;
  }

  com.slave.queue = g_async_queue_new();
  if (!com.slave.queue)
  {
    ERR("could not create GAsyncQueue for slave.\n");
    gstuff_shutdown();
    return 0;
  }

  return 1;
}

/**
 * Schedule function to run on the other thread.
 *
 * @param from_thread: thread we should be at, used to avoid adding calls
 *        from the incorrect thread.
 * @param fd: where to write to in order to wakeup the remote thread.
 * @param queue: where to push pending call.
 * @param ctxt: context of the pending call.
 * @param dbgname: name to use in debug, usually describes the fd name.
 *
 * @return: 0 on success, -1 on lesser error, 1 on fatal error.
 */
static int
_gstuff_run(pthread_t from_thread, int fd, GAsyncQueue *queue, struct func_ctxt *ctxt, const char *dbgname)
{
  ssize_t ret;

  if (pthread_self() != from_thread)
  {
    fprintf(stderr,
            "ERROR: trying to add function from incorrect thread. (%zd, %zd)\n",
            pthread_self(), from_thread);
    ctxt->free(ctxt);
    return -1;
  }

  g_async_queue_push(queue, ctxt);

  do
  {
    char c = 0;
    ret = write(fd, &c, 1);

    if (ret < 0)
    {
      if ((errno != EAGAIN) && (errno != EINTR))
      {
        fprintf(stderr, "ERROR: could not write to %s (%d): %s. Exit.\n",
                dbgname, fd, strerror(errno));
        ctxt->free(ctxt);
        return 1;
      }
    }
  } while (ret != 1);

  return 0;
}

/**
 * Have the given function context to run on GLib thread/main loop.
 *
 * @warning: just call this function from Ecore thread!
 *
 * @param ctxt: context to call function on GLib thread.
 * @return: 1 on success, 0 on error (ctxt->free(ctxt) is called on errors).
 */
Eina_Bool
gstuff_glib_run(struct func_ctxt *ctxt)
{
  int r;

  if (!ctxt)
  {
    ERR("gstuff_glib_run(NULL)\n");
    return 0;
  }

  r = _gstuff_run(com.master.thread, com.slave.out, com.slave.queue, ctxt,
                  "com.slave.out");
  if (r == 0)
    return 1;
  else if (r == -1)
  {
    ERR("gstuff_glib_run(%p): lesser error.\n", ctxt);
    return 0;
  }
  else if (r == 1)
  {
    ERR("gstuff_glib_run(%p): major error. Exit\n", ctxt);
    ecore_main_loop_quit();
    return 0;
  }
  ERR("gstuff_glib_run(%p): unknown error, r=%d\n", ctxt, r);
  return 0;
}

/**
 * Have the given function context to run on Ecore thread/main loop.
 *
 * @warning: just call this function from GLib thread!
 *
 * @param ctxt: context to call function on Ecore thread.
 * @return: 1 on success, 0 on error (ctxt->free(ctxt) is called on errors).
 */
Eina_Bool
gstuff_ecore_run(struct func_ctxt *ctxt)
{
  int r;

  if (!ctxt)
  {
    g_warning("gstuff_ecore_run(NULL)");
    return 0;
  }

  r = _gstuff_run(com.slave.thread, com.master.out, com.master.queue, ctxt,
                  "com.master.out");
  if (r == 0)
    return 1;
  else if (r == -1)
  {
    g_warning("gstuff_ecore_run(%p): lesser error.", ctxt);
    return 0;
  }
  else if (r == 1)
  {
    g_warning("gstuff_ecore_run(%p): major error. Exit", ctxt);
    gtk_main_quit();
    return 0;
  }
  g_warning("gstuff_ecore_run(%p): unknown error, r=%d", ctxt, r);
  return 0;
}

static void
_gstuff_func_ctxt_dumb_free(void *data __UNUSED__)
{
}

static void
_gstuff_glib_exit(void *data __UNUSED__)
{
  g_debug("quit gtk main loop.");
  gtk_main_quit();
}

/**
 * Request glib main loop to exit.
 */
Eina_Bool
gstuff_glib_exit(void)
{
  static struct func_ctxt ctxt = {
    _gstuff_glib_exit,
    _gstuff_func_ctxt_dumb_free,
    NULL
  };
  return gstuff_glib_run(&ctxt);
}

static void
_gstuff_ecore_exit(void *data __UNUSED__)
{
  ecore_main_loop_quit();
}

/**
 * Request ecore main loop to exit.
 */
Eina_Bool
gstuff_ecore_exit(void)
{
  static struct func_ctxt ctxt = {
    _gstuff_ecore_exit,
    _gstuff_func_ctxt_dumb_free,
    NULL
  };
  return gstuff_ecore_run(&ctxt);
}

struct func_ctxt_noargs
{
  struct func_ctxt base;
  void (*real_func)(void);
};

static void
_gstuff_noargs(void *data)
{
  struct func_ctxt_noargs *ctxt = data;
  ctxt->real_func();
}

static struct func_ctxt *
_gstuff_noargs_ctxt_new(void (*func)(void))
{
  struct func_ctxt_noargs *ctxt;

  if (!func)
    return NULL;

  ctxt = malloc(sizeof(*ctxt));
  if (!ctxt)
    return NULL;
  ctxt->base.func = _gstuff_noargs;
  ctxt->base.free = free;
  ctxt->base.data = ctxt;
  ctxt->real_func = func;
  return &ctxt->base;
}

/**
 * Request function to be called at GLib thread.
 */
Eina_Bool
gstuff_glib_run_noargs(void (*func)(void))
{
  return gstuff_glib_run(_gstuff_noargs_ctxt_new(func));
}

/**
 * Request function to be called at Ecore thread.
 */
Eina_Bool
gstuff_ecore_run_noargs(void (*func)(void))
{
  return gstuff_ecore_run(_gstuff_noargs_ctxt_new(func));
}

static struct func_ctxt *
_gstuff_simple_ctxt_new(void (*func)(void *data), const void *data)
{
  struct func_ctxt *ctxt;

  if (!func)
    return NULL;

  ctxt = malloc(sizeof(*ctxt));
  if (!ctxt)
    return NULL;
  ctxt->func = func;
  ctxt->free = free;
  ctxt->data = (void *)data;
  return ctxt;
}

/**
 * Request function to be called at GLib thread.
 *
 * @param data: any pointer to give to @a func, will not be copied.
 */
Eina_Bool
gstuff_glib_run_simple(void (*func)(void *data), const void *data)
{
  return gstuff_glib_run(_gstuff_simple_ctxt_new(func, data));
}

/**
 * Request function to be called at Ecore thread.
 *
 * @param data: any pointer to give to @a func, will not be copied.
 */
Eina_Bool
gstuff_ecore_run_simple(void (*func)(void *data), const void *data)
{
  return gstuff_ecore_run(_gstuff_simple_ctxt_new(func, data));
}

struct func_ctxt_string
{
  struct func_ctxt base;
  void (*real_func)(const char *);
  char str[];
};

static void
_gstuff_string(void *data)
{
  struct func_ctxt_string *ctxt = data;
  ctxt->real_func(ctxt->str);
}

static struct func_ctxt *
_gstuff_string_ctxt_new(void (*func)(const char *str), const char *str)
{
  struct func_ctxt_string *ctxt;
  size_t len;

  if (!func)
    return NULL;

  if (!str)
    return NULL;
  len = strlen(str);
  ctxt = malloc(sizeof(*ctxt) + len + 1);
  if (!ctxt)
    return NULL;

  ctxt->base.func = _gstuff_string;
  ctxt->base.free = free;
  ctxt->base.data = ctxt;
  ctxt->real_func = func;
  memcpy(ctxt->str, str, len + 1);
  return &ctxt->base;
}

/**
 * Request function to be called with a copy of str at GLib thread.
 *
 * @param str: string to be COPIED, no references will be kept after return.
 */
Eina_Bool
gstuff_glib_run_string(void (*func)(const char *str), const char *str)
{
  return gstuff_glib_run(_gstuff_string_ctxt_new(func, str));
}

/**
 * Request function to be called with a copy of str at Ecore thread.
 *
 * @param str: string to be COPIED, no references will be kept after return.
 */
Eina_Bool
gstuff_ecore_run_string(void (*func)(const char *str), const char *str)
{
  return gstuff_ecore_run(_gstuff_string_ctxt_new(func, str));
}

struct func_ctxt_ptr_string
{
  struct func_ctxt base;
  void (*real_func)(void *, const char *);
  const void *real_data;
  char str[];
};

static void
_gstuff_ptr_string(void *data)
{
  struct func_ctxt_ptr_string *ctxt = data;
  ctxt->real_func((void *)ctxt->real_data, ctxt->str);
}

static struct func_ctxt *
_gstuff_ptr_string_ctxt_new(void (*func)(void *data, const char *str), const void *data, const char *str)
{
  struct func_ctxt_ptr_string *ctxt;
  size_t len;

  if (!func)
    return NULL;

  if (!str)
    return NULL;
  len = strlen(str);
  ctxt = malloc(sizeof(*ctxt) + len + 1);
  if (!ctxt)
    return NULL;

  ctxt->base.func = _gstuff_ptr_string;
  ctxt->base.free = free;
  ctxt->base.data = ctxt;
  ctxt->real_func = func;
  ctxt->real_data = data;
  memcpy(ctxt->str, str, len + 1);
  return &ctxt->base;
}

/**
 * Request function to be called with a copy of str at GLib thread.
 *
 * @param data: extra pointer to give to function, not copied or touched.
 * @param str: string to be COPIED, no references will be kept after return.
 */
Eina_Bool
gstuff_glib_run_ptr_string(void (*func)(void *data, const char *str), const void *data, const char *str)
{
  return gstuff_glib_run(_gstuff_ptr_string_ctxt_new(func, data, str));
}

/**
 * Request function to be called with a copy of str at Ecore thread.
 *
 * @param data: extra pointer to give to function, not copied or touched.
 * @param str: string to be COPIED, no references will be kept after return.
 */
Eina_Bool
gstuff_ecore_run_ptr_string(void (*func)(void *data, const char *str), const void *data, const char *str)
{
  return gstuff_ecore_run(_gstuff_ptr_string_ctxt_new(func, data, str));
}

struct func_ctxt_string_array
{
  struct func_ctxt base;
  void (*real_func)(unsigned int, const char **);
  unsigned int count;
  char **array;
};

static void
_gstuff_string_array(void *data)
{
  struct func_ctxt_string_array *ctxt = data;
  ctxt->real_func(ctxt->count, (const char **)ctxt->array);
}

static struct func_ctxt *
_gstuff_string_array_ctxt_new(void (*func)(unsigned int count, const char **array), unsigned int count, const char **array)
{
  struct func_ctxt_string_array *ctxt;
  unsigned int i;
  size_t len;
  char *s;

  if (!func)
    return NULL;

  if (!array)
    return NULL;

  if (count == 0)
    while (array[count] != NULL)
      count++;

  len = 0;
  for (i = 0; i < count; i++)
    len += strlen(array[i]) + 1;

  ctxt = malloc(sizeof(*ctxt) + ((count + 1) * sizeof(char *)) + len);
  if (!ctxt)
    return NULL;

  ctxt->base.func = _gstuff_string_array;
  ctxt->base.free = free;
  ctxt->base.data = ctxt;
  ctxt->real_func = func;
  ctxt->count = count;

  ctxt->array = (char **)((char *)ctxt + sizeof(*ctxt));
  s = (char *)(ctxt->array + count + 1);
  for (i = 0; i < count;  i++)
  {
    len = strlen(array[i]) + 1;
    ctxt->array[i] = s;
    memcpy(s, array[i], len);
    s += len;
  }
  ctxt->array[i] = NULL;

  return &ctxt->base;
}

/**
 * Request function to run on GLib thread with an array of strings.
 *
 * @note: function will always receive the number of elements and the
 *        copied array will always be NULL terminated. Functions can
 *        always rely on such values even if they were not given as
 *        original parameter.
 *
 * @param func: function to call, first argument is the number of
 *        elements and second is the array, with a NULL terminator.
 * @param count: elements in array or 0 to count @a array until a NULL
 *        terminator is found.
 * @param array: elements to COPY. If @a count is 0, then it must be NULL
 *        terminated. Strings and array will be copied, no reference will
 *        be kept after function returns.
 */
Eina_Bool
gstuff_glib_run_string_array(void (*func)(unsigned int count, const char **array), unsigned int count, const char **array)
{
  return gstuff_glib_run(_gstuff_string_array_ctxt_new(func, count, array));
}

/**
 * Request function to run on Ecore thread with an array of strings.
 *
 * @note: function will always receive the number of elements and the
 *        copied array will always be NULL terminated. Functions can
 *        always rely on such values even if they were not given as
 *        original parameter.
 *
 * @param func: function to call, first argument is the number of
 *        elements and second is the array, with a NULL terminator.
 * @param count: elements in array or 0 to count @a array until a NULL
 *        terminator is found.
 * @param array: elements to COPY. If @a count is 0, then it must be NULL
 *        terminated. Strings and array will be copied, no reference will
 *        be kept after function returns.
 */
Eina_Bool
gstuff_ecore_run_string_array(void (*func)(unsigned int count, const char **array), unsigned int count, const char **array)
{
  return gstuff_ecore_run(_gstuff_string_array_ctxt_new(func, count, array));
}

struct func_ctxt_ptr_string_array
{
  struct func_ctxt base;
  void (*real_func)(void *, unsigned int, const char **);
  void *real_data;
  unsigned int count;
  char **array;
};

static void
_gstuff_ptr_string_array(void *data)
{
  struct func_ctxt_ptr_string_array *ctxt = data;
  ctxt->real_func(ctxt->real_data, ctxt->count, (const char **)ctxt->array);
}

static struct func_ctxt *
_gstuff_ptr_string_array_ctxt_new(void (*func)(void *data, unsigned int count, const char **array), const void *data, unsigned int count, const char **array)
{
  struct func_ctxt_ptr_string_array *ctxt;
  unsigned int i;
  size_t len;
  char *s;

  if (!func)
    return NULL;

  if (!array)
    return NULL;

  if (count == 0)
    while (array[count] != NULL)
      count++;

  len = 0;
  for (i = 0; i < count; i++)
    len += strlen(array[i]) + 1;

  ctxt = malloc(sizeof(*ctxt) + ((count + 1) * sizeof(char *)) + len);
  if (!ctxt)
    return NULL;

  ctxt->base.func = _gstuff_ptr_string_array;
  ctxt->base.free = free;
  ctxt->base.data = ctxt;
  ctxt->real_func = func;
  ctxt->real_data = (void *)data;
  ctxt->count = count;

  ctxt->array = (char **)((char *)ctxt + sizeof(*ctxt));
  s = (char *)(ctxt->array + count + 1);
  for (i = 0; i < count;  i++)
  {
    len = strlen(array[i]) + 1;
    ctxt->array[i] = s;
    memcpy(s, array[i], len);
    s += len;
  }
  ctxt->array[i] = NULL;

  return &ctxt->base;
}

/**
 * Request function to run on GLib thread with data and array of strings.
 *
 * @note: function will always receive the number of elements and the
 *        copied array will always be NULL terminated. Functions can
 *        always rely on such values even if they were not given as
 *        original parameter.
 *
 * @param func: function to call, first argument is the number of
 *        elements and second is the array, with a NULL terminator.
 * @param data: extra data to give to function.
 * @param count: elements in array or 0 to count @a array until a NULL
 *        terminator is found.
 * @param array: elements to COPY. If @a count is 0, then it must be NULL
 *        terminated. Strings and array will be copied, no reference will
 *        be kept after function returns.
 */
Eina_Bool
gstuff_glib_run_ptr_string_array(void (*func)(void *data, unsigned int count, const char **array), const void *data, unsigned int count, const char **array)
{
  return gstuff_glib_run(
    _gstuff_ptr_string_array_ctxt_new(func, data, count, array));
}

/**
 * Request function to run on Ecore thread with data and array of strings.
 *
 * @note: function will always receive the number of elements and the
 *        copied array will always be NULL terminated. Functions can
 *        always rely on such values even if they were not given as
 *        original parameter.
 *
 * @param func: function to call, first argument is the number of
 *        elements and second is the array, with a NULL terminator.
 * @param data: extra data to give to function.
 * @param count: elements in array or 0 to count @a array until a NULL
 *        terminator is found.
 * @param array: elements to COPY. If @a count is 0, then it must be NULL
 *        terminated. Strings and array will be copied, no reference will
 *        be kept after function returns.
 */
Eina_Bool
gstuff_ecore_run_ptr_string_array(void (*func)(void *data, unsigned int count, const char **array), const void *data, unsigned int count, const char **array)
{
  return gstuff_ecore_run
    (_gstuff_ptr_string_array_ctxt_new(func, data, count, array));
}
