/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <glib/gi18n.h>
#include <libgnomeui/libgnomeui.h>

#include "launcher-link-editor.h"
#include "launcher-app.h"
#include "launcher-defines.h"
#include "launcher-menu.h"
#include "launcher-util.h"

G_DEFINE_TYPE (LauncherLinkEditor, launcher_link_editor, GTK_TYPE_DIALOG);

#define LAUNCHER_LINK_EDITOR_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  LAUNCHER_TYPE_LINK_EDITOR, \
  LauncherLinkEditorPrivate))

#define WEB_ICON_NAME "applications-internet"
#define FOLDER_ICON_NAME "gnome-fs-directory"
#define APP_ICON_NAME "gnome-panel-launcher"

struct _LauncherLinkEditorPrivate
{
  LauncherMenuCategory *cat;

  GtkWidget *combo;
  GtkWidget *name_entry;
  GtkWidget *loc_entry;
  GtkWidget *icon;
  GtkWidget *browse;

  gboolean icon_change_lock;
};

enum
{
	PROP_0,
  PROP_TYPE,
	PROP_NAME,
  PROP_EXEC,
  PROP_ICON
};

static void
set_icon_path (GnomeIconEntry *entry, const gchar *icon_name)
{
  GtkIconTheme *theme = gtk_icon_theme_get_default ();
  GtkIconInfo *info = NULL;

  info = gtk_icon_theme_lookup_icon (theme, icon_name, 
                                     48, 
                                     GTK_ICON_LOOKUP_FORCE_SVG);

  if (info)
  {
    gnome_icon_entry_set_filename (entry, 
                                        gtk_icon_info_get_filename (info));
    gtk_icon_info_free (info);
  }
  else
  {
    gnome_icon_entry_set_filename (entry, 
                                   icon_name);
   }
}

void
launcher_link_editor_set_name (LauncherLinkEditor *self, 
                                   const gchar            *name)
{
  g_return_if_fail (LAUNCHER_IS_LINK_EDITOR (self));

  gtk_entry_set_text (GTK_ENTRY (self->priv->name_entry), name);
}

void
launcher_link_editor_set_exec (LauncherLinkEditor *self, 
                                   const gchar            *exec)
{
  LauncherLinkEditorPrivate *priv;
  static gint open_len = 0;

  g_return_if_fail (LAUNCHER_IS_LINK_EDITOR (self));
  priv = self->priv;

  if (!open_len)
    open_len = strlen (XDG_OPEN) + 1;

  gtk_entry_set_text (GTK_ENTRY (priv->loc_entry), exec);
}

gchar *
launcher_link_editor_get_exec (LauncherLinkEditor *self)
{
  LauncherLinkEditorPrivate *priv = self->priv;
  gchar *exec = NULL;

  exec = g_strdup_printf ("%s %s", XDG_OPEN,
                          gtk_entry_get_text (GTK_ENTRY (priv->loc_entry)));


  return exec;
}

void
launcher_link_editor_set_icon (LauncherLinkEditor *self, 
                                   const gchar            *icon_name)
{
  g_return_if_fail (LAUNCHER_IS_LINK_EDITOR (self));

  set_icon_path (GNOME_ICON_ENTRY (self->priv->icon), icon_name);
  self->priv->icon_change_lock = TRUE;
}

/* GObject stuff */
static void
launcher_link_editor_finalize (GObject *link_editor)
{
  LauncherLinkEditorPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_LINK_EDITOR (link_editor));
  priv = LAUNCHER_LINK_EDITOR_GET_PRIVATE (link_editor);

  G_OBJECT_CLASS (launcher_link_editor_parent_class)->finalize (link_editor);
}

static void
set_property (GObject      *object, 
              guint         prop_id,
              const GValue *value, 
              GParamSpec   *pspec)
{
  LauncherLinkEditor *self = LAUNCHER_LINK_EDITOR (object);
  LauncherLinkEditorPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_LINK_EDITOR (object));
  priv = LAUNCHER_LINK_EDITOR_GET_PRIVATE (object);

  switch (prop_id) 
  {
    case PROP_TYPE:
      gtk_combo_box_set_active (GTK_COMBO_BOX (priv->combo),  
                                g_value_get_int (value));
      break;
    case PROP_NAME:
      launcher_link_editor_set_name (self, g_value_get_string (value));
      break;
    case PROP_EXEC:
      launcher_link_editor_set_exec (self, g_value_get_string (value));
      break;
    case PROP_ICON:
      launcher_link_editor_set_icon (self, g_value_get_string (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
get_property (GObject      *object, 
              guint         prop_id,
              GValue       *value, 
              GParamSpec   *pspec)
{
  LauncherLinkEditorPrivate *priv;
  gchar *exec;

  g_return_if_fail (LAUNCHER_IS_LINK_EDITOR (object));
  priv = LAUNCHER_LINK_EDITOR_GET_PRIVATE (object);

  switch (prop_id) 
  {
    case PROP_TYPE:
      g_value_set_int (value,
                       gtk_combo_box_get_active (GTK_COMBO_BOX (priv->combo)));

      break;
    case PROP_NAME:
      g_value_set_string (value, 
                          gtk_entry_get_text (GTK_ENTRY (priv->name_entry)));
      break;
    case PROP_EXEC:
      exec = launcher_link_editor_get_exec (
                                            LAUNCHER_LINK_EDITOR(object));
      g_value_set_string (value, exec);
      g_free (exec);
      break;
    case PROP_ICON:
      g_value_set_string (value, 
                gnome_icon_entry_get_filename (GNOME_ICON_ENTRY (priv->icon)));
       break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
launcher_link_editor_class_init (LauncherLinkEditorClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);
  
  obj_class->finalize = launcher_link_editor_finalize;
  obj_class->get_property = get_property;
  obj_class->set_property = set_property;


  /* Class properties */
  g_object_class_install_property (obj_class,
                                   PROP_TYPE,
                                   g_param_spec_int ("link-category",
                                   "Link Type",
                                   "Type of link",
                                   0, 100, 0, 
                                   G_PARAM_READWRITE));

  g_object_class_install_property (obj_class,
                                   PROP_NAME,
                                   g_param_spec_string ("link-name",
                                   "Link Name",
                                   "Name of link",
                                   NULL,
                                   G_PARAM_READWRITE));

  g_object_class_install_property (obj_class,
                                   PROP_EXEC,
                                   g_param_spec_string ("link-exec",
                                   "Link Exec",
                                   "Exec of link",
                                   NULL,
                                   G_PARAM_READWRITE));

  g_object_class_install_property (obj_class,
                                   PROP_ICON,
                                   g_param_spec_string ("link-icon",
                                   "Link Icon",
                                   "Icon of link",
                                   NULL,
                                   G_PARAM_READWRITE));


  g_type_class_add_private (obj_class, sizeof (LauncherLinkEditorPrivate));
}

      
static void
launcher_link_editor_init (LauncherLinkEditor *self)
{
  LauncherLinkEditorPrivate *priv;
  GtkWidget *window = GTK_WIDGET (self);
  GtkWidget *align, *vbox, *hbox;
  GtkWidget *label, *combo, *entry, *icon;
  GtkSizeGroup *group;
  gint active_cat = 0;
  
  priv = self->priv = LAUNCHER_LINK_EDITOR_GET_PRIVATE (self);

  priv->icon_change_lock = FALSE;

  group = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);

  align = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
  gtk_alignment_set_padding (GTK_ALIGNMENT (align), 12, 0, 12, 12);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->vbox), align, 
                      0, TRUE, TRUE);
  
  g_object_set (GTK_DIALOG(window)->vbox,  "spacing", 8, NULL);  
 
  vbox = gtk_vbox_new (FALSE, 8);
  gtk_container_add (GTK_CONTAINER (align), vbox); 

  hbox = gtk_hbox_new (FALSE, 8);
  
  label = gtk_label_new (_("Category:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_size_group_add_widget (group, label);

  combo = priv->combo = gtk_combo_box_new_text ();
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo), active_cat);
  gtk_widget_set_size_request (combo, 300, -1);

  hbox = gtk_hbox_new (FALSE, 8);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, 0, TRUE, TRUE);
  
  label = gtk_label_new (_("Name:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (hbox), label, 0, FALSE, FALSE);
  gtk_size_group_add_widget (group, label);

  entry = priv->name_entry = gtk_entry_new ();
  gtk_widget_set_size_request (entry, 300, -1);
  gtk_box_pack_start (GTK_BOX (hbox), entry, 0, TRUE, TRUE);
  gtk_widget_grab_focus (entry);

  hbox = gtk_hbox_new (FALSE, 8);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, 0, TRUE, TRUE);
  
  label = gtk_label_new (_("Location:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (hbox), label, 0, FALSE, FALSE);
  gtk_size_group_add_widget (group, label);

  entry = priv->loc_entry = gtk_entry_new ();
  gtk_widget_set_size_request (entry, 300, -1);
  gtk_box_pack_start (GTK_BOX (hbox), entry, 0, TRUE, TRUE);

  hbox = gtk_hbox_new (FALSE, 8);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, 0, TRUE, TRUE);
  
  label = gtk_label_new (_("Icon:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (hbox), label, 0, FALSE, FALSE);
  gtk_size_group_add_widget (group, label);

  icon = priv->icon = gnome_icon_entry_new ("1", _("Choose Favorite Icon"));
  set_icon_path (GNOME_ICON_ENTRY (icon), "applications-internet");
  gtk_box_pack_start (GTK_BOX (hbox), icon, 0, TRUE, TRUE);
  
  gtk_widget_show_all (align);
}

GtkWidget *
launcher_link_editor_new (void)

{
  GtkWidget *link_editor = NULL;

  link_editor = g_object_new (LAUNCHER_TYPE_LINK_EDITOR, 
                              "window-position", GTK_WIN_POS_CENTER_ALWAYS,
                              "type", GTK_WINDOW_TOPLEVEL,
                              "decorated", TRUE,
                              NULL);

  return link_editor;
}
