/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "nl-favorite-uri.h"

#include <glib.h>
#include <gio/gio.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <clutk/clutk.h>
#include <clutter/clutter.h>
#include <clutter-gtk/clutter-gtk.h>
#include <launcher/launcher.h>

G_DEFINE_TYPE (NlFavoriteUri, nl_favorite_uri,
               NL_TYPE_FAVORITE);

#define NL_FAVORITE_URI_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE \
        ((obj), NL_TYPE_FAVORITE_URI, NlFavoriteUriPrivate))

#define THUMBNAIL_DIR "netbook-launcher/cache"

struct _NlFavoriteUriPrivate
{
  LauncherFavorites *favorites;
  gchar         *uid;
  gchar         *uri;
  gchar         *name;
  
  ClutterActor *button;
  ClutterActor *image;
};

enum
{
  PROP_0,
  PROP_UID
};

typedef struct
{
  NlFavoriteUri *self;
  gchar         *thumbfile;
  gchar         *uri;
  gchar         *name;
  GdkPixbuf     *pixbuf;

} UriThumbnailData;

/* Forwards */
static void          on_clicked                   (ClutterActor       *button,
                                                   NlFavoriteUri      *self);
static const gchar * get_name                     (NlFavorite         *self);
static GdkPixbuf   * get_icon                     (NlFavorite         *self);
static gboolean      nl_favorite_uri_motion_event (ClutterActor       *actor,
                                                   ClutterMotionEvent *event);
static void          on_show_context_menu         (ClutterActor       *button,
                                                   guint32             event_t,
                                                   NlFavoriteUri      *self);
static void          removed                      (NlFavorite         *self);
static gboolean      on_activated                 (ClutterActor       *actor,
                                                   ClutterEvent       *event,
                                                   NlFavorite         *self);
static gpointer      create_web_thumbnail         (UriThumbnailData   *data);

static NlFavoriteViewType get_view_type           (NlFavorite         *self);

/* GObject stuff */
static void
set_property (GObject      *object,
              guint         prop_id,
              const GValue *value,
              GParamSpec   *pspec)
{
  NlFavoriteUriPrivate *priv;

  g_return_if_fail (NL_IS_FAVORITE_URI (object));
  priv = NL_FAVORITE_URI_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_UID:
      if (priv->uid)
        g_free (priv->uid);
      priv->uid = g_value_dup_string (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
get_property (GObject      *object,
              guint         prop_id,
              GValue       *value,
              GParamSpec   *pspec)
{
  NlFavoriteUriPrivate *priv;

  g_return_if_fail (NL_IS_FAVORITE_URI (object));
  priv = NL_FAVORITE_URI_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_UID:
      g_value_set_string (value, priv->uid);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
nl_favorite_uri_finalize (GObject *object)
{
  NlFavoriteUriPrivate *priv;

  priv = NL_FAVORITE_URI_GET_PRIVATE (object);

  if (priv->uid)
    {
      g_free (priv->uid);
      priv->uid = NULL;
    }
  if (priv->name)
    {
      g_free (priv->name);
      priv->name = NULL;
    }

  G_OBJECT_CLASS (nl_favorite_uri_parent_class)->finalize (object);
}

static gchar *
make_thumbnail_hash (const gchar *name)
{
  gchar *temp;
  gchar *ret;

  temp = g_compute_checksum_for_string (G_CHECKSUM_MD5, name, -1);
  ret = g_strconcat (temp, ".png", NULL);

  g_free (temp);
  return ret;
}

static void
nl_favorite_uri_constructed (GObject *object)
{
  NlFavoriteUriPrivate *priv;
  NlPixbufCache *cache = nl_pixbuf_cache_get_default ();
  GdkPixbuf     *pixbuf;
  gchar         *name;
  gint           pixbuf_size = 64;

  priv = NL_FAVORITE_URI_GET_PRIVATE (object);

  priv->favorites = launcher_favorites_get_default ();

  priv->uri = launcher_favorites_get_string (priv->favorites, priv->uid, "uri");

  if (g_str_has_prefix (priv->uri, "http"))
    {
      gchar *thumbhash;
      gchar *thumbfile;
      gchar *escaped;

      priv->name = launcher_favorites_get_string (priv->favorites,
                                                  priv->uid,
                                                  "name");
      escaped = g_markup_escape_text (priv->name, -1);
      name = g_strdup_printf ("%s", escaped);
      thumbhash = make_thumbnail_hash (name);
      thumbfile = g_build_filename (g_get_user_config_dir (),
                                    THUMBNAIL_DIR,
                                    thumbhash,
                                    NULL);
      pixbuf = gdk_pixbuf_new_from_file (thumbfile, NULL);
      if (!pixbuf)
        {
          UriThumbnailData *data;
          GError           *error = NULL;

          data = g_slice_new0 (UriThumbnailData);
          data->self = NL_FAVORITE_URI (object);
          data->uri = g_strdup (priv->uri);
          data->thumbfile = g_strdup (thumbfile);
          data->pixbuf = NULL;

          g_thread_create ((GThreadFunc)create_web_thumbnail,
                           data,
                           FALSE,
                           &error);
          if (error)
            {
              g_warning ("Unable to create thread for website thumbnail "
                         "generation %s", error->message);

              g_free (data->uri);
              g_free (data->thumbfile);

              g_slice_free (UriThumbnailData, data);
            }
          pixbuf = nl_pixbuf_cache_icon_for_name (cache, "text-html", 64);
        }

      pixbuf_size = 96;
      g_free (thumbhash);
      g_free (thumbfile);
    }
  else
    {
      GFile     *file;
      GFileInfo *info;
      GIcon     *icon;
      GError    *error = NULL;
      gchar     *escaped;

      file = g_file_new_for_uri (priv->uri);
      if (!file)
        {
          g_warning ("Unable to open %s, cannot display favorite",priv->uri);
          return;
        }
      
      info = g_file_query_info (file, "standard::*", 0, NULL, &error);
      if (error || !info)
        {
          g_warning ("Unable to read uri attributes for %s: %s",
                     priv->uri,
                     error ? error->message : "Unknown");
          if (error)
            g_error_free (error);
          return;
        }

      priv->name = g_strdup (g_file_info_get_display_name (info));
      escaped = g_markup_escape_text (priv->name, -1);
      name = g_strdup_printf ("%s", escaped);
      icon = g_file_info_get_icon (info);
      pixbuf = nl_pixbuf_cache_icon_for_gicon (cache, icon, pixbuf_size);

      g_free (escaped);
      g_object_unref (info);
      g_object_unref (file);
    }

  priv->button = nl_icon_tile_new (name, NULL, pixbuf);
  clutter_container_add_actor (CLUTTER_CONTAINER (object), priv->button);
  clutter_actor_show (priv->button);

  priv->image = (ClutterActor*)ctk_button_get_image (CTK_BUTTON (priv->button));
  ctk_image_set_size (CTK_IMAGE (priv->image), pixbuf_size);
  ctk_image_set_from_pixbuf (CTK_IMAGE (priv->image), pixbuf);
#if 0
  priv->button = ctk_button_new (CTK_ORIENTATION_VERTICAL);
  ctk_button_set_label (CTK_BUTTON (priv->button), name);
  clutter_text_set_markup (CLUTTER_TEXT (ctk_button_get_text
                                         (CTK_BUTTON (priv->button))), name);
  clutter_container_add_actor (CLUTTER_CONTAINER (object), priv->button);
  clutter_actor_show (priv->button);

  priv->image = (ClutterActor*)ctk_button_get_image (CTK_BUTTON (priv->button));
  ctk_image_set_size (CTK_IMAGE (priv->image), pixbuf_size);
  ctk_image_set_from_pixbuf (CTK_IMAGE (priv->image), pixbuf);
#endif
  g_signal_connect (priv->button, "clicked", G_CALLBACK (on_clicked), object);
  g_signal_connect (object, "button-press-event",
                    G_CALLBACK (on_activated),NULL);
  g_signal_connect (priv->button, "show-context-menu",
                    G_CALLBACK (on_show_context_menu), object);

  g_free (name);
  if (pixbuf)
    g_object_unref (pixbuf);
  g_object_unref (cache);
}

static void
nl_favorite_uri_class_init (NlFavoriteUriClass *klass)
{
  GObjectClass      *obj_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *act_class = CLUTTER_ACTOR_CLASS (klass);
  NlFavoriteClass   *fav_class = NL_FAVORITE_CLASS (klass);
  GParamSpec        *pspec;

  obj_class->finalize     = nl_favorite_uri_finalize;
  obj_class->constructed  = nl_favorite_uri_constructed;
  obj_class->set_property = set_property;
  obj_class->get_property = get_property;
  
  act_class->motion_event = nl_favorite_uri_motion_event;

  fav_class->get_name = get_name;
  fav_class->get_icon = get_icon;
  fav_class->removed  = removed;
  fav_class->get_view_type = get_view_type;

  pspec = g_param_spec_string ("uid", "uid", "uid",  "",
                               G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_UID, pspec);

  g_type_class_add_private (obj_class, sizeof (NlFavoriteUriPrivate));
}

static void
nl_favorite_uri_init (NlFavoriteUri *self)
{
  NlFavoriteUriPrivate *priv;

  priv = self->priv = NL_FAVORITE_URI_GET_PRIVATE (self);

  clutter_actor_set_reactive (CLUTTER_ACTOR (self), TRUE);
}

/*
 * Public methods
 */
NlFavorite *
nl_favorite_uri_new (const gchar *uid)

{
  NlFavorite *favorite_uri;

  favorite_uri = g_object_new (NL_TYPE_FAVORITE_URI,
                               "uid", uid,
                               NULL);

  return favorite_uri;
}

/*
 * Private methods
 */
static gboolean
load_thumbnail (UriThumbnailData *data)
{
  if (!data)
    return FALSE;
  
  
  if (G_IS_OBJECT (data->self) && data->pixbuf)
    {
      NlFavoriteUriPrivate *priv = data->self->priv;

      ctk_image_set_from_pixbuf (CTK_IMAGE (priv->image), data->pixbuf);
    }

  g_free (data->uri);
  g_free (data->thumbfile);
  if (data->pixbuf)
    g_object_unref (data->pixbuf);
  g_slice_free (UriThumbnailData, data);

  return FALSE;
}

static gpointer
create_web_thumbnail (UriThumbnailData   *data)
{
  gchar *cmd;

  cmd = g_strdup_printf ("gnome-web-photo --mode=thumbnail \"%s\" %s",
                         data->uri,
                         data->thumbfile);

  g_print ("cmd: %s\n", cmd);
  g_spawn_command_line_sync (cmd, NULL, NULL, NULL, NULL);

  data->pixbuf = gdk_pixbuf_new_from_file (data->thumbfile, NULL);

  g_idle_add ((GSourceFunc)load_thumbnail, data);

  return NULL;
}

static void
removed (NlFavorite *self)
{
  NlFavoriteUriPrivate *priv;

  g_return_if_fail (NL_IS_FAVORITE (self));
  priv = NL_FAVORITE_URI (self)->priv;

  launcher_favorites_remove_favorite (priv->favorites, priv->uid);
}

void
_activate (NlFavoriteUri *self)
{
  g_app_info_launch_default_for_uri (self->priv->uri, NULL, NULL);
}

static void
on_clicked (ClutterActor *button, NlFavoriteUri *self)
{
  g_return_if_fail (NL_IS_FAVORITE_URI (self));
  _activate (self);
}

static void
on_user_removed (GtkMenuItem *item, NlFavoriteUri *self)
{
  g_return_if_fail (NL_IS_FAVORITE_URI (self));
  g_signal_emit_by_name (self, "remove-me");
}

static void
on_show_context_menu (ClutterActor  *button,
                      guint32        event_time,
                      NlFavoriteUri *self)
{
  GtkWidget *menu, *item;

  g_return_if_fail (NL_IS_FAVORITE_URI (self));

  menu = gtk_menu_new ();

  item = gtk_image_menu_item_new_from_stock (GTK_STOCK_OPEN, NULL);
  gtk_widget_show (item);
  g_signal_connect (item, "activate", G_CALLBACK (on_clicked), self);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

  item = gtk_image_menu_item_new_from_stock (GTK_STOCK_REMOVE, NULL);
  gtk_widget_show (item);
  g_signal_connect (item, "activate", G_CALLBACK (on_user_removed), self);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

  gtk_menu_popup (GTK_MENU (menu),
                  NULL, NULL,
                  NULL, NULL,
                  3, event_time);

}

static const gchar *
get_name (NlFavorite *self)
{
  NlFavoriteUriPrivate *priv;

  g_return_val_if_fail (NL_IS_FAVORITE_URI (self), NULL);
  priv = NL_FAVORITE_URI (self)->priv;

  return priv->name;
}

static GdkPixbuf *
get_icon (NlFavorite   *self)
{
  NlFavoriteUriPrivate *priv;

  g_return_val_if_fail (NL_IS_FAVORITE_URI (self), NULL);
  priv = NL_FAVORITE_URI (self)->priv;

  return ctk_image_get_pixbuf (CTK_IMAGE (priv->image));
}

static NlFavoriteViewType
get_view_type (NlFavorite *self)
{
  return NL_FAVORITE_VIEW_FIXED;
}

static gboolean
nl_favorite_uri_motion_event (ClutterActor       *actor,
                              ClutterMotionEvent *event)
{
  g_return_val_if_fail (NL_IS_FAVORITE_URI (actor), FALSE);

  if (event->modifier_state & CLUTTER_BUTTON1_MASK)
    {
      g_signal_emit_by_name (actor, "begin-drag-move", event);
      return TRUE;
    }
  return FALSE;
}

static gboolean
on_activated (ClutterActor *actor,
              ClutterEvent *event,
              NlFavorite *self)
{
  g_signal_emit_by_name (actor, "active");

  return FALSE;
}
