/*
 * Copyright (C) 2009 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib-object.h>

#include "nl-shell.h"

enum
{
  FAVORITE_ADDED,
  FAVORITE_REMOVED,

  LAST_SIGNAL
};

static guint32 _shell_signals[LAST_SIGNAL] = { 0, };

static void
nl_shell_base_init (gpointer g_iface)
{
  static gboolean initialised = FALSE;

  if (!initialised)
    {
      GType iface_type = G_TYPE_FROM_INTERFACE (g_iface);

      initialised = TRUE;

      _shell_signals[FAVORITE_ADDED] =
        g_signal_new ("favorite-added",
                      iface_type,
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (NlShellIface, favorite_added),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__STRING,
                      G_TYPE_NONE, 1, G_TYPE_STRING);

      _shell_signals[FAVORITE_REMOVED] =
        g_signal_new ("favorite-removed",
                      iface_type,
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (NlShellIface, favorite_removed),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__STRING,
                      G_TYPE_NONE, 1, G_TYPE_STRING);
    }
}

GType
nl_shell_get_type (void)
{
  static GType shell_type = 0;

  if (G_UNLIKELY (!shell_type))
    {
      GTypeInfo shell_info =
      {
        sizeof (NlShellIface),
        nl_shell_base_init,
        NULL, /* iface_finalize */
      };

      shell_type = g_type_register_static (G_TYPE_INTERFACE,
                                           "NlShell",
                                           &shell_info, 0);
    }

  return shell_type;
}

ClutterActor *
nl_shell_get_stage (NlShell *shell)
{
  g_return_val_if_fail (NL_IS_SHELL (shell), NULL);

  return NL_SHELL_GET_IFACE (shell)->get_stage (shell);
}

NlContentView *
nl_shell_get_content_view (NlShell *shell)
{
  g_return_val_if_fail (NL_IS_SHELL (shell), NULL);

  return NL_SHELL_GET_IFACE (shell)->get_content_view (shell);
}

NlSidebar *
nl_shell_get_sidebar (NlShell *shell)
{
  g_return_val_if_fail (NL_IS_SHELL (shell), NULL);

  return NL_SHELL_GET_IFACE (shell)->get_sidebar (shell);
}

GSList *
nl_shell_get_favorites_loaders (NlShell *shell)
{
  g_return_val_if_fail (NL_IS_SHELL (shell), NULL);

  return NL_SHELL_GET_IFACE (shell)->get_favorites_loaders (shell);
}

void
nl_shell_add_places_source (NlShell *shell,
                            ClutterActor  *source)
{
  g_return_if_fail (NL_IS_SHELL (shell));
  g_return_if_fail (CLUTTER_IS_ACTOR (source));

  NL_SHELL_GET_IFACE (shell)->add_places_source (shell, source);
}

void
nl_shell_add_favorites_loader (NlShell           *shell,
                               NlFavoritesLoader *loader)
{
  g_return_if_fail (NL_IS_SHELL (shell));
  g_return_if_fail (NL_IS_FAVORITES_LOADER (loader));

  NL_SHELL_GET_IFACE (shell)->add_favorites_loader (shell, loader);
}

void
nl_shell_show_favorites (NlShell *shell)
{
  g_return_if_fail (NL_IS_SHELL (shell));

  NL_SHELL_GET_IFACE (shell)->show_favorites (shell);
}

GtkWidget *
nl_shell_get_toplevel (NlShell *shell)
{
  g_return_val_if_fail (NL_IS_SHELL (shell), NULL);

  return NL_SHELL_GET_IFACE (shell)->get_toplevel (shell);
}

void
nl_shell_add_favorite (NlShell           *self,
                       const gchar       *uid)
{
  g_return_if_fail (NL_IS_SHELL (self));
  g_return_if_fail (uid);

  g_signal_emit (self, _shell_signals[FAVORITE_ADDED], 0, uid);
}

void
nl_shell_remove_favorite (NlShell           *self,
                          const gchar       *uid)
{
  g_return_if_fail (NL_IS_SHELL (self));
  g_return_if_fail (uid);

  g_signal_emit (self, _shell_signals[FAVORITE_REMOVED], 0, uid);
}

void
nl_shell_begin_favorite_drag   (NlShell           *shell,
                                GObject           *origin,
                                ClutterActor      *clone,
                                NlShellGetFavoriteDataFunc dfunc,
                                NlShellDragFinishedFunc ffunc)
{
  g_return_if_fail (NL_IS_SHELL (shell));
  g_return_if_fail (G_IS_OBJECT (origin));
  g_return_if_fail (CLUTTER_IS_ACTOR (clone));
  g_return_if_fail (dfunc);

  NL_SHELL_GET_IFACE (shell)->begin_favorite_drag (shell, origin, clone,
      dfunc, ffunc);
}

