/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

#include <glib/gi18n.h>
#include "nmn-item.h"
#include "nmn-icon-cache.h"

G_DEFINE_TYPE (NmnItem, nmn_item, GTK_TYPE_EVENT_BOX)

enum {
    CONNECT_REQUESTED,
    DISCONNECT_REQUESTED,
    REMOVE_REQUESTED,
    DELETE,
    PRIORITY_CHANGED,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL];

#define NMN_ITEM_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), NMN_TYPE_ITEM, NmnItemPrivate))

typedef struct {
    GtkBox *vbox; /* child of self */
    GtkBox *hbox; /* child of vbox */

    GtkWidget *icon;
    GtkLabel *name;
    GtkWidget *status_separator;
    GtkLabel *status_label;
    GtkLabel *security;
    GtkWidget *advanced;
    GtkButton *connect;
    GtkWidget *remove;
    NmnItemStatus status;
} NmnItemPrivate;

GtkWidget *
nmn_item_new (void)
{
    return GTK_WIDGET (g_object_new (NMN_TYPE_ITEM, NULL));
}

void
nmn_item_set_name (NmnItem *item,
                   const char *name)
{
    NmnItemPrivate *priv;
    char *s;

    g_return_if_fail (NMN_IS_ITEM (item));
    g_return_if_fail (name != NULL);

    priv = NMN_ITEM_GET_PRIVATE (item);

    s = g_strdup_printf ("<big><b>%s</b></big>", name);
    gtk_label_set_markup (GTK_LABEL (priv->name), s);
    g_free (s);
}

void
nmn_item_set_status_visible (NmnItem *item,
                             gboolean visible)
{
    NmnItemPrivate *priv;

    g_return_if_fail (NMN_IS_ITEM (item));

    priv = NMN_ITEM_GET_PRIVATE (item);
    if (visible) {
        gtk_widget_show (priv->status_separator);
        gtk_widget_show (GTK_WIDGET (priv->status_label));
    } else {
        gtk_widget_hide (priv->status_separator);
        gtk_widget_hide (GTK_WIDGET (priv->status_label));
    }
}

void
nmn_item_set_status (NmnItem *item,
                     NmnItemStatus status)
{
    NmnItemPrivate *priv;

    g_return_if_fail (NMN_IS_ITEM (item));

    priv = NMN_ITEM_GET_PRIVATE (item);
    if (priv->status != status) {
        const char *status_msg;
        char *str;

        priv->status = status;

        switch (status) {
        case NMN_ITEM_STATUS_DISCONNECTED:
            status_msg = _("Disconnected");
            gtk_button_set_label (priv->connect, _("Connect"));
            break;
        case NMN_ITEM_STATUS_CONNECTING:
            status_msg = _("Connecting...");
            gtk_button_set_label (priv->connect, _("Disconnect"));
            break;
        case NMN_ITEM_STATUS_CONNECTED:
            status_msg = _("Connected");
            gtk_button_set_label (priv->connect, _("Disconnect"));
            break;
        default:
            g_assert_not_reached ();
        }

        str = g_strconcat ("<big><b>", status_msg, "</b></big>", NULL);
        gtk_label_set_markup (priv->status_label, str);
        g_free (str);

        nmn_item_set_status_visible (item, TRUE);
    }
}

NmnItemStatus
nmn_item_get_status (NmnItem *item)
{
    g_return_val_if_fail (NMN_IS_ITEM (item), NMN_ITEM_STATUS_DISCONNECTED);

    return NMN_ITEM_GET_PRIVATE (item)->status;
}

void
nmn_item_set_icon (NmnItem *item,
                   const char *icon_name)
{
    GtkImage *image;
    GdkPixbuf *pixbuf;

    g_return_if_fail (NMN_IS_ITEM (item));
    g_return_if_fail (icon_name != NULL);

    pixbuf = nmn_icon_cache_get (icon_name);
    if (!pixbuf)
        return;

    image = GTK_IMAGE (NMN_ITEM_GET_PRIVATE (item)->icon);

    if (gtk_image_get_storage_type (image) == GTK_IMAGE_PIXBUF && gtk_image_get_pixbuf (image) == pixbuf)
        return;

    gtk_image_set_from_pixbuf (image, pixbuf);
}

void
nmn_item_set_security (NmnItem *item,
                       const char *security_string)
{
    g_return_if_fail (NMN_IS_ITEM (item));

    gtk_label_set_text (GTK_LABEL (NMN_ITEM_GET_PRIVATE (item)->security),
                        security_string ? security_string : "");
}

static void
nmn_item_delete (NmnItem *self)
{
    g_signal_emit (self, signals[DELETE], 0);
}

void
nmn_item_set_delete_visible (NmnItem *item,
                             gboolean visible)
{
    NmnItemPrivate *priv = NMN_ITEM_GET_PRIVATE (item);

    g_return_if_fail (NMN_IS_ITEM (item));

    if (visible && !priv->remove) {
        priv->remove = gtk_button_new ();
        gtk_widget_set_tooltip_text (priv->remove, "Remove connection");
        gtk_container_add (GTK_CONTAINER (priv->remove),
                           gtk_image_new_from_stock (GTK_STOCK_DELETE, GTK_ICON_SIZE_BUTTON));

        gtk_box_pack_end (priv->hbox, priv->remove, FALSE, FALSE, 0);
        gtk_widget_show_all (priv->remove);
        g_signal_connect_swapped (priv->remove, "clicked",
                                  G_CALLBACK (nmn_item_delete),
                                  item);
    } else if (!visible && priv->remove) {
        gtk_container_remove (GTK_CONTAINER (priv->hbox), priv->remove);
        priv->remove = NULL;
    }
}

void
nmn_item_connect_request (NmnItem *self)
{
    g_return_if_fail (NMN_IS_ITEM (self));

    NMN_ITEM_GET_CLASS (self)->connect (self);
    g_signal_emit (self, signals[CONNECT_REQUESTED], 0);
}

void
nmn_item_disconnect_request (NmnItem *self)
{
    g_return_if_fail (NMN_IS_ITEM (self));

    NMN_ITEM_GET_CLASS (self)->disconnect (self);
    g_signal_emit (self, signals[DISCONNECT_REQUESTED], 0);
}

void
nmn_item_remove_request (NmnItem *self)
{
    g_return_if_fail (NMN_IS_ITEM (self));

    g_signal_emit (self, signals[REMOVE_REQUESTED], 0);
}

guint
nmn_item_get_priority (NmnItem *self)
{
    g_return_val_if_fail (NMN_IS_ITEM (self), 0);

    if (NMN_ITEM_GET_CLASS (self)->get_priority)
        return NMN_ITEM_GET_CLASS (self)->get_priority (self);

    return 0;
}

void
nmn_item_priority_changed (NmnItem *self)
{
    g_return_if_fail (NMN_IS_ITEM (self));

    g_signal_emit (self, signals[PRIORITY_CHANGED], 0);
}

static void
advanced_expanded (GtkExpander *expander,
                   GParamSpec *param_spec,
                   gpointer user_data)
{
    NmnItem *self = NMN_ITEM (user_data);
    NmnItemPrivate *priv = NMN_ITEM_GET_PRIVATE (self);
    GtkWidget *child;

    if (gtk_expander_get_expanded (expander)) {
        /* create widgets */

        if (NMN_ITEM_GET_CLASS (self)->create_advanced_information)
            child = NMN_ITEM_GET_CLASS (self)->create_advanced_information (self);
        else
            child = NULL;

        if (child) {
            GtkWidget *alignment;

            alignment = gtk_alignment_new (0, 0, 0, 0);
            gtk_container_add (GTK_CONTAINER (alignment), child);
            gtk_widget_show (alignment);

            gtk_box_pack_end (priv->vbox, alignment, FALSE, FALSE, 0);
            g_object_set_data (G_OBJECT (priv->vbox), "expanded-child", alignment);
        }
    } else {
        /* Hide or destroy widgets */
        child = GTK_WIDGET (g_object_get_data (G_OBJECT (priv->vbox), "expanded-child"));
        if (child) {
            g_object_set_data (G_OBJECT (priv->vbox), "expanded-child", NULL);
            gtk_container_remove (GTK_CONTAINER (priv->vbox), child);
        }
    }
}

static void
nmn_item_connect (NmnItem *item)
{
    g_warning ("Connect not overriden");
}

static void
nmn_item_disconnect (NmnItem *item)
{
    g_warning ("Disconnect not overriden");
}

static void
connect_clicked (GtkButton *button, gpointer user_data)
{
    NmnItem *item = NMN_ITEM (user_data);
    NmnItemPrivate *priv = NMN_ITEM_GET_PRIVATE (item);

    if (priv->status == NMN_ITEM_STATUS_DISCONNECTED)
        nmn_item_connect_request (item);
    else
        nmn_item_disconnect_request (item);
}

static void
nmn_item_init (NmnItem *item)
{
    NmnItemPrivate *priv = NMN_ITEM_GET_PRIVATE (item);
    GtkWidget *vbox;
    GtkWidget *hbox;
    GtkWidget *w;

    priv->vbox = GTK_BOX (gtk_vbox_new (FALSE, 6));
    gtk_container_set_border_width (GTK_CONTAINER (priv->vbox), 6);
    gtk_container_add (GTK_CONTAINER (item), GTK_WIDGET (priv->vbox));

    priv->hbox = GTK_BOX (gtk_hbox_new (FALSE, 6));
    gtk_container_set_border_width (GTK_CONTAINER (priv->hbox), 6);
    gtk_box_pack_start (priv->vbox, GTK_WIDGET (priv->hbox), TRUE, TRUE, 0);

    priv->icon = gtk_image_new ();
    gtk_box_pack_start (priv->hbox, priv->icon, FALSE, FALSE, 0);

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_box_pack_start (priv->hbox, vbox, TRUE, TRUE, 0);

    hbox = gtk_hbox_new (FALSE, 12);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

    w = gtk_label_new ("");
    gtk_label_set_use_markup (GTK_LABEL (w), TRUE);
    gtk_misc_set_alignment (GTK_MISC (w), 0.0, 0.5);
    gtk_box_pack_start (GTK_BOX (hbox), w, FALSE, FALSE, 0);
    priv->name = GTK_LABEL (w);

    w = gtk_label_new (" - ");
    gtk_box_pack_start (GTK_BOX (hbox), w, FALSE, FALSE, 0);
    priv->status_separator = w;

    w = gtk_label_new ("");
    gtk_label_set_use_markup (GTK_LABEL (w), TRUE);
    gtk_misc_set_alignment (GTK_MISC (w), 0.0, 0.5);
    gtk_box_pack_start (GTK_BOX (hbox), w, FALSE, FALSE, 0);
    priv->status_label = GTK_LABEL (w);

    hbox = gtk_hbox_new (FALSE, 12);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

    w = gtk_label_new ("");
    gtk_box_pack_start (GTK_BOX (hbox), w, FALSE, FALSE, 0);
    priv->security = GTK_LABEL (w);

    w = gtk_expander_new (_("Advanced"));
    gtk_box_pack_start (GTK_BOX (hbox), w, FALSE, FALSE, 0);
    priv->advanced = w;
    g_signal_connect (w, "notify::expanded", G_CALLBACK (advanced_expanded), item);

    w = gtk_button_new_with_label (_("Connect"));
    gtk_box_pack_start (GTK_BOX (hbox), w, FALSE, FALSE, 0);
    priv->connect = GTK_BUTTON (w);
    g_signal_connect (w, "clicked", G_CALLBACK (connect_clicked), item);

    priv->remove = NULL;

    gtk_widget_show_all (GTK_WIDGET (priv->vbox));
}

static void
nmn_item_class_init (NmnItemClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);

    g_type_class_add_private (object_class, sizeof (NmnItemPrivate));

    class->connect = nmn_item_connect;
    class->disconnect = nmn_item_disconnect;

    /* Signals */
    signals[CONNECT_REQUESTED] = g_signal_new 
        ("connect-requested",
         G_OBJECT_CLASS_TYPE (class),
         G_SIGNAL_RUN_LAST,
         G_STRUCT_OFFSET (NmnItemClass, connect_requested),
         NULL, NULL,
         g_cclosure_marshal_VOID__VOID,
         G_TYPE_NONE, 0);

    signals[DISCONNECT_REQUESTED] = g_signal_new 
        ("disconnect-requested",
         G_OBJECT_CLASS_TYPE (class),
         G_SIGNAL_RUN_LAST,
         G_STRUCT_OFFSET (NmnItemClass, disconnect_requested),
         NULL, NULL,
         g_cclosure_marshal_VOID__VOID,
         G_TYPE_NONE, 0);

    signals[REMOVE_REQUESTED] = g_signal_new 
        ("remove-requested",
         G_OBJECT_CLASS_TYPE (class),
         G_SIGNAL_RUN_LAST,
         G_STRUCT_OFFSET (NmnItemClass, remove_requested),
         NULL, NULL,
         g_cclosure_marshal_VOID__VOID,
         G_TYPE_NONE, 0);

    signals[DELETE] = g_signal_new 
        ("delete",
         G_OBJECT_CLASS_TYPE (class),
         G_SIGNAL_RUN_LAST,
         G_STRUCT_OFFSET (NmnItemClass, delete),
         NULL, NULL,
         g_cclosure_marshal_VOID__VOID,
         G_TYPE_NONE, 0);

    signals[PRIORITY_CHANGED] = g_signal_new 
        ("priority-changed",
         G_OBJECT_CLASS_TYPE (class),
         G_SIGNAL_RUN_LAST,
         G_STRUCT_OFFSET (NmnItemClass, priority_changed),
         NULL, NULL,
         g_cclosure_marshal_VOID__VOID,
         G_TYPE_NONE, 0);
}
