/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

#include <nm-dbus-settings-system.h>
#include "nmn-nm-data.h"
#include "nma-gconf-settings.h"

G_DEFINE_TYPE (NmnNMData, nmn_nm_data, NM_TYPE_CLIENT)

enum {
    ETHERNET_TOGGLED,
    WIFI_TOGGLED,
    MODEMS_TOGGLED,
    FLIGHT_MODE_TOGGLED,

    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL];

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), NMN_TYPE_NM_DATA, NmnNMDataPrivate))

typedef struct {
    NMSettings *user_settings;
    NMSettings *system_settings;

    gboolean ethernet_active;
    gboolean wifi_active;
    gboolean modems_active;
    gboolean flight_mode_active;

    gboolean disposed;
} NmnNMDataPrivate;

NmnNMData *
nmn_nm_data_new (DBusGConnection *bus)
{
    return NMN_NM_DATA (g_object_new (NMN_TYPE_NM_DATA,
                                      NM_OBJECT_DBUS_CONNECTION, bus,
                                      NM_OBJECT_DBUS_PATH, NM_DBUS_PATH,
                                      NULL));
}

NMSettings *
nmn_nm_data_get_user_settings (NmnNMData *self)
{
    g_return_val_if_fail (NMN_IS_NM_DATA (self), NULL);

    return GET_PRIVATE (self)->user_settings;
}

NMSettings *
nmn_nm_data_get_system_settings (NmnNMData *self)
{
    g_return_val_if_fail (NMN_IS_NM_DATA (self), NULL);

    return GET_PRIVATE (self)->system_settings;
}

gboolean
nmn_nm_data_ethernet_get_active (NmnNMData *self)
{
    g_return_val_if_fail (NMN_IS_NM_DATA (self), FALSE);

    return GET_PRIVATE (self)->ethernet_active;
}

void
nmn_nm_data_ethernet_toggled (NmnNMData *self,
                              gboolean active)
{
    NmnNMDataPrivate *priv;

    g_return_if_fail (NMN_IS_NM_DATA (self));

    priv = GET_PRIVATE (self);
    if (priv->ethernet_active != active) {
        /* FIXME: Save in gconf? */
        priv->ethernet_active = active;
        g_signal_emit (self, signals[ETHERNET_TOGGLED], 0, active);
    }
}

gboolean
nmn_nm_data_wifi_get_active (NmnNMData *self)
{
    g_return_val_if_fail (NMN_IS_NM_DATA (self), FALSE);

    return GET_PRIVATE (self)->wifi_active;
}

void
nmn_nm_data_wifi_toggled (NmnNMData *self,
                          gboolean active)
{
    NmnNMDataPrivate *priv;

    g_return_if_fail (NMN_IS_NM_DATA (self));

    priv = GET_PRIVATE (self);
    if (priv->wifi_active != active) {
        /* FIXME: Save in gconf? */
        priv->wifi_active = active;
        g_signal_emit (self, signals[WIFI_TOGGLED], 0, active);
    }
}

gboolean
nmn_nm_data_modems_get_active (NmnNMData *self)
{
    g_return_val_if_fail (NMN_IS_NM_DATA (self), FALSE);

    return GET_PRIVATE (self)->modems_active;
}

void
nmn_nm_data_modems_toggled (NmnNMData *self,
                            gboolean active)
{
    NmnNMDataPrivate *priv;

    g_return_if_fail (NMN_IS_NM_DATA (self));

    priv = GET_PRIVATE (self);
    if (priv->modems_active != active) {
        /* FIXME: Save in gconf? */
        priv->modems_active = active;
        g_signal_emit (self, signals[MODEMS_TOGGLED], 0, active);
    }
}

gboolean
nmn_nm_data_flight_mode_can_change (NmnNMData *self)
{
    g_return_val_if_fail (NMN_IS_NM_DATA (self), FALSE);

    return nm_client_wireless_hardware_get_enabled (NM_CLIENT (self));
}

gboolean
nmn_nm_data_flight_mode_get_active (NmnNMData *self)
{
    g_return_val_if_fail (NMN_IS_NM_DATA (self), FALSE);

    return GET_PRIVATE (self)->flight_mode_active;
}

static void
flight_mode_toggled_internal (NmnNMData *self, gboolean active)
{
    NmnNMDataPrivate *priv = GET_PRIVATE (self);

    if (priv->flight_mode_active != active) {
        /* FIXME: Save in gconf? */
        priv->flight_mode_active = active;
        g_signal_emit (self, signals[FLIGHT_MODE_TOGGLED], 0, active);
    }
}

void
nmn_nm_data_flight_mode_toggled (NmnNMData *self,
                                 gboolean active)
{
    g_return_if_fail (NMN_IS_NM_DATA (self));

    flight_mode_toggled_internal (self, active);
    nm_client_wireless_set_enabled (NM_CLIENT (self), !active);
}

static void
nm_wireless_state_changed (NMClient *client,
                           GParamSpec *gobject,
                           gpointer user_data)
{
    flight_mode_toggled_internal (NMN_NM_DATA (client),
                                  !nm_client_wireless_get_enabled (client));
}

static void
nmn_nm_data_init (NmnNMData *self)
{
    NmnNMDataPrivate *priv = GET_PRIVATE (self);

    /* FIXME: Load from gconf? */
    priv->ethernet_active = TRUE;
    priv->wifi_active = TRUE;
    priv->modems_active = TRUE;
    priv->flight_mode_active = FALSE;
}

static GObject*
constructor (GType type,
             guint n_construct_params,
             GObjectConstructParam *construct_params)
{
	GObject *object;
    NmnNMDataPrivate *priv;
    DBusGConnection *bus;

	object = G_OBJECT_CLASS (nmn_nm_data_parent_class)->constructor (type,
                                                                     n_construct_params,
                                                                     construct_params);
	if (!object)
		return NULL;

	priv = GET_PRIVATE (object);

    bus = nm_object_get_connection (NM_OBJECT (object));
    priv->user_settings = NM_SETTINGS (nma_gconf_settings_new ());
    priv->system_settings = NM_SETTINGS (nm_dbus_settings_system_new (bus));

    g_signal_connect (object,
                      "notify::" NM_CLIENT_WIRELESS_ENABLED,
                      G_CALLBACK (nm_wireless_state_changed),
                      NULL);

    g_signal_connect (object,
                      "notify::" NM_CLIENT_WIRELESS_HARDWARE_ENABLED,
                      G_CALLBACK (nm_wireless_state_changed),
                      NULL);

    nm_wireless_state_changed (NM_CLIENT (object), NULL, NULL);

    return object;
}

static void
dispose (GObject *object)
{
    NmnNMDataPrivate *priv = GET_PRIVATE (object);

    if (priv->disposed)
        return;

    g_object_unref (priv->user_settings);
    g_object_unref (priv->system_settings);
    priv->disposed = TRUE;

    G_OBJECT_CLASS (nmn_nm_data_parent_class)->dispose (object);
}

static void
nmn_nm_data_class_init (NmnNMDataClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);

    g_type_class_add_private (object_class, sizeof (NmnNMDataPrivate));

    object_class->constructor = constructor;
    object_class->dispose = dispose;

    /* signals */
    signals[ETHERNET_TOGGLED] = g_signal_new 
        ("ethernet-toggled",
         G_OBJECT_CLASS_TYPE (class),
         G_SIGNAL_RUN_LAST,
         G_STRUCT_OFFSET (NmnNMDataClass, ethernet_toggled),
         NULL, NULL,
         g_cclosure_marshal_VOID__BOOLEAN,
         G_TYPE_NONE, 1,
         G_TYPE_BOOLEAN);

    signals[WIFI_TOGGLED] = g_signal_new 
        ("wifi-toggled",
         G_OBJECT_CLASS_TYPE (class),
         G_SIGNAL_RUN_LAST,
         G_STRUCT_OFFSET (NmnNMDataClass, wifi_toggled),
         NULL, NULL,
         g_cclosure_marshal_VOID__BOOLEAN,
         G_TYPE_NONE, 1,
         G_TYPE_BOOLEAN);

    signals[MODEMS_TOGGLED] = g_signal_new 
        ("modems-toggled",
         G_OBJECT_CLASS_TYPE (class),
         G_SIGNAL_RUN_LAST,
         G_STRUCT_OFFSET (NmnNMDataClass, modems_toggled),
         NULL, NULL,
         g_cclosure_marshal_VOID__BOOLEAN,
         G_TYPE_NONE, 1,
         G_TYPE_BOOLEAN);

    signals[FLIGHT_MODE_TOGGLED] = g_signal_new 
        ("flight-mode-toggled",
         G_OBJECT_CLASS_TYPE (class),
         G_SIGNAL_RUN_LAST,
         G_STRUCT_OFFSET (NmnNMDataClass, flight_mode_toggled),
         NULL, NULL,
         g_cclosure_marshal_VOID__BOOLEAN,
         G_TYPE_NONE, 1,
         G_TYPE_BOOLEAN);
}
