/*******************************************************************************
**3456789 123456789 123456789 123456789 123456789 123456789 123456789 123456789
**      10        20        30        40        50        60        70        80
**
** notify-osd
**
** split-button.c - a split-button class for snap-decisions
**
** Copyright 2012 Canonical Ltd.
**
** Authors:
**    Mirco "MacSlow" Mueller <mirco.mueller@canonical.com>
**
** This program is free software: you can redistribute it and/or modify it
** under the terms of the GNU General Public License version 3, as published
** by the Free Software Foundation.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranties of
** MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
** PURPOSE.  See the GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License along
** with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*******************************************************************************/

#include <pango/pango.h>

#include "bubble.h"
#include "split-button.h"

G_DEFINE_TYPE (SplitButton, split_button, G_TYPE_OBJECT);

#define GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), SPLIT_BUTTON_TYPE, SplitButtonPrivate))

struct _SplitButtonPrivate {
	gint                 x;
	gint                 y;
	guint                num_labels;
	gchar**              labels;
	SplitButtonState     state;
	SplitButtonForm      form;
	gint                 hit_label_index;            
	cairo_surface_t*     collapsed_normal_surface;
	cairo_surface_t*     expanded_normal_surface;
	cairo_surface_t*     collapsed_hover_surface;
	cairo_surface_t*     expanded_hover_surface;
	cairo_surface_t*     collapsed_pressed_surface;
	cairo_surface_t*     expanded_pressed_surface;
	cairo_surface_t*     arrow_hover_surface[2];
	cairo_surface_t*     arrow_pressed_surface[2];
	cairo_surface_t*     entry0_hover_surface[2];
	cairo_surface_t*     entry0_pressed_surface[2];
	cairo_surface_t*     entry_hover_surface;
	cairo_surface_t*     entry_pressed_surface;
	SettingsButton*      settings_b;
	SettingsSplitButton* settings_sb;
	gboolean             arrow_hit;
};

/*-- internal API ------------------------------------------------------------*/

static void
split_button_dispose (GObject* gobject)
{
	/* chain up to the parent class */
	G_OBJECT_CLASS (split_button_parent_class)->dispose (gobject);
}

static void
split_button_finalize (GObject* gobject)
{
	SplitButton*        split_button = NULL;
	SplitButtonPrivate* priv         = NULL;

	// sanity checks
	g_assert (gobject);
	split_button = SPLIT_BUTTON (gobject);
	g_assert (split_button);
	g_assert (IS_SPLIT_BUTTON (split_button));
	priv = GET_PRIVATE (split_button);
	g_assert (priv);

	if (priv->labels != NULL)
	{
		guint i = 0;

		for (; i < priv->num_labels; i++)
			g_free (priv->labels[i]);

		g_free (priv->labels);
		priv->num_labels = 0;
		priv->labels = NULL;
	}

	if (priv->collapsed_normal_surface != NULL)
		cairo_surface_destroy (priv->collapsed_normal_surface);

	if (priv->expanded_normal_surface != NULL)
		cairo_surface_destroy (priv->expanded_normal_surface);

	if (priv->collapsed_hover_surface != NULL)
		cairo_surface_destroy (priv->collapsed_hover_surface);

	if (priv->expanded_hover_surface != NULL)
		cairo_surface_destroy (priv->expanded_hover_surface);

	if (priv->collapsed_pressed_surface != NULL)
		cairo_surface_destroy (priv->collapsed_pressed_surface);

	if (priv->expanded_pressed_surface != NULL)
		cairo_surface_destroy (priv->expanded_pressed_surface);

	if (priv->arrow_hover_surface[0] != NULL)
		cairo_surface_destroy (priv->arrow_hover_surface[0]);

	if (priv->arrow_pressed_surface[0] != NULL)
		cairo_surface_destroy (priv->arrow_pressed_surface[0]);

	if (priv->entry0_hover_surface[0] != NULL)
		cairo_surface_destroy (priv->entry0_hover_surface[0]);

	if (priv->entry0_pressed_surface[0] != NULL)
		cairo_surface_destroy (priv->entry0_pressed_surface[0]);

	if (priv->arrow_hover_surface[1] != NULL)
		cairo_surface_destroy (priv->arrow_hover_surface[1]);

	if (priv->arrow_pressed_surface[1] != NULL)
		cairo_surface_destroy (priv->arrow_pressed_surface[1]);

	if (priv->entry0_hover_surface[1] != NULL)
		cairo_surface_destroy (priv->entry0_hover_surface[1]);

	if (priv->entry0_pressed_surface[1] != NULL)
		cairo_surface_destroy (priv->entry0_pressed_surface[1]);

	if (priv->entry_hover_surface != NULL)
		cairo_surface_destroy (priv->entry_hover_surface);

	if (priv->entry_pressed_surface != NULL)
		cairo_surface_destroy (priv->entry_pressed_surface);

	/* chain up to the parent class */
	G_OBJECT_CLASS (split_button_parent_class)->finalize (gobject);
}

static void
split_button_init (SplitButton* self)
{
	/* If you need specific construction properties to complete
	** initialization, delay initialization completion until the
	** property is set. */
}

static void
split_button_get_property (GObject*    gobject,
                           guint       prop,
                           GValue*     value,
                           GParamSpec* spec)
{
	switch (prop)
	{
		default :
			G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop, spec);
		break;
	}
}

static void
split_button_class_init (SplitButtonClass* klass)
{
	GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

	g_type_class_add_private (klass, sizeof (SplitButtonPrivate));

	gobject_class->dispose      = split_button_dispose;
	gobject_class->finalize     = split_button_finalize;
	gobject_class->get_property = split_button_get_property;
}

void
_refresh_split_button_surfaces (SplitButton* self)
{
	g_return_if_fail (self && IS_SPLIT_BUTTON (self));

	SplitButtonPrivate* priv          = GET_PRIVATE (self);
	gint                i             = 0;
	guint               num_labels    = split_button_get_num_actions (self); 
	gint                rows          = num_labels / 2;

	gdouble expanded_height = rows * priv->settings_sb->expanded_height;

	if (priv->collapsed_normal_surface)
		cairo_surface_destroy (priv->collapsed_normal_surface);

	if (priv->expanded_normal_surface)
		cairo_surface_destroy (priv->expanded_normal_surface);

	if (priv->collapsed_hover_surface)
		cairo_surface_destroy (priv->collapsed_hover_surface);

	if (priv->expanded_hover_surface)
		cairo_surface_destroy (priv->expanded_hover_surface);

	if (priv->collapsed_pressed_surface)
		cairo_surface_destroy (priv->collapsed_pressed_surface);

	if (priv->expanded_pressed_surface)
		cairo_surface_destroy (priv->expanded_pressed_surface);

	if (priv->arrow_hover_surface[0])
		cairo_surface_destroy (priv->arrow_hover_surface[0]);

	if (priv->arrow_pressed_surface[0])
		cairo_surface_destroy (priv->arrow_pressed_surface[0]);

	if (priv->entry0_hover_surface[0])
		cairo_surface_destroy (priv->entry0_hover_surface[0]);

	if (priv->entry0_pressed_surface[0])
		cairo_surface_destroy (priv->entry0_pressed_surface[0]);

	if (priv->arrow_hover_surface[1])
		cairo_surface_destroy (priv->arrow_hover_surface[1]);

	if (priv->arrow_pressed_surface[1])
		cairo_surface_destroy (priv->arrow_pressed_surface[1]);

	if (priv->entry0_hover_surface[1])
		cairo_surface_destroy (priv->entry0_hover_surface[1]);

	if (priv->entry0_pressed_surface[1])
		cairo_surface_destroy (priv->entry0_pressed_surface[1]);

	if (priv->entry_hover_surface)
		cairo_surface_destroy (priv->entry_hover_surface);

	if (priv->entry_pressed_surface)
		cairo_surface_destroy (priv->entry_pressed_surface);

	priv->collapsed_normal_surface = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
	                                    priv->settings_sb->contracted_width,
	                                    priv->settings_sb->contracted_height);
	priv->collapsed_hover_surface = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
										priv->settings_sb->contracted_width,
										priv->settings_sb->contracted_height);
	priv->collapsed_pressed_surface = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
										priv->settings_sb->contracted_width,
										priv->settings_sb->contracted_height);
	priv->expanded_normal_surface = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
	                                    priv->settings_sb->expanded_width,
	                                    expanded_height);
	priv->expanded_hover_surface = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
										priv->settings_sb->expanded_width,
										expanded_height);
	priv->expanded_pressed_surface = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
										priv->settings_sb->expanded_width,
										expanded_height);

	priv->arrow_hover_surface[0] = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
										priv->settings_sb->contracted_arrow_button_size,
										priv->settings_sb->contracted_arrow_button_size);

	priv->arrow_pressed_surface[0] = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
										priv->settings_sb->contracted_arrow_button_size,
										priv->settings_sb->contracted_arrow_button_size);

	priv->entry0_hover_surface[0] = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
	                                    priv->settings_sb->contracted_width,
	                                    priv->settings_sb->contracted_height);

	priv->entry0_pressed_surface[0] = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
	                                    priv->settings_sb->contracted_width,
	                                    priv->settings_sb->contracted_height);

	priv->arrow_hover_surface[1] = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
										priv->settings_sb->expanded_arrow_button_size,
										priv->settings_sb->expanded_arrow_button_size);

	priv->arrow_pressed_surface[1] = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
										priv->settings_sb->expanded_arrow_button_size,
										priv->settings_sb->expanded_arrow_button_size);

	priv->entry0_hover_surface[1] = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
	                                    priv->settings_sb->expanded_width,
	                                    priv->settings_sb->expanded_height);

	priv->entry0_pressed_surface[1] = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
	                                    priv->settings_sb->expanded_width,
	                                    priv->settings_sb->expanded_height);

	priv->entry_hover_surface = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
	                                    priv->settings_sb->expanded_width,
	                                    priv->settings_sb->expanded_height);

	priv->entry_pressed_surface = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
	                                    priv->settings_sb->expanded_width,
	                                    priv->settings_sb->expanded_height);

	PangoWeight font_weight = PANGO_WEIGHT_NORMAL;

	setup_font_weight (&font_weight, priv->settings_b->text_weight);

	gdouble text_normal_color[4]    = {0.0, 0.0, 0.0, 0.0};
	gdouble text_hover_color[4]     = {0.0, 0.0, 0.0, 0.0};
	gdouble text_pressed_color[4]   = {0.0, 0.0, 0.0, 0.0};
	gdouble fill_normal_color[4]    = {0.0, 0.0, 0.0, 0.0};
	gdouble fill_hover_color[4]     = {0.0, 0.0, 0.0, 0.0};
	gdouble fill_pressed_color[4]   = {0.0, 0.0, 0.0, 0.0};
	gdouble stroke_normal_color[4]  = {0.0, 0.0, 0.0, 0.0};
	gdouble stroke_hover_color[4]   = {0.0, 0.0, 0.0, 0.0};
	gdouble stroke_pressed_color[4] = {0.0, 0.0, 0.0, 0.0};
	gdouble arrow_color[4]          = {0.0, 0.0, 0.0, 0.0};
	gdouble separator_color[4]      = {0.0, 0.0, 0.0, 0.0};

	gboolean grid_align = priv->settings_b->line_width == 1.0;

	setup_color (text_normal_color,
	             &priv->settings_b->text_colors[0],
	             priv->settings_b->text_opacities[0]);

	setup_color (text_hover_color,
	             &priv->settings_b->text_colors[1],
	             priv->settings_b->text_opacities[1]);

	setup_color (text_pressed_color,
	             &priv->settings_b->text_colors[2],
	             priv->settings_b->text_opacities[2]);

	setup_color (fill_normal_color,
	             &priv->settings_b->fill_colors[0],
	             priv->settings_b->fill_opacities[0]);

	setup_color (fill_hover_color,
	             &priv->settings_b->fill_colors[1],
	             priv->settings_b->fill_opacities[1]);

	setup_color (fill_pressed_color,
	             &priv->settings_b->fill_colors[2],
	             priv->settings_b->fill_opacities[2]);

	setup_color (stroke_normal_color,
	             &priv->settings_b->outline_colors[0],
	             priv->settings_b->outline_opacities[0]);

	setup_color (stroke_hover_color,
	             &priv->settings_b->outline_colors[1],
	             priv->settings_b->outline_opacities[1]);

	setup_color (stroke_pressed_color,
	             &priv->settings_b->outline_colors[2],
	             priv->settings_b->outline_opacities[2]);

	setup_color (arrow_color,
	             &priv->settings_sb->arrow_color,
	             priv->settings_sb->arrow_opacity);

	setup_color (separator_color,
	             &priv->settings_sb->separator_color,
	             priv->settings_sb->separator_opacity);

	cairo_t* cr = NULL;

	/* collapsed normal-state surface - fill */
	cr = cairo_create (priv->collapsed_normal_surface);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, FALSE),
	                 align (0.0, FALSE),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->contracted_width, FALSE),
	                 align (priv->settings_sb->contracted_height, FALSE));
	cairo_set_source_rgba (cr,
	                       fill_normal_color[0],
	                       fill_normal_color[1],
	                       fill_normal_color[2],
	                       fill_normal_color[3]);
	cairo_fill (cr);

	/* collapsed normal-state surface - outline */
	cairo_set_line_width (cr, priv->settings_b->line_width);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, grid_align),
	                 align (0.0, grid_align),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->contracted_width, grid_align) - 3.0 * align (0.0, grid_align),
	                 align (priv->settings_sb->contracted_height, grid_align) - 3.0 * align (0.0, grid_align));
	cairo_set_source_rgba (cr,
	                       stroke_normal_color[0],
	                       stroke_normal_color[1],
	                       stroke_normal_color[2],
	                       stroke_normal_color[3]);
	cairo_stroke (cr);

	/* collapsed normal-state surface - text */
	gint x = priv->settings_sb->contracted_arrow_button_size;
	gint y = 0;
	if (priv->num_labels != 0)
		draw_button_label (cr,
		                   split_button_get_action_by_index (self, 1),
		                   x,
		                   y,
		                   priv->settings_sb->contracted_width - priv->settings_sb->contracted_arrow_button_size,
		                   priv->settings_sb->contracted_height,
		                   priv->settings_b->text_margin_horiz,
		                   priv->settings_b->text_family,
		                   priv->settings_b->text_size,
		                   font_weight,
		                   text_normal_color);

	/* collapsed normal-state surface - arrow */
	draw_round_rect (cr,
	                 1.0,
	                 align (priv->settings_b->line_width + priv->settings_sb->contracted_arrow_button_margin, grid_align),
	                 align (priv->settings_b->line_width + priv->settings_sb->contracted_arrow_button_margin, grid_align),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->contracted_arrow_button_size - (3.0 * priv->settings_b->line_width + 2.0 * priv->settings_sb->contracted_arrow_button_margin), FALSE),
	                 align (priv->settings_sb->contracted_arrow_button_size - (3.0 * priv->settings_b->line_width + 2.0 * priv->settings_sb->contracted_arrow_button_margin), FALSE));
	cairo_set_source_rgba (cr,
	                       stroke_normal_color[0],
	                       stroke_normal_color[1],
	                       stroke_normal_color[2],
	                       stroke_normal_color[3]);
	cairo_stroke (cr);
	cairo_set_line_width (cr, priv->settings_sb->arrow_thickness);
	cairo_set_source_rgba (cr,
	                       arrow_color[0],
	                       arrow_color[1],
	                       arrow_color[2],
	                       arrow_color[3]);
	cairo_set_line_cap (cr, CAIRO_LINE_CAP_ROUND);

	cairo_move_to (cr,
	               priv->settings_sb->contracted_arrow_button_size / 2.0 -
	               priv->settings_sb->contracted_arrow_width / 2.0,
	               priv->settings_sb->contracted_arrow_button_size / 2.0 -
	               priv->settings_sb->contracted_arrow_height / 2.0);
	cairo_rel_line_to (cr,
	                   priv->settings_sb->contracted_arrow_width / 2.0,
	                   priv->settings_sb->contracted_arrow_height);
	cairo_rel_line_to (cr,
	                   priv->settings_sb->contracted_arrow_width / 2.0,
	                   -priv->settings_sb->contracted_arrow_height);
	cairo_stroke (cr);
	cairo_destroy (cr);
	cr = NULL;

	/* expanded normal-state surface - fill */
	cr = cairo_create (priv->expanded_normal_surface);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	expanded_height = rows * priv->settings_sb->expanded_height;
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, FALSE),
	                 align (0.0, FALSE),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->expanded_width, FALSE),
	                 align (expanded_height, FALSE));
	cairo_set_source_rgba (cr,
	                       fill_normal_color[0],
	                       fill_normal_color[1],
	                       fill_normal_color[2],
	                       fill_normal_color[3]);
	cairo_fill (cr);

	/* expanded normal-state surface - outline */
	cairo_set_line_width (cr, priv->settings_b->line_width);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, grid_align),
	                 align (0.0, grid_align),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->expanded_width, grid_align) - 3.0 * align (0.0, grid_align),
	                 align (expanded_height, grid_align) - 3.0 * align (0.0, grid_align));
	cairo_set_source_rgba (cr,
	                       stroke_normal_color[0],
	                       stroke_normal_color[1],
	                       stroke_normal_color[2],
	                       stroke_normal_color[3]);
	cairo_stroke (cr);

	/* expanded normal-state surface - separators */
	grid_align = priv->settings_sb->separator_thickness == 1.0;
	cairo_set_line_width (cr, priv->settings_sb->separator_thickness);
	cairo_set_source_rgba (cr,
	                       separator_color[0],
	                       separator_color[1],
	                       separator_color[2],
	                       separator_color[3]);
	for (i = 0; i < rows; i++)
	{
		cairo_move_to (cr,
		               align (priv->settings_sb->separator_margin, grid_align),
		               align ((i + 1) * priv->settings_sb->expanded_height, grid_align));
		cairo_line_to (cr,
		               align (priv->settings_sb->expanded_width -
		               priv->settings_sb->separator_margin, grid_align),
		               align ((i + 1) * priv->settings_sb->expanded_height, grid_align));
	}
	cairo_stroke (cr);

	/* expanded normal-state surface - text */
	x = 0;
	y = 0;
	if (priv->num_labels != 0)
		for (i = 0; i < priv->num_labels; i += 2)
			draw_button_label (cr,
			                   split_button_get_action_by_index (self, i + 1),
			                   x,
			                   y + i/2 * priv->settings_sb->expanded_height,
			                   priv->settings_sb->expanded_width,
			                   priv->settings_sb->expanded_height,
			                   priv->settings_b->text_margin_horiz,
			                   priv->settings_b->text_family,
			                   priv->settings_b->text_size,
			                   font_weight,
			                   text_normal_color);

	/* expanded normal-state surface - arrow */
	draw_round_rect (cr,
	                 1.0,
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, grid_align),
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, grid_align),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->expanded_arrow_button_size - (3.0 * priv->settings_b->line_width + 2.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE),
	                 align (priv->settings_sb->expanded_arrow_button_size - (3.0 * priv->settings_b->line_width + 2.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE));
	cairo_set_source_rgba (cr,
	                       stroke_normal_color[0],
	                       stroke_normal_color[1],
	                       stroke_normal_color[2],
	                       stroke_normal_color[3]);
	cairo_stroke (cr);
	cairo_set_line_width (cr, priv->settings_sb->arrow_thickness);
	cairo_set_source_rgba (cr,
	                       arrow_color[0],
	                       arrow_color[1],
	                       arrow_color[2],
	                       arrow_color[3]);
	cairo_set_line_cap (cr, CAIRO_LINE_CAP_ROUND);
	cairo_move_to (cr,
	               priv->settings_sb->expanded_arrow_button_size / 2.0 +
	               priv->settings_sb->expanded_arrow_width / 2.0,
	               priv->settings_sb->expanded_arrow_button_size / 2.0 -
	               priv->settings_sb->expanded_arrow_height / 2.0);
	cairo_rel_line_to (cr,
	                   -priv->settings_sb->expanded_arrow_width,
	                   priv->settings_sb->expanded_arrow_height / 2.0);
	cairo_rel_line_to (cr,
	                   priv->settings_sb->expanded_arrow_width,
	                   priv->settings_sb->expanded_arrow_height / 2.0);
	cairo_stroke (cr);
	cairo_destroy (cr);
	cr = NULL;

	/* collapsed hover-state surface - fill*/
	cr = cairo_create (priv->collapsed_hover_surface);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, FALSE),
	                 align (0.0, FALSE),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->contracted_width, FALSE),
	                 align (priv->settings_sb->contracted_height, FALSE));
	cairo_set_source_rgba (cr,
	                       fill_normal_color[0],
	                       fill_normal_color[1],
	                       fill_normal_color[2],
	                       fill_normal_color[3]);
	cairo_fill (cr);

	/* collapsed hover-state surface - outline */
	cairo_set_line_width (cr, priv->settings_b->line_width);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, grid_align),
	                 align (0.0, grid_align),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->contracted_width, grid_align) - 3.0 * align (0.0, grid_align),
	                 align (priv->settings_sb->contracted_height, grid_align) - 3.0 * align (0.0, grid_align));
	cairo_set_source_rgba (cr,
	                       stroke_hover_color[0],
	                       stroke_hover_color[1],
	                       stroke_hover_color[2],
	                       stroke_hover_color[3]);
	cairo_stroke (cr);

	/* collapsed hover-state surface - text */
	x = priv->settings_sb->contracted_arrow_button_size;
	y = 0;
	if (priv->num_labels != 0)
		draw_button_label (cr,
		                   split_button_get_action_by_index (self, 1),
		                   x,
		                   y,
		                   priv->settings_sb->contracted_width - priv->settings_sb->contracted_arrow_button_size,
		                   priv->settings_sb->contracted_height,
		                   priv->settings_b->text_margin_horiz,
		                   priv->settings_b->text_family,
		                   priv->settings_b->text_size,
		                   font_weight,
		                   text_hover_color);

	/* collapsed hover-state surface - arrow */
	draw_round_rect (cr,
	                 1.0,
	                 align (priv->settings_b->line_width + priv->settings_sb->contracted_arrow_button_margin, grid_align),
	                 align (priv->settings_b->line_width + priv->settings_sb->contracted_arrow_button_margin, grid_align),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->contracted_arrow_button_size - (3.0 * priv->settings_b->line_width + 2.0 * priv->settings_sb->contracted_arrow_button_margin), FALSE),
	                 align (priv->settings_sb->contracted_arrow_button_size - (3.0 * priv->settings_b->line_width + 2.0 * priv->settings_sb->contracted_arrow_button_margin), FALSE));
	cairo_set_source_rgba (cr,
	                       stroke_hover_color[0],
	                       stroke_hover_color[1],
	                       stroke_hover_color[2],
	                       stroke_hover_color[3]);
	cairo_stroke (cr);
	cairo_set_line_width (cr, priv->settings_sb->arrow_thickness);
	cairo_set_source_rgba (cr,
	                       arrow_color[0],
	                       arrow_color[1],
	                       arrow_color[2],
	                       arrow_color[3]);
	cairo_set_line_cap (cr, CAIRO_LINE_CAP_ROUND);
	cairo_move_to (cr,
	               priv->settings_sb->contracted_arrow_button_size / 2.0 -
	               priv->settings_sb->contracted_arrow_width / 2.0,
	               priv->settings_sb->contracted_arrow_button_size / 2.0 -
	               priv->settings_sb->contracted_arrow_height / 2.0);
	cairo_rel_line_to (cr,
	                   priv->settings_sb->contracted_arrow_width / 2.0,
	                   priv->settings_sb->contracted_arrow_height);
	cairo_rel_line_to (cr,
	                   priv->settings_sb->contracted_arrow_width / 2.0,
	                   -priv->settings_sb->contracted_arrow_height);
	cairo_stroke (cr);
	cairo_destroy (cr);
	cr = NULL;

	/* expanded hover-state surface - fill */
	cr = cairo_create (priv->expanded_hover_surface);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	expanded_height = rows * priv->settings_sb->expanded_height;
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, FALSE),
	                 align (0.0, FALSE),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->expanded_width, FALSE),
	                 align (expanded_height, FALSE));
	cairo_set_source_rgba (cr,
	                       fill_normal_color[0],  // use the normal fill-color
	                       fill_normal_color[1],  // intentionally here, as the
	                       fill_normal_color[2],  // hover-color will only be
	                       fill_normal_color[3]); // use for the hit label
	cairo_fill (cr);

	/* expanded hover-state surface - outline */
	cairo_set_line_width (cr, priv->settings_b->line_width);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, grid_align),
	                 align (0.0, grid_align),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->expanded_width, grid_align) - 3.0 * align (0.0, grid_align),
	                 align (expanded_height, grid_align) - 3.0 * align (0.0, grid_align));
	cairo_set_source_rgba (cr,
	                       stroke_hover_color[0],
	                       stroke_hover_color[1],
	                       stroke_hover_color[2],
	                       stroke_hover_color[3]);
	cairo_stroke (cr);

	/* expanded hover-state surface - separator */
	grid_align = priv->settings_sb->separator_thickness == 1.0;
	cairo_set_line_width (cr, priv->settings_sb->separator_thickness);
	cairo_set_source_rgba (cr,
	                       separator_color[0],
	                       separator_color[1],
	                       separator_color[2],
	                       separator_color[3]);
	for (i = 0; i < rows; i++)
	{
		cairo_move_to (cr,
		               align (priv->settings_sb->separator_margin, grid_align),
		               align ((i + 1) * priv->settings_sb->expanded_height, grid_align));
		cairo_line_to (cr,
		               align (priv->settings_sb->expanded_width -
		               priv->settings_sb->separator_margin, grid_align),
		               align ((i + 1) * priv->settings_sb->expanded_height, grid_align));
	}
	cairo_stroke (cr);

	/* expanded hover-state surface - text */
	x = 0;
	y = 0;
	if (priv->num_labels != 0)
		for (i = 0; i < priv->num_labels; i += 2)
			draw_button_label (cr,
			                   split_button_get_action_by_index (self, i + 1),
			                   x,
			                   y + i/2 * priv->settings_sb->expanded_height,
			                   priv->settings_sb->expanded_width,
			                   priv->settings_sb->expanded_height,
			                   priv->settings_b->text_margin_horiz,
			                   priv->settings_b->text_family,
			                   priv->settings_b->text_size,
			                   font_weight,
			                   text_hover_color);

	/* expanded hover-state surface - arrow */
	draw_round_rect (cr,
	                 1.0,
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, grid_align),
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, grid_align),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->expanded_arrow_button_size - (3.0 * priv->settings_b->line_width + 2.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE),
	                 align (priv->settings_sb->expanded_arrow_button_size - (3.0 * priv->settings_b->line_width + 2.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE));
	cairo_set_source_rgba (cr,
	                       stroke_hover_color[0],
	                       stroke_hover_color[1],
	                       stroke_hover_color[2],
	                       stroke_hover_color[3]);
	cairo_stroke (cr);
	cairo_set_line_width (cr, priv->settings_sb->arrow_thickness);
	cairo_set_source_rgba (cr,
	                       arrow_color[0],
	                       arrow_color[1],
	                       arrow_color[2],
	                       arrow_color[3]);
	cairo_set_line_cap (cr, CAIRO_LINE_CAP_ROUND);
	cairo_move_to (cr,
	               priv->settings_sb->expanded_arrow_button_size / 2.0 +
	               priv->settings_sb->expanded_arrow_width / 2.0,
	               priv->settings_sb->expanded_arrow_button_size / 2.0 -
	               priv->settings_sb->expanded_arrow_height / 2.0);
	cairo_rel_line_to (cr,
	                   -priv->settings_sb->expanded_arrow_width,
	                   priv->settings_sb->expanded_arrow_height / 2.0);
	cairo_rel_line_to (cr,
	                   priv->settings_sb->expanded_arrow_width,
	                   priv->settings_sb->expanded_arrow_height / 2.0);
	cairo_stroke (cr);
	cairo_destroy (cr);
	cr = NULL;

	/* collapsed pressed-state surface - fill */
	cr = cairo_create (priv->collapsed_pressed_surface);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, FALSE),
	                 align (0.0, FALSE),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->contracted_width, FALSE),
	                 align (priv->settings_sb->contracted_height, FALSE));
	cairo_set_source_rgba (cr,
	                       fill_normal_color[0],
	                       fill_normal_color[1],
	                       fill_normal_color[2],
	                       fill_normal_color[3]);
	cairo_fill (cr);

	/* collapsed pressed-state surface - outline */
	cairo_set_line_width (cr, priv->settings_b->line_width);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, grid_align),
	                 align (0.0, grid_align),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->contracted_width, grid_align) - 3.0 * align (0.0, grid_align),
	                 align (priv->settings_sb->contracted_height, grid_align) - 3.0 * align (0.0, grid_align));
	cairo_set_source_rgba (cr,
	                       stroke_pressed_color[0],
	                       stroke_pressed_color[1],
	                       stroke_pressed_color[2],
	                       stroke_pressed_color[3]);
	cairo_stroke (cr);

	/* collapsed pressed-state surface - text */
	x = priv->settings_sb->contracted_arrow_button_size;
	y = 0;
	if (priv->num_labels != 0)
		draw_button_label (cr,
		                   split_button_get_action_by_index (self, 1),
		                   x,
		                   y,
		                   priv->settings_sb->contracted_width - priv->settings_sb->contracted_arrow_button_size,
		                   priv->settings_sb->contracted_height,
		                   priv->settings_b->text_margin_horiz,
		                   priv->settings_b->text_family,
		                   priv->settings_b->text_size,
		                   font_weight,
		                   text_pressed_color);

	/* collapsed pressed-state surface - arrow */
	draw_round_rect (cr,
	                 1.0,
	                 align (priv->settings_b->line_width + priv->settings_sb->contracted_arrow_button_margin, grid_align),
	                 align (priv->settings_b->line_width + priv->settings_sb->contracted_arrow_button_margin, grid_align),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->contracted_arrow_button_size - (3.0 * priv->settings_b->line_width + 2.0 * priv->settings_sb->contracted_arrow_button_margin), FALSE),
	                 align (priv->settings_sb->contracted_arrow_button_size - (3.0 * priv->settings_b->line_width + 2.0 * priv->settings_sb->contracted_arrow_button_margin), FALSE));
	cairo_set_source_rgba (cr,
	                       stroke_pressed_color[0],
	                       stroke_pressed_color[1],
	                       stroke_pressed_color[2],
	                       stroke_pressed_color[3]);
	cairo_stroke (cr);
	cairo_set_line_width (cr, priv->settings_sb->arrow_thickness);
	cairo_set_source_rgba (cr,
	                       arrow_color[0],
	                       arrow_color[1],
	                       arrow_color[2],
	                       arrow_color[3]);
	cairo_set_line_cap (cr, CAIRO_LINE_CAP_ROUND);
	cairo_move_to (cr,
	               priv->settings_sb->contracted_arrow_button_size / 2.0 -
	               priv->settings_sb->contracted_arrow_width / 2.0,
	               priv->settings_sb->contracted_arrow_button_size / 2.0 -
	               priv->settings_sb->contracted_arrow_height / 2.0);
	cairo_rel_line_to (cr,
	                   priv->settings_sb->contracted_arrow_width / 2.0,
	                   priv->settings_sb->contracted_arrow_height);
	cairo_rel_line_to (cr,
	                   priv->settings_sb->contracted_arrow_width / 2.0,
	                   -priv->settings_sb->contracted_arrow_height);
	cairo_stroke (cr);
	cairo_destroy (cr);
	cr = NULL;

	/* expanded pressed-state surface - fill */
	cr = cairo_create (priv->expanded_pressed_surface);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	expanded_height = rows * priv->settings_sb->expanded_height;
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, FALSE),
	                 align (0.0, FALSE),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->expanded_width, FALSE),
	                 align (expanded_height, FALSE));
	cairo_set_source_rgba (cr,
	                       fill_normal_color[0],  // use the normal fill-color
	                       fill_normal_color[1],  // intentionally here, as the
	                       fill_normal_color[2],  // pressed-color will only be
	                       fill_normal_color[3]); // use for the hit label
	cairo_fill (cr);

	/* expanded pressed-state surface - outline */
	cairo_set_line_width (cr, priv->settings_b->line_width);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, grid_align),
	                 align (0.0, grid_align),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->expanded_width, grid_align) - 3.0 * align (0.0, grid_align),
	                 align (expanded_height, grid_align) - 3.0 * align (0.0, grid_align));
	cairo_set_source_rgba (cr,
	                       stroke_pressed_color[0],
	                       stroke_pressed_color[1],
	                       stroke_pressed_color[2],
	                       stroke_pressed_color[3]);
	cairo_stroke (cr);

	/* expanded pressed-state surface - separators */
	grid_align = priv->settings_sb->separator_thickness == 1.0;
	cairo_set_line_width (cr, priv->settings_sb->separator_thickness);
	cairo_set_source_rgba (cr,
	                       separator_color[0],
	                       separator_color[1],
	                       separator_color[2],
	                       separator_color[3]);
	for (i = 0; i < rows; i++)
	{
		cairo_move_to (cr,
		               align (priv->settings_sb->separator_margin, grid_align),
		               align ((i + 1) * priv->settings_sb->expanded_height, grid_align));
		cairo_line_to (cr,
		               align (priv->settings_sb->expanded_width -
		               priv->settings_sb->separator_margin, grid_align),
		               align ((i + 1) * priv->settings_sb->expanded_height, grid_align));
	}
	cairo_stroke (cr);

	/* expanded pressed-state surface - text */
	x = 0;
	y = 0;
	if (priv->num_labels != 0)
		for (i = 0; i < priv->num_labels; i += 2)
			draw_button_label (cr,
			                   split_button_get_action_by_index (self, i + 1),
			                   x,
			                   y + i/2 * priv->settings_sb->expanded_height,
			                   priv->settings_sb->expanded_width,
			                   priv->settings_sb->expanded_height,
			                   priv->settings_b->text_margin_horiz,
			                   priv->settings_b->text_family,
			                   priv->settings_b->text_size,
			                   font_weight,
			                   text_pressed_color);

	/* expanded pressed-state surface - arrow */
	draw_round_rect (cr,
	                 1.0,
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, grid_align),
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, grid_align),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->expanded_arrow_button_size - (3.0 * priv->settings_b->line_width + 2.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE),
	                 align (priv->settings_sb->expanded_arrow_button_size - (3.0 * priv->settings_b->line_width + 2.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE));
	cairo_set_source_rgba (cr,
	                       stroke_pressed_color[0],
	                       stroke_pressed_color[1],
	                       stroke_pressed_color[2],
	                       stroke_pressed_color[3]);
	cairo_stroke (cr);
	cairo_set_line_width (cr, priv->settings_sb->arrow_thickness);
	cairo_set_source_rgba (cr,
	                       arrow_color[0],
	                       arrow_color[1],
	                       arrow_color[2],
	                       arrow_color[3]);
	cairo_set_line_cap (cr, CAIRO_LINE_CAP_ROUND);
	cairo_move_to (cr,
	               priv->settings_sb->expanded_arrow_button_size / 2.0 +
	               priv->settings_sb->expanded_arrow_width / 2.0,
	               priv->settings_sb->expanded_arrow_button_size / 2.0 -
	               priv->settings_sb->expanded_arrow_height / 2.0);
	cairo_rel_line_to (cr,
	                   -priv->settings_sb->expanded_arrow_width,
	                   priv->settings_sb->expanded_arrow_height / 2.0);
	cairo_rel_line_to (cr,
	                   priv->settings_sb->expanded_arrow_width,
	                   priv->settings_sb->expanded_arrow_height / 2.0);
	cairo_stroke (cr);
	cairo_destroy (cr);
	cr = NULL;

	/* contracted arrow hover */
	cr = cairo_create (priv->arrow_hover_surface[0]);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	draw_round_rect (cr,
	                 1.0,
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, FALSE),
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, FALSE),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->expanded_arrow_button_size - (5.0 * priv->settings_b->line_width + 4.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE),
	                 align (priv->settings_sb->expanded_arrow_button_size - (5.0 * priv->settings_b->line_width + 4.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE));
	cairo_set_source_rgba (cr,
	                       fill_hover_color[0],
	                       fill_hover_color[1],
	                       fill_hover_color[2],
	                       fill_hover_color[3]);
	cairo_fill (cr);
	cairo_destroy (cr);
	cr = NULL;

	/* contracted arrow pressed */
	cr = cairo_create (priv->arrow_pressed_surface[0]);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	draw_round_rect (cr,
	                 1.0,
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, FALSE),
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, FALSE),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->expanded_arrow_button_size - (5.0 * priv->settings_b->line_width + 4.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE),
	                 align (priv->settings_sb->expanded_arrow_button_size - (5.0 * priv->settings_b->line_width + 4.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE));
	cairo_set_source_rgba (cr,
	                       fill_pressed_color[0],
	                       fill_pressed_color[1],
	                       fill_pressed_color[2],
	                       fill_pressed_color[3]);
	cairo_fill (cr);
	cairo_destroy (cr);
	cr = NULL;

	/* contracted entry 0 hover */
	cr = cairo_create (priv->entry0_hover_surface[0]);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, FALSE),
	                 align (0.0, FALSE),
	                 priv->settings_b->radius,
					 priv->settings_sb->contracted_width,
					 priv->settings_sb->contracted_height);
	cairo_set_source_rgba (cr,
	                       fill_hover_color[0],
	                       fill_hover_color[1],
	                       fill_hover_color[2],
	                       fill_hover_color[3]);
	cairo_fill (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	draw_round_rect (cr,
	                 1.0,
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, FALSE),
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, FALSE),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->expanded_arrow_button_size - (5.0 * priv->settings_b->line_width + 4.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE),
	                 align (priv->settings_sb->expanded_arrow_button_size - (5.0 * priv->settings_b->line_width + 4.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE));
	cairo_fill (cr);
	cairo_destroy (cr);
	cr = NULL;

	/* contracted entry 0 pressed */
	cr = cairo_create (priv->entry0_pressed_surface[0]);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, FALSE),
	                 align (0.0, FALSE),
	                 priv->settings_b->radius,
					 priv->settings_sb->contracted_width,
					 priv->settings_sb->contracted_height);
	cairo_set_source_rgba (cr,
	                       fill_pressed_color[0],
	                       fill_pressed_color[1],
	                       fill_pressed_color[2],
	                       fill_pressed_color[3]);
	cairo_fill (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	draw_round_rect (cr,
	                 1.0,
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, FALSE),
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, FALSE),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->expanded_arrow_button_size - (5.0 * priv->settings_b->line_width + 4.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE),
	                 align (priv->settings_sb->expanded_arrow_button_size - (5.0 * priv->settings_b->line_width + 4.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE));
	cairo_fill (cr);
	cairo_destroy (cr);
	cr = NULL;

	/* expanded arrow hover */
	cr = cairo_create (priv->arrow_hover_surface[1]);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	draw_round_rect (cr,
	                 1.0,
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, FALSE),
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, FALSE),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->expanded_arrow_button_size - (3.0 * priv->settings_b->line_width + 2.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE),
	                 align (priv->settings_sb->expanded_arrow_button_size - (3.0 * priv->settings_b->line_width + 2.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE));
	cairo_set_source_rgba (cr,
	                       fill_hover_color[0],
	                       fill_hover_color[1],
	                       fill_hover_color[2],
	                       fill_hover_color[3]);
	cairo_fill (cr);
	cairo_destroy (cr);
	cr = NULL;

	/* expanded arrow pressed */
	cr = cairo_create (priv->arrow_pressed_surface[1]);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	draw_round_rect (cr,
	                 1.0,
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, FALSE),
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, FALSE),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->expanded_arrow_button_size - (3.0 * priv->settings_b->line_width + 2.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE),
	                 align (priv->settings_sb->expanded_arrow_button_size - (3.0 * priv->settings_b->line_width + 2.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE));
	cairo_set_source_rgba (cr,
	                       fill_pressed_color[0],
	                       fill_pressed_color[1],
	                       fill_pressed_color[2],
	                       fill_pressed_color[3]);
	cairo_fill (cr);
	cairo_destroy (cr);
	cr = NULL;

	/* entry 0 hover */
	cr = cairo_create (priv->entry0_hover_surface[1]);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, FALSE),
	                 align (0.0, FALSE),
	                 priv->settings_b->radius,
					 priv->settings_sb->expanded_width,
					 priv->settings_sb->expanded_height);
	cairo_set_source_rgba (cr,
	                       fill_hover_color[0],
	                       fill_hover_color[1],
	                       fill_hover_color[2],
	                       fill_hover_color[3]);
	cairo_fill (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	draw_round_rect (cr,
	                 1.0,
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, FALSE),
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, FALSE),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->expanded_arrow_button_size - (3.0 * priv->settings_b->line_width + 2.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE),
	                 align (priv->settings_sb->expanded_arrow_button_size - (3.0 * priv->settings_b->line_width + 2.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE));
	cairo_fill (cr);
	cairo_destroy (cr);
	cr = NULL;

	/* entry 0 pressed */
	cr = cairo_create (priv->entry0_pressed_surface[1]);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, FALSE),
	                 align (0.0, FALSE),
	                 priv->settings_b->radius,
					 priv->settings_sb->expanded_width,
					 priv->settings_sb->expanded_height);
	cairo_set_source_rgba (cr,
	                       fill_pressed_color[0],
	                       fill_pressed_color[1],
	                       fill_pressed_color[2],
	                       fill_pressed_color[3]);
	cairo_fill (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	draw_round_rect (cr,
	                 1.0,
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, FALSE),
	                 align (priv->settings_b->line_width + priv->settings_sb->expanded_arrow_button_margin, FALSE),
	                 priv->settings_b->radius,
	                 align (priv->settings_sb->expanded_arrow_button_size - (3.0 * priv->settings_b->line_width + 2.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE),
	                 align (priv->settings_sb->expanded_arrow_button_size - (3.0 * priv->settings_b->line_width + 2.0 * priv->settings_sb->expanded_arrow_button_margin), FALSE));
	cairo_fill (cr);
	cairo_destroy (cr);
	cr = NULL;

	/* entry hover */
	cr = cairo_create (priv->entry_hover_surface);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, FALSE),
	                 align (0.0, FALSE),
	                 priv->settings_b->radius,
					 priv->settings_sb->expanded_width,
					 priv->settings_sb->expanded_height);
	cairo_set_source_rgba (cr,
	                       fill_hover_color[0],
	                       fill_hover_color[1],
	                       fill_hover_color[2],
	                       fill_hover_color[3]);
	cairo_fill (cr);
	cairo_destroy (cr);
	cr = NULL;

	/* entry pressed */
	cr = cairo_create (priv->entry_pressed_surface);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, FALSE),
	                 align (0.0, FALSE),
	                 priv->settings_b->radius,
					 priv->settings_sb->expanded_width,
					 priv->settings_sb->expanded_height);
	cairo_set_source_rgba (cr,
	                       fill_pressed_color[0],
	                       fill_pressed_color[1],
	                       fill_pressed_color[2],
	                       fill_pressed_color[3]);
	cairo_fill (cr);
	cairo_destroy (cr);
	cr = NULL;
}

/*-- public API --------------------------------------------------------------*/

SplitButton*
split_button_new (gint                 x,
                  gint                 y,
                  SettingsButton*      settings_b,
                  SettingsSplitButton* settings_sb,
                  gchar**              labels,
                  guint                size)
{
	SplitButton*        self = NULL;
	SplitButtonPrivate* priv = NULL;

	g_return_val_if_fail (settings_b &&
	                      settings_sb &&
	                      settings_sb->contracted_width != 0 &&
	                      settings_sb->contracted_height != 0 &&
	                      settings_sb->expanded_width != 0 &&
	                      settings_sb->expanded_height != 0,
	                      NULL);

	self = g_object_new (SPLIT_BUTTON_TYPE, NULL);
	if (!self)
		return NULL;

	priv = GET_PRIVATE (self);

	priv->settings_b               = settings_b;
	priv->settings_sb              = settings_sb;

	split_button_set_actions (self, labels, size);

	gint    rows            = split_button_get_num_actions (self) / 2;
	gdouble expanded_height = rows * priv->settings_sb->expanded_height;

	priv->x                        = x;
	priv->y                        = y;
	priv->state                    = SPLIT_BUTTON_STATE_NORMAL;
	priv->form                     = SPLIT_BUTTON_FORM_COLLAPSED;
	priv->hit_label_index          = -1;
	priv->arrow_hit                = FALSE;
	priv->collapsed_normal_surface = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
	                                    settings_sb->contracted_width,
	                                    settings_sb->contracted_height);
	priv->collapsed_hover_surface = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
										settings_sb->contracted_width,
										settings_sb->contracted_height);
	priv->collapsed_pressed_surface = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
										settings_sb->contracted_width,
										settings_sb->contracted_height);
	priv->expanded_normal_surface = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
	                                    settings_sb->expanded_width,
	                                    expanded_height);
	priv->expanded_hover_surface = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
										settings_sb->expanded_width,
										expanded_height);
	priv->expanded_pressed_surface = cairo_image_surface_create (
										CAIRO_FORMAT_ARGB32,
										settings_sb->expanded_width,
										expanded_height);

	_refresh_split_button_surfaces (self);

	return self;
}

void
split_button_del (SplitButton* self)
{
	g_return_if_fail (self && IS_SPLIT_BUTTON (self));
	g_object_unref (self);
}

void
split_button_set_position (SplitButton* self,
                           gint         x,
                           gint         y)
{
	g_return_if_fail (self && IS_SPLIT_BUTTON (self));

	SplitButtonPrivate* priv = GET_PRIVATE (self);

	priv->x = x;
	priv->y = y;
}

void
split_button_get_position (SplitButton* self,
                           gint*        x,
                           gint*        y)
{
	g_return_if_fail (self && IS_SPLIT_BUTTON (self) && x && y);

	SplitButtonPrivate* priv = GET_PRIVATE (self);

	*x = priv->x;
	*y = priv->y;
}

guint
split_button_get_expanded_height (SplitButton* self)
{
	g_return_val_if_fail (self && IS_SPLIT_BUTTON (self), 0);

	gint rows = split_button_get_num_actions (self) / 2;
	
	return (rows * GET_PRIVATE (self)->settings_sb->expanded_height);
}

gboolean
split_button_is_arrow_hit (SplitButton* self,
                           gint         x,
                           gint         y)
{
	g_return_val_if_fail (self && IS_SPLIT_BUTTON (self), FALSE);

	SplitButtonPrivate*   priv      = GET_PRIVATE (self);
	gboolean              result    = FALSE;
	cairo_rectangle_int_t rect      = {0, 0, 0, 0};
	cairo_region_t*       region    = NULL;
	SplitButtonForm       form      = SPLIT_BUTTON_FORM_NONE;

	priv = GET_PRIVATE (self);

	rect.x = priv->x;
	rect.y = priv->y;
	form = split_button_get_form (self);
	if (form == SPLIT_BUTTON_FORM_COLLAPSED)
	{
		rect.width  = priv->settings_sb->contracted_arrow_button_size;
		rect.height = priv->settings_sb->contracted_arrow_button_size;
	}
	else if (form == SPLIT_BUTTON_FORM_EXPANDED)
	{
		rect.width  = priv->settings_sb->expanded_arrow_button_size;
		rect.height = priv->settings_sb->expanded_arrow_button_size;
	}

	region = cairo_region_create_rectangle (&rect);
	result = cairo_region_contains_point (region, x, y);
	cairo_region_destroy (region);

	priv->arrow_hit = result;

	return result;
}

gboolean
split_button_is_hit (SplitButton* self,
                     gint         x,
                     gint         y)
{
	g_return_val_if_fail (self && IS_SPLIT_BUTTON (self), FALSE);

	SplitButtonPrivate*   priv      = GET_PRIVATE (self);
	gboolean              result    = FALSE;
	cairo_rectangle_int_t rect      = {0, 0, 0, 0};
	cairo_region_t*       region    = NULL;
	SplitButtonForm       form      = SPLIT_BUTTON_FORM_NONE;
	gint                  hit_index = -1;

	priv = GET_PRIVATE (self);

	rect.x = priv->x;
	rect.y = priv->y;
	form = split_button_get_form (self);
	if (form == SPLIT_BUTTON_FORM_COLLAPSED)
	{
		rect.width  = priv->settings_sb->contracted_width;
		rect.height = priv->settings_sb->contracted_height;
	}
	else if (form == SPLIT_BUTTON_FORM_EXPANDED)
	{
		guint rows = split_button_get_num_actions (self) / 2;
		rect.width  = priv->settings_sb->expanded_width;
		rect.height = priv->settings_sb->expanded_height * rows;
	}

	region = cairo_region_create_rectangle (&rect);
	result = cairo_region_contains_point (region, x, y);
	cairo_region_destroy (region);

	if (form == SPLIT_BUTTON_FORM_EXPANDED)
	{
		guint    i            = 0;
		gboolean is_label_hit = FALSE;

		for (; i < priv->num_labels / 2; i++)
		{
			rect.x      = priv->x;
			rect.y      = priv->y + i * priv->settings_sb->expanded_height;
			rect.width  = priv->settings_sb->expanded_width;
			rect.height = priv->settings_sb->expanded_height;

			region       = cairo_region_create_rectangle (&rect);
			is_label_hit = cairo_region_contains_point (region, x, y);
			cairo_region_destroy (region);

			if (is_label_hit)
				hit_index = i;
		}

		split_button_set_hit_label (self, hit_index);
	}

	split_button_is_arrow_hit (self, x, y);

	return result;
}

void
split_button_set_form (SplitButton*    self,
                       SplitButtonForm form)
{
	g_return_if_fail (self && IS_SPLIT_BUTTON (self));

	GET_PRIVATE (self)->form = form;
}

SplitButtonForm
split_button_get_form (SplitButton* self)
{
	g_return_val_if_fail (self && IS_SPLIT_BUTTON (self),
	                      SPLIT_BUTTON_FORM_NONE);

	return GET_PRIVATE (self)->form;
}

void
split_button_set_state (SplitButton*     self,
                        SplitButtonState state)
{
	g_return_if_fail (self && IS_SPLIT_BUTTON (self));

	GET_PRIVATE (self)->state = state;
}

SplitButtonState
split_button_get_state (SplitButton* self)
{
	g_return_val_if_fail (self && IS_SPLIT_BUTTON (self),
	                      SPLIT_BUTTON_STATE_NONE);

	return GET_PRIVATE (self)->state;
}

void
split_button_set_hit_label (SplitButton* self,
                            gint         index)
{
	g_return_if_fail (self && IS_SPLIT_BUTTON (self));

	SplitButtonPrivate* priv = GET_PRIVATE (self);

	if (priv->hit_label_index == index)
		return;

	if (index == -1)
		priv->hit_label_index = -1;
	else if (priv->num_labels / 2 > index)
	{
		priv->hit_label_index = index;
		_refresh_split_button_surfaces (self);
	}
}

gint
split_button_get_hit_label (SplitButton* self)
{
	g_return_val_if_fail (self && IS_SPLIT_BUTTON (self), -1);

	return GET_PRIVATE (self)->hit_label_index;
}

gboolean
split_button_set_actions (SplitButton* self,
                          gchar**      actions,
                          guint        size)
{
	g_return_val_if_fail (self &&
	                      IS_SPLIT_BUTTON (self) &&
	                      actions &&
	                      size > 2,
	                      FALSE);

	SplitButtonPrivate* priv = GET_PRIVATE (self);
	guint               i    = 0;

	if (priv->labels != NULL)
	{
		for (i = 0; i < priv->num_labels; i++)
			g_free (priv->labels[i]);

		g_free (priv->labels);
		priv->labels = NULL;
		priv->num_labels = 0;
	}

	priv->num_labels = size;
	priv->labels = (gchar**) g_malloc0 (priv->num_labels * sizeof (gchar*));
	for (i = 0; i < priv->num_labels; i++)
		priv->labels[i] = g_strdup (actions[i]);

	_refresh_split_button_surfaces (self);

	return TRUE;
}

guint
split_button_get_num_actions (SplitButton* self)
{
	g_return_val_if_fail (self && IS_SPLIT_BUTTON (self), 0);

	return GET_PRIVATE (self)->num_labels;
}

const gchar*
split_button_get_action_by_index (SplitButton* self,
                                  guint        index)
{
	g_return_val_if_fail (self &&
	                      IS_SPLIT_BUTTON (self) &&
	                      GET_PRIVATE (self)->num_labels - 1 >= index,
	                      NULL);

	return GET_PRIVATE (self)->labels[index];
}

void
split_button_paint (SplitButton* self,
                    cairo_t*     cr)
{
	g_return_if_fail (self && IS_SPLIT_BUTTON (self) && cr);

	SplitButtonPrivate* priv = GET_PRIVATE (self);

	if (priv->form == SPLIT_BUTTON_FORM_COLLAPSED)
	{
		if (priv->state == SPLIT_BUTTON_STATE_NORMAL)
		{
			cairo_set_source_surface (cr,
			                          priv->collapsed_normal_surface,
			                          priv->x,
			                          priv->y);
			cairo_paint (cr);
		}
		if (priv->state == SPLIT_BUTTON_STATE_HOVER)
		{
			cairo_set_source_surface (cr,
			                          priv->collapsed_hover_surface,
			                          priv->x,
			                          priv->y);
			cairo_paint (cr);

			if (priv->arrow_hit)
				cairo_set_source_surface (cr,
					                      priv->arrow_hover_surface[0],
					                      priv->x,
					                      priv->y);
			else
				cairo_set_source_surface (cr,
					                      priv->entry0_hover_surface[0],
					                      priv->x,
					                      priv->y);

			cairo_paint (cr);
		}
		if (priv->state == SPLIT_BUTTON_STATE_PRESSED)
		{
			cairo_set_source_surface (cr,
			                          priv->collapsed_pressed_surface,
			                          priv->x,
			                          priv->y);
			cairo_paint (cr);

			if (priv->arrow_hit)
				cairo_set_source_surface (cr,
					                      priv->arrow_pressed_surface[0],
					                      priv->x,
					                      priv->y);
			else
				cairo_set_source_surface (cr,
					                      priv->entry0_pressed_surface[0],
					                      priv->x,
					                      priv->y);

			cairo_paint (cr);
		}
	}
	else if (priv->form == SPLIT_BUTTON_FORM_EXPANDED)
	{
		if (priv->state == SPLIT_BUTTON_STATE_NORMAL)
		{
			cairo_set_source_surface (cr,
			                          priv->expanded_normal_surface,
			                          priv->x,
			                          priv->y);
			cairo_paint (cr);
		}
		if (priv->state == SPLIT_BUTTON_STATE_HOVER)
		{
			cairo_set_source_surface (cr,
			                          priv->expanded_hover_surface,
			                          priv->x,
			                          priv->y);
			cairo_paint (cr);

			if (priv->arrow_hit)
				cairo_set_source_surface (cr,
					                      priv->arrow_hover_surface[1],
					                      priv->x,
					                      priv->y);
			else if (priv->hit_label_index == 0)
				cairo_set_source_surface (cr,
					                      priv->entry0_hover_surface[1],
					                      priv->x,
					                      priv->y);
			else if (priv->hit_label_index != -1)
				cairo_set_source_surface (cr,
					                      priv->entry_hover_surface,
					                      priv->x,
					                      priv->y + priv->hit_label_index * priv->settings_sb->expanded_height);

			cairo_paint (cr);
		}
		if (priv->state == SPLIT_BUTTON_STATE_PRESSED)
		{
			cairo_set_source_surface (cr,
			                          priv->expanded_pressed_surface,
			                          priv->x,
			                          priv->y);
			cairo_paint (cr);

			if (priv->arrow_hit)
				cairo_set_source_surface (cr,
					                      priv->arrow_pressed_surface[1],
					                      priv->x,
					                      priv->y);
			else if (priv->hit_label_index == 0)
				cairo_set_source_surface (cr,
					                      priv->entry0_pressed_surface[1],
					                      priv->x,
					                      priv->y);
			else if (priv->hit_label_index != -1)
				cairo_set_source_surface (cr,
					                      priv->entry_pressed_surface,
					                      priv->x,
					                      priv->y + priv->hit_label_index * priv->settings_sb->expanded_height);

			cairo_paint (cr);
		}
	}
}
