/*******************************************************************************
**3456789 123456789 123456789 123456789 123456789 123456789 123456789 123456789
**      10        20        30        40        50        60        70        80
**
** notify-osd
**
** test-queue.c - unit-tests for sd-queue class
**
** Copyright 2012 Canonical Ltd.
**
** Authors:
**    Mirco "MacSlow" Mueller <mirco.mueller@canonical.com>
**
** This program is free software: you can redistribute it and/or modify it
** under the terms of the GNU General Public License version 3, as published
** by the Free Software Foundation.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranties of
** MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
** PURPOSE.  See the GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License along
** with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*******************************************************************************/

#include <glib.h>

#include "queue.h"
#include "notification.h"

static void
test_queue_new ()
{
	Queue* queue = NULL;

	queue = queue_new ();
	g_assert (queue != NULL);
	g_assert (IS_QUEUE(queue));
	g_object_unref (queue);
}

static void
test_queue_del ()
{
	Queue* queue = NULL;

	queue = queue_new ();
	g_assert (queue != NULL);
	g_assert (IS_QUEUE(queue));
	queue_del (queue);
	g_assert (!IS_QUEUE(queue));
}

static void
test_queue_inject ()
{
	Queue*        queue = NULL;
	Notification* n1    = NULL;
	Notification* n2    = NULL;
	gchar* actions[]    = {"Accept", "Decline", "Away", "Busy", NULL};

	queue = queue_new ();
	g_assert (queue != NULL);
	g_assert (IS_QUEUE(queue));

	n1 = notification_new ();
	notification_set_summary (n1, "Title 1");
	notification_set_body (n1, "Summary 1");
	notification_set_sender_name (n1, "application1");
	notification_set_sender_pid (n1, 42);
	notification_set_urgency (n1, URGENCY_NORMAL);
	notification_set_actions (n1, actions);

	n2 = notification_new ();
	notification_set_summary (n2, "Title 2");
	notification_set_body (n2, "Summary 2");
	notification_set_sender_name (n2, "application2");
	notification_set_sender_pid (n2, 13);
	notification_set_urgency (n2, URGENCY_NORMAL);
	notification_set_actions (n2, actions);

	g_assert_cmpuint (queue_get_size (queue), ==, 0);
	queue_inject (queue, n1);
	g_assert_cmpuint (queue_get_size (queue), ==, 1);
	queue_inject (queue, n2);
	g_assert_cmpuint (queue_get_size (queue), ==, 2);

	queue_del (queue);
	g_assert (!IS_QUEUE(queue));
}

static void
test_queue_remove ()
{
	Queue*        queue = NULL;
	Notification* n1    = NULL;
	Notification* n2    = NULL;
	gchar* actions[]    = {"Accept", "Decline", "Away", "Busy", NULL};

	queue = queue_new ();
	g_assert (queue != NULL);
	g_assert (IS_QUEUE(queue));

	n1 = notification_new ();
	notification_set_summary (n1, "Title 1");
	notification_set_body (n1, "Summary 1");
	notification_set_sender_name (n1, "application1");
	notification_set_sender_pid (n1, 42);
	notification_set_urgency (n1, URGENCY_NORMAL);
	notification_set_actions (n1, actions);

	n2 = notification_new ();
	notification_set_summary (n2, "Title 2");
	notification_set_body (n2, "Summary 2");
	notification_set_sender_name (n2, "application2");
	notification_set_sender_pid (n2, 13);
	notification_set_urgency (n2, URGENCY_NORMAL);
	notification_set_actions (n2, actions);

	queue_inject (queue, n1);
	queue_inject (queue, n2);
	g_assert_cmpuint (queue_get_size (queue), ==, 2);
	queue_remove (queue, n1);
	g_assert_cmpuint (queue_get_size (queue), ==, 1);
	queue_remove (queue, n2);
	g_assert_cmpuint (queue_get_size (queue), ==, 0);

	queue_del (queue);
	g_assert (!IS_QUEUE(queue));
}

static void
test_queue_remove_by_pid ()
{
	Queue*        queue = NULL;
	Notification* n1    = NULL;
	Notification* n2    = NULL;
	gchar* actions[]    = {"Accept", "Decline", "Away", "Busy", NULL};

	queue = queue_new ();
	g_assert (queue != NULL);
	g_assert (IS_QUEUE(queue));

	n1 = notification_new ();
	notification_set_summary (n1, "Title 1");
	notification_set_body (n1, "Summary 1");
	notification_set_sender_name (n1, "application1");
	notification_set_sender_pid (n1, 42);
	notification_set_urgency (n1, URGENCY_NORMAL);
	notification_set_actions (n1, actions);

	n2 = notification_new ();
	notification_set_summary (n2, "Title 2");
	notification_set_body (n2, "Summary 2");
	notification_set_sender_name (n2, "application2");
	notification_set_sender_pid (n2, 13);
	notification_set_urgency (n2, URGENCY_NORMAL);
	notification_set_actions (n2, actions);

	queue_inject (queue, n1);
	queue_inject (queue, n2);
	g_assert_cmpuint (queue_get_size (queue), ==, 2);
	queue_remove_by_pid (queue, 42);
	g_assert_cmpuint (queue_get_size (queue), ==, 1);
	queue_remove_by_pid (queue, 13);
	g_assert_cmpuint (queue_get_size (queue), ==, 0);

	queue_del (queue);
	g_assert (!IS_QUEUE(queue));
}

static void
test_queue_remove_by_id ()
{
	Queue*        queue = NULL;
	Notification* n1    = NULL;
	Notification* n2    = NULL;
	gchar* actions[]    = {"Accept", "Decline", "Away", "Busy", NULL};

	queue = queue_new ();
	g_assert (queue != NULL);
	g_assert (IS_QUEUE(queue));

	n1 = notification_new ();
	notification_set_id (n1, 0);
	notification_set_summary (n1, "Title 1");
	notification_set_body (n1, "Summary 1");
	notification_set_sender_name (n1, "application1");
	notification_set_sender_pid (n1, 42);
	notification_set_urgency (n1, URGENCY_NORMAL);
	notification_set_actions (n1, actions);

	n2 = notification_new ();
	notification_set_id (n2, 1);
	notification_set_summary (n2, "Title 2");
	notification_set_body (n2, "Summary 2");
	notification_set_sender_name (n2, "application2");
	notification_set_sender_pid (n2, 13);
	notification_set_urgency (n2, URGENCY_NORMAL);
	notification_set_actions (n2, actions);

	queue_inject (queue, n1);
	queue_inject (queue, n2);
	g_assert_cmpuint (queue_get_size (queue), ==, 2);
	queue_remove_by_id (queue, 0);
	g_assert_cmpuint (queue_get_size (queue), ==, 1);
	queue_remove_by_id (queue, 1);
	g_assert_cmpuint (queue_get_size (queue), ==, 0);

	queue_del (queue);
	g_assert (!IS_QUEUE(queue));
}

static void
test_queue_find_by_pid ()
{
	Queue*        queue = NULL;
	Notification* n1    = NULL;
	Notification* n2    = NULL;
	Notification* n3    = NULL;
	Notification* sd    = NULL;
	gchar* actions[]    = {"Accept", "Decline", "Away", "Busy", NULL};

	queue = queue_new ();
	g_assert (queue != NULL);
	g_assert (IS_QUEUE(queue));

	n1 = notification_new ();
	notification_set_summary (n1, "Title 1");
	notification_set_body (n1, "Summary 1");
	notification_set_sender_name (n1, "application1");
	notification_set_sender_pid (n1, 87);
	notification_set_urgency (n1, URGENCY_NORMAL);
	notification_set_actions (n1, actions);

	n2 = notification_new ();
	notification_set_summary (n2, "Title 2");
	notification_set_body (n2, "Summary 2");
	notification_set_sender_name (n2, "application2");
	notification_set_sender_pid (n2, 215);
	notification_set_urgency (n2, URGENCY_NORMAL);
	notification_set_actions (n2, actions);

	n3 = notification_new ();
	notification_set_summary (n3, "Title 3");
	notification_set_body (n3, "Summary 3");
	notification_set_sender_name (n3, "application3");
	notification_set_sender_pid (n3, 121);
	notification_set_urgency (n3, URGENCY_NORMAL);
	notification_set_actions (n3, actions);

	queue_inject (queue, n1);
	queue_inject (queue, n2);
	queue_inject (queue, n3);
	g_assert_cmpuint (queue_get_size (queue), ==, 3);

	sd = queue_find_by_pid (queue, notification_get_sender_pid (n2));
	g_assert_cmpuint (notification_get_sender_pid (n2),
	                  ==,
	                  notification_get_sender_pid (sd));

	queue_del (queue);
	g_assert (!IS_QUEUE(queue));
}

static void
test_queue_find_by_id ()
{
	Queue*        queue = NULL;
	Notification* n1    = NULL;
	Notification* n2    = NULL;
	Notification* n3    = NULL;
	Notification* sd    = NULL;
	gchar* actions[]    = {"Accept", "Decline", "Away", "Busy", NULL};

	queue = queue_new ();
	g_assert (queue != NULL);
	g_assert (IS_QUEUE(queue));

	n1 = notification_new ();
	notification_set_summary (n1, "Title 1");
	notification_set_body (n1, "Summary 1");
	notification_set_sender_name (n1, "application1");
	notification_set_sender_pid (n1, 87);
	notification_set_urgency (n1, URGENCY_NORMAL);
	notification_set_actions (n1, actions);

	n2 = notification_new ();
	notification_set_summary (n2, "Title 2");
	notification_set_body (n2, "Summary 2");
	notification_set_sender_name (n2, "application2");
	notification_set_sender_pid (n2, 215);
	notification_set_urgency (n2, URGENCY_NORMAL);
	notification_set_actions (n2, actions);

	n3 = notification_new ();
	notification_set_summary (n3, "Title 3");
	notification_set_body (n3, "Summary 3");
	notification_set_sender_name (n3, "application3");
	notification_set_sender_pid (n3, 121);
	notification_set_urgency (n3, URGENCY_NORMAL);
	notification_set_actions (n3, actions);

	queue_inject (queue, n1);
	queue_inject (queue, n2);
	queue_inject (queue, n3);
	g_assert_cmpuint (queue_get_size (queue), ==, 3);

	sd = queue_find_by_id (queue, notification_get_id (n1));
	g_assert_cmpint (notification_get_id (n1),
	                 ==,
	                 notification_get_id (sd));

	queue_del (queue);
	g_assert (!IS_QUEUE(queue));
}

static void
test_queue_get_size ()
{
	Queue*        queue = NULL;
	Notification* n1    = NULL;
	Notification* n2    = NULL;
	Notification* n3    = NULL;
	Notification* n4    = NULL;
	gchar* actions[]    = {"Accept", "Decline", "Away", "Busy", NULL};

	queue = queue_new ();
	g_assert (queue != NULL);
	g_assert (IS_QUEUE(queue));

	n1 = notification_new ();
	notification_set_id (n1, 0);
	notification_set_summary (n1, "Title 1");
	notification_set_body (n1, "Summary 1");
	notification_set_sender_name (n1, "application1");
	notification_set_sender_pid (n1, 42);
	notification_set_urgency (n1, URGENCY_NORMAL);
	notification_set_actions (n1, actions);

	n2 = notification_new ();
	notification_set_id (n2, 1);
	notification_set_summary (n2, "Title 2");
	notification_set_body (n2, "Summary 2");
	notification_set_sender_name (n2, "application2");
	notification_set_sender_pid (n2, 13);
	notification_set_urgency (n2, URGENCY_NORMAL);
	notification_set_actions (n2, actions);

	n3 = notification_new ();
	notification_set_id (n3, 2);
	notification_set_summary (n3, "Title 3");
	notification_set_body (n3, "Summary 3");
	notification_set_sender_name (n3, "application3");
	notification_set_sender_pid (n3, 321);
	notification_set_urgency (n3, URGENCY_NORMAL);
	notification_set_actions (n3, actions);

	n4 = notification_new ();
	notification_set_id (n4, 3);
	notification_set_summary (n4, "Title 4");
	notification_set_body (n4, "Summary 4");
	notification_set_sender_name (n4, "application4");
	notification_set_sender_pid (n4, 4711);
	notification_set_urgency (n4, URGENCY_NORMAL);
	notification_set_actions (n4, actions);

	queue_inject (queue, n1);
	queue_inject (queue, n2);
	queue_inject (queue, n3);
	queue_inject (queue, n4);
	g_assert_cmpuint (queue_get_size (queue), ==, 4);
	queue_remove_by_id (queue, 0);
	queue_remove_by_id (queue, 1);
	queue_remove_by_id (queue, 2);
	queue_remove_by_id (queue, 3);
	g_assert_cmpuint (queue_get_size (queue), ==, 0);

	queue_del (queue);
	g_assert (!IS_QUEUE(queue));
}

static void
test_queue_is_full ()
{
	Queue*        queue = NULL;
	Notification* n[MAX_QUEUE_SIZE];
	Notification* too_much;

	gchar*        actions[] = {"Accept", "Decline", "Away", "Busy", NULL};
	gint          i = 0;

	queue = queue_new ();
	g_assert (queue != NULL);
	g_assert (IS_QUEUE(queue));

	for (i = 0; i < MAX_QUEUE_SIZE; i++)
	{
		n[i] = notification_new ();
		notification_set_id (n[i], i);
		notification_set_summary (n[i], "Title");
		notification_set_body (n[i], "Summary");
		GString* app_name = g_string_new ("application_");
		g_string_append_printf (app_name, "%d", i);
		notification_set_sender_name (n[i], app_name->str);
		g_string_free (app_name, TRUE);
		notification_set_sender_pid (n[i], i);
		notification_set_urgency (n[i], URGENCY_NORMAL);
		notification_set_actions (n[i], actions);
		queue_inject (queue, n[i]);
	}

	g_assert_cmpuint (queue_get_size (queue), ==, MAX_QUEUE_SIZE);

	too_much = notification_new ();
	notification_set_summary (too_much, "Title");
	notification_set_body (too_much, "Summary");
	notification_set_sender_name (too_much, "application");
	notification_set_sender_pid (too_much, 4711);
	notification_set_urgency (too_much, URGENCY_NORMAL);
	notification_set_actions (too_much, actions);
	queue_inject (queue, too_much);

	g_assert_cmpuint (queue_get_size (queue), ==, MAX_QUEUE_SIZE);

	for (i = 0; i < MAX_QUEUE_SIZE; i++)
		queue_remove_by_id (queue, i);

	g_assert_cmpuint (queue_get_size (queue), ==, 0);

	queue_del (queue);
	g_assert (!IS_QUEUE(queue));
}

/* FIXME: this unit-test is currently disabled, because when used in a real
** world scenario, the PID of the application triggering a notification isn't
** available through DBus */
static void
test_queue_only_one_sd_per_pid ()
{
	Queue*        queue = NULL;
	Notification* n1    = NULL;
	Notification* n2    = NULL;
	gchar* actions[]    = {"Accept", "Decline", "Away", "Busy", NULL};

	queue = queue_new ();
	g_assert (queue != NULL);
	g_assert (IS_QUEUE(queue));

	n1 = notification_new ();
	notification_set_summary (n1, "Title 1");
	notification_set_body (n1, "Summary 1");
	notification_set_sender_name (n1, "application1");
	notification_set_sender_pid (n1, 42);
	notification_set_urgency (n1, URGENCY_NORMAL);
	notification_set_actions (n1, actions);

	n2 = notification_new ();
	notification_set_summary (n2, "Title 2");
	notification_set_body (n2, "Summary 2");
	notification_set_sender_name (n2, "application1");
	notification_set_sender_pid (n2, 42);
	notification_set_urgency (n2, URGENCY_NORMAL);
	notification_set_actions (n2, actions);

	queue_inject (queue, n1);
	queue_inject (queue, n2);
	g_assert_cmpint (notification_get_id (n2), ==, -1);
	g_assert_cmpuint (queue_get_size (queue), ==, 1);
	queue_remove_by_id (queue, notification_get_id (n1));
	g_assert_cmpuint (queue_get_size (queue), ==, 0);

	queue_del (queue);
	g_assert (!IS_QUEUE(queue));
}

static void
test_queue_reorder ()
{
	Queue*        queue = NULL;
	Notification* n1    = NULL;
	Notification* n2    = NULL;
	Notification* n3    = NULL;
	Notification* n4    = NULL;
	Notification* n5    = NULL;
	gchar* actions[]    = {"Accept", "Decline", "Away", "Busy", NULL};

	queue = queue_new ();
	g_assert (queue != NULL);
	g_assert (IS_QUEUE(queue));

	n1 = notification_new ();
	notification_set_summary (n1, "Title 1");
	notification_set_body (n1, "Summary 1");
	notification_set_sender_name (n1, "application1");
	notification_set_sender_pid (n1, 123);
	notification_set_urgency (n1, URGENCY_LOW);
	notification_set_actions (n1, actions);

	n2 = notification_new ();
	notification_set_summary (n2, "Title 2");
	notification_set_body (n2, "Summary 2");
	notification_set_sender_name (n2, "application2");
	notification_set_sender_pid (n2, 43);
	notification_set_urgency (n2, URGENCY_NORMAL);
	notification_set_actions (n2, actions);

	n3 = notification_new ();
	notification_set_summary (n3, "Title 3");
	notification_set_body (n3, "Summary 3");
	notification_set_sender_name (n3, "application3");
	notification_set_sender_pid (n3, 432);
	notification_set_urgency (n3, URGENCY_HIGH);
	notification_set_actions (n3, actions);

	n4 = notification_new ();
	notification_set_summary (n4, "Title 4");
	notification_set_body (n4, "Summary 4");
	notification_set_sender_name (n4, "application4");
	notification_set_sender_pid (n4, 7);
	notification_set_urgency (n4, URGENCY_LOW);
	notification_set_actions (n4, actions);

	n5 = notification_new ();
	notification_set_summary (n5, "Title 5");
	notification_set_body (n5, "Summary 5");
	notification_set_sender_name (n5, "application5");
	notification_set_sender_pid (n5, 72);
	notification_set_urgency (n5, URGENCY_HIGH);
	notification_set_actions (n5, actions);

	queue_inject (queue, n1);
	queue_inject (queue, n2);
	queue_inject (queue, n3);
	queue_inject (queue, n4);
	queue_inject (queue, n5);

	g_assert_cmpstr (notification_get_sender_name (NOTIFICATION (g_list_nth_data (queue->list, 0))),
	                 ==,
	                 "application3");
	g_assert_cmpstr (notification_get_sender_name (NOTIFICATION (g_list_nth_data (queue->list, 1))),
	                 ==,
	                 "application5");
	g_assert_cmpstr (notification_get_sender_name (NOTIFICATION (g_list_nth_data (queue->list, 2))),
	                 ==,
	                 "application2");
	g_assert_cmpstr (notification_get_sender_name (NOTIFICATION (g_list_nth_data (queue->list, 3))),
	                 ==,
	                 "application1");
	g_assert_cmpstr (notification_get_sender_name (NOTIFICATION (g_list_nth_data (queue->list, 4))),
	                 ==,
	                 "application4");

	queue_del (queue);
	g_assert (!IS_QUEUE(queue));
}

static void
test_queue_pop ()
{
	Queue*        queue = NULL;
	Notification* n1    = NULL;
	Notification* n2    = NULL;
	Notification* n3    = NULL;
	Notification* check = NULL;

	queue = queue_new ();
	g_assert (queue != NULL);
	g_assert (IS_QUEUE(queue));

	n1 = notification_new ();
	notification_set_id (n1, 0);
	notification_set_sender_pid (n1, 123);

	n2 = notification_new ();
	notification_set_id (n2, 1);
	notification_set_sender_pid (n2, 534);

	n3 = notification_new ();
	notification_set_id (n3, 2);
	notification_set_sender_pid (n3, 645);

	queue_inject (queue, n1);
	queue_inject (queue, n2);
	queue_inject (queue, n3);
	g_assert_cmpint (queue_get_size (queue), ==, 3);

	check = queue_pop (queue);
	g_assert_cmpint (notification_get_id (check), ==, notification_get_id (n1));

	notification_set_visible (n1);
	check = queue_pop (queue);
	g_assert_cmpint (notification_get_id (check), ==, notification_get_id (n2));

	notification_set_visible (n2);
	check = queue_pop (queue);
	g_assert_cmpint (notification_get_id (check), ==, notification_get_id (n3));

	notification_set_visible (n3);
	check = queue_pop (queue);
	g_assert (check == NULL);

	queue_del (queue);
	g_assert (!IS_QUEUE(queue));
}

GTestSuite *
test_queue_create_test_suite (void)
{
	GTestSuite *ts = NULL;

	ts = g_test_create_suite ("queue");

#define TC(x) g_test_create_case(#x, 0, NULL, NULL, x, NULL)

	g_test_suite_add(ts, TC(test_queue_new));
	g_test_suite_add(ts, TC(test_queue_del));
	g_test_suite_add(ts, TC(test_queue_inject));
	g_test_suite_add(ts, TC(test_queue_remove));
	g_test_suite_add(ts, TC(test_queue_remove_by_pid));
	g_test_suite_add(ts, TC(test_queue_remove_by_id));
	g_test_suite_add(ts, TC(test_queue_find_by_pid));
	g_test_suite_add(ts, TC(test_queue_find_by_id));
	g_test_suite_add(ts, TC(test_queue_get_size));
	g_test_suite_add(ts, TC(test_queue_is_full));
	//g_test_suite_add(ts, TC(test_queue_only_one_sd_per_pid));
	g_test_suite_add(ts, TC(test_queue_reorder));
	g_test_suite_add(ts, TC(test_queue_pop));

	return ts;
}
